from pandas.api.extensions import register_dataframe_accessor, register_series_accessor

from censaurus.geography import Geography
from censaurus.variable import Variable, VariableCollection


@register_dataframe_accessor('census')
class DataFrameCensusAccessor:
    """
    Custom :class:`pandas.DataFrame` accessors. Allows users to access the geography and 
    variables associated with a :class:`pandas.DataFrame` object that was generated
    by ``censaurus``. Properties can be accessed by calling 
    ``dataset.census.<property>``.
    """
    def __init__(self, pandas_obj) -> None:
        self._obj = pandas_obj
        self._geography : Geography = None
        self._variables : VariableCollection = None
    
    @property
    def geography(self) -> Geography:
        """
        Returns the geography hierarchy associated with the :class:`pandas.DataFrame`.
        """
        return self._geography

    @geography.setter
    def geography(self, geography: Geography):
        self._geography = geography

    @property
    def variables(self) -> VariableCollection:
        """
        Returns the variable collection associated with the :class:`pandas.DataFrame`.
        """
        return self._variables

    @variables.setter
    def variables(self, variables: VariableCollection):
        self._variables = variables


@register_series_accessor('census')
class SeriesCensusAccessor:
    """
    Custom :class:`pandas.Series` accessors. Allows users to access the variable 
    associated with a :class:`pandas.Series` object from within a 
    :class:`pandas.DataFrame` that was generated by ``censaurus``. Properties can be 
    accessed by calling ``dataset["column"].census.<property>``.
    """
    def __init__(self, pandas_obj):
        self._obj = pandas_obj
        self._variable : Variable = None

    @property
    def variable(self) -> Variable:
        """
        Returns the variable associated with the :class:`pandas.Series`. 
        """
        return self._variable
    
    @variable.setter
    def variable(self, variable: Variable):
        self._variable = variable