"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const kinesis = require("@aws-cdk/aws-kinesis");
// --------------------------------------------------------------
// Test minimal deployment snapshot
// --------------------------------------------------------------
test('Test minimal deployment snapshot', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {});
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new core_1.Stack();
    const pattern = new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {});
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
    expect(pattern.kinesisStream !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        kinesisStreamProps: {
            shardCount: 1,
            streamName: 'my-stream'
        },
        putRecordRequestTemplate: `{ "Data": "$util.base64Encode($input.json('$.foo'))", "PartitionKey": "$input.path('$.bar')" }`,
        putRecordRequestModel: { schema: {} },
        putRecordsRequestTemplate: `{ "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.foo)", "PartitionKey": "$elem.bar"}#if($foreach.hasNext),#end #end ] }`,
        putRecordsRequestModel: { schema: {} }
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: false,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 1,
        Name: 'my-stream'
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing stream
// --------------------------------------------------------------
test('Test deployment w/ existing stream', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {},
        existingStreamObj: new kinesis.Stream(stack, 'ExistingStream', {
            shardCount: 5,
            retentionPeriod: core_1.Duration.days(4)
        })
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 96
    });
});
//# sourceMappingURL=data:application/json;base64,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