"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionInvokeAction = exports.StateMachineInput = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Represents the input for the StateMachine.
 */
class StateMachineInput {
    constructor(input, inputArtifact, inputType) {
        this.input = input;
        this.inputArtifact = inputArtifact;
        this.inputType = inputType;
    }
    /**
     * When the input type is FilePath, input artifact and filepath must be specified.
     */
    static filePath(inputFile) {
        return new StateMachineInput(inputFile.location, inputFile.artifact, 'FilePath');
    }
    /**
     * When the input type is Literal, input value is passed directly to the state machine input.
     */
    static literal(object) {
        return new StateMachineInput(JSON.stringify(object), undefined, 'Literal');
    }
}
exports.StateMachineInput = StateMachineInput;
/**
 * StepFunctionInvokeAction that is provided by an AWS CodePipeline.
 */
class StepFunctionInvokeAction extends action_1.Action {
    /**
     *
     */
    constructor(props) {
        super({
            ...props,
            resource: props.stateMachine,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'StepFunctions',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 1,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs: (props.stateMachineInput && props.stateMachineInput.inputArtifact) ? [props.stateMachineInput.inputArtifact] : [],
            outputs: (props.output) ? [props.output] : [],
        });
        this.props = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(_scope, _stage, options) {
        var _a, _b, _c, _d, _e;
        // allow pipeline to invoke this step function
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution', 'states:DescribeStateMachine'],
            resources: [this.props.stateMachine.stateMachineArn],
        }));
        // allow state machine executions to be inspected
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution'],
            resources: [cdk.Stack.of(this.props.stateMachine).formatArn({
                    service: 'states',
                    resource: 'execution',
                    resourceName: `${this.props.stateMachine.stateMachineArn}:${(_a = this.props.executionNamePrefix) !== null && _a !== void 0 ? _a : ''}*`,
                    sep: ':',
                })],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if (((_b = this.actionProperties.inputs) !== null && _b !== void 0 ? _b : []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if (((_c = this.actionProperties.outputs) !== null && _c !== void 0 ? _c : []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        return {
            configuration: {
                StateMachineArn: this.props.stateMachine.stateMachineArn,
                Input: (_d = this.props.stateMachineInput) === null || _d === void 0 ? void 0 : _d.input,
                InputType: (_e = this.props.stateMachineInput) === null || _e === void 0 ? void 0 : _e.inputType,
                ExecutionNamePrefix: this.props.executionNamePrefix,
            },
        };
    }
}
exports.StepFunctionInvokeAction = StepFunctionInvokeAction;
//# sourceMappingURL=data:application/json;base64,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