"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 *
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    S3Trigger["NONE"] = "None";
    S3Trigger["POLL"] = "Poll";
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    /**
     *
     */
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node) + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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