"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsDeployAction = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * CodePipeline Action to deploy an ECS Service.
 */
class EcsDeployAction extends action_1.Action {
    /**
     *
     */
    constructor(props) {
        var _a;
        super({
            ...props,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'ECS',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [determineInputArtifact(props)],
            resource: props.service,
        });
        const deploymentTimeout = (_a = props.deploymentTimeout) === null || _a === void 0 ? void 0 : _a.toMinutes({ integral: true });
        if (deploymentTimeout !== undefined && (deploymentTimeout < 1 || deploymentTimeout > 60)) {
            throw new Error(`Deployment timeout must be between 1 and 60 minutes, got: ${deploymentTimeout}`);
        }
        this.props = props;
        this.deploymentTimeout = deploymentTimeout;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(_scope, _stage, options) {
        var _a;
        // permissions based on CodePipeline documentation:
        // https://docs.aws.amazon.com/codepipeline/latest/userguide/how-to-custom-role.html#how-to-update-role-new-services
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'ecs:DescribeTaskDefinition',
                'ecs:DescribeTasks',
                'ecs:ListTasks',
                'ecs:RegisterTaskDefinition',
                'ecs:UpdateService',
            ],
            resources: ['*'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ec2.amazonaws.com',
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        options.bucket.grantRead(options.role);
        return {
            configuration: {
                ClusterName: this.props.service.cluster.clusterName,
                ServiceName: this.props.service.serviceName,
                FileName: (_a = this.props.imageFile) === null || _a === void 0 ? void 0 : _a.fileName,
                DeploymentTimeout: this.deploymentTimeout,
            },
        };
    }
}
exports.EcsDeployAction = EcsDeployAction;
function determineInputArtifact(props) {
    if (props.imageFile && props.input) {
        throw new Error("Exactly one of 'input' or 'imageFile' can be provided in the ECS deploy Action");
    }
    if (props.imageFile) {
        return props.imageFile.artifact;
    }
    if (props.input) {
        return props.input;
    }
    throw new Error("Specifying one of 'input' or 'imageFile' is required for the ECS deploy Action");
}
//# sourceMappingURL=data:application/json;base64,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