"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheQueryStringBehavior = exports.CacheHeaderBehavior = exports.CacheCookieBehavior = exports.CachePolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Cache Policy configuration.
 *
 * @resource AWS::CloudFront::CachePolicy
 * @link https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html
 */
class CachePolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.cachePolicyName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_CachePolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const cachePolicyName = (_e = props.cachePolicyName) !== null && _e !== void 0 ? _e : `${core_1.Names.uniqueId(this).slice(0, 110)}-${core_1.Stack.of(this).region}`;
        if (!core_1.Token.isUnresolved(cachePolicyName) && !cachePolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'cachePolicyName' can only include '-', '_', and alphanumeric characters, got: '${cachePolicyName}'`);
        }
        if (cachePolicyName.length > 128) {
            throw new Error(`'cachePolicyName' cannot be longer than 128 characters, got: '${cachePolicyName.length}'`);
        }
        const minTtl = ((_f = props.minTtl) !== null && _f !== void 0 ? _f : core_1.Duration.seconds(0)).toSeconds();
        const defaultTtl = Math.max(((_g = props.defaultTtl) !== null && _g !== void 0 ? _g : core_1.Duration.days(1)).toSeconds(), minTtl);
        const maxTtl = Math.max(((_h = props.maxTtl) !== null && _h !== void 0 ? _h : core_1.Duration.days(365)).toSeconds(), defaultTtl);
        const resource = new cloudfront_generated_1.CfnCachePolicy(this, 'Resource', {
            cachePolicyConfig: {
                name: cachePolicyName,
                comment: props.comment,
                minTtl,
                maxTtl,
                defaultTtl,
                parametersInCacheKeyAndForwardedToOrigin: this.renderCacheKey(props),
            },
        });
        this.cachePolicyId = resource.ref;
    }
    /** Imports a Cache Policy from its id. */
    static fromCachePolicyId(scope, id, cachePolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.cachePolicyId = cachePolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed cache policy. */
    static fromManagedCachePolicy(managedCachePolicyId) {
        return new class {
            constructor() {
                this.cachePolicyId = managedCachePolicyId;
            }
        }();
    }
    renderCacheKey(props) {
        var _e, _f, _g, _h, _j;
        const cookies = (_e = props.cookieBehavior) !== null && _e !== void 0 ? _e : CacheCookieBehavior.none();
        const headers = (_f = props.headerBehavior) !== null && _f !== void 0 ? _f : CacheHeaderBehavior.none();
        const queryStrings = (_g = props.queryStringBehavior) !== null && _g !== void 0 ? _g : CacheQueryStringBehavior.none();
        return {
            cookiesConfig: {
                cookieBehavior: cookies.behavior,
                cookies: cookies.cookies,
            },
            headersConfig: {
                headerBehavior: headers.behavior,
                headers: headers.headers,
            },
            enableAcceptEncodingGzip: (_h = props.enableAcceptEncodingGzip) !== null && _h !== void 0 ? _h : false,
            enableAcceptEncodingBrotli: (_j = props.enableAcceptEncodingBrotli) !== null && _j !== void 0 ? _j : false,
            queryStringsConfig: {
                queryStringBehavior: queryStrings.behavior,
                queryStrings: queryStrings.queryStrings,
            },
        };
    }
}
exports.CachePolicy = CachePolicy;
_a = JSII_RTTI_SYMBOL_1;
CachePolicy[_a] = { fqn: "@aws-cdk/aws-cloudfront.CachePolicy", version: "1.156.1" };
/**
 * This policy is designed for use with an origin that is an AWS Amplify web app.
 */
CachePolicy.AMPLIFY = CachePolicy.fromManagedCachePolicy('2e54312d-136d-493c-8eb9-b001f22f67d2');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 */
CachePolicy.CACHING_OPTIMIZED = CachePolicy.fromManagedCachePolicy('658327ea-f89d-4fab-a63d-7e88639e58f6');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 * Disables cache compression.
 */
CachePolicy.CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS = CachePolicy.fromManagedCachePolicy('b2884449-e4de-46a7-ac36-70bc7f1ddd6d');
/** Disables caching. This policy is useful for dynamic content and for requests that are not cacheable. */
CachePolicy.CACHING_DISABLED = CachePolicy.fromManagedCachePolicy('4135ea2d-6df8-44a3-9df3-4b5a84be39ad');
/** Designed for use with an origin that is an AWS Elemental MediaPackage endpoint. */
CachePolicy.ELEMENTAL_MEDIA_PACKAGE = CachePolicy.fromManagedCachePolicy('08627262-05a9-4f76-9ded-b50ca2e3a84f');
/**
 * Determines whether any cookies in viewer requests are included in the cache key and
 * automatically included in requests that CloudFront sends to the origin.
 */
class CacheCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in the cache key and
     * are not automatically included in requests that CloudFront sends to the origin.
     */
    static none() { return new CacheCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     */
    static all() { return new CacheCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new CacheCookieBehavior('whitelist', cookies);
    }
    /**
     * All cookies except the provided `cookies` are included in the cache key and
     * automatically included in requests that CloudFront sends to the origin.
     */
    static denyList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to deny must be provided');
        }
        return new CacheCookieBehavior('allExcept', cookies);
    }
}
exports.CacheCookieBehavior = CacheCookieBehavior;
_b = JSII_RTTI_SYMBOL_1;
CacheCookieBehavior[_b] = { fqn: "@aws-cdk/aws-cloudfront.CacheCookieBehavior", version: "1.156.1" };
/**
 * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 */
class CacheHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /** HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. */
    static none() { return new CacheHeaderBehavior('none'); }
    /** Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin. */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new CacheHeaderBehavior('whitelist', headers);
    }
}
exports.CacheHeaderBehavior = CacheHeaderBehavior;
_c = JSII_RTTI_SYMBOL_1;
CacheHeaderBehavior[_c] = { fqn: "@aws-cdk/aws-cloudfront.CacheHeaderBehavior", version: "1.156.1" };
/**
 * Determines whether any URL query strings in viewer requests are included in the cache key
 * and automatically included in requests that CloudFront sends to the origin.
 */
class CacheQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in the cache key and
     * are not automatically included in requests that CloudFront sends to the origin.
     */
    static none() { return new CacheQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     */
    static all() { return new CacheQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new CacheQueryStringBehavior('whitelist', queryStrings);
    }
    /**
     * All query strings except the provided `queryStrings` are included in the cache key and
     * automatically included in requests that CloudFront sends to the origin.
     */
    static denyList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to deny must be provided');
        }
        return new CacheQueryStringBehavior('allExcept', queryStrings);
    }
}
exports.CacheQueryStringBehavior = CacheQueryStringBehavior;
_d = JSII_RTTI_SYMBOL_1;
CacheQueryStringBehavior[_d] = { fqn: "@aws-cdk/aws-cloudfront.CacheQueryStringBehavior", version: "1.156.1" };
//# sourceMappingURL=data:application/json;base64,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