"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
class FakeTarget {
    constructor() {
        this.connections = new aws_ec2_1.Connections({
            peer: aws_ec2_1.Peer.ipv4('666.666.666.666/666')
        });
    }
    attachToClassicLB(_loadBalancer) {
        // Nothing to do. Normally we set a property on ourselves so
        // our instances know to bind to the LB on startup.
    }
}
module.exports = {
    'test specifying nonstandard port works'(test) {
        const stack = new core_1.Stack(undefined, undefined, { env: { account: '1234', region: 'test' } });
        stack.node.setContext('availability-zones:1234:test', ['test-1a', 'test-1b']);
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        const lb = new lib_1.LoadBalancer(stack, 'LB', { vpc });
        lb.addListener({
            externalProtocol: lib_1.LoadBalancingProtocol.HTTP,
            externalPort: 8080,
            internalProtocol: lib_1.LoadBalancingProtocol.HTTP,
            internalPort: 8080,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            Listeners: [{
                    InstancePort: '8080',
                    InstanceProtocol: 'http',
                    LoadBalancerPort: '8080',
                    Protocol: 'http'
                }]
        }));
        test.done();
    },
    'add a health check'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            healthCheck: {
                interval: core_1.Duration.minutes(1),
                path: '/ping',
                protocol: lib_1.LoadBalancingProtocol.HTTPS,
                port: 443,
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            HealthCheck: {
                HealthyThreshold: '2',
                Interval: '60',
                Target: 'HTTPS:443/ping',
                Timeout: '5',
                UnhealthyThreshold: '5'
            },
        }));
        test.done();
    },
    'add a listener and load balancing target'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        const elb = new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            healthCheck: {
                interval: core_1.Duration.minutes(1),
                path: '/ping',
                protocol: lib_1.LoadBalancingProtocol.HTTPS,
                port: 443,
            }
        });
        // WHEN
        elb.addListener({ externalPort: 80, internalPort: 8080 });
        elb.addTarget(new FakeTarget());
        // THEN: at the very least it added a security group rule for the backend
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    Description: 'Port 8080 LB to fleet',
                    CidrIp: '666.666.666.666/666',
                    FromPort: 8080,
                    IpProtocol: 'tcp',
                    ToPort: 8080
                }
            ],
        }));
        test.done();
    },
    'enable cross zone load balancing'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            crossZone: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: true
        }));
        test.done();
    },
    'disable cross zone load balancing'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            crossZone: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: false
        }));
        test.done();
    },
    'cross zone load balancing enabled by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: true
        }));
        test.done();
    },
    'use specified subnet'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP', {
            subnetConfiguration: [
                {
                    name: 'public',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 21
                },
                {
                    name: 'private1',
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    cidrMask: 21
                },
                {
                    name: 'private2',
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    cidrMask: 21
                }
            ],
        });
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            subnetSelection: {
                subnetName: 'private1'
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            Subnets: vpc.selectSubnets({
                subnetName: 'private1'
            }).subnetIds.map((subnetId) => stack.resolve(subnetId))
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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