"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to run AWS Batch
 */
class RunBatchJob {
    constructor(props) {
        var _a;
        this.props = props;
        // validate integrationPattern
        this.integrationPattern =
            props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call RunBatchJob.`);
        }
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`timeout must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is reuqired since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_a = props.containerOverrides) === null || _a === void 0 ? void 0 : _a.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('batch', 'submitJob', this.integrationPattern),
            policyStatements: this.configurePolicyStatements(_task),
            parameters: {
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: this.props.payload,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined
            }
        };
    }
    configurePolicyStatements(task) {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*'
                    }),
                    this.props.jobQueue.jobQueueArn
                ],
                actions: ['batch:SubmitJob']
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule'
                    })
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule']
            })
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`
                }
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: containerOverrides.memory,
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus
        };
    }
}
exports.RunBatchJob = RunBatchJob;
//# sourceMappingURL=data:application/json;base64,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