## `cmlkit.tune.run`: Execution 💥

This module takes care of actually running optimisations (or "searches"). It's implemented as a collection of specialised objects, all tied together with a `Run` instance.

From the parent module (where the exact interfaces are defined), we need:

- `Search`: Something which gives us models to try ("suggestions") based on previous losses we have given it (the results of "trials"). Treated like a black box by this module.
- `Evaluator`: Something which computes losses, with the signature `evalutor(config) -> dict`, which we use to run "trials".

To run an optimisation, we query the `Search` for new suggestions, try them out in parallel on a `Pool` of workers, and submit results back to the search. Each `suggest` or `submit` step is tracked on a sequential `tape`, which we can use to fully reconstruct and restart any given optimisation run.

### Terms

First, let's get the terminology clearly defined.

In this module, we have:
- Suggestion: Emitted by the search, it represents one possible model that should be tried. Normally a `config`, but in general any (serialisable) `dict`.
- Trial: The task of performing the evaluation of such a suggestion. (From the perspective of the search, so to speak.)
- Evaluation: The unique task of evaluating a particular dictionary/config. Since Searches are free to make the same suggestion multiple times, this is not identical to the concept of a Trial. Think of it as the "de-duplicated version" of a Trial, or the act of calling `evaluator(suggestion)`.
- `tid`/`eid`: Trial ID and Evaluation ID. Trial IDs are generated by the Search and must only occur once during a search, Evaluation IDs are simple the hash of a suggestion.
- `tape`: The "trajectory" of the optimisation, a series of steps that, if retraced in the same order with the same arguments, yield an identical optimisation state.
- `result`: The result of an evaluation, with a `state` (`ok` or `error`) and an `outcome` (loss, maybe additional data).

We say "search" as opposed to "optimisation" to reflect the fact that this is fundamentally different from "normal" sequential optimisation methods: There are no guarantees/expectations on the order in which candidate models are evaluated, everything is asynchronous in nature.

### Architecture

- `Run` represents the execution of an optimisation.
- `State` advances and tracks the state of the optimisation.
- `Pool` takes care of distributing the work to many parallel processes. 
- `Tape` is an append-only record of every optimisation step in order.
- `Stop` decides when to stop optimising.
- `ResultDB` is a database of all results, where you can look up which suggestion currently has the lowest loss, how many errors have been tracked, and so on and so on *Žižek sniff*

I'm not sure it's very helpful to write much more about what's going on here beyond this, the code is probably faster to read at this point! I'd suggest taking a look at `State` first, and going from there.

Despite that, here is a short account of what happens during the "event loop" of the optimisation.

At each step, `Run` checks whether the `Pool` has finished any evaluations. If yes, it calls `state.submit` with the result, which write this `submit` event to `tape` and also calls `search.submit`. Internally, the search can then take this information into account when giving future suggestions. Having completed this step, `Run` calls `state.suggest` to get some new trials to run, and forwards the suggestions to the `Pool` for actual evaluation. As `state.suggest` is called, this `suggest` event is also stored on the `tape`.
