#pragma once

#ifdef WIN32
#define __align__(X) __declspec(align(X))
#else
#define __align__(X) __attribute__((aligned(X)))
#endif

template <typename T>
struct Record {
    __align__(OPTIX_SBT_RECORD_ALIGNMENT) char header[OPTIX_SBT_RECORD_HEADER_SIZE];
    T data;
};

struct Empty {
};

typedef Record<Empty>   RayGenSbtRecord;
typedef Record<Empty>     MissSbtRecord;
typedef Record<Empty> HitGroupSbtRecord;

#define OPTIX_CHECK( call )                                                    \
    do                                                                         \
    {                                                                          \
        OptixResult res = call;                                                \
        if( res != OPTIX_SUCCESS )                                             \
        {                                                                      \
            std::stringstream ss;                                              \
            ss << "Optix call '" << #call << "' failed with code("             \
               << res << "): " __FILE__ ":"                                    \
               << __LINE__ << ")\n";                                           \
            fprintf(stderr, "[OptiX Error] %s\n", ss.str().c_str());           \
            return res;                                                        \
        }                                                                      \
    } while( 0 )

#define OPTIX_CHECK_LOG( call )                                                \
    do                                                                         \
    {                                                                          \
        OptixResult res = call;                                                \
        const size_t sizeof_log_returned = logSize;                            \
        logSize = sizeof( log ); /* reset sizeof_log for future calls */       \
        if( res != OPTIX_SUCCESS )                                             \
        {                                                                      \
            std::stringstream ss;                                              \
            ss << "Optix call '" << #call << "' failed with code("             \
            << res << "): " __FILE__ ":"                                       \
               << __LINE__ << ")\nLog:\n" << log                               \
               << ( sizeof_log_returned > sizeof( log ) ? "<TRUNCATED>" : "" ) \
               << "\n";                                                        \
            fprintf(stderr, "[OptiX Error] %s\n", ss.str().c_str());           \
            return res;                                                        \
        }                                                                      \
    } while( 0 )


struct Scene {
    Scene() {
        accelerationStructureHandle = 0;
        memory = 0;
        hash = uint64_t(-1);
    }
    ~Scene() {
        freeMem();
    }
    void freeMem() {
        if (memory) {
            cuMemFree(memory);
            memory = 0;
        }
    }

    uint64_t hash;
    CUdeviceptr memory;
    OptixTraversableHandle accelerationStructureHandle;
};

struct State {
    OptixDeviceContext          context = 0;
    Scene                       scene;

    OptixPipelineCompileOptions pipeline_compile_options = {};
    OptixModule                 ptx_module = 0;
    OptixPipeline               pipeline = 0;

    OptixProgramGroup raygen = 0;
    OptixProgramGroup miss = 0;
    OptixProgramGroup hit = 0;

    OptixShaderBindingTable sbt = {};

    CUdeviceptr d_params = 0;

    CUdeviceptr d_rays = 0;
    size_t d_rays_size = 0;
    CUdeviceptr d_hits = 0;
    size_t d_hits_size = 0;

    struct {
        CUdevice device = 0;
        CUstream stream = 0;
        CUcontext context = 0;
    } cuda;

    bool valid = false;
};

inline void checkCuda(int err, const char* msg, const char* fn, int line) {
    if (!msg) {
        fprintf(stderr, "CUDA Error[%d] at %s[%d]\n", err, fn, line);
    }
    else {
        fprintf(stderr, "CUDA Error[%d] at %s[%d] : %s\n", err, fn, line, msg);
    }
}

#define CHECK_CUDA_LOG(err, msg) \
    do { \
        if (err != CUDA_SUCCESS) { \
            checkCuda(int(err), msg, __FUNCTION__, __LINE__); \
            return err; \
        }\
    } while (false)

#define CHECK_CUDA(err) CHECK_CUDA_LOG(err, "")

// Our GPU source code
static const uint32_t buff[] = {
	0x2f0a2f2f, 0x6547202f, 0x6172656e, 0x20646574, 0x4e207962, 0x49444956, 0x564e2041, 0x43204d56, 0x69706d6f, 0xa72656c, 0x2f0a2f2f, 0x6f43202f, 0x6c69706d, 0x42207265, 0x646c6975, 0x3a444920, 0x2d4c4320, 0x33303033, 0x31313433, 0x202f2f0a, 0x61647543,
	0x6d6f6320, 0x616c6970, 0x6e6f6974, 0x6f6f7420, 0x202c736c, 0x656c6572, 0x20657361, 0x342e3131, 0x3156202c, 0x2e342e31, 0x2f0a3834, 0x6142202f, 0x20646573, 0x4e206e6f, 0x204d5656, 0x2e302e37, 0x2f2f0a31, 0x762e0a0a, 0x69737265, 0x37206e6f,
	0x2e0a342e, 0x67726174, 0x73207465, 0x30355f6d, 0x64612e0a, 0x73657264, 0x69735f73, 0x3620657a, 0x90a0a34, 0x2e202f2f, 0x626f6c67, 0x5f5f096c, 0x67796172, 0x5f5f6e65, 0x6e69616d, 0x6f632e0a, 0x2074736e, 0x696c612e, 0x38206e67, 0x38622e20,
	0x72617020, 0x5b736d61, 0x3b5d3432, 0x762e0a0a, 0x62697369, 0x2e20656c, 0x72746e65, 0x5f5f2079, 0x67796172, 0x5f5f6e65, 0x6e69616d, 0x7b0a2928, 0x722e090a, 0x2e206765, 0x20323366, 0x3c662509, 0x3b3e3031, 0x722e090a, 0x2e206765, 0x20323362,
	0x3c722509, 0x3b3e3632, 0x722e090a, 0x2e206765, 0x20343662, 0x64722509, 0x3e30313c, 0xa0a0a3b, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120, 0x6163090a, 0x28206c6c, 0x29317225, 0x6f5f202c, 0x78697470, 0x7465675f, 0x75616c5f,
	0x5f68636e, 0x65646e69, 0x2c785f78, 0x3b292820, 0x2f2f090a, 0x646e6520, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120, 0x6163090a, 0x28206c6c, 0x29327225, 0x6f5f202c, 0x78697470, 0x7465675f,
	0x75616c5f, 0x5f68636e, 0x65646e69, 0x2c795f78, 0x3b292820, 0x2f2f090a, 0x646e6520, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120, 0x6163090a, 0x28206c6c, 0x29337225, 0x6f5f202c, 0x78697470,
	0x7465675f, 0x75616c5f, 0x5f68636e, 0x65646e69, 0x2c7a5f78, 0x3b292820, 0x2f2f090a, 0x646e6520, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120, 0x6163090a, 0x28206c6c, 0x29347225, 0x6f5f202c,
	0x78697470, 0x7465675f, 0x75616c5f, 0x5f68636e, 0x656d6964, 0x6f69736e, 0x2c785f6e, 0x3b292820, 0x2f2f090a, 0x646e6520, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120, 0x6163090a, 0x28206c6c,
	0x29357225, 0x6f5f202c, 0x78697470, 0x7465675f, 0x75616c5f, 0x5f68636e, 0x656d6964, 0x6f69736e, 0x2c795f6e, 0x3b292820, 0x2f2f090a, 0x646e6520, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x64616d09, 0x2e6f6c2e, 0x20323373, 0x32722509, 0x25202c30,
	0x202c3572, 0x2c337225, 0x32722520, 0x6d090a3b, 0x6c2e6461, 0x33732e6f, 0x25092032, 0x2c313272, 0x32722520, 0x25202c30, 0x202c3472, 0x3b317225, 0x646c090a, 0x6e6f632e, 0x752e7473, 0x9203436, 0x32647225, 0x705b202c, 0x6d617261, 0x5d382b73,
	0x63090a3b, 0x2e617476, 0x672e6f74, 0x61626f6c, 0x36752e6c, 0x25092034, 0x2c336472, 0x64722520, 0x90a3b32, 0x2e6c756d, 0x65646977, 0x3233752e, 0x72250920, 0x202c3464, 0x31327225, 0x3233202c, 0x61090a3b, 0x732e6464, 0x9203436, 0x35647225,
	0x7225202c, 0x202c3364, 0x34647225, 0x6c090a3b, 0x6c672e64, 0x6c61626f, 0x3233662e, 0x66250920, 0x5b202c31, 0x35647225, 0x90a3b5d, 0x672e646c, 0x61626f6c, 0x33662e6c, 0x25092032, 0x202c3266, 0x6472255b, 0x5d342b35, 0x6c090a3b, 0x6c672e64,
	0x6c61626f, 0x3233662e, 0x66250920, 0x5b202c33, 0x35647225, 0x3b5d382b, 0x646c090a, 0x6f6c672e, 0x2e6c6162, 0x20323366, 0x37662509, 0x255b202c, 0x2b356472, 0x3b5d3231, 0x646c090a, 0x6f6c672e, 0x2e6c6162, 0x20323366, 0x34662509, 0x255b202c,
	0x2b356472, 0x3b5d3631, 0x646c090a, 0x6f6c672e, 0x2e6c6162, 0x20323366, 0x35662509, 0x255b202c, 0x2b356472, 0x3b5d3032, 0x646c090a, 0x6f6c672e, 0x2e6c6162, 0x20323366, 0x36662509, 0x255b202c, 0x2b356472, 0x3b5d3432, 0x646c090a, 0x6f6c672e,
	0x2e6c6162, 0x20323366, 0x38662509, 0x255b202c, 0x2b356472, 0x3b5d3832, 0x646c090a, 0x6e6f632e, 0x752e7473, 0x9203436, 0x31647225, 0x705b202c, 0x6d617261, 0xa3b5d73, 0x766f6d09, 0x3233662e, 0x66250920, 0x30202c39, 0x30303066, 0x30303030,
	0x90a3b30, 0x2e766f6d, 0x20323375, 0x31722509, 0x31202c34, 0x6d090a3b, 0x752e766f, 0x9203233, 0x35317225, 0x3b30202c, 0x2f2f090a, 0x67656220, 0x69206e69, 0x6e696c6e, 0x73612065, 0x63090a6d, 0x206c6c61, 0x37722528, 0x7225202c, 0x25202c38,
	0x202c3972, 0x30317225, 0x5f202c29, 0x6974706f, 0x72745f78, 0x5f656361, 0x28202c34, 0x31647225, 0x6625202c, 0x25202c31, 0x202c3266, 0x2c336625, 0x34662520, 0x6625202c, 0x25202c35, 0x202c3666, 0x2c376625, 0x38662520, 0x6625202c, 0x25202c39,
	0x2c343172, 0x31722520, 0x25202c35, 0x2c353172, 0x31722520, 0x25202c34, 0x2c353172, 0x32722520, 0x25202c32, 0x2c333272, 0x32722520, 0x25202c34, 0x29353272, 0x2f090a3b, 0x6e65202f, 0x6e692064, 0x656e696c, 0x6d736120, 0x646c090a, 0x6e6f632e,
	0x752e7473, 0x9203436, 0x36647225, 0x705b202c, 0x6d617261, 0x36312b73, 0x90a3b5d, 0x61747663, 0x2e6f742e, 0x626f6c67, 0x752e6c61, 0x9203436, 0x37647225, 0x7225202c, 0xa3b3664, 0x6c756d09, 0x6469772e, 0x33752e65, 0x25092032, 0x2c386472,
	0x32722520, 0x31202c31, 0x90a3b36, 0x2e646461, 0x20343673, 0x64722509, 0x25202c39, 0x2c376472, 0x64722520, 0x90a3b38, 0x672e7473, 0x61626f6c, 0x33752e6c, 0x5b092032, 0x39647225, 0x25202c5d, 0xa3b3772, 0x2e747309, 0x626f6c67, 0x752e6c61,
	0x9203233, 0x6472255b, 0x5d342b39, 0x7225202c, 0x90a3b38, 0x672e7473, 0x61626f6c, 0x33752e6c, 0x5b092032, 0x39647225, 0x2c5d382b, 0x39722520, 0x73090a3b, 0x6c672e74, 0x6c61626f, 0x3233752e, 0x255b0920, 0x2b396472, 0x2c5d3231, 0x31722520,
	0x90a3b30, 0x3b746572, 0xa7d0a0a, 0x202f2f09, 0x6f6c672e, 0x5f096c62, 0x73696d5f, 0x6d5f5f73, 0xa737369, 0x7369762e, 0x656c6269, 0x6e652e20, 0x20797274, 0x696d5f5f, 0x5f5f7373, 0x7373696d, 0x7b0a2928, 0x722e090a, 0x2e206765, 0x20323362,
	0x3c722509, 0xa3b3e35, 0x6d090a0a, 0x752e766f, 0x9203233, 0x2c317225, 0x30312d20, 0x33313238, 0x32333430, 0x2f090a3b, 0x6562202f, 0x206e6967, 0x696c6e69, 0x6120656e, 0x90a6d73, 0x6c6c6163, 0x706f5f20, 0x5f786974, 0x5f746573, 0x6c796170,
	0x5f64616f, 0x28202c30, 0x29317225, 0x2f090a3b, 0x6e65202f, 0x6e692064, 0x656e696c, 0x6d736120, 0x6f6d090a, 0x33752e76, 0x25092032, 0x202c3272, 0x35363031, 0x32333533, 0xa3b3631, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120,
	0x6163090a, 0x5f206c6c, 0x6974706f, 0x65735f78, 0x61705f74, 0x616f6c79, 0x2c315f64, 0x72252820, 0xa3b2932, 0x202f2f09, 0x20646e65, 0x696c6e69, 0x6120656e, 0x90a6d73, 0x2e766f6d, 0x20323375, 0x34722509, 0x3b30202c, 0x2f2f090a, 0x67656220,
	0x69206e69, 0x6e696c6e, 0x73612065, 0x63090a6d, 0x206c6c61, 0x74706f5f, 0x735f7869, 0x705f7465, 0x6f6c7961, 0x325f6461, 0x2528202c, 0x3b293472, 0x2f2f090a, 0x646e6520, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x202f2f09, 0x69676562, 0x6e69206e,
	0x656e696c, 0x6d736120, 0x6163090a, 0x5f206c6c, 0x6974706f, 0x65735f78, 0x61705f74, 0x616f6c79, 0x2c335f64, 0x72252820, 0xa3b2934, 0x202f2f09, 0x20646e65, 0x696c6e69, 0x6120656e, 0x90a6d73, 0x3b746572, 0xa7d0a0a, 0x202f2f09, 0x6f6c672e,
	0x5f096c62, 0x6f6c635f, 0x74736573, 0x5f746968, 0x6968635f, 0x762e0a74, 0x62697369, 0x2e20656c, 0x72746e65, 0x5f5f2079, 0x736f6c63, 0x68747365, 0x5f5f7469, 0x74696863, 0x7b0a2928, 0x722e090a, 0x2e206765, 0x20323366, 0x3c662509, 0x3b3e3733,
	0x722e090a, 0x2e206765, 0x20323362, 0x3c722509, 0x3b3e3031, 0x722e090a, 0x2e206765, 0x20343662, 0x64722509, 0x3b3e333c, 0x90a0a0a, 0x62202f2f, 0x6e696765, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x6c616309, 0x2528206c, 0x2c293166, 0x706f5f20,
	0x5f786974, 0x5f746567, 0x5f796172, 0x78616d74, 0x2928202c, 0x2f090a3b, 0x6e65202f, 0x6e692064, 0x656e696c, 0x6d736120, 0x7663090a, 0x7a722e74, 0x74662e69, 0x33752e7a, 0x33662e32, 0x25092032, 0x202c3972, 0x3b316625, 0x2f2f090a, 0x67656220,
	0x69206e69, 0x6e696c6e, 0x73612065, 0x63090a6d, 0x206c6c61, 0x64722528, 0x202c2931, 0x74706f5f, 0x675f7869, 0x675f7465, 0x745f7361, 0x65766172, 0x62617372, 0x685f656c, 0x6c646e61, 0x28202c65, 0x90a3b29, 0x65202f2f, 0x6920646e, 0x6e696c6e,
	0x73612065, 0x2f090a6d, 0x6562202f, 0x206e6967, 0x696c6e69, 0x6120656e, 0x90a6d73, 0x6c6c6163, 0x72252820, 0x202c2931, 0x74706f5f, 0x725f7869, 0x5f646165, 0x6d697270, 0x76697469, 0x64695f65, 0x28202c78, 0x90a3b29, 0x65202f2f, 0x6920646e,
	0x6e696c6e, 0x73612065, 0x2f090a6d, 0x6562202f, 0x206e6967, 0x696c6e69, 0x6120656e, 0x90a6d73, 0x6c6c6163, 0x72252820, 0x202c2932, 0x74706f5f, 0x725f7869, 0x5f646165, 0x5f746273, 0x5f736167, 0x2c786469, 0x3b292820, 0x2f2f090a, 0x646e6520,
	0x6c6e6920, 0x20656e69, 0xa6d7361, 0x202f2f09, 0x69676562, 0x6e69206e, 0x656e696c, 0x6d736120, 0x6163090a, 0x28206c6c, 0x29326625, 0x6f5f202c, 0x78697470, 0x7465675f, 0x7961725f, 0x6d69745f, 0x28202c65, 0x90a3b29, 0x65202f2f, 0x6920646e,
	0x6e696c6e, 0x73612065, 0x2f090a6d, 0x6562202f, 0x206e6967, 0x696c6e69, 0x6120656e, 0x90a6d73, 0x6c6c6163, 0x66252820, 0x25202c33, 0x202c3466, 0x2c356625, 0x36662520, 0x6625202c, 0x25202c37, 0x202c3866, 0x2c396625, 0x31662520, 0x25202c30,
	0x29313166, 0x6f5f202c, 0x78697470, 0x7465675f, 0x6972745f, 0x6c676e61, 0x65765f65, 0x78657472, 0x7461645f, 0x28202c61, 0x31647225, 0x7225202c, 0x25202c31, 0x202c3272, 0x29326625, 0x2f090a3b, 0x6e65202f, 0x6e692064, 0x656e696c, 0x6d736120,
	0x7573090a, 0x74662e62, 0x33662e7a, 0x25092032, 0x2c333166, 0x36662520, 0x6625202c, 0x90a3b33, 0x2e627573, 0x2e7a7466, 0x20323366, 0x31662509, 0x25202c34, 0x202c3766, 0x3b346625, 0x7573090a, 0x74662e62, 0x33662e7a, 0x25092032, 0x2c353166,
	0x38662520, 0x6625202c, 0x90a3b35, 0x2e627573, 0x2e7a7466, 0x20323366, 0x31662509, 0x25202c36, 0x202c3966, 0x3b336625, 0x7573090a, 0x74662e62, 0x33662e7a, 0x25092032, 0x2c373166, 0x31662520, 0x25202c30, 0xa3b3466, 0x62757309, 0x7a74662e,
	0x3233662e, 0x66250920, 0x202c3831, 0x31316625, 0x6625202c, 0x90a3b35, 0x2e6c756d, 0x2e7a7466, 0x20323366, 0x31662509, 0x25202c39, 0x2c343166, 0x31662520, 0x90a3b38, 0x2e6c756d, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c30, 0x2c353166,
	0x31662520, 0x90a3b37, 0x2e627573, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c31, 0x2c393166, 0x32662520, 0x90a3b30, 0x2e6c756d, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c32, 0x2c333166, 0x31662520, 0x90a3b38, 0x2e6c756d, 0x2e7a7466,
	0x20323366, 0x32662509, 0x25202c33, 0x2c353166, 0x31662520, 0x90a3b36, 0x2e627573, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c34, 0x2c323266, 0x32662520, 0x90a3b33, 0x2e6c756d, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c35, 0x2c333166,
	0x31662520, 0x90a3b37, 0x2e6c756d, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c36, 0x2c343166, 0x31662520, 0x90a3b36, 0x2e627573, 0x2e7a7466, 0x20323366, 0x32662509, 0x25202c37, 0x2c353266, 0x32662520, 0x90a3b36, 0x2e6c756d, 0x2e7a7466,
	0x20323366, 0x32662509, 0x25202c38, 0x2c343266, 0x32662520, 0x90a3b34, 0x2e616d66, 0x662e6e72, 0x662e7a74, 0x9203233, 0x39326625, 0x6625202c, 0x202c3132, 0x31326625, 0x6625202c, 0xa3b3832, 0x616d6609, 0x2e6e722e, 0x2e7a7466, 0x20323366,
	0x33662509, 0x25202c30, 0x2c373266, 0x32662520, 0x25202c37, 0x3b393266, 0x7372090a, 0x2e747271, 0x72707061, 0x662e786f, 0x662e7a74, 0x9203233, 0x31336625, 0x6625202c, 0xa3b3033, 0x6c756d09, 0x7a74662e, 0x3233662e, 0x66250920, 0x202c3233,
	0x31336625, 0x6625202c, 0xa3b3132, 0x6c756d09, 0x7a74662e, 0x3233662e, 0x66250920, 0x202c3333, 0x34326625, 0x6625202c, 0xa3b3133, 0x67656e09, 0x7a74662e, 0x3233662e, 0x66250920, 0x202c3433, 0x33336625, 0x6d090a3b, 0x662e6c75, 0x662e7a74,
	0x9203233, 0x35336625, 0x6625202c, 0x202c3133, 0x37326625, 0x63090a3b, 0x722e7476, 0x33662e6e, 0x33752e32, 0x25092032, 0x2c363366, 0x39722520, 0x6d090a3b, 0x622e766f, 0x9203233, 0x2c357225, 0x33662520, 0x90a3b36, 0x62202f2f, 0x6e696765,
	0x6c6e6920, 0x20656e69, 0xa6d7361, 0x6c616309, 0x6f5f206c, 0x78697470, 0x7465735f, 0x7961705f, 0x64616f6c, 0x202c305f, 0x35722528, 0x90a3b29, 0x65202f2f, 0x6920646e, 0x6e696c6e, 0x73612065, 0x6d090a6d, 0x622e766f, 0x9203233, 0x2c367225,
	0x33662520, 0x90a3b32, 0x62202f2f, 0x6e696765, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x6c616309, 0x6f5f206c, 0x78697470, 0x7465735f, 0x7961705f, 0x64616f6c, 0x202c315f, 0x36722528, 0x90a3b29, 0x65202f2f, 0x6920646e, 0x6e696c6e, 0x73612065,
	0x6d090a6d, 0x622e766f, 0x9203233, 0x2c377225, 0x33662520, 0x90a3b34, 0x62202f2f, 0x6e696765, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x6c616309, 0x6f5f206c, 0x78697470, 0x7465735f, 0x7961705f, 0x64616f6c, 0x202c325f, 0x37722528, 0x90a3b29,
	0x65202f2f, 0x6920646e, 0x6e696c6e, 0x73612065, 0x6d090a6d, 0x622e766f, 0x9203233, 0x2c387225, 0x33662520, 0x90a3b35, 0x62202f2f, 0x6e696765, 0x6c6e6920, 0x20656e69, 0xa6d7361, 0x6c616309, 0x6f5f206c, 0x78697470, 0x7465735f, 0x7961705f,
	0x64616f6c, 0x202c335f, 0x38722528, 0x90a3b29, 0x65202f2f, 0x6920646e, 0x6e696c6e, 0x73612065, 0x72090a6d, 0xa3b7465, 0xa0a7d0a, 0x0
};