"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const client_vpn_endpoint_1 = require("./client-vpn-endpoint");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet.
 *
 * @stability stable
 */
var SubnetType;
(function (SubnetType) {
    SubnetType["ISOLATED"] = "Isolated";
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["PRIVATE_ISOLATED"] = "Isolated";
    SubnetType["PRIVATE"] = "Private";
    /**
     * Subnet that routes to the internet (via a NAT gateway), but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE_WITH_NAT"] = "Private";
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * (deprecated) Dependencies for NAT connectivity.
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets.
         *
         * @stability stable
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets.
     *
     * @stability stable
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
        };
    }
    /**
     * Adds a VPN Gateway to this VPC.
     *
     * @stability stable
     */
    enableVpnGateway(options) {
        var _e;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_e = options.vpnRoutePropagation) !== null && _e !== void 0 ? _e : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC.
     *
     * @stability stable
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new client VPN endpoint to this VPC.
     *
     * @stability stable
     */
    addClientVpnEndpoint(id, options) {
        return new client_vpn_endpoint_1.ClientVpnEndpoint(this, id, {
            ...options,
            vpc: this,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC.
     *
     * @stability stable
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC.
     *
     * @stability stable
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC.
     *
     * @stability stable
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled).
     *
     * @stability stable
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy.
     *
     * @stability stable
     */
    selectSubnetObjects(selection = {}) {
        var _e;
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE_WITH_NAT;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, (_e = selection.subnetFilters) !== null && _e !== void 0 ? _e : []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.PRIVATE_ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE_WITH_NAT]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        var _e;
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length
                ? SubnetType.PRIVATE_WITH_NAT : this.isolatedSubnets.length ? SubnetType.PRIVATE_ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = (_e = placement.subnetFilters) !== null && _e !== void 0 ? _e : [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 *
 * @stability stable
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    DefaultInstanceTenancy["DEFAULT"] = "default";
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud.
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *    cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *    subnetType: ec2.SubnetType.PRIVATE_WITH_NAT
 * });
 *
 * for (const subnet of selection.subnets) {
 *    // ...
 * }
 * ```
 *
 * @stability stable
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     *
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     *
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _e;
        super(scope, id);
        /**
         * List of public subnets in this VPC.
         *
         * @stability stable
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC.
         *
         * @stability stable
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC.
         *
         * @stability stable
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcProps(props);
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = (_e = props.maxAzs) !== null && _e !== void 0 ? _e : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, stack);
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.PRIVATE_ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import a VPC by supplying all attributes directly.
     *
     * NOTE: using `fromVpcAttributes()` with deploy-time parameters (like a `Fn.importValue()` or
     * `CfnParameter` to represent a list of subnet IDs) sometimes accidentally works. It happens
     * to work for constructs that need a list of subnets (like `AutoScalingGroup` and `eks.Cluster`)
     * but it does not work for constructs that need individual subnets (like
     * `Instance`). See https://github.com/aws/aws-cdk/issues/4118 for more
     * information.
     *
     * Prefer to use `Vpc.fromLookup()` instead.
     *
     * @stability stable
     */
    static fromVpcAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcAttributes(attrs);
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * @stability stable
     */
    static fromLookup(scope, id, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_VpcLookupOptions(options);
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const overrides = {};
        if (options.region) {
            overrides.region = options.region;
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                ...overrides,
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * (deprecated) Adds a new S3 gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.Vpc#addS3Endpoint", "use `addGatewayEndpoint()` instead");
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * (deprecated) Adds a new DynamoDB gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-ec2.Vpc#addDynamoDbEndpoint", "use `addGatewayEndpoint()` instead");
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaining = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaining);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE_WITH_NAT:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.PRIVATE_ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
exports.Vpc = Vpc;
_a = JSII_RTTI_SYMBOL_1;
Vpc[_a] = { fqn: "@aws-cdk/aws-ec2.Vpc", version: "1.134.0" };
/**
 * The default CIDR range used when creating VPCs.
 *
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 *
 * @stability stable
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration.
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 *
 * @stability stable
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_WITH_NAT,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE_WITH_NAT),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0.
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 *
 * @stability stable
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE_ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE_ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE_WITH_NAT: return 'Private';
        case SubnetType.PRIVATE_ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource.
 *
 * @stability stable
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet.
         *
         * @stability stable
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_SubnetProps(props);
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        this.subnetOutpostArn = subnet.attrOutpostArn;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.string({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    /**
     * @stability stable
     */
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    /**
     * @stability stable
     */
    static fromSubnetAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_SubnetAttributes(attrs);
        return new ImportedSubnet(scope, id, attrs);
    }
    // eslint-disable-next-line @typescript-eslint/no-shadow
    /**
     * Import existing subnet from id.
     *
     * @stability stable
     */
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC.
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency.
     * @stability stable
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet.
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explicit ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     *
     * @stability stable
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatewayId.
     *
     * @param natGatewayId The ID of the NAT gateway.
     * @stability stable
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table.
     *
     * @stability stable
     */
    addRoute(id, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_AddRouteOptions(options);
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    /**
     * Associate a Network ACL with this subnet.
     *
     * @stability stable
     */
    associateNetworkAcl(id, networkAcl) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_INetworkAcl(networkAcl);
        this._networkAcl = networkAcl;
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
_b = JSII_RTTI_SYMBOL_1;
Subnet[_b] = { fqn: "@aws-cdk/aws-ec2.Subnet", version: "1.134.0" };
/**
 * Type of router used in route.
 *
 * @stability stable
 */
var RouterType;
(function (RouterType) {
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    RouterType["GATEWAY"] = "Gateway";
    RouterType["INSTANCE"] = "Instance";
    RouterType["NAT_GATEWAY"] = "NatGateway";
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource.
 *
 * @stability stable
 */
class PublicSubnet extends Subnet {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_PublicSubnetProps(props);
    }
    /**
     * @stability stable
     */
    static fromPublicSubnetAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_PublicSubnetAttributes(attrs);
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     *
     * Also adds the EIP for the managed NAT.
     *
     * @returns A ref to the the NAT Gateway ID
     * @stability stable
     */
    addNatGateway(eipAllocationId) {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: eipAllocationId !== null && eipAllocationId !== void 0 ? eipAllocationId : new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
_c = JSII_RTTI_SYMBOL_1;
PublicSubnet[_c] = { fqn: "@aws-cdk/aws-ec2.PublicSubnet", version: "1.134.0" };
/**
 * Represents a private VPC subnet resource.
 *
 * @stability stable
 */
class PrivateSubnet extends Subnet {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnetProps(props);
    }
    /**
     * @stability stable
     */
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_PrivateSubnetAttributes(attrs);
        return new ImportedSubnet(scope, id, attrs);
    }
}
exports.PrivateSubnet = PrivateSubnet;
_d = JSII_RTTI_SYMBOL_1;
PrivateSubnet[_d] = { fqn: "@aws-cdk/aws-ec2.PrivateSubnet", version: "1.134.0" };
function ifUndefined(value, defaultValue) {
    return value !== null && value !== void 0 ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        // None of the values may be unresolved list tokens
        for (const k of Object.keys(props)) {
            if (Array.isArray(props[k]) && core_1.Token.isUnresolved(props[k])) {
                core_1.Annotations.of(this).addWarning(`fromVpcAttributes: '${k}' is a list token: the imported VPC will not work with constructs that require a list of subnets at synthesis time. Use 'Vpc.fromLookup()' or 'Fn.importListValue' instead.`);
            }
        }
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE_WITH_NAT, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.PRIVATE_ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.vpcArn = core_1.Arn.format({
            service: 'ec2',
            resource: 'vpc',
            resourceName: this.vpcId,
        }, core_1.Stack.of(this));
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
                ipv4CidrBlock: vpcSubnet.cidr,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            // The following looks a little weird, but comes down to:
            //
            // * Is the subnetId itself unresolved ({ Ref: Subnet }); or
            // * Was it the accidentally extracted first element of a list-encoded
            //   token? ({ Fn::ImportValue: Subnets } => ['#{Token[1234]}'] =>
            //   '#{Token[1234]}'
            //
            // There's no other API to test for the second case than to the string back into
            // a list and see if the combination is Unresolved.
            //
            // In both cases we can't output the subnetId literally into the metadata (because it'll
            // be useless). In the 2nd case even, if we output it to metadata, the `resolve()` call
            // that gets done on the metadata will even `throw`, because the '#{Token}' value will
            // occur in an illegal position (not in a list context).
            const ref = core_1.Token.isUnresolved(attrs.subnetId) || core_1.Token.isUnresolved([attrs.subnetId])
                ? `at '${constructs_1.Node.of(scope).path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE_WITH_NAT && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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