# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/optimization.ipynb.

# %% auto 0
__all__ = ['mlforecast_objective']

# %% ../nbs/optimization.ipynb 3
import copy
from typing import Any, Callable, Dict, Optional, Union

import numpy as np
import optuna
import utilsforecast.processing as ufp
from sklearn.base import BaseEstimator, clone
from utilsforecast.compat import DataFrame

from . import MLForecast
from .compat import CatBoostRegressor
from .core import Freq

# %% ../nbs/optimization.ipynb 4
_TrialToConfig = Callable[[optuna.Trial], Dict[str, Any]]

# %% ../nbs/optimization.ipynb 5
def mlforecast_objective(
    df: DataFrame,
    config_fn: _TrialToConfig,
    loss: Callable,
    model: BaseEstimator,
    freq: Freq,
    n_windows: int,
    h: int,
    step_size: Optional[int] = None,
    input_size: Optional[int] = None,
    refit: Union[bool, int] = False,
    id_col: str = "unique_id",
    time_col: str = "ds",
    target_col: str = "y",
) -> Callable[[optuna.Trial], float]:
    """optuna objective function for the MLForecast class

    Parameters
    ----------
    config_fn : callable
        Function that takes an optuna trial and produces a configuration with the following keys:
        - model_params
        - mlf_init_params
        - mlf_fit_params
    loss : callable
        Function that takes the validation and train dataframes and produces a float.
    model : BaseEstimator
        scikit-learn compatible model to be trained
    freq : str or int
        pandas' or polars' offset alias or integer denoting the frequency of the series.
    n_windows : int
        Number of windows to evaluate.
    h : int
        Forecast horizon.
    step_size : int, optional (default=None)
        Step size between each cross validation window. If None it will be equal to `h`.
    input_size : int, optional (default=None)
        Maximum training samples per serie in each window. If None, will use an expanding window.
    refit : bool or int (default=False)
        Retrain model for each cross validation window.
        If False, the models are trained at the beginning and then used to predict each window.
        If positive int, the models are retrained every `refit` windows.
    id_col : str (default='unique_id')
        Column that identifies each serie.
    time_col : str (default='ds')
        Column that identifies each timestep, its values can be timestamps or integers.
    target_col : str (default='y')
        Column that contains the target.
    study_kwargs : dict, optional (default=None)
    """

    def objective(trial: optuna.Trial) -> float:
        config = config_fn(trial)
        trial.set_user_attr("config", copy.deepcopy(config))
        if all(
            config["mlf_init_params"].get(k, None) is None
            for k in ["lags", "lag_transforms", "date_features"]
        ):
            # no features
            return np.inf
        splits = ufp.backtest_splits(
            df,
            n_windows=n_windows,
            h=h,
            id_col=id_col,
            time_col=time_col,
            freq=freq,
            step_size=step_size,
            input_size=input_size,
        )
        model_copy = clone(model)
        model_params = config["model_params"]
        if config["mlf_fit_params"].get("static_features", []) and isinstance(
            model, CatBoostRegressor
        ):
            # catboost needs the categorical features in the init signature
            # we assume all statics are categoricals
            model_params["cat_features"] = config["mlf_fit_params"]["static_features"]
        model_copy.set_params(**config["model_params"])
        metrics = []
        mlf = MLForecast(
            models={"model": model_copy},
            freq=freq,
            **config["mlf_init_params"],
        )
        for i, (_, train, valid) in enumerate(splits):
            should_fit = i == 0 or (refit > 0 and i % refit == 0)
            if should_fit:
                mlf.fit(
                    train,
                    id_col=id_col,
                    time_col=time_col,
                    target_col=target_col,
                    **config["mlf_fit_params"],
                )
            static = [c for c in mlf.ts.static_features_.columns if c != id_col]
            dynamic = [
                c
                for c in valid.columns
                if c not in static + [id_col, time_col, target_col]
            ]
            if dynamic:
                X_df: Optional[DataFrame] = ufp.drop_columns(
                    valid, static + [target_col]
                )
            else:
                X_df = None
            preds = mlf.predict(
                h=h,
                X_df=X_df,
                new_df=None if should_fit else train,
            )
            result = ufp.join(
                valid[[id_col, time_col, target_col]],
                preds,
                on=[id_col, time_col],
            )
            if result.shape[0] < valid.shape[0]:
                raise ValueError(
                    "Cross validation result produced less results than expected. "
                    "Please verify that the passed frequency (freq) matches your series' "
                    "and that there aren't any missing periods."
                )
            metric = loss(result, train_df=train)
            metrics.append(metric)
            trial.report(metric, step=i)
            if trial.should_prune():
                raise optuna.TrialPruned()
        return np.mean(metrics).item()

    return objective
