# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/auto.ipynb.

# %% auto 0
__all__ = ['lightgbm_space', 'xgboost_space', 'catboost_space', 'linear_regression_space', 'ridge_space', 'lasso_space',
           'elastic_net_space', 'random_forest_space', 'AutoModel', 'AutoLightGBM', 'AutoXGBoost', 'AutoCatboost',
           'AutoLinearRegression', 'AutoRidge', 'AutoLasso', 'AutoElasticNet', 'AutoRandomForest', 'AutoMLForecast']

# %% ../nbs/auto.ipynb 3
import warnings
from pathlib import Path
from typing import Any, Callable, Dict, List, Optional, Union

import numpy as np
import optuna
import utilsforecast.processing as ufp
from sklearn.base import BaseEstimator, clone
from sklearn.preprocessing import FunctionTransformer
from utilsforecast.compat import DataFrame
from utilsforecast.losses import smape
from utilsforecast.validation import validate_freq

from . import MLForecast
from .core import Freq, _get_model_name, _name_models
from .lag_transforms import ExponentiallyWeightedMean, RollingMean
from .optimization import _TrialToConfig, mlforecast_objective
from mlforecast.target_transforms import (
    Differences,
    LocalStandardScaler,
    GlobalSklearnTransformer,
)
from .utils import PredictionIntervals

# %% ../nbs/auto.ipynb 4
def lightgbm_space(trial: optuna.Trial):
    return {
        "bagging_freq": 1,
        "learning_rate": 0.05,
        "verbosity": -1,
        "n_estimators": trial.suggest_int("n_estimators", 20, 1000, log=True),
        "lambda_l1": trial.suggest_float("lambda_l1", 1e-8, 10.0, log=True),
        "lambda_l2": trial.suggest_float("lambda_l2", 1e-8, 10.0, log=True),
        "num_leaves": trial.suggest_int("num_leaves", 2, 4096, log=True),
        "feature_fraction": trial.suggest_float("feature_fraction", 0.5, 1.0),
        "bagging_fraction": trial.suggest_float("bagging_fraction", 0.5, 1.0),
        "objective": trial.suggest_categorical("objective", ["l1", "l2"]),
    }


def xgboost_space(trial: optuna.Trial):
    return {
        "n_estimators": trial.suggest_int("n_estimators", 20, 1000),
        "max_depth": trial.suggest_int("max_depth", 1, 10),
        "learning_rate": trial.suggest_float("learning_rate", 1e-3, 0.2, log=True),
        "subsample": trial.suggest_float("subsample", 0.1, 1.0),
        "colsample_bytree": trial.suggest_float("colsample_bytree", 0.1, 1.0),
        "reg_lambda": trial.suggest_float("reg_lambda", 1e-8, 1.0, log=True),
        "reg_alpha": trial.suggest_float("reg_alpha", 1e-8, 1.0, log=True),
        "min_child_weight": trial.suggest_int("min_child_weight", 2, 10),
    }


def catboost_space(trial: optuna.Trial):
    return {
        "silent": True,
        "n_estimators": trial.suggest_int("n_estimators", 50, 1000),
        "depth": trial.suggest_int("depth", 1, 10),
        "learning_rate": trial.suggest_float("learning_rate", 1e-3, 0.2, log=True),
        "subsample": trial.suggest_float("subsample", 0.1, 1.0),
        "colsample_bylevel": trial.suggest_float("colsample_bylevel", 0.1, 1.0),
        "min_data_in_leaf": trial.suggest_float("min_data_in_leaf", 1, 100),
    }


def linear_regression_space(trial: optuna.Trial):
    return {"fit_intercept": trial.suggest_categorical("fit_intercept", [True, False])}


def ridge_space(trial: optuna.Trial):
    return {
        "fit_intercept": trial.suggest_categorical("fit_intercept", [True, False]),
        "alpha": trial.suggest_float("alpha", 0.001, 10.0),
    }


def lasso_space(trial: optuna.Trial):
    return {
        "fit_intercept": trial.suggest_categorical("fit_intercept", [True, False]),
        "alpha": trial.suggest_float("alpha", 0.001, 10.0),
    }


def elastic_net_space(trial: optuna.Trial):
    return {
        "fit_intercept": trial.suggest_categorical("fit_intercept", [True, False]),
        "alpha": trial.suggest_float("alpha", 0.001, 10.0),
        "l1_ratio": trial.suggest_float("l1_ratio", 0.0, 1.0),
    }


def random_forest_space(trial: optuna.Trial):
    return {
        "n_estimators": trial.suggest_int("n_estimators", 50, 1000),
        "max_depth": trial.suggest_int("max_depth", 1, 10),
        "min_samples_split": trial.suggest_int("min_samples_split", 2, 100),
        "max_features": trial.suggest_float("max_features", 0.5, 1.0),
        "criterion": trial.suggest_categorical(
            "criterion", ["squared_error", "absolute_error"]
        ),
    }


class AutoModel:
    """Structure to hold a model and its search space

    Parameters
    ----------
    model : BaseEstimator
        scikit-learn compatible regressor
    config : callable
        function that takes an optuna trial and produces a configuration
    """

    def __init__(
        self,
        model: BaseEstimator,
        config: _TrialToConfig,
    ):
        self.model = model
        self.config = config

    def __repr__(self):
        return f"AutoModel(model={_get_model_name(self.model)})"


class AutoLightGBM(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from mlforecast.compat import LGBMRegressor

        super().__init__(
            LGBMRegressor(),
            config if config is not None else lightgbm_space,
        )


class AutoXGBoost(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from mlforecast.compat import XGBRegressor

        super().__init__(
            XGBRegressor(),
            config if config is not None else xgboost_space,
        )


class AutoCatboost(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from mlforecast.compat import CatBoostRegressor

        super().__init__(
            CatBoostRegressor(),
            config if config is not None else catboost_space,
        )


class AutoLinearRegression(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from sklearn.linear_model import LinearRegression

        super().__init__(
            LinearRegression(),
            config if config is not None else linear_regression_space,
        )


class AutoRidge(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from sklearn.linear_model import Ridge

        super().__init__(
            Ridge(),
            config if config is not None else ridge_space,
        )


class AutoLasso(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from sklearn.linear_model import Lasso

        super().__init__(
            Lasso(),
            config if config is not None else lasso_space,
        )


class AutoElasticNet(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from sklearn.linear_model import ElasticNet

        super().__init__(
            ElasticNet(),
            config if config is not None else elastic_net_space,
        )


class AutoRandomForest(AutoModel):
    def __init__(
        self,
        config: Optional[_TrialToConfig] = None,
    ):
        from sklearn.ensemble import RandomForestRegressor

        super().__init__(
            RandomForestRegressor(),
            config if config is not None else random_forest_space,
        )

# %% ../nbs/auto.ipynb 7
class AutoMLForecast:
    """Hyperparameter optimization helper

    Parameters
    ----------
    models : list or dict
        Auto models to be optimized.
    freq : str or int
        pandas' or polars' offset alias or integer denoting the frequency of the series.
    season_length : int, optional (default=None)
        Length of the seasonal period. This is used for producing the feature space.
        Only required if `init_config` is None.
    init_config : callable, optional (default=None)
        Function that takes an optuna trial and produces a configuration passed to the MLForecast constructor.
    fit_config : callable, optional (default=None)
        Function that takes an optuna trial and produces a configuration passed to the MLForecast fit method.
    num_threads : int (default=1)
        Number of threads to use when computing the features.
    """

    def __init__(
        self,
        models: Union[List[AutoModel], Dict[str, AutoModel]],
        freq: Freq,
        season_length: Optional[int] = None,
        init_config: Optional[_TrialToConfig] = None,
        fit_config: Optional[_TrialToConfig] = None,
        num_threads: int = 1,
    ):
        self.freq = freq
        if season_length is None and init_config is None:
            raise ValueError(
                "`season_length` is required when `init_config` is not provided."
            )
        if init_config is not None and not callable(init_config):
            raise ValueError("`init_config` must be a function.")
        if season_length is not None and init_config is not None:
            warnings.warn("`season_length` is not used when `init_config` is provided.")
        self.init_config = init_config
        self.season_length = season_length
        if fit_config is not None:
            if not callable(fit_config):
                raise ValueError("`fit_config` must be a function.")
            self.fit_config = fit_config
        else:
            self.fit_config = lambda trial: {}  # noqa: ARG005
        self.num_threads = num_threads
        if isinstance(models, list):
            model_names = _name_models([_get_model_name(m) for m in models])
            models_with_names = dict(zip(model_names, models))
        else:
            models_with_names = models
        self.models = models_with_names

    def __repr__(self):
        return f"AutoMLForecast(models={self.models})"

    def _seasonality_based_config(
        self,
        h: int,
        min_samples: int,
        min_value: float,
    ) -> _TrialToConfig:
        assert self.season_length is not None
        # target transforms
        candidate_targ_tfms: List[Any] = [
            None,
            [LocalStandardScaler()],
            [Differences([1]), LocalStandardScaler()],
        ]
        log1p_tfm = GlobalSklearnTransformer(
            FunctionTransformer(func=np.log1p, inverse_func=np.expm1)
        )
        if min_value >= 0:
            candidate_targ_tfms.extend(
                [
                    [log1p_tfm, LocalStandardScaler()],
                    [log1p_tfm, Differences([1]), LocalStandardScaler()],
                ]
            )
        # we leave two seasonal periods for the features and model
        if self.season_length > 1 and min_samples > 3 * self.season_length + 1:
            candidate_targ_tfms.append(
                [Differences([1, self.season_length]), LocalStandardScaler()]
            )
            if min_value >= 0:
                candidate_targ_tfms.append(
                    [
                        log1p_tfm,
                        Differences([1, self.season_length]),
                        LocalStandardScaler(),
                    ],
                )

        # lags
        candidate_lags = [None, [self.season_length]]
        seasonality2extra_candidate_lags = {
            7: [
                [7, 14],
                [7, 28],
            ],
            12: [range(1, 13)],
            24: [
                range(1, 25),
                range(24, 24 * 7 + 1, 24),
            ],
            52: [
                range(4, 53, 4),
            ],
        }
        if self.season_length in seasonality2extra_candidate_lags:
            candidate_lags.extend(
                seasonality2extra_candidate_lags[self.season_length]  # type: ignore
            )
        if h >= 2 * self.season_length:
            candidate_lags.extend(
                [
                    range(self.season_length, h + 1, self.season_length),  # type: ignore
                    [h],
                    [self.season_length, h],
                ]
            )

        # lag transforms
        candidate_lag_tfms = [None, {1: [ExponentiallyWeightedMean(0.9)]}]
        if self.season_length > 1:
            candidate_lag_tfms.append(
                {
                    1: [ExponentiallyWeightedMean(0.9)],
                    self.season_length: [
                        RollingMean(window_size=self.season_length, min_samples=1),
                    ],
                }
            )
        if self.season_length != h:
            candidate_lag_tfms.append(
                {
                    1: [ExponentiallyWeightedMean(0.9)],
                    self.season_length: [
                        RollingMean(window_size=self.season_length, min_samples=1),
                    ],
                    h: [
                        RollingMean(window_size=self.season_length, min_samples=1),
                    ],
                }
            )

        # date features
        seasonality2date_features = {
            1: ["year"],
            4: ["quarter", "year"],
            7: ["weekday", "month", "year"],
            12: ["month", "year"],
            24: ["hour", "weekday", "month", "year"],
            52: ["week", "year"],
            60: ["weekday", "hour", "second"],
        }
        candidate_date_features = seasonality2date_features.get(self.season_length, [])
        if isinstance(self.freq, int):
            candidate_date_features = []

        def config(trial):
            # target transforms
            targ_tfms_idx = trial.suggest_categorical(
                "target_transforms_idx", range(len(candidate_targ_tfms))
            )
            target_transforms = candidate_targ_tfms[targ_tfms_idx]

            # lags
            lags_idx = trial.suggest_categorical("lags_idx", range(len(candidate_lags)))
            lags = candidate_lags[lags_idx]

            # lag transforms
            if candidate_lag_tfms:
                lag_tfms_idx = trial.suggest_categorical(
                    "lag_transforms_idx", range(len(candidate_lag_tfms))
                )
                lag_transforms = candidate_lag_tfms[lag_tfms_idx]
            else:
                lag_transforms = None

            # date features
            if candidate_date_features:
                use_date_features = trial.suggest_int("use_date_features", 0, 1)
                if use_date_features:
                    date_features = candidate_date_features
                else:
                    date_features = None
            else:
                date_features = None

            return {
                "lags": lags,
                "target_transforms": target_transforms,
                "lag_transforms": lag_transforms,
                "date_features": date_features,
            }

        return config

    def fit(
        self,
        df: DataFrame,
        n_windows: int,
        h: int,
        num_samples: int,
        step_size: Optional[int] = None,
        input_size: Optional[int] = None,
        refit: Union[bool, int] = False,
        loss: Optional[Callable[[DataFrame, DataFrame], float]] = None,
        id_col: str = "unique_id",
        time_col: str = "ds",
        target_col: str = "y",
        study_kwargs: Optional[Dict[str, Any]] = None,
        optimize_kwargs: Optional[Dict[str, Any]] = None,
        fitted: bool = False,
        prediction_intervals: Optional[PredictionIntervals] = None,
    ) -> "AutoMLForecast":
        """Carry out the optimization process.
        Each model is optimized independently and the best one is trained on all data

        Parameters
        ----------
        df : pandas or polars DataFrame
            Series data in long format.
        n_windows : int
            Number of windows to evaluate.
        h : int
            Forecast horizon.
        num_samples : int
            Number of trials to run
        step_size : int, optional (default=None)
            Step size between each cross validation window. If None it will be equal to `h`.
        input_size : int, optional (default=None)
            Maximum training samples per serie in each window. If None, will use an expanding window.
        refit : bool or int (default=False)
            Retrain model for each cross validation window.
            If False, the models are trained at the beginning and then used to predict each window.
            If positive int, the models are retrained every `refit` windows.
        loss : callable, optional (default=None)
            Function that takes the validation and train dataframes and produces a float.
            If `None` will use the average SMAPE across series.
        id_col : str (default='unique_id')
            Column that identifies each serie.
        time_col : str (default='ds')
            Column that identifies each timestep, its values can be timestamps or integers.
        target_col : str (default='y')
            Column that contains the target.
        study_kwargs : dict, optional (default=None)
            Keyword arguments to be passed to the optuna.Study constructor.
        optimize_kwargs : dict, optional (default=None)
            Keyword arguments to be passed to the optuna.Study.optimize method.
        fitted : bool (default=False)
            Whether to compute the fitted values when retraining the best model.
        prediction_intervals :
            Configuration to calibrate prediction intervals when retraining the best model.

        Returns
        -------
        AutoMLForecast
            object with best models and optimization results
        """
        validate_freq(df[time_col], self.freq)
        if self.init_config is not None:
            init_config = self.init_config
        else:
            min_size = ufp.counts_by_id(df, id_col)["counts"].min()
            min_train_size = min_size - n_windows * h
            init_config = self._seasonality_based_config(
                h=h,
                min_samples=min_train_size,
                min_value=df[target_col].min(),
            )

        if loss is None:

            def loss(df, train_df):  # noqa: ARG001
                return smape(
                    df,
                    models=["model"],
                    id_col=id_col,
                    target_col=target_col,
                )["model"].mean()

        if study_kwargs is None:
            study_kwargs = {}
        if "sampler" not in study_kwargs:
            # for reproducibility
            study_kwargs["sampler"] = optuna.samplers.TPESampler(seed=0)
        if optimize_kwargs is None:
            optimize_kwargs = {}

        self.results_ = {}
        self.models_ = {}
        for name, auto_model in self.models.items():

            def config_fn(trial: optuna.Trial) -> Dict[str, Any]:
                return {
                    "model_params": auto_model.config(trial),
                    "mlf_init_params": {
                        **init_config(trial),
                        "num_threads": self.num_threads,
                    },
                    "mlf_fit_params": self.fit_config(trial),
                }

            objective = mlforecast_objective(
                df=df,
                config_fn=config_fn,
                loss=loss,
                model=auto_model.model,
                freq=self.freq,
                n_windows=n_windows,
                h=h,
                step_size=step_size,
                input_size=input_size,
                refit=refit,
                id_col=id_col,
                time_col=time_col,
                target_col=target_col,
            )
            study = optuna.create_study(direction="minimize", **study_kwargs)
            study.optimize(objective, n_trials=num_samples, **optimize_kwargs)
            self.results_[name] = study
            best_config = study.best_trial.user_attrs["config"]
            for arg in (
                "fitted",
                "prediction_intervals",
                "id_col",
                "time_col",
                "target_col",
            ):
                best_config["mlf_fit_params"].pop(arg, None)
            best_model = clone(auto_model.model)
            best_model.set_params(**best_config["model_params"])
            self.models_[name] = MLForecast(
                models={name: best_model},
                freq=self.freq,
                **best_config["mlf_init_params"],
            )
            self.models_[name].fit(
                df,
                fitted=fitted,
                prediction_intervals=prediction_intervals,
                id_col=id_col,
                time_col=time_col,
                target_col=target_col,
                **best_config["mlf_fit_params"],
            )
        return self

    def predict(
        self,
        h: int,
        X_df: Optional[DataFrame] = None,
        level: Optional[List[Union[int, float]]] = None,
    ) -> DataFrame:
        """ "Compute forecasts

        Parameters
        ----------
        h : int
            Number of periods to predict.
        X_df : pandas or polars DataFrame, optional (default=None)
            Dataframe with the future exogenous features. Should have the id column and the time column.
        level : list of ints or floats, optional (default=None)
            Confidence levels between 0 and 100 for prediction intervals.

        Returns
        -------
        pandas or polars DataFrame
            Predictions for each serie and timestep, with one column per model.
        """
        all_preds = None
        for name, model in self.models_.items():
            preds = model.predict(h=h, X_df=X_df, level=level)
            if all_preds is None:
                all_preds = preds
            else:
                model_cols = [c for c in preds.columns if c not in all_preds.columns]
                all_preds = ufp.horizontal_concat([all_preds, preds[model_cols]])
        return all_preds

    def save(self, path: Union[str, Path]) -> None:
        """Save AutoMLForecast objects

        Parameters
        ----------
        path : str or pathlib.Path
            Directory where artifacts will be stored."""
        for name, model in self.models_.items():
            model.save(f"{path}/{name}")

    def forecast_fitted_values(
        self,
        level: Optional[List[Union[int, float]]] = None,
    ) -> DataFrame:
        """Access in-sample predictions.

        Parameters
        ----------
        level : list of ints or floats, optional (default=None)
            Confidence levels between 0 and 100 for prediction intervals.

        Returns
        -------
        pandas or polars DataFrame
            Dataframe with predictions for the training set
        """
        fitted_vals = None
        for name, model in self.models_.items():
            model_fitted = model.forecast_fitted_values(level=level)
            if fitted_vals is None:
                fitted_vals = model_fitted
            else:
                fitted_vals = ufp.join(
                    fitted_vals,
                    ufp.drop_columns(model_fitted, model.ts.target_col),
                    on=[model.ts.id_col, model.ts.time_col],
                    how="inner",
                )
        return fitted_vals
