# -*- coding: utf-8 -*-
# Author: Felipe Bogaerts de Mattos
# Contact me at felipe.bogaerts@engenharia.ufjf.br.
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.

"""
Stores utility functions, used in various locations throughtout the 
application.
"""

from pathlib import Path

import numpy as np


def generate_eng(
    time: np.ndarray,
    thrust: np.ndarray,
    propellant_mass: np.ndarray,
    name: str,
    manufacturer: str,
    chamber_length: float,
    outer_diameter: float,
    motor_mass: float,
    eng_res: int = 25,
):
    # Forming a new time vector that has exactly 'eng_res' length:
    t_out = np.linspace(0, time[-1], eng_res)
    # Interpolating old thrust-time data into new time vector:
    thrust_out = np.interp(t_out, time, thrust, left=0, right=0)

    eng_header = (
        f"{name} {outer_diameter * 1e3:.4f} {chamber_length * 1e3:.4f} P "
        f"{propellant_mass[0]:.4f} {propellant_mass[0] + motor_mass:.4f} "
        f"{manufacturer}\n"
    )

    with open(Path("output", f"{name}.eng"), "w") as file:
        # Header:
        file.write(
            "; Generated by RocketSolver program written by Felipe Bogaerts de Mattos\n; Juiz de Fora, Brasil\n"
        )
        file.write(eng_header)

        # Content:
        for i in range(eng_res):
            file.write("   %.2f %.0f\n" % ((t_out[i]), (thrust_out[i])))

        file.write(";")


def motor_to_eng(
    t,
    F,
    dt,
    V_prop_CP,
    D_out,
    L_chamber,
    eng_res,
    pp,
    m_motor,
    manufacturer,
    name,
):
    """
    Exports an eng file to the directory 'outputs' inside the program folder.
    """

    # Making the received volume array the same length as the time vector.
    V_prop = np.zeros(np.size(t))
    for i, volume in enumerate(V_prop_CP):
        V_prop[i] += volume

    # Forming a new time vector that has exactly 'eng_res' points
    # (independent on time step input):
    t_out = np.linspace(0, t[-1] + dt, eng_res)
    # Interpolating old thrust-time data into new time vector:
    F_out = np.interp(t_out, t, F, left=0, right=0)
    # Interpolating the Propellant volume with the new time vector
    # (to find propellant. mass with t):
    m_prop_out = pp * np.interp(t_out, t, V_prop, right=0)

    # Writing to the ENG file:
    eng_header = (
        f"{name} {D_out * 1e3:.4f} {L_chamber * 1e3:.4f} P "
        f"{m_prop_out[0]:.4f} {m_prop_out[0] + m_motor:.4f} "
        f"{manufacturer}\n"
    )
    saveFile = open(f"output/{name}.eng", "w")
    saveFile.write(
        "; Generated by RocketSolver program written by Felipe Bogaerts de Mattos\n; Juiz de Fora, Brasil\n"
    )
    saveFile.write(eng_header)
    for i in range(eng_res):
        saveFile.write("   %.2f %.0f\n" % ((t_out[i]), (F_out[i])))
    saveFile.write(";")
    saveFile.close()


def output_eng_csv(
    time: np.ndarray,
    burn_time: float,
    thrust: np.ndarray,
    propellant_volume: np.ndarray,
    dt: float,
    chamber_od: float,
    chamber_length: float,
    eng_resolution: int,
    propellant_density: float,
    motor_dry_mass: float,
    manufacturer: str,
    name: str,
):
    """
    This program exports the motor data into three separate files.
    The .eng file is compatible with most rocket ballistic simulators such as openRocket and RASAero.
    The output .csv file contains thrust, time, propellant mass, Kn, chamber pressure, web thickness and burn rate data.
    The input .csv file contains all info used in the input section.
    """
    # Writing the ENG file:
    index = np.where(time == burn_time)
    time = time[: index[0][0]]
    thrust = thrust[: index[0][0]]
    prop_vol = propellant_volume[: index[0][0]]

    motor_to_eng(
        time,
        thrust,
        dt,
        prop_vol,
        chamber_od,
        chamber_length,
        eng_resolution,
        propellant_density,
        motor_dry_mass,
        manufacturer,
        name,
    )


def obtain_attributes_from_object(obj) -> dict:
    try:
        return vars(obj)
    except TypeError:  # if does not have __dict__ method
        return {}
