'''
Settings
========

Methods described in this section relate to the the Settings API.
These methods can be accessed at ``Nessus.settings``.

.. rst-class:: hide-signature
.. autoclass:: SettingsAPI
    :members:
'''
from typing import List, Dict, Optional
from typing_extensions import Literal
from restfly.utils import dict_clean
from tenable.base.endpoint import APIEndpoint
from .schema.settings import SettingsListSchema


class SettingsAPI(APIEndpoint):
    _path = 'settings'

    def modify(self, settings: List[Dict]) -> Dict:
        '''
        Modifies the advanced settings on the Nessus scanner.  Settings objects
        must contain an action and a name field.  They may also require a value
        field and/or an id field depending on the nature of the change.

        Args:
            settings (list[dict]): List of settings change objects

        Examples:

            Adding a new value:

            >>> nessus.settings.modify([{
            ...     'action': 'add',
            ...     'name': 'new_value',
            ...     'value': 'value_contents'
            ... }])

            Updating a default setting value:

            >>> nessus.settings.modify([{
            ...     'action': 'edit',
            ...     'name': 'allow_post_scan_editing',
            ...     'value': 'no'
            ... }])

            Removing a setting:

            >>> nessus.settings.modify([{
            ...     'action': 'remove',
            ...     'name': 'old_setting',
            ...     'id': 'abcdef1234567890abcdef'
            ... }])
        '''
        schema = SettingsListSchema()
        payload = schema.dump(schema.load({'settings': settings}))
        return self._put('advanced', json=payload)

    def list(self) -> List[Dict]:
        '''
        Returns the list of advanced settings

        Returns:
            List[Dict]:
                List of settings objects.

        Example:

            >>> nessus.settings.list()
        '''
        return self._get('advanced')['preferences']

    def health(self) -> Dict:
        '''
        Returns the current health statistics fore the Nessus scanner

        Returns:
            Dict:
                Health stats information

        Example:

            >>> nessus.settings.health()
        '''
        return self._get('health/stats')

    def alerts(self,
               start_time: Optional[int] = None,
               end_time: Optional[int] = None
               ) -> List[Dict]:
        '''
        Returns the list of health alerts generated by the scanner

        Args:
            start_time (int, optional):
                Start time to query the historical data for.  Defaults to 24hrs
                ago.
            end_time (int, optional):
                End time to query the historicat data for.  Defaults to now.

        Returns:
            List[Dict]:
                List of alert objects matching the specified time range

        Example:

            >>> nessus.settings.alerts()
        '''
        return self._get('health/alerts', params=dict_clean({
            'start_time': start_time,
            'end_time': end_time
        }))
