"""Extension that replaces reStructuredText by Markdown"""
from functools import partial, reduce
from textwrap import dedent
from typing import List

from configupdater import ConfigUpdater, Option
from pyscaffold.actions import Action, ActionParams, ScaffoldOpts, Structure
from pyscaffold.extensions import Extension
from pyscaffold.identification import dasherize
from pyscaffold.operations import no_overwrite
from pyscaffold.structure import merge, reify_leaf, reject
from pyscaffold.templates import get_template

from . import templates

__author__ = "Florian Wilhelm"
__copyright__ = "Florian Wilhelm"
__license__ = "MIT"

DESC_KEY = "long_description"
TYPE_KEY = "long_description_content_type"

DOC_REQUIREMENTS = ["myst-parser[linkify]"]


template = partial(get_template, relative_to=templates)


class Markdown(Extension):
    """Replace reStructuredText by Markdown"""

    def activate(self, actions: List[Action]) -> List[Action]:
        """Activate extension. See :obj:`pyscaffold.extension.Extension.activate`."""
        actions = self.register(actions, add_doc_requirements)
        return self.register(actions, replace_files, before="verify_project_dir")


def add_long_desc(content: str) -> str:
    updater = ConfigUpdater()
    updater.read_string(content)
    metadata = updater["metadata"]

    dash_key = dasherize(DESC_KEY)
    default = Option(DESC_KEY)
    long_desc = metadata.get(dash_key) or metadata.setdefault(DESC_KEY, default)
    long_desc.key = DESC_KEY  # dash-separated keys are deprecated in setuptools
    long_desc.value = "file: README.md"
    long_desc_value = "text/markdown; charset=UTF-8; variant=GFM"

    long_desc_type = metadata.get(dasherize(TYPE_KEY)) or metadata.get(TYPE_KEY)
    if long_desc_type:
        long_desc_type.key = TYPE_KEY
        long_desc_type.value = long_desc_value
    else:
        long_desc.add_after.option(TYPE_KEY, long_desc_value)

    return str(updater)


def add_myst(original: str) -> str:
    """Change docs/conf.py to use MyST-Parser, enabling md files"""
    # add myst_parser extension and its own extensions configuration
    content = original.splitlines()
    myst = '\n# Enable markdown\nextensions.append("myst_parser")\n'
    myst_extensions = template("myst_extensions").template  # raw string
    j = next(i for i, line in enumerate(content) if line.startswith("source_suffix ="))
    content[j] = 'source_suffix = [".rst", ".md"]'
    content.insert(j - 1, myst)
    content.insert(j, myst_extensions)
    return "\n".join(content)


def default_myst_include(root_file: str) -> str:
    """Return string which is a MyST include directive pointing to `root_file`,
    located under `..` relative to `docs`. See:
    https://myst-parser.readthedocs.io/en/latest/sphinx/use.html#howto-include-readme
    """
    template_include = dedent(
        """\
        ```{include} ../{root_file}
        :relative-docs: docs/
        :relative-images:
        ```
        """
    )
    return template_include.replace("{root_file}", root_file)


def add_doc_requirements(struct: Structure, opts: ScaffoldOpts) -> ActionParams:
    """In order to build the docs new requirements are necessary now.

    The default ``tox.ini`` generated by PyScaffold should already include
    ``-e {toxinidir}/docs/requirements.txt`` in its dependencies. Therefore,
    this action will make sure ``tox -e docs`` run without problems.

    It is important to sort the requirements otherwise pre-commit will raise an error
    for a newly generated file and that would correspond to a bad user experience.
    """
    leaf = struct.get("docs", {}).get("requirements.txt")
    original, file_op = reify_leaf(leaf, opts)
    contents = original or ""

    missing = [req for req in DOC_REQUIREMENTS if req not in contents]
    requirements = [*contents.splitlines(), *missing]

    # It is not trivial to sort the requirements because they include a comment header
    j = (i for (i, line) in enumerate(requirements) if line and not is_commented(line))
    comments_end = next(j, 0)  # first element of the iterator is a non commented line
    comments = requirements[:comments_end]
    sorted_requirements = sorted(requirements[comments_end:])

    new_contents = "\n".join([*comments, *sorted_requirements]) + "\n"
    # ^  pre-commit requires a new line at the end of the file

    files: Structure = {"docs": {"requirements.txt": (new_contents, file_op)}}

    return merge(struct, files), opts


# NOTE: Avoid renaming/removing replace_files.
#       The dsproject extension depends on that name, any changes in the function
#       signature here should be followed by a PR to that repository.
def replace_files(struct: Structure, opts: ScaffoldOpts) -> ActionParams:
    """Replace all rst files to proper md and activate Sphinx md.
    See :obj:`pyscaffold.actions.Action`
    """
    # Define new files
    NO_OVERWRITE = no_overwrite()
    files: Structure = {
        "README.md": (template("readme"), NO_OVERWRITE),
        "AUTHORS.md": (template("authors"), NO_OVERWRITE),
        "CHANGELOG.md": (template("changelog"), NO_OVERWRITE),
        "CONTRIBUTING.md": (template("contributing"), NO_OVERWRITE),
        "docs": {
            "index.md": (template("index"), NO_OVERWRITE),
            "readme.md": (default_myst_include("README.md"), NO_OVERWRITE),
            "license.md": (template("license"), NO_OVERWRITE),
            "authors.md": (default_myst_include("AUTHORS.md"), NO_OVERWRITE),
            "changelog.md": (default_myst_include("CHANGELOG.md"), NO_OVERWRITE),
            "contributing.md": (default_myst_include("CONTRIBUTING.md"), NO_OVERWRITE),
        },
    }

    # TODO: Automatically convert RST to MD
    #
    # >>> content, file_op = reify_leaf(struct.get("CONTRIBUTING.rst"), opts)
    # >>> md_content = rst_to_myst(content or "", **RST2MYST_OPTS).text
    # >>> files["CONTRIBUTING.md"] = (md_content, file_op)
    #
    # Currently there is a problem in rst-to-myst, preventing automatic conversion:
    # https://github.com/executablebooks/rst-to-myst/issues/33#issuecomment-922264030

    # Modify pre-existing files
    content, file_op = reify_leaf(struct["setup.cfg"], opts)
    files["setup.cfg"] = (add_long_desc(content), file_op)

    content, file_op = reify_leaf(struct["docs"]["conf.py"], opts)
    files["docs"]["conf.py"] = (add_myst(content), file_op)

    # Remove all unnecessary .rst files from struct
    unnecessary = [
        "README.rst",
        "AUTHORS.rst",
        "CHANGELOG.rst",
        "CONTRIBUTING.rst",
        "docs/index.rst",
        "docs/readme.rst",
        "docs/license.rst",
        "docs/authors.rst",
        "docs/changelog.rst",
        "docs/contributing.rst",
    ]
    struct = reduce(reject, unnecessary, struct)

    return merge(struct, files), opts


def is_commented(line):
    return line.strip().startswith("#")
