# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_training.ipynb.

# %% auto 0
__all__ = ['SimODEData', 'create_sim_dataset', 'train']

# %% ../nbs/01_training.ipynb 2
import torch 
import torch.nn as nn
from torchdiffeq import odeint
import pylab as plt
from torch.utils.data import Dataset, DataLoader
from typing import Callable, List, Tuple, Union, Optional
from .models import *

# %% ../nbs/01_training.ipynb 3
class SimODEData(Dataset):
    """ 
        A very simple dataset class for simulating ODEs, really could just use
        a tensor of the values directly but for general use it is very nice to
        define a Dataset class for your data. This can handle batching and 
        other nice things.
    """
    def __init__(self,
                 ts: List[torch.Tensor], # List of time points
                 values: List[torch.Tensor], # List of dynamical state values at each time point 
                 ) -> None:
        self.ts = ts 
        self.values = values 
        
    def __len__(self) -> int:
        return len(self.ts)
    
    def __getitem__(self, index: int) -> Tuple[torch.Tensor, torch.Tensor]:
        return self.ts[index], self.values[index]

      

# %% ../nbs/01_training.ipynb 4
def create_sim_dataset(model: nn.Module, # model to simulate from
                       ts: torch.Tensor, # Time points to simulate at
                       num_samples: int = 10, # Number of samples to generate
                       sigma_noise: float = 0.1, # Noise level to add to the data
                       initial_conditions_default: torch.Tensor = torch.tensor([0.0, 0.0]), # Default initial conditions
                       sigma_initial_conditions: float = 0.1, # Noise level to add to the initial conditions
                       ) -> SimODEData:
    ts_list = [] 
    states_list = [] 
    for i in range(num_samples):
        x0 = sigma_initial_conditions * torch.randn((1,2)).detach() + initial_conditions_default
        ys = odeint(model, x0, ts).squeeze(1).detach() 
        ys += sigma_noise*torch.randn_like(ys)
        ts_list.append(ts)
        states_list.append(ys)
    return SimODEData(ts_list, states_list)
    


# %% ../nbs/01_training.ipynb 5
def train(model: torch.nn.Module, 
          data: SimODEData, 
          lr: float = 1e-2, 
          epochs: int = 10):
    trainloader = DataLoader(data, batch_size=5, shuffle=True)
    optimizer = torch.optim.Adam(model.parameters(), lr=1e-2)
    criterion = torch.nn.MSELoss()
    
    for epoch in range(epochs):
        running_loss = 0.0 
        for data in trainloader:
            optimizer.zero_grad()
            ts, states = data 
            initial_state = states[:,0,:]
            pred = odeint(model, initial_state, ts[0]).transpose(0,1)
            loss = criterion(pred, states)
            loss.backward()
            optimizer.step()
            running_loss += loss.item()
        if epoch % 10 == 0:
            print(f"Loss at {epoch}: {running_loss}")


