# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_models.ipynb.

# %% auto 0
__all__ = ['VDP', 'Lorenz', 'LotkaVolterra']

# %% ../nbs/00_models.ipynb 3
import torch 
import torch.nn as nn
if torch.cuda.is_available():
    # If you have a gpu then integrate with adjoint method
    from torchdiffeq import odeint_adjoint as odeint
else:
    from torchdiffeq import odeint
import pylab as plt
from torch.utils.data import Dataset, DataLoader
from typing import Callable, List, Tuple, Union, Optional

# %% ../nbs/00_models.ipynb 5
class VDP(nn.Module):
    """ 
    Define the Van der Pol oscillator as a PyTorch module.
    """
    def __init__(self, 
                 mu: float, # Stiffness parameter of the VDP oscillator
                 ):
        super().__init__() 
        self.mu = torch.nn.Parameter(torch.tensor(mu))
        
    def forward(self, 
                t: float, # time index
                state: torch.TensorType, # state of the system first dimension is the batch size
                ) -> torch.Tensor:
        x = state[:, 0]
        y = state[:, 1]
        dX = self.mu*(x-1/3*x**3 - y)
        dY = 1/self.mu*x
        dfunc = torch.zeros_like(state)
        dfunc[:, 0] = dX
        dfunc[:, 1] = dY
        return dfunc
    
    def __repr__(self):
        return f" mu: {self.mu.item()}"
    
    
        
        

# %% ../nbs/00_models.ipynb 11
class Lorenz(nn.Module):
    """ 
    Define the Lorenz system as a PyTorch module.
    """
    def __init__(self, 
                 sigma: float =10.0, # The sigma parameter of the Lorenz system
                 rho: float=28.0, # The rho parameter of the Lorenz system
                beta: float=8.0/3, # The beta parameter of the Lorenz system
                ):
        super().__init__() 
        self.model_params = torch.nn.Parameter(torch.tensor([sigma, rho, beta]))
        
        
    def forward(self, t, state):
        x = state[:,0]      #variables are part of vector array u 
        y = state[:,1]
        z = state[:,2]
        sol = torch.zeros_like(state)
        
        sigma, rho, beta = self.model_params    #coefficients are part of vector array p
        sol[:,0] = sigma*(y-x)
        sol[:,1] = x*(rho-z) - y
        sol[:,2] = x*y - beta*z
        return sol
    
    def __repr__(self):
        return f" sigma: {self.model_params[0].item()}, rho: {self.model_params[1].item()}, beta: {self.model_params[2].item()}"
    

# %% ../nbs/00_models.ipynb 19
class LotkaVolterra(nn.Module):
    """ 
     The Lotka-Volterra equations are a pair of first-order, non-linear, differential equations
     describing the dynamics of two species interacting in a predator-prey relationship.
    """
    def __init__(self,
                 alpha: float = 1.5, # The alpha parameter of the Lotka-Volterra system
                 beta: float = 1.0, # The beta parameter of the Lotka-Volterra system
                 delta: float = 3.0, # The delta parameter of the Lotka-Volterra system
                 gamma: float = 1.0 # The gamma parameter of the Lotka-Volterra system
                 ) -> None:
        super().__init__()
        self.model_params = torch.nn.Parameter(torch.tensor([1.5,1.0,3.0,1.0]))
        
        
    def forward(self, t, state):
        x = state[:,0]      #variables are part of vector array u 
        y = state[:,1]
        sol = torch.zeros_like(state)
        
        alpha, beta, delta, gamma = self.model_params    #coefficients are part of vector array p
        sol[:,0] = alpha*x - beta*x*y
        sol[:,1] = -delta*y + gamma*x*y
        return sol
    
    def __repr__(self):
        return f" alpha: {self.model_params[0].item()}, beta: {self.model_params[1].item()}, delta: {self.model_params[2].item()}, gamma: {self.model_params[3].item()}"

