from typing import Dict, Callable, Any, Optional, Union

import numpy as np
from skimage.color import rgb2gray
from skimage.filters import sobel
from skimage.segmentation import felzenszwalb, slic, quickshift, watershed
from sklearn.linear_model import BayesianRidge, Lasso, LinearRegression
from sklearn.metrics import pairwise_distances


def _watershed(image: np.ndarray, **kwargs):
    gradient = sobel(rgb2gray(image))
    return watershed(image=gradient, **kwargs)


SEGMENTATION_METHODS = {
    "felzenszwalb": (felzenszwalb, {"scale": 250, "sigma": 0.6, "min_size": 45}),
    "slic": (
        slic,
        {
            "n_segments": 250,
            "compactness": 2,
            "convert2lab": True,
            "sigma": 1,
            "start_label": 0,
        },
    ),
    "quickshift": (quickshift, {"kernel_size": 5, "max_dist": 6, "ratio": 0.7}),
    "watershed": (_watershed, {"markers": 250, "compactness": 0.001}),
}


def create_segments(
    image: np.ndarray,
    segmentation_method: str,
    segmentation_settings: Optional[Dict[str, Any]] = None,
) -> np.ndarray:
    """Divide the image into segments (superpixels).

    Proper segmentation of the images is key to producing meaningful explanations with LIME.
    Which method and settings are appropriate is highly use-case specific.

    For an introduction into image segmentation and a comparison of the different methods, see
    [this tutorial in the `scikit-learn` documentation](https://scikit-image.org/docs/stable/auto_examples/segmentation/plot_segmentations.html).

    Parameters
    ----------
    image : np.ndarray
        The image to segment as a three-dimensional numpy array
        of shape (image_width, image_height, 3) where the last dimension are the RGB channels.

    segmentation_method : str
        The method used to segment the image into superpixels.
        Available options are `"felzenszwalb"`, `"slic"`, `"quickshift"`, and `"watershed"`.

        See the [documentation for `scikit-image`](https://scikit-image.org/docs/stable/api/skimage.segmentation.html)
        for details.

    segmentation_settings : Dict[str, Any], optional
        The keyword arguments passed to the segmentation methods.

        See the [documentation for `scikit-image`](https://scikit-image.org/docs/stable/api/skimage.segmentation.html)
        for details.

    Returns
    -------
    A numpy array of shape (image_width, image_height) where each entry is an integer
    that corresponds to the segment number.

    Segment numbers start at 0 and are continuous. The number of segments can be computed
    by determining the maximum value in the array and adding 1.
    """
    segmentation_settings = segmentation_settings or {}

    try:
        segmentation_fn, default_settings = SEGMENTATION_METHODS[segmentation_method]
    except KeyError:
        raise ValueError(
            f"Unknown segmentation_method '{segmentation_method}'."
            f" Available options: {list(SEGMENTATION_METHODS.keys())}"
        )

    settings = {**default_settings, **segmentation_settings}

    _segment_mask = segmentation_fn(image=image, **settings)

    return _segment_mask - np.min(_segment_mask)


def generate_samples(
    segment_mask: np.ndarray, num_of_samples: int = 64, p: float = 0.5
) -> np.ndarray:
    """Generate samples by randomly selecting a subset of the segments.

    Parameters
    ----------
    segment_mask : np.ndarray
        The mask generated by `create_segments()`: An array of shape (image_width, image_height).

    num_of_samples : int
        The number of samples to generate.

    p : float
        The probability for each segment to be removed from a sample.

    Returns
    -------
    samples : np.ndarray
        A two-dimensional array of size (num_of_samples, num_of_segments).
    """
    num_of_segments = int(np.max(segment_mask) + 1)

    return np.random.binomial(n=1, p=p, size=(num_of_samples, num_of_segments))


def generate_images(
    image: np.ndarray,
    segment_mask: np.ndarray,
    samples: np.ndarray,
    background: Optional[Union[np.ndarray, int, float]] = None,
) -> np.ndarray:
    """Generate images from a list of samples.

    Parameters
    ----------
    image : np.ndarray
        The image to explain: An array of shape (image_width, image_height, 3).

    segment_mask : np.ndarray
        The mask generated by `create_segments()`: An array of shape (image_width, image_height).

    samples : np.ndarray
        The samples generated by `generate_samples()`: An array of shape (num_of_samples, num_of_segments).

    background : {np.ndarray, int, float}, optional
        The background to replace the excluded segments with.
        Can be a single number or an array of the same shape as the image.
        If not given, excluded segments are replaced with `0`.

    Returns
    -------
        An array of shape (num_of_samples, image_width, image_height, 3).
    """
    binary_segment_mask = np.zeros(
        shape=(samples.shape[0], image.shape[0], image.shape[1]), dtype=np.int8
    )

    for sample_idx in range(binary_segment_mask.shape[0]):
        binary_segment_mask[sample_idx, :, :] = np.isin(
            segment_mask, np.nonzero(samples[sample_idx])
        ).astype(np.int8)

    images = binary_segment_mask[:, :, :, None] * image

    if background is not None:
        images += (1 - binary_segment_mask)[:, :, :, None] * background

    return images


def predict_images(
    images: np.ndarray, predict_fn: Callable[[np.ndarray], np.ndarray]
) -> np.ndarray:
    """Obtain model predictions for all images.

    Parameters
    ----------
    images : np.ndarray
        Images as an array of dimension (num_of_samples, image_width, image_height, 3)

    predict_fn : Callable
        A function that takes an input of size (num_of_samples, image_width, image_height, 3)
        and returns an array of size (num_of_samples, num_of_classes), where num_of_classes
        is the number of output classes (labels) assigned by the model.

        Commonly, `predict_fn()` feeds the images to the image classification model to be
        explained and takes care of any preprocessing and batching.
        When building explanation pipelines, it might be preferable to replace `predict_images()`
        entirely.

    Returns
    -------
    An array of size (num_of_samples, num_of_classes).
    """
    return predict_fn(images)


LINEAR_MODELS = {
    "bayesian_ridge": BayesianRidge,
    "lasso": Lasso,
    "linear_regression": LinearRegression,
}


def image_distances(
    image: np.ndarray, images: np.ndarray, norm: Optional[Union[str, int]] = None, select: str = "sum"
) -> np.ndarray:
    """

    Parameters
    ----------
    image : np.ndarray
        The original image

    images : np.ndarray
        The sample images

    norm : {non-zero int, str}, optional
        To compute the distance between two images, a norm of the difference between each color channel
        is calculated.

        Defaults to the Frobenius norm if not given.

        For all available options, see
        [the documentation for `numpy.linalg.norm`](https://numpy.org/doc/stable/reference/generated/numpy.linalg.norm.html)

    select : {"sum", "max"}, default "sum"
        Method to combine the channel-wise distances to the final distance.

        There are two options:

        - `"sum"` (the default): Sum the channel-wise distances

        - `"max"`: Take the maximum of the channel-wise distances


    Returns
    -------

    """
    distances_per_channel = np.linalg.norm(images - image, axis=(1, 2), ord=norm)
    if select == "sum":
        return np.sum(distances_per_channel, axis=1)
    elif select == "max":
        return np.max(distances_per_channel, axis=1)
    else:
        raise ValueError(f"Invalid value '{select}' for parameter 'select'.")


def weigh_segments(
    samples: np.ndarray,
    predictions: np.ndarray,
    label_idx: int,
    model_type: str = "bayesian_ridge",
    distances: Optional[np.ndarray] = None,
) -> np.ndarray:
    """Generating list of coefficients to weigh segments

    Parameters
    ----------
    samples : np.ndarray
        The samples generated by `generate_samples()`: An array of shape (num_of_samples, num_of_segments).

    predictions : np.ndarray
        The predictions produced by `predict_images()`: An array of shape (num_of_samples, num_of_classes).

    label_idx : int
        The index of the label to explain in the output of `predict_fn()`.
        Can be the class predicted by the model, or a different class.

    model_type : str
        The type of linear model to fit.
        Available options are `"bayesian_ridge"`, `"lasso"`, and `"linear_regression"`.

        See [the `scikit-learn` documentation](https://scikit-learn.org/stable/modules/classes.html#module-sklearn.linear_model)
        for details on each of the methods.

    distances : np.ndarray, optional
        The distances between the images and the original images used as sample weights when
        fitting the linear model.

        If not given, the cosine distance between a sample and the original image is used.
        Note that this is only a rough approximation and not a good measure if the image
        contains a lot of variation or the segments are of very different size.

    Returns
    -------
    Array of length num_of_segments where each entry corresponds to the segment's coefficient
    in the fitted linear model.
    """
    try:
        linear_model = LINEAR_MODELS[model_type]()
    except KeyError:
        raise ValueError(
            f"Unknown model_type '{model_type}'. Available options: {list(LINEAR_MODELS.keys())}"
        )

    if distances is None:
        distances = np.sum(samples, axis=1)/np.linalg.norm(samples, axis=1)

    # TODO: Make configurable
    linear_model.fit(samples, predictions[:, label_idx], sample_weight=distances)

    return linear_model.coef_
