#!/usr/bin/env python
#
# Author: Markus Stabrin 2016-2019 (markus.stabrin@mpi-dortmund.mpg.de)
# Author: Fabian Schoenfeld 2019 (fabian.schoenfeld@mpi-dortmund.mpg.de)
# Author: Thorsten Wagner 2019 (thorsten.wagner@mpi-dortmund.mpg.de)
# Author: Tapu Shaikh 2019 (tapu.shaikh@mpi-dortmund.mpg.de)
# Author: Adnan Ali 2019 (adnan.ali@mpi-dortmund.mpg.de)
# Author: Luca Lusnig 2019 (luca.lusnig@mpi-dortmund.mpg.de)
# Author: Toshio Moriya 2015-2019 (toshio.moriya@kek.jp)
#
# Copyright (c) 2015-2019 Max Planck Institute of Molecular Physiology
#
# This software is issued under a joint BSD/GNU license. You may use the
# source code in this file under either license. However, note that the
# complete EMAN2 and SPHIRE software packages have some GPL dependencies,
# so you are responsible for compliance with the licenses of these packages
# if you opt to use BSD licensing. The warranty disclaimer below holds
# in either instance.
#
# This complete copyright notice must be included in any revised version of the
# source code. Additional authorship citations may be added, but existing
# author citations must be preserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
#

import os
import subprocess
import sys
import collections
from builtins import range
from builtins import object
try:
    from PyQt4.Qt import *
    from PyQt4 import QtGui
    from PyQt4 import QtCore
except ImportError:
    from PyQt5.Qt import *
    from PyQt5 import QtWidgets as QtGui
    from PyQt5 import QtCore
from EMAN2 import *
from EMAN2_cppwrap import *
try:
	from sphire.libpy import sp_global_def  #### from ..libpy.sp_global_def import *
	from sphire.libpy.sp_sparx import *     #### from ..libpy.sp_sparx import *
	import sphire
except ImportError as e:
	#####print("Import error raised. Ignore.")
	#####print(e)
	from sp_global_def import *
	from sp_sparx import *
from optparse import OptionParser
from functools import partial  # Use to connect event-source widget and event handler
from subprocess import *
import re


# ========================================================================================
# Helper Functions
# 
# This function is added here because db_convert_path in EMAN2db.py has a bug.
# 
def translate_to_bdb_path(std_path):
	'''
	Translate a standard file path (std_path) to bdb syntax (return value). 
	The path pass must contain at lease EMAN2DB directory and .bdb file name.
	For instance, if the path is particles/EMAN2DB/data.bdb,
	will return bdb:particles#data.
	'''
	
	# Check error conditions
	if not isinstance(std_path,str): 
		raise RuntimeError("Path has to be a string")
	path_tokens = std_path.split("/")
	
	if len(path_tokens) < 2: 
		raise ValueError("Invalid file path. The path pass must contain at least \'EMAN2DB\' directory and \'.bdb\' file name (e.g \'./EMAN2DB/data.bdb\'). ")

	if path_tokens[-2] != "EMAN2DB": 
		raise ValueError("Invalid file path. The path pass must contain \'EMAN2DB\' directory (e.g \'./EMAN2DB/data.bdb\').")
	
	if os.path.splitext(path_tokens[-1])[1] != ".bdb": 
		raise ValueError("Path is invalid. The path pass must contain \'.bdb\' file name (e.g \'./EMAN2DB/data.bdb\').")
	
	# If necessary, compose directory path as a relative path at first
	dir = ""
	if len(path_tokens) > 2:
		for idx in range(0, len(path_tokens) - 2):
			if idx != 0:
				dir += "/"
			dir += path_tokens[idx] # accrue the directory
	
	# if the input file path is a absolute path, add '/' at the head of the path
	if std_path[0] == "/" and dir[0] != "/": 
		dir = "/" + dir
	
	# Add '#' before the database name (file basename without extension)
	bdb_path = "bdb:"
	if dir != "":
		bdb_path += dir + "#"
	# Finally, add file basename (without .bdb extension)
	assert(os.path.splitext(path_tokens[-1])[1] == ".bdb")
	bdb_path += os.path.splitext(path_tokens[-1])[0]
	
	return bdb_path

# ========================================================================================
# Inherited by SXcmd_category, SXconst_set, and SXoperand_set
# SXMainWindow use this class to handle events from menu item buttons
class SXmenu_item(object):
	def __init__(self, name = "", label = "", short_info = ""):
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.name = name              # Name of this menu item, used as a key of dictionary
		self.label = label            # User friendly name of this menu item
		self.short_info = short_info  # Short description of this menu item
		self.btn = None               # <Used only in sxgui.py> QPushButton button instance associating with this menu item
		self.widget = None            # <Used only in sxgui.py> SXCmdWidget instance associating with this menu item
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

# ========================================================================================
class SXcmd_token(object):
	def __init__(self):
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.key_base = ""            # key base name of command token (argument or option) in command line
		self.key_prefix = ""          # key prefix of of command token. None for argument, "--" or "-" for option
		self.label = ""               # User friendly name of argument or option
		self.help = ""                # Help info
		self.group = ""               # Tab group: main or advanced
		self.dependency_group = [["", "", ""]]    # Depencency group: Disables or enables widgets
		self.is_required = False      # Required argument or options. No default values are available
		self.is_locked = False        # The restore value will be used as the locked value.
		self.is_reversed = False      # Reversed default value of bool. The flag will be added if the value is same as default 
		self.default = ""             # Default value
		self.restore = [[""], [""]]             # Restore value
		self.type = ""                # Type of value
		self.filament_tab = ""                # Type of value
		# NOTE: Toshio Moriya 2018/01/19
		# self.is_in_io should be removed after cleaning up MoinMoin related codes.
		self.is_in_io = False         # <Used only in wikiparser.py> To check consistency between "usage in command line" and list in "== Input ==" and "== Output ==" sections
		self.restore_widget = None    # <Used only in sxgui.py> Restore widget instance associating with this command token
		self.widget = None            # <Used only in sxgui.py> Main widget instance associating with this command token
		self.subwidget_left = None    # <Used only in sxgui.py> Subwidget instance at the left associating with the helper utility of this command token (e.g. conversion calculator)
		self.subwidget_right = None   # <Used only in sxgui.py> SubWidget instance at the right associating with the helper utility of this command token (e.g. conversion calculator)
		self.calculator_dialog = None # <Used only in sxgui.py> Calculator dialog instance associating with the helper utility of this command token (e.g. conversion calculator)
		self.other_dialog_list = []   # <Used only in sxgui.py> List of the other calculator dialog instances associating with this command (e.g. conversion calculator)
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

	def initialize_edit(self, key_base):
		self.key_base = key_base
		self.key_prefix = None
		self.label = None
		self.help = None
		self.group = None
		self.is_required = None
		self.is_locked = None
		self.is_reversed = None
		self.default = None
		self.restore = None
		self.type = None

# ========================================================================================
class SXcmd(object):
	def __init__(self, category = "", role = "", is_submittable = True):
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.name = ""                        # Name of this command (i.e. name of sx*.py script but without .py extension), used for generating command line
		self.subname = ""                     # Subname of this command (i.e. sub-command name), used for generating command line. For fullset command, use empty string
		self.mode = ""                        # Key base name of a command option token, defining mode/subset of this command (i.e. option mode name). For fullset command, use empty string
		self.subset_config = ""               # Unique name to differentiate subset configuration of this command. For example, to name a command argument mode, which dependes on the number of input arguments. If not necessary, use empty string 
		self.label = ""                       # User friendly name of this command
		self.short_info = ""                  # Short description of this command
		self.mpi_support = False              # Flag to indicate if this command suppors MPI version
		self.mpi_add_flag = False             # DESIGN_NOTE: 2015/11/12 Toshio Moriya. This can be removed when --MPI flag is removed from all sx*.py scripts
		self.category = category              # Category of this command: sxc_movie, sxc_cter, sxc_window, sxc_isac, sxc_viper, sxc_meridien, sxc_sort3d, sxc_localres, sxc_utilities
		self.role = role                      # Role of this command; sxr_pipe (pipeline), sxr_alt (alternative) sxr_util (utility)
		self.is_submittable = is_submittable  # External GUI Application (e.g. sxgui_cter.py) should not be submitted to job queue
		self.token_list = []                  # List of command tokens. Need this to keep the order of command tokens
		self.token_dict = {}                  # Dictionary of command tokens, organised by key base name of command token. Easy to access a command token but looses their order
		self.dependency_dict = collections.OrderedDict()             # Dictionary of command tokens, containing the dependencies
		self.btn = None                       # <Used only in sxgui.py> QPushButton button instance associating with this command
		self.widget = None                    # <Used only in sxgui.py> SXCmdWidget instance associating with this command
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

	def get_mode_name_for(self, target_name):
		mode_name = self.name
		
		if self.subname != "":
			if target_name in ["file_path"]:
				mode_name = "%s_%s" % (mode_name, self.subname)
			elif target_name in ["human"]:
				mode_name = "%s %s" % (mode_name, self.subname)
		
		if self.mode != "":
			if target_name in ["file_path"]:
				mode_name = "%s_%s" % (mode_name, self.mode)
			elif target_name in ["human"]:
				mode_name = "%s %s%s" % (mode_name, self.token_dict[self.mode].key_prefix, self.mode)
		
		if self.subset_config != "":
			if target_name in ["file_path"]:
				mode_name = "%s_%s" % (mode_name, self.subset_config.replace(" ", "_"))
			elif target_name in ["human"]:
				mode_name = "%s (%s)" % (mode_name, self.subset_config)
		
		return mode_name

	def get_category_dir_path(self, parent_dir_path = ""):
		category_dir_path = self.category.replace("sxc_", "")
		if parent_dir_path != "":
			category_dir_path = os.path.join(parent_dir_path, category_dir_path)

		return category_dir_path

# ========================================================================================
class SXcmd_category(SXmenu_item):
	def __init__(self, name = "", label = "", short_info = ""):
		super(SXcmd_category, self).__init__(name, label, short_info)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		# self.name = name              # <Inherit from SXmenu_item> Name of this command category (i.e. sxc_movie, sxc_cter, sxc_window, sxc_isac, sxc_viper, sxc_meridien, sxc_sort3d, sxc_localres, sxc_utilities), used as a key of dictionary
		# self.label = label            # <Inherit from SXmenu_item> User friendly name of this command category
		# self.short_info = short_info  # <Inherit from SXmenu_item> Short description of this command category
		self.cmd_list = []              # <Used only in sxgui.py> list of commands in this category. Need this to keep the order of commands
#		self.cmd_dict = {}              # <Used only in sxgui.py> dictionary of commands in this category, organised by names of commands. Easy to access a command but looses their order
		# self.btn = None               # <Inherit from SXmenu_item> QPushButton button instance associating with this category
		# self.widget = None            # <Inherit from SXmenu_item> SXCmdWidget instance associating with this category

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

# ========================================================================================
class SXconst(object):
	def __init__(self):
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.key = ""                # <Used only in sxgui.py> key of constant parameter
		self.label = ""              # <Used only in sxgui.py> User friendly name of constant parameter
		self.help = ""               # <Used only in sxgui.py> Help info
		self.register = ""           # <Used only in sxgui.py> Default value
		self.type = ""               # <Used only in sxgui.py> Type of value
		self.register_widget = None  # <Used only in sxgui.py> Restore widget instance associating with this command token
		self.widget = None           # <Used only in sxgui.py> Widget instance associating with this constant parameter
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

# ========================================================================================
class SXconst_set(SXmenu_item):
	def __init__(self):
		super(SXmenu_item, self).__init__()
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		# self.name = ""        # <Inherit from SXmenu_item> Name of this constant parameter set
		# self.label = ""       # <Inherit from SXmenu_item> User friendly name of this set
		# self.short_info = ""  # <Inherit from SXmenu_item> Short description of this set
		self.list = []          # <Used only in sxgui.py> list of constant parameters. Need this to keep the order of constant parameters
		self.dict = {}          # <Used only in sxgui.py> dictionary of constant parameters, organised by keys of constant parameters. Easy to access each constant parameter but looses their order
		# self.btn = None       # <Inherit from SXmenu_item> QPushButton button instance associating with this set
		# self.widget = None    # <Inherit from SXmenu_item> Widget instance associating with this set
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

# ========================================================================================
class SXoperand(object):
	def __init__(self):
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.key = ""                  # <Used only in sxgui.py> key of calculator operand
		self.label = ""                # <Used only in sxgui.py> User friendly name of calculator operand
		self.help = ""                 # <Used only in sxgui.py> Help info
		self.register = ""             # <Used only in sxgui.py> Default value
		self.type = ""                 # <Used only in sxgui.py> Type of value
		self.is_input = True           # <Used only in sxgui.py> Flag to indicate if this operand is input or output
		self.validated_register = None # <Used only in sxgui.py> Contain validated numerical value of register only if the value is valid. If not, it is None. This is used as a flag to indicate if the registered value is valid or not
		self.validated = None          # <Used only in sxgui.py> Contain Validated numerical value of command widget only if the value is valid. If not, it is None. This is used as a flag to indicate if the registered value is valid or not
		self.register_widget = None    # <Used only in sxgui.py> Restore widget instance associating with this calculator operand
		self.widget = None             # <Used only in sxgui.py> Widget instance associating with this calculator operand
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

# ========================================================================================
class SXoperand_set(SXmenu_item):
	def __init__(self):
		super(SXmenu_item, self).__init__()
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		# self.name = ""        # <Inherit from SXmenu_item> Name of this calculator operand set
		# self.label = ""       # <Inherit from SXmenu_item> User friendly name of this set
		# self.short_info = ""  # <Inherit from SXmenu_item> Short description of this set
		self.list = []          # <Used only in sxgui.py> list of calculator operands. Need this to keep the order of calculator operands
		self.dict = {}          # <Used only in sxgui.py> dictionary of calculator operands, organised by keys of calculator operands. Easy to access each calculator operand but looses their order
		# self.btn = None       # <Inherit from SXmenu_item> QPushButton button instance associating with this set
		# self.widget = None    # <Inherit from SXmenu_item> Widget instance associating with this set
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

# ========================================================================================
class SXLookFeelConst(object):
	# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
	# static class variables
	# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
	default_bg_color = QColor(229, 229, 229, 192) # default_bg_color = QColor(229, 229, 229, 242) # Greyish-White Transparent
	default_bg_color_string = 'rgba(229, 229, 229, 192)' # default_bg_color = QColor(229, 229, 229, 242) # Greyish-White Transparent
	default_bg_color_calculator = QColor(214, 214, 214) # Greyish-White
	sxinfo_widget_bg_color = QColor(0, 0, 0, 10) # Almost-Completely Transparent
	sxcmd_widget_bg_color = QColor(0, 0, 0, 0) # Completely Transparent
	sxcmd_tab_bg_color = QColor(229, 229, 229, 200) # White Transparent
	sxcmd_tab_bg_color_string = 'rgba(229, 229, 229, 200)' # White Transparent

	# Constants
	project_dir_raw = "sxgui_settings"
	project_dir = project_dir_raw
	sxmain_window_left = 0
	sxmain_window_top = 0
	sxmain_window_min_width = 1500 # Requirement of specification
	sxmain_window_min_height = 360 # Requirement of specification
	expected_cmd_counts = 32
	grid_margin = 6 # grid_margin = 12
	grid_spacing = 6

	# Constants initialised with invalid values.
	# Valid values should be set by initialise() function
	screen_height = -1
	screen_width = -1
	sxmain_window_width = -1
	sxmain_window_height = -1
	sxmenu_item_btn_width = -1
	grid_distance = -1
	sxmenu_btn_area_min_width = -1
	sxcmd_btn_area_min_width = -1
	sxcmd_widget_area_min_width = -1

	file_dialog_dir = ""
	
	@staticmethod
	def initialise(sxapp, version):
		# Set the directory for all file dialogs to script directory
		version_main = ".".join(version.split('.')[:2])
		SXLookFeelConst.project_dir += '/{0}'.format(version_main)
		SXLookFeelConst.file_dialog_dir = os.getcwd()
		
		# Check whether current settings directory exists
		if os.path.exists(SXLookFeelConst.project_dir):
			current_settings_exist = True
		else:
			current_settings_exist = False
			
		do_longer_warning = False
		older_settings_exist = False
		print('\nCurrent SPHIRE settings directory: {}'.format(SXLookFeelConst.project_dir))
		try:
			for file_name in os.listdir(SXLookFeelConst.project_dir_raw):
				if file_name != os.path.basename(SXLookFeelConst.project_dir):
					print('INFORMATION: Old settings directory %s/%s detected that belongs to another version of SPHIRE.' % (SXLookFeelConst.project_dir_raw, file_name))
					older_settings_exist = True

					if not current_settings_exist:
						do_longer_warning = True
		except FileNotFoundError:
			pass
				
		if do_longer_warning:
			print('To load old settings, please load the gui settings manually.')
			print('Backwards compatibility cannot be guaranteed.')
		
		if not current_settings_exist and not older_settings_exist:
			print("\nSettings directory for current SPHIRE version %s doesn't exist\nWould you like to create a new project directory and continue?\nYou need to run the sphire command in the foreground (without &) to answer this question\n[y/n] "% SXLookFeelConst.project_dir)
			answer = input()
			while answer.lower() not in ('y', 'n'):
				print('Answer needs to be y or n')
				answer = input()
			if answer.lower() == 'n':
				print("\nbye bye")
				exit()
		
		monitor_index = 0
		# Search for maximum screen height and set it to SXLookFeelConst singleton class
		max_screen_height = sxapp.desktop().screenGeometry().height()
		for index in range(sxapp.desktop().screenCount()):
			screen_height = sxapp.desktop().screenGeometry(index).height()
			if max_screen_height < screen_height:
				monitor_index = index
				max_screen_height = screen_height
		SXLookFeelConst.screen_height = max_screen_height
		# Search for maximum screen width and set it to SXLookFeelConst singleton class
		SXLookFeelConst.screen_width = sxapp.desktop().screenGeometry(monitor_index).width()

		# Set size of the main window depending on the screen size
		SXLookFeelConst.sxmain_window_height = SXLookFeelConst.screen_height / 2
		if SXLookFeelConst.sxmain_window_height <= SXLookFeelConst.sxmain_window_min_height:
			SXLookFeelConst.sxmain_window_height = SXLookFeelConst.sxmain_window_min_height

		SXLookFeelConst.sxmain_window_width = SXLookFeelConst.sxmain_window_min_width
		if SXLookFeelConst.sxmain_window_width >= SXLookFeelConst.screen_width * 3 / 4:
			SXLookFeelConst.sxmain_window_width = SXLookFeelConst.screen_width * 3 / 4
			if SXLookFeelConst.sxmain_window_width < 960:
				SXLookFeelConst.sxmain_window_width = 960

		# SXLookFeelConst.sxmain_window_height = SXLookFeelConst.screen_height / 2
		# SXLookFeelConst.sxmain_window_width =SXLookFeelConst.sxmain_window_min_width

		SXLookFeelConst.sxmenu_item_btn_width = SXLookFeelConst.sxmain_window_height * 0.125
		SXLookFeelConst.grid_distance = SXLookFeelConst.sxmenu_item_btn_width / 10

		SXLookFeelConst.sxmenu_btn_area_min_width = 2 * SXLookFeelConst.sxmenu_item_btn_width + SXLookFeelConst.grid_distance + 18
		SXLookFeelConst.sxcmd_btn_area_min_width = 240
		SXLookFeelConst.sxcmd_widget_area_min_width = SXLookFeelConst.sxmain_window_width - SXLookFeelConst.sxmenu_btn_area_min_width - SXLookFeelConst.sxcmd_btn_area_min_width

	@staticmethod
	def format_path(path):
		formatted_path = os.path.relpath(str(path))
		if formatted_path[:len("../")] == "../":
			# if the path is above the project root directory (current directory)
			# use absolute path
			formatted_path = path
		# else:
			# if the path is project subdirectory
			# use relative path

		return formatted_path

	@staticmethod
	def generate_sxcmd_wiki_url(sxcmd, wiki_type = "SPHIRE"):
		if wiki_type == "SPHIRE":
			# First, handle exceptional cases
			if sxcmd.name in ["e2display", "sxpdb2em", "sxrelion2sphire", "sxprocess", "e2proc3d", "sxheader", "e2bdb"]:
				sxcmd_category_name = "utilities"
			elif sxcmd.name in ["sxpipe"] and sxcmd.subname in ["resample_micrographs", "organize_micrographs", "restacking", "moon_eliminator"]:
				sxcmd_category_name = "utilities"
			# elif sxcmd.name in ["sxpipe"] and sxcmd.subname in ["reboxing"]:
			# 	sxcmd_category_name = "meridien"
			elif sxcmd.name in ["sxmeridien", "sxmeridien_20171120"]:
				sxcmd_category_name = "meridien"
			elif sxcmd.name in ["sxunblur", "sxsummovie"]:
				sxcmd_category_name = "movie"
			else:
				sxcmd_category_name = sxcmd.category.replace("sxc_", "")
			# URL Format: "http://sphire.mpg.de/wiki/doku.php?id=pipeline:CMD_CATEGORY:CMD_BASE
			sxcmd_wiki_url = "http://sphire.mpg.de/wiki/doku.php?id=pipeline:%s:%s" % (sxcmd_category_name, sxcmd.name)
			if sxcmd.subname != "":
				sxcmd_wiki_url = "%s_%s" % (sxcmd_wiki_url, sxcmd.subname)
		else:
			assert (wiki_type == "SPARX")
			sxcmd_wiki_url = "%s%s" % (SPARX_DOCUMENTATION_WEBSITE, sxcmd.name)
			if sxcmd.subname != "":
				sxcmd_wiki_url = "%s_%s" % (sxcmd_wiki_url, sxcmd.subname)
		
		return sxcmd_wiki_url

	@staticmethod
	def generate_sxmenu_item_wiki_url(sxmenu_item):
		# First, handle exceptional cases
		sxmenu_item_wiki_url = None
		if sxmenu_item.name in ["sxc_gui_info"]:
			sxmenu_item_wiki_url = "http://sphire.mpg.de/wiki/doku.php?id=start"
		else:
			# URL Format: "http://sphire.mpg.de/wiki/doku.php?id=pipeline:CMD_CATEGORY:start"
			sxmenu_item_wiki_url = "http://sphire.mpg.de/wiki/doku.php?id=pipeline:%s:start" % (sxmenu_item.name.replace("sxc_", ""))
		assert (sxmenu_item_wiki_url is not None)
		
		return sxmenu_item_wiki_url

# ========================================================================================
class SXLogoButton(QPushButton):
	def __init__(self, logo_file_path, parent = None):
		super(SXLogoButton, self).__init__(parent)

		# print "MRK_DEBUG: logo_file_path = %s" % logo_file_path
		# print "MRK_DEBUG: os.path.exists(logo_file_path) %s" % os.path.exists(logo_file_path)

		# Width of logo image
		logo_width = SXLookFeelConst.sxmenu_item_btn_width * 2 + SXLookFeelConst.grid_distance

		# Style of widget
		self.setFixedSize(logo_width, 0.434 * logo_width)
		self.customButtonStyle = """
			SXLogoButton {{background-color: rgba(0, 0, 0, 0); border: 0px solid black; border-radius: 0px; image: url("{0}");}}
			SXLogoButton:focus {{background-color: rgba(0, 0, 0, 0); border: 0px solid grey; border-radius: 0px; image: url("{0}");}}
			SXLogoButton:pressed {{background-color: rgba(0, 0, 0, 0); border: 0px solid red; border-radius: 0px; image: url("{0}");}}
			""".format(logo_file_path)
		self.customButtonStyleClicked = """
			SXLogoButton {{background-color: rgba(0, 0, 0, 0); border: 0px solid black; border-radius: 0px; image: url("{0}");}}
			SXLogoButton:focus {{background-color: rgba(0, 0, 0, 0); border: 0px solid grey; border-radius: 0px; image: url("{0}");}}
			SXLogoButton:pressed {{background-color: rgba(0, 0, 0, 0); border: 0px solid red; border-radius: 0px; image: url("{0}");}}
			""".format(logo_file_path)

		# Set style and add click event
		self.setStyleSheet(self.customButtonStyle)

		# Add ToolTip
		self.setToolTip('HELP')

# ========================================================================================
class SXPictogramButton(QPushButton):
	def __init__(self, pictogram_name, pictogram_file_path, parent = None):
		super(SXPictogramButton, self).__init__(parent)

		# print "MRK_DEBUG: pictogram_file_path = %s" % pictogram_file_path
		# print "MRK_DEBUG: os.path.exists(logo_file_path) %s" % os.path.exists(pictogram_file_path)

		# Width of pictogram image
		self.parent = parent
		pictogram_width = SXLookFeelConst.sxmenu_item_btn_width

		# Style of widget
		self.setFixedSize(pictogram_width, pictogram_width)
		self.customButtonStyle = """
			SXPictogramButton {{background-color: rgba(0, 0, 0, 0); border: 2px solid rgba(0, 0, 0, 0); border-radius: {1}px; image: url("{0}");}}
			SXPictogramButton:focus {{background-color: rgba(0, 0, 0, 0); border: 2px solid grey; border-radius: {1}px; image: url("{0}");}}
			SXPictogramButton:pressed {{background-color: rgba(0, 0, 0, 0); border: 2px solid rgb(153, 153, 153); border-radius: {1}px; image: url("{0}");}}
			""".format(pictogram_file_path, pictogram_width / 6)
		self.customButtonStyleClicked = """
			SXPictogramButton:pressed {{background-color: rgba(0, 0, 0, 0); border: 2px solid rgb(153, 153, 153); border-radius: {1}px; image: url("{0}");}}
			SXPictogramButton {{background-color: rgba(0, 0, 0, 0); border: 2px solid rgb(220, 220, 220); border-radius: {1}px; image: url("{0}");}}
			""".format(pictogram_file_path, pictogram_width / 6)

		# Set style and add click event
		self.setStyleSheet(self.customButtonStyle)

		# Add tooltipp
		self.setToolTip(pictogram_name.upper())

class SXMenuItemBtnAreaWidget(QWidget):
	def __init__(self, sxconst_set, sxcmd_category_list, sxinfo, helical, parent = None):
		super(SXMenuItemBtnAreaWidget, self).__init__(parent)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>

		# Create widgets for pipeline command category button area and miscellaneous function button area
		sxcmd_category_btn_subarea_widget = self.create_sxmenu_item_btn_subarea_widget()
		misc_func_btn_subarea_widget = self.create_sxmenu_item_btn_subarea_widget()
		self.add_sxmenu_item_btn_widget(sxconst_set, sxcmd_category_btn_subarea_widget)
		self.sxconst_set = sxconst_set
		self.sxcmd_category_list = sxcmd_category_list
		for sxcmd_category in sxcmd_category_list:
			if sxcmd_category.name != "sxc_utilities" and sxcmd_category.name != "sxc_movie":
				self.add_sxmenu_item_btn_widget(sxcmd_category, sxcmd_category_btn_subarea_widget)
			else: # assert(sxcmd_category.name == "sxc_utilities")
				self.add_sxmenu_item_btn_widget(sxcmd_category, misc_func_btn_subarea_widget)

		global_layout = QVBoxLayout()
		global_layout.setContentsMargins(0, 0, 0, 0)

		sxmenu_item_btn_area_widget = QWidget(self)
		sxmenu_item_btn_area_widget.setObjectName('SXMenuItemBtnAreaWidget')
		sxmenu_item_btn_area_widget.setStyleSheet('QWidget#SXMenuItemBtnAreaWidget {background-color: rgba(0, 0, 0, 153);}')
		sxmenu_item_btn_area_widget.setFixedWidth(SXLookFeelConst.sxmenu_btn_area_min_width)

		sxmenu_item_btn_area_layout = QVBoxLayout()

		# Add widget of pipeline command category button area to layout
		sxmenu_item_btn_area_layout.addWidget(sxcmd_category_btn_subarea_widget)

		# Create and Add separator label
		layout_label = QHBoxLayout()
		line_label = QLabel(sxmenu_item_btn_area_widget)
		line_label.setFixedHeight(1)
		line_label.setFixedWidth(SXLookFeelConst.sxmenu_item_btn_width * 2)
		line_label.setStyleSheet('background-color: rgba(220, 220, 220, 100)')
		layout_label.addWidget(line_label)
		layout_label.setContentsMargins(0, 7, 0, 7)

		sxmenu_item_btn_area_layout.addLayout(layout_label)

		# Add widget of miscellaneous function button area to layout
		sxmenu_item_btn_area_layout.addWidget(misc_func_btn_subarea_widget)

		# Add stretch to make a space and keep sizes of the other widgets to be constant
		sxmenu_item_btn_area_layout.addStretch(1)

		# Add menu item button for application information
		if helical:
			sxmenu_item_btn_pictograph_file_path = '{0}sxgui_logo_sphire_helix.png'.format(get_image_directory())
		else:
			sxmenu_item_btn_pictograph_file_path = '{0}sxgui_logo_sphire.png'.format(get_image_directory())
		sxmenu_item_btn = SXLogoButton(sxmenu_item_btn_pictograph_file_path)
		sxinfo.btn = sxmenu_item_btn

		sxmenu_item_btn_area_layout.addWidget(sxmenu_item_btn)

		# Set menu item button area layout to the widget
		sxmenu_item_btn_area_widget.setLayout(sxmenu_item_btn_area_layout)

		# self related settings
		global_layout.addWidget(sxmenu_item_btn_area_widget)
		self.setLayout(global_layout)

	def create_sxmenu_item_btn_subarea_widget(self):
		sxmenu_item_btn_subarea_widget = QWidget()

		grid_layout = QGridLayout()
		grid_layout.setSpacing(SXLookFeelConst.grid_distance)
		grid_layout.setContentsMargins(0, 0, 0, 0)

		sxmenu_item_btn_subarea_widget.setLayout(grid_layout)

		return sxmenu_item_btn_subarea_widget

	def add_sxmenu_item_btn_widget(self, sxmenu_item, sxmenu_item_btn_subarea_widget):
		assert(isinstance(sxmenu_item, SXmenu_item) == True) # Assuming the sxmenu_item is an instance of class SXmenu_item

		sxmenu_item_btn_pictograph_file_path = "{0}sxgui_pictograph_{1}.png".format(get_image_directory(), sxmenu_item.name.replace("sxc_", ""))
		sxmenu_item.btn = SXPictogramButton(sxmenu_item.name.replace("sxc_", ""), sxmenu_item_btn_pictograph_file_path, self)
		cur_widget_counts = sxmenu_item_btn_subarea_widget.layout().count()
		sxmenu_item_btn_subarea_widget.layout().addWidget(sxmenu_item.btn, cur_widget_counts // 2, cur_widget_counts % 2)

# ========================================================================================
# Provides all necessary functionarity
# tabs only provides widgets and knows how to layout them
class SXCmdWidget(QWidget):
	### process_started = pyqtSignal()

	def __init__(self, sxconst_set, sxcmd, helical, parent = None):
		super(SXCmdWidget, self).__init__(parent)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.sxconst_set = sxconst_set
		self.sxcmd = sxcmd

		self.sxcmd_tab_main = None
		self.sxcmd_tab_advance = None

		self.child_application_list = []

		self.gui_settings_file_path = "%s/gui_settings_%s_%d.txt" % (self.sxcmd.get_category_dir_path(SXLookFeelConst.project_dir), self.sxcmd.get_mode_name_for("file_path"), int(helical))

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

		# Set grid layout
		grid_layout = QGridLayout(self)
		# grid_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		# grid_layout.setSpacing(SXLookFeelConst.grid_spacing)

		self.setAutoFillBackground(True)
		palette = QPalette()
		palette.setBrush(QPalette.Background, QBrush(SXLookFeelConst.sxcmd_widget_bg_color))
		self.setPalette(palette)

		self.sxcmd_tab_main = SXCmdTab("Main", helical, self)
		self.sxcmd_tab_advance = SXCmdTab("Advanced", helical, self)
		for key in self.sxcmd.dependency_dict:
			if key not in self.sxcmd.token_dict:
				continue
			self.sxcmd.token_dict[key].widget.my_changed_state.connect(self.change_state)
			if isinstance(self.sxcmd.token_dict[key].widget, SXCheckBox):
				idx = self.sxcmd.token_dict[key].widget.checkState()
				if idx == 0:
					self.sxcmd.token_dict[key].widget.setCheckState(2)
				elif idx == 1:
					self.sxcmd.token_dict[key].widget.setCheckState(2)
				elif idx == 2:
					self.sxcmd.token_dict[key].widget.setCheckState(0)
				self.sxcmd.token_dict[key].widget.setCheckState(idx)
			elif isinstance(self.sxcmd.token_dict[key].widget, SXLineEdit):
				text = self.sxcmd.token_dict[key].widget.text()
				self.sxcmd.token_dict[key].widget.setText('0000000000000000099999')
				self.sxcmd.token_dict[key].widget.setText(text)


		tab_widget = QTabWidget()
		tab_widget.insertTab(0, self.sxcmd_tab_main, self.sxcmd_tab_main.name)
		tab_widget.insertTab(1, self.sxcmd_tab_advance, self.sxcmd_tab_advance.name)
		tab_widget.setAutoFillBackground(True)
		tab_widget.setStyleSheet("""QTabWidget::pane {
			border-top: 2px solid #C2C7CB;
			position: absolute;
			top: -0.5em;
		}

		QTabWidget::tab-bar {
			alignment: center;
		}

		QTabBar::tab {
			background: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1,
					stop: 0 #E1E1E1, stop: 0.4 #DDDDDD,
					stop: 0.5 #D8D8D8, stop: 1.0 #D3D3D3);
			border: 2px solid #C4C4C3;
			border-bottom-color: #C2C7CB; /* same as the pane color */
			border-top-left-radius: 4px;
			border-top-right-radius: 4px;
			min-width: 8ex;
			padding: 2px;
		}

		QTabBar::tab:selected, QTabBar::tab:hover {
			background: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1,
					stop: 0 #fafafa, stop: 0.4 #f4f4f4,
					stop: 0.5 #e7e7e7, stop: 1.0 #fafafa);
		}

		QTabBar::tab:selected {
			border-color: #9B9B9B;
			border-bottom-color: #C2C7CB; /* same as pane color */
		}""")
		palette = tab_widget.palette()
		palette.setBrush(QPalette.Background, QBrush(SXLookFeelConst.sxcmd_widget_bg_color))
		tab_widget.setPalette(palette)
		grid_layout.addWidget(tab_widget, 0, 0)

	@QtCore.pyqtSlot(str, str)
	@QtCore.pyqtSlot(str, int)
	def change_state(self, dependency, state, old_dependency=None, sender=None, prev_sender=None):
		"""
		Change the state of widgets based on the choice of the corresponding combo box

		name - Name of the group to change status (Emitted by the combo box)

		Returns:
		None
		"""

		#print(dependency, state, old_dependency)
		if sender is None:
			sender = self.sender()
		else:
			sender = sender

		dependency = str(dependency)
		if old_dependency is None:
			old_dependency = [dependency]
		else:
			old_dependency.append(dependency)
		parent = sender.parent()
		while True:
			if hasattr(parent, 'sxcmd'):
				break
			else:
				parent = parent.parent()

		final_state = True
		for name, exp_state, inverse in parent.sxcmd.token_dict[dependency].dependency_group:
			if not name:
				continue

			try:
				widget = parent.sxcmd.token_dict[name].widget
			except KeyError:
				continue

			try:
				curr_state = widget.checkState()
			except AttributeError:
				curr_state = widget.text()

			if isinstance(widget, SXCheckBox):
				type_check = bool
			elif isinstance(widget, SXLineEdit):
				type_check = str
			else:
				assert False

			if str(type_check(curr_state)) != exp_state and inverse == 'True':
				is_enabled = True
			elif str(type_check(curr_state)) == exp_state and inverse == 'True':
				is_enabled = False
			elif str(type_check(curr_state)) == exp_state:
				is_enabled = True
			else:
				is_enabled = False

			if len(parent.sxcmd.token_dict[dependency].dependency_group) == 2 and is_enabled:
				if prev_sender is not None:
					is_enabled = prev_sender.isEnabled()

			if not is_enabled:
				final_state = False
		sender.setEnabled(final_state)

		try:
			chain_deps = parent.sxcmd.dependency_dict[dependency]
		except KeyError:
			pass
		else:
			for name, exp_state, inverse in chain_deps:
				if not name:
					continue

				try:
					widget = parent.sxcmd.token_dict[name].widget
				except KeyError:
					continue

				if not name in old_dependency:
					self.change_state(name, exp_state, old_dependency=old_dependency, sender=widget, prev_sender=sender)

		#try:
		#	for name, exp_state, inverse in parent.sxcmd.dependency_dict[str(dependency)]:
		#		if not parent.sxcmd.token_dict[str(dependency)].widget.isEnabled():
		#			is_enabled = False
		#		elif str(type_check(state)) != exp_state and inverse == 'True':
		#			is_enabled = True
		#		elif str(type_check(state)) == exp_state and inverse == 'True':
		#			is_enabled = False
		#		elif str(type_check(state)) == exp_state:
		#			is_enabled = True
		#		else:
		#			is_enabled = False
		#		print('STATE', name, exp_state, state, is_enabled, type_check, sender)
		#		SXCmdTab.set_text_entry_widget_enable_state(parent.sxcmd.token_dict[name].widget, is_enabled)
		#	child_dict = {}
		#	for name, exp_state, inverse in parent.sxcmd.dependency_dict[str(dependency)]:
		#		for entry in parent.sxcmd.token_dict[name].dependency_group:
		#			if not entry[0]:
		#				continue
		#			if name not in old_dependency and str(dependency) != entry[0]:
		#				try:
		#					new_state = parent.sxcmd.token_dict[name].widget.checkState()
		#				except AttributeError:
		#					new_state = parent.sxcmd.token_dict[name].widget.text()
		#				child_dict[name] = new_state
		#	for key, value in child_dict.items():
		#		print(key, value)
		#		self.change_state(key, value, old_dependency, parent.sxcmd.token_dict[key].widget)
		#except KeyError:
		#	return None
		#print(state)
		#try:
		#	for entry in self.cmd[name]:
		#		widget = entry[0]
		#		state = entry[1]
		#		sub_name = entry[2]
		#		if not self.content[name].isEnabled():
		#			widget.setEnabled(False)
		#		elif self.content[name].edit.currentText() == state:
		#			widget.setEnabled(True)
		#		else:
		#			widget.setEnabled(False)
		#		self.change_state(name=sub_name)
		#except KeyError:
		#	return None


	def map_widgets_to_sxcmd_line(self):
		# Add program name to command line
		sxcmd_line = "%s.py" % (self.sxcmd.name)
		
		if self.sxcmd.subname != "":
			sxcmd_line += " %s" % (self.sxcmd.subname)

		# Loop through all command tokens
		for sxcmd_token in self.sxcmd.token_list:
			# First, handle very special cases
			if not isinstance(sxcmd_token.widget, list):
				widgets = [sxcmd_token.widget]
			else:
				widgets = sxcmd_token.widget

			do_continue = False
			if not sxcmd_token.is_locked:
				for widget in widgets:
					if not widget.isEnabled():
						do_continue = True
			if do_continue:
				continue
			elif sxcmd_token.type == "user_func":
				user_func_name_index = 0
				external_file_path_index = 1
				user_func_name = str(sxcmd_token.widget[user_func_name_index].text())
				external_file_path = str(sxcmd_token.widget[external_file_path_index].text())

				# This is not default value
				if external_file_path not in ["", sxcmd_token.default[external_file_path_index]]:
					# Case 1: User specified an exteranl function different from default or empty string
					if os.path.splitext(external_file_path)[1] != ".py":
						QMessageBox.warning(self, "Invalid parameter value", "Exteranl File Path (%s) should include the python script extension (.py)." % (external_file_path))
						return ""
					dir_path, file_basename = os.path.split(external_file_path)
					file_basename = file_basename.replace(".py", "")
					sxcmd_line += " %s%s=[%s,%s,%s]" % (sxcmd_token.key_prefix, sxcmd_token.key_base, dir_path, file_basename, user_func_name)
				elif user_func_name != sxcmd_token.default[user_func_name_index]:
					# Case 2: User specified an internal function different from default
					sxcmd_line += " %s%s=%s" % (sxcmd_token.key_prefix, sxcmd_token.key_base, user_func_name)
				# else: User left default value. Do nothing
			# Then, handle the other cases//
			else:
				if sxcmd_token.type == "bool_ignore":
					### Possbile cases:
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == False: # Add this token to command line
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == False: # Do not add this token to command line
					### 
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == False: # Do not add this token to command line
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == False: # Add this token to command line
					### 
					#else: # Do not add this token to command line
					continue
					
				elif sxcmd_token.type == "bool":
					### Possbile cases:
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == False: # Add this token to command line
					### if not sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == False: # Do not add this token to command line
					### 
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == True:  # Add this token to command line
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default  and  sxcmd_token.is_required == False: # Do not add this token to command line
					### if     sxcmd_token.is_reversed  and  (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default  and  sxcmd_token.is_required == False: # Add this token to command line
					### 
					is_flag_required = False
					if sxcmd_token.is_required:
						is_flag_required = True
					else:
						assert (not sxcmd_token.is_required)
						if not sxcmd_token.is_reversed and (sxcmd_token.widget.checkState() == Qt.Checked) != sxcmd_token.default:
							is_flag_required = True
						elif sxcmd_token.is_reversed and (sxcmd_token.widget.checkState() == Qt.Checked) == sxcmd_token.default:
							is_flag_required = True
						else:
							assert (not is_flag_required)
					
					if is_flag_required:
						sxcmd_line += " %s%s" % (sxcmd_token.key_prefix, sxcmd_token.key_base)
					#else: # Do not add this token to command line
					
				else:
					if sxcmd_token.widget.text() == sxcmd_token.default:
						### if sxcmd_token.widget.text() == sxcmd_token.default and sxcmd_token.is_required == True:  # Error case
						if sxcmd_token.is_required == True:
							QMessageBox.warning(self, "Invalid parameter value", "Token (%s) of command (%s) is required. Please set the value for this." % (sxcmd_token.label, self.sxcmd.get_mode_name_for("human")))
							return ""
						### if sxcmd_token.widget.text() == sxcmd_token.default and sxcmd_token.is_required == False: # Do not add this token to command line
						# else: # assert(sxcmd_token.is_required == False) # Do not add to this command line
					else: # sxcmd_token.widget.text() != sxcmd_token.default
						### if sxcmd_token.widget.text() != sxcmd_token.default and sxcmd_token.is_required == True:  # Add this token to command line
						### if sxcmd_token.widget.text() != sxcmd_token.default and sxcmd_token.is_required == False: # Add this token to command line

						# For now, using line edit box for the other type
						widget_text = str(sxcmd_token.widget.text())
###						if sxcmd_token.type not in ["int", "float", "apix", "ctfwin", "box", "radius", "mass", "displayable_list", "mic_one_list", "any_file_list", "any_image_list", "dir_list"]:
						if sxcmd_token.type not in ["int", "float", "abs_freq", "apix", "ctfwin", "box", "radius", "mass", "displayable_list", "mic_one_list", "dir_list", "filament_width"]:
							# Always enclose the string value with single quotes (')
							widget_text = widget_text.strip("\'")  # make sure the string is not enclosed by (')
							widget_text = widget_text.strip("\"")  # make sure the string is not enclosed by (")
							widget_text = "\'%s\'" % (widget_text) # then, enclose the string value with single quotes (')

						if sxcmd_token.key_prefix == "":
							sxcmd_line += " %s" % (widget_text)
						elif sxcmd_token.key_prefix == "--":
							sxcmd_line += " %s%s=%s" % (sxcmd_token.key_prefix, sxcmd_token.key_base, widget_text)
						else:
							ERROR("Logical Error: Encountered unexpected prefix for token (%s) of command (%s). Consult with the developer." % (sxcmd_token.key_base, self.sxcmd.get_mode_name_for("human")), "%s in %s" % (__name__, os.path.basename(__file__)))
						# else: # assert(sxcmd_token.widget.text() == sxcmd_token.default) # Do not add to this command line

		return sxcmd_line

	def generate_cmd_line(self):
		# Generate SX command line
		sxcmd_line = self.map_widgets_to_sxcmd_line()

		if sxcmd_line:
			# SX command line is not empty
			# If mpi is not supported set number of MPI processer (np) to 1
			np = 1
			if self.sxcmd.mpi_support:
				# mpi is supported
				np = int(str(self.sxcmd_tab_main.mpi_nproc_edit.text()))
				#
				# DESIGN_NOTE: 2016/03/17 Toshio Moriya
				# The MPI policy below has changed!!! An example of this exception is sxcter.py.
				# Don't add --MPI flag if np == 1
				#
				# DESIGN_NOTE: 2015/10/27 Toshio Moriya
				# Since we now assume sx*.py exists in only MPI version, always add --MPI flag if necessary
				# This is not elegant but can be removed when --MPI flag is removed from all sx*.py scripts
				#
				if self.sxcmd.mpi_add_flag and np > 1:
					sxcmd_line += " --MPI"

				# DESIGN_NOTE: 2016/02/11 Toshio Moriya
				# Ideally, the following exceptional cases should not handled in here
				# because it will remove the generality from the software design
				required_key_base = None
				if self.sxcmd.name == "sxisac":
					required_key_base = "indep_run"
				elif self.sxcmd.name == "sxviper":
					required_key_base = "nruns"
				elif self.sxcmd.name == "sxrviper":
					required_key_base = "n_shc_runs"
				# else: # Do nothing

				if required_key_base != None:
					required_divisor = int(str(self.sxcmd.token_dict[required_key_base].widget.text()))
					required_label =  self.sxcmd.token_dict[required_key_base].label
					if required_divisor == 0:
						QMessageBox.warning(self, "Invalid parameter value", "\"%s\" must be larger than 0. Please check the setting" % (required_label))
						return ""

					valid_np = np
					if valid_np % required_divisor != 0:
						if valid_np < required_divisor:
							valid_np = required_divisor
						else:
							valid_np = valid_np - (valid_np % required_divisor)
						QMessageBox.warning(self, "Invalid parameter value", "The number of \"MPI processes\" (%d) is invalid. It MUST BE multiplicity of \"%s\" (%d). Please check the setting. A close valid number is %d." % (np, required_label, required_divisor,valid_np))
						return ""

			# else: assert(np == 1) # because the "MPI Processes" is disabled for sx*.py process which does not support mpi

			# Generate command line according to the case
			cmd_line = ""
			if self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Checked:
				# Case 1: queue submission is enabled (MPI can be supported or unsupported)
				# Create script for queue submission from a give template
				if os.path.exists(self.sxcmd_tab_main.qsub_script_edit.text()) != True:
					QMessageBox.warning(self, "Invalid parameter value", "Invalid file path for qsub script template (%s)." % (self.sxcmd_tab_main.qsub_script_edit.text()))
					return ""

				file_template = open(self.sxcmd_tab_main.qsub_script_edit.text(),"r")
				# Extract command line from qsub script template
				for line in file_template:
					if line.find("XXX_SXCMD_LINE_XXX") != -1:
						if np > 1:
							cmd_line = line
						else:
							cmd_line = "XXX_SXCMD_LINE_XXX " + line.split('XXX_SXCMD_LINE_XXX')[-1]
						cmd_line = cmd_line.replace("XXX_SXCMD_LINE_XXX", sxcmd_line)
						if cmd_line.find("XXX_SXMPI_NPROC_XXX") != -1:
							cmd_line = cmd_line.replace("XXX_SXMPI_NPROC_XXX", str(np))
						if cmd_line.find("XXX_SXMPI_JOB_NAME_XXX") != -1:
							cmd_line = cmd_line.replace("XXX_SXMPI_JOB_NAME_XXX", str(self.sxcmd_tab_main.qsub_job_name_edit.text()))
				file_template.close()
			elif self.sxcmd.mpi_support:
				# Case 2: queue submission is disabled, but MPI is supported
				if self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Checked: ERROR("Logical Error: Encountered unexpected condition for sxcmd_tab_main.qsub_enable_checkbox.checkState. Consult with the developer.", "%s in %s" % (__name__, os.path.basename(__file__)))
				# Add MPI execution to command line
				cmd_line = str(self.sxcmd_tab_main.mpi_cmd_line_edit.text())
				# If empty string is entered, use a default template
				if cmd_line == "":
					if np > 1:
						cmd_line = "mpirun -np XXX_SXMPI_NPROC_XXX XXX_SXCMD_LINE_XXX"
					else:
						cmd_line = "XXX_SXCMD_LINE_XXX " + cmd_line.split('XXX_SXCMD_LINE_XXX')[-1]
				if cmd_line.find("XXX_SXMPI_NPROC_XXX") != -1:
					cmd_line = cmd_line.replace("XXX_SXMPI_NPROC_XXX", str(np))
				if cmd_line.find("XXX_SXCMD_LINE_XXX") != -1:
					cmd_line = cmd_line.replace("XXX_SXCMD_LINE_XXX", sxcmd_line)
			else:
				# Case 3: queue submission is disabled, and MPI is not supported
				if self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Checked: ERROR("Logical Error: Encountered unexpected condition for sxcmd_tab_main.qsub_enable_checkbox.checkState. Consult with the developer.", "%s in %s" % (__name__, os.path.basename(__file__)))
				# Use sx command as it is
				cmd_line = sxcmd_line
		else:
			# SX command line is be empty because an error happens in map_widgets_to_sxcmd_line
			cmd_line = ""

		return cmd_line

	@QtCore.pyqtSlot()
	def execute_cmd_line(self, execute=True, output_dir='.', number=None):
		# Disable the run command button
		execute_btn = self.sender()
		execute_btn.setEnabled(False)
		QtCore.QTimer.singleShot(5000, lambda: execute_btn.setEnabled(True))

		# Generate command line
		cmd_line = self.generate_cmd_line()

		if cmd_line:
			# Command line is not empty
			# First, check existence of outputs
			for sxcmd_token in self.sxcmd.token_list:
				if sxcmd_token.type == "output" or sxcmd_token.type == "output_continue" or sxcmd_token.type == "output_bdb2d_stack":
					if os.path.exists(sxcmd_token.widget.text()) or db_check_dict(str(sxcmd_token.widget.text())):
						# DESIGN_NOTE: 2015/11/24 Toshio Moriya
						# This special case needs to be handled with more general method...
						if sxcmd_token.type == "output_continue":
							reply = QMessageBox.question(self, "Output Directory/File", "Output Directory/File (%s) already exists. Do you really want to run the program with continue mode?" % (sxcmd_token.widget.text()), QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
							if reply == QMessageBox.No:
								return
							# else: # Do nothing
						else:
							assert(sxcmd_token.type == "output" or sxcmd_token.type == "output_bdb2d_stack")
							QMessageBox.warning(self, "Output Directory/File", "Output Directory/File (%s) already exists. Please change the name and try it again. Aborting execution ..." % (sxcmd_token.widget.text()))
							return

			# If mpi is not supported set number of MPI processer (np) to 1
			np = 1
			if self.sxcmd.mpi_support:
				np = int(str(self.sxcmd_tab_main.mpi_nproc_edit.text()))

			if self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Checked:
				# Case 1: queue submission is enabled (MPI can be supported or unsupported)
				# Create script for queue submission from a give template
				submit_command_prefix = str(self.sxcmd_tab_main.qsub_cmd_edit.text()).split()[0]+"_"

				template_file_path = self.sxcmd_tab_main.qsub_script_edit.text()
				if os.path.exists(template_file_path) == False:
					QMessageBox.warning(self, "Invalid parameter value", "Invalid file path for qsub script template (%s). Aborting execution ..." % (template_file_path))
					return
				file_template = open(self.sxcmd_tab_main.qsub_script_edit.text(),"r")
				if number is not None:
					file_name_qsub_script = os.path.join(output_dir, "{0:04d}_".format(number) + submit_command_prefix + str(self.sxcmd_tab_main.qsub_job_name_edit.text()) + ".sh")
				else:
					file_name_qsub_script = os.path.join(output_dir, submit_command_prefix + str(self.sxcmd_tab_main.qsub_job_name_edit.text()) + ".sh")
				file_qsub_script = open(file_name_qsub_script,"w")
				for line_io in file_template:
					if line_io.find("XXX_SXCMD_LINE_XXX") != -1:
						line_io = cmd_line
					else:
						if line_io.find("XXX_SXMPI_NPROC_XXX") != -1:
							line_io = line_io.replace("XXX_SXMPI_NPROC_XXX", str(np))
						if line_io.find("XXX_SXMPI_JOB_NAME_XXX") != -1:
							line_io = line_io.replace("XXX_SXMPI_JOB_NAME_XXX", str(self.sxcmd_tab_main.qsub_job_name_edit.text()))
					file_qsub_script.write(line_io)
				file_template.close()
				file_qsub_script.close()
				# Generate command line for queue submission
				cmd_line_in_script = cmd_line
				cmd_line = str(self.sxcmd_tab_main.qsub_cmd_edit.text()) + " " + file_name_qsub_script
				print("Wrote the following command line in the queue submission script: ")
				print(cmd_line_in_script)
				if execute:
					print("Submitted a job by the following command: ")
					print(cmd_line)
				else:
					print("Saved command to: ")
					print(file_name_qsub_script)
			elif self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Unchecked and not execute:
				QMessageBox.warning(self, "Qsub template needs to be specified for pipeline option", "Qsub template needs to be specified for pipeline option")
				return
			else:
				# Case 2: queue submission is disabled (MPI can be supported or unsupported)
				if self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Checked: ERROR("Logical Error: Encountered unexpected condition for sxcmd_tab_main.qsub_enable_checkbox.checkState. Consult with the developer.", "%s in %s" % (__name__, os.path.basename(__file__)))
				print("Executed the following command: ")
				print(cmd_line)

			# Execute the generated command line
			if execute:
				process = subprocess.Popen(cmd_line, shell=True)
				### self.process_started.emit(process.pid)
				if self.sxcmd.is_submittable == False:
					assert(self.sxcmd.mpi_support == False)
					# Register to This is a GUI application
					self.child_application_list.append(process)

			# Save the current state of GUI settings
			if os.path.exists(self.sxcmd.get_category_dir_path(SXLookFeelConst.project_dir)) == False:
				os.makedirs(self.sxcmd.get_category_dir_path(SXLookFeelConst.project_dir))
			self.write_params(self.gui_settings_file_path)
		# else: SX command line is be empty because an error happens in generate_cmd_line. Let's do nothing

	def add_to_pipeline(self):
		name = QtGui.QFileDialog.getExistingDirectory(self, 'Pipeline output directory', SXLookFeelConst.file_dialog_dir, options = QFileDialog.DontUseNativeDialog)
		if isinstance(name, tuple):
			dir_name = str(name[0])
		else:
			dir_name = str(name)
		if dir_name:
			number_match = re.compile('(\d{4})_')
			number = -1
			for file_name in sorted(os.listdir(dir_name)):
				try:
					number = number_match.match(file_name).group(1)
				except AttributeError:
					QMessageBox.warning(self, "Invalid file in pipeline directory", "There is a file in the pipeline directory not starting with four digits. Abort...")
					return

			self.execute_cmd_line(execute=False, output_dir=dir_name, number=int(number)+1)

	def print_cmd_line(self):
		# Generate command line
		cmd_line = self.generate_cmd_line()
		if cmd_line:
			message_line = "Generated the following command line:"
			print(message_line)
			print(cmd_line)
			QtGui.QMessageBox.information(self, "Information","%s \n\n%s" % (message_line, cmd_line))

			# Save the current state of GUI settings
			if os.path.exists(self.sxcmd.get_category_dir_path(SXLookFeelConst.project_dir)) == False:
				os.makedirs(self.sxcmd.get_category_dir_path(SXLookFeelConst.project_dir))
			self.write_params(self.gui_settings_file_path)
		# else: Do nothing

	def show_dialog_calculated_res(self):
		# Generate command line
		calculate = FSCPlot(self)
		fsc_plot.plot(*args)

	def write_params(self, file_path_out):
		file_out = open(file_path_out,"w")

		# Write script name for consistency check upon loading
		file_out.write("@@@@@ %s gui settings - " % (self.sxcmd.get_mode_name_for("human")))
		# file_out.write(EMANVERSION + " (CVS" + CVSDATESTAMP[6:-2] +")")
		file_out.write(EMANVERSION + " (GITHUB: " + DATESTAMP +")" )
		file_out.write(" @@@@@ \n")

		# Define list of (tab) groups
		group_main = "main"
		group_advanced = "advanced"

		# Loop through all groups. First write out values of widgets in main tab, then ones in advanced
		for group in [group_main, group_advanced]:
			# Loop through all command tokens
			for cmd_token in self.sxcmd.token_list:
				if cmd_token.group == group:
					# First, handle very special cases
					if cmd_token.type == "user_func":
						# This type has two line edit boxes as a list of widget
						n_widgets = 2
						for widget_index in range(n_widgets):
							val_str = str(cmd_token.widget[widget_index].text())
							file_out.write("<%s> %s (default %s) == %s \n" % (cmd_token.key_base, cmd_token.label[widget_index], cmd_token.default[widget_index], val_str))
					# Then, handle the other cases
					else:
						val_str = ""
						if cmd_token.type in ("bool", "bool_ignore"):
							if cmd_token.widget.checkState() == Qt.Checked:
								val_str = "YES"
							else:
								val_str = "NO"
						else:
							# The other type has only one line edit box
							val_str = str(cmd_token.widget.text())

						if cmd_token.is_required == False:
							file_out.write("<%s> %s (default %s) == %s \n" % (cmd_token.key_base, cmd_token.label, cmd_token.default, val_str))
						else:
							file_out.write("<%s> %s (default required %s) == %s \n" % (cmd_token.key_base, cmd_token.label, cmd_token.type, val_str))
				# else: do nothig

		# At the end of parameter file...
		# Write MPI parameters
		file_out.write("%s == %s \n" % ("MPI processors", str(self.sxcmd_tab_main.mpi_nproc_edit.text())))
		file_out.write("%s == %s \n" % ("MPI Command Line Template", str(self.sxcmd_tab_main.mpi_cmd_line_edit.text())))
		# Write Qsub parameters
		if self.sxcmd_tab_main.qsub_enable_checkbox.checkState() == Qt.Checked:
			val_str = "YES"
		else:
			val_str = "NO"
		file_out.write("%s == %s \n" % ("Submit Job to Queue", val_str))
		file_out.write("%s == %s \n" % ("Job Name", str(self.sxcmd_tab_main.qsub_job_name_edit.text())))
		file_out.write("%s == %s \n" % ("Submission Command", str(self.sxcmd_tab_main.qsub_cmd_edit.text())))
		file_out.write("%s == %s \n" % ("Submission Script Template", str(self.sxcmd_tab_main.qsub_script_edit.text())))

		file_out.close()

	def read_params(self, file_path_in):
		file_in = open(file_path_in,"r")

		# Check if this parameter file is for this sx script
		line_in = file_in.readline()
		if line_in.find("@@@@@ %s gui settings" % (self.sxcmd.get_mode_name_for("human"))) != -1:
			n_function_type_lines = 2
			function_type_line_counter = 0
			cmd_token_apix = None
			# loop through the rest of lines
			for line_in in file_in:
				# Extract label (which should be left of "=="). Also strip the ending spaces
				label_in = line_in.split("==")[0].strip()
				# Extract value (which should be right of "=="). Also strip all spaces
				val_str_in = line_in.split("==")[1].strip()

				if label_in == "MPI processors":
					self.sxcmd_tab_main.mpi_nproc_edit.setText(val_str_in)
				elif label_in == "MPI Command Line Template":
					self.sxcmd_tab_main.mpi_cmd_line_edit.setText(val_str_in)
				elif label_in == "Submit Job to Queue":
					if val_str_in == "YES":
						self.sxcmd_tab_main.qsub_enable_checkbox.setChecked(Qt.Checked)
					else: # assert(val_str_in == "NO")
						self.sxcmd_tab_main.qsub_enable_checkbox.setChecked(Qt.Unchecked)
					# self.sxcmd_tab_main.set_qsub_enable_state() # Somehow this place does not paint the text boxes upon application startup
				elif label_in == "Job Name":
					self.sxcmd_tab_main.qsub_job_name_edit.setText(val_str_in)
				elif label_in == "Submission Command":
					self.sxcmd_tab_main.qsub_cmd_edit.setText(val_str_in)
				elif label_in == "Submission Script Template":
					self.sxcmd_tab_main.qsub_script_edit.setText(val_str_in)
				else:
					# Extract key_base of this command token
					target_operator = "<"
					item_tail = label_in.find(target_operator)
					if item_tail != 0:
						QMessageBox.warning(self, "Invalid Parameter File Format", "Command token entry should start from \"%s\" for key base name in line (%s) of file (%s). The format of this file might be corrupted. Please save the parameter file again." % (target_operator, line_in, file_path_in))
					label_in = label_in[item_tail + len(target_operator):].strip() # Get the rest of line
					target_operator = ">"
					item_tail = label_in.find(target_operator)
					if item_tail == -1:
						QMessageBox.warning(self, "Invalid Parameter File Format", "Command token entry should have \"%s\" closing key base name in line (%s) of file (%s). The format of this file might be corrupted. Please save the parameter file again." % (target_operator, line_in, file_path_in))
					key_base = label_in[0:item_tail]
					# Get corresponding cmd_token
					if key_base not in list(self.sxcmd.token_dict.keys()):
						QMessageBox.warning(self, "Invalid Parameter File Format", "Invalid base name of command token \"%s\" is found in line (%s) of file (%s). This parameter file might be incompatible with the current version. Please save the parameter file again." % (key_base, line_in, file_path_in))
					else:
						cmd_token = self.sxcmd.token_dict[key_base]
					if not cmd_token.is_locked: 
						# First, handle very special cases
						if cmd_token.type == "user_func":
							cmd_token.widget[function_type_line_counter].setText(val_str_in)
							function_type_line_counter += 1
							function_type_line_counter %= n_function_type_lines # function have two line edit boxes
						else:
							if cmd_token.type in ("bool", "bool_ignore"):
								# construct new widget(s) for this command token
								if val_str_in == "YES":
									cmd_token.widget.setChecked(Qt.Checked)
								else: # val_str_in == "NO"
									cmd_token.widget.setChecked(Qt.Unchecked)
							# Then, handle the other cases
							else:
								# For now, use line edit box for the other type
								cmd_token.widget.setText(val_str_in)
								if cmd_token.type == "apix":
									cmd_token_apix = cmd_token 
			if cmd_token_apix is not None:
				assert (cmd_token_apix.type == "apix")
				# if len(cmd_token_apix.other_dialog_list) > 0:
				# 	print("MRK_DEBUG: ")
				# 	print("MRK_DEBUG: ----- SXCmdWidget.read_params() ----- ")
				# 	print("MRK_DEBUG: cmd_token_apix.widget.text() := \"{}\"".format(cmd_token_apix.widget.text()))
				# 	print("MRK_DEBUG: len(cmd_token_apix.other_dialog_list) := \"{}\"".format(len(cmd_token_apix.other_dialog_list)))
				for sxcmd_token_apix_other_dialog in cmd_token_apix.other_dialog_list:
				# 	print("MRK_DEBUG: BEFORE sxcmd_token_apix_other_dialog.sxconst_register_widget_apix.text() := \"{}\"".format(sxcmd_token_apix_other_dialog.sxconst_register_widget_apix.text()))
				# 	print("MRK_DEBUG: BEFORE sxcmd_token_apix_other_dialog.sxcmd_token_widget_apix.text() := \"{}\"".format(sxcmd_token_apix_other_dialog.sxcmd_token_widget_apix.text()))
				# 	print("MRK_DEBUG: BEFORE sxcmd_token_apix_other_dialog.sxcmd_token_widget_abs_freq.text() := \"{}\"".format(sxcmd_token_apix_other_dialog.sxcmd_token_widget_abs_freq.text()))
					sxcmd_token_apix_other_dialog.reflect_external_local_update_apix_and_abs_freq()
				# 	print("MRK_DEBUG: AFTER sxcmd_token_apix_other_dialog.sxconst_register_widget_apix.text() := \"{}\"".format(sxcmd_token_apix_other_dialog.sxconst_register_widget_apix.text()))
				# 	print("MRK_DEBUG: AFTER sxcmd_token_apix_other_dialog.sxcmd_token_widget_apix.text() := \"{}\"".format(sxcmd_token_apix_other_dialog.sxcmd_token_widget_apix.text()))
				# 	print("MRK_DEBUG: AFTER sxcmd_token_apix_other_dialog.sxcmd_token_widget_abs_freq.text() := \"{}\"".format(sxcmd_token_apix_other_dialog.sxcmd_token_widget_abs_freq.text()))
		else:
			QMessageBox.warning(self, "Fail to load parameters", "The specified file parameter file for %s could not be read." % self.sxcmd.get_mode_name_for("human"))

		file_in.close()

	def save_params(self):
		name = QFileDialog.getSaveFileName(self, "Save Parameters", SXLookFeelConst.file_dialog_dir, options = QFileDialog.DontUseNativeDialog)
		if isinstance(name, tuple):
			file_path = str(name[0])
		else:
			file_path = str(name)
		if file_path != "":
			self.write_params(file_path)

	def load_params(self):
		name = QFileDialog.getOpenFileName(self, "Load parameters", SXLookFeelConst.file_dialog_dir, options = QFileDialog.DontUseNativeDialog)
		if isinstance(name, tuple):
			file_path = str(name[0])
		else:
			file_path = str(name)
		if file_path != "":
			self.read_params(file_path)
			self.sxcmd_tab_main.set_qsub_enable_state()

	def select_file(self, target_widget, file_format = ""):
		file_path = ""
		# NOTE: Toshio Moriya 2018/01/25
		# All supported image/volume formats according to http://blake.bcm.edu/emanwiki/EMAN2ImageFormats
		# ;; HDF (*.hdf);; MRC (*.mrc);; MRCS (*.mrcs);; Spider (*.spi);; Imagic (*.img *hed);; TIFF (*.tif *.tiff);; PNG (*.png);; JPEG (*.jpg *.jpeg);; BDB (*.bdb);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.hdr *.img);; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor)
		# 
		if file_format == "displayable_list":
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			name = QFileDialog.getOpenFileNames(self, "Select any displayable files", SXLookFeelConst.file_dialog_dir, "Typical displayable files (*.hdf *.bdb *.mrc *.mrcs *.spi *.img *.tif *.tiff *.png *.txt);; HDF (*.hdf);; BDB (*.bdb);; MRC (*.mrc);; MRCS (*.mrcs);; Spider (*.spi);; Imagic (*.img *hed);; TIFF (*.tif *.tiff);; PNG (*.png);; Text (*.txt);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.hdr *.img);; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path_list = name[0]
			else:
				file_path_list = name
			for a_file_path in file_path_list:
				# Use relative path.
				a_file_path = str(SXLookFeelConst.format_path(a_file_path))
				try: # Check if the path is bdb
					a_file_path = translate_to_bdb_path(a_file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
				file_path += a_file_path + " "
		elif file_format == "data2d3d_both":
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			name = QFileDialog.getOpenFileName(self, "Select any image/volume file", SXLookFeelConst.file_dialog_dir, "Typical image & volume files (*.hdf *.bdb *.mrc *.mrcs *.spi *.img *.tif *.tiff *.png);; HDF (*.hdf);; BDB (*.bdb);; MRC (*.mrc);; MRCS (*.mrcs);; Spider (*.spi);; Imagic (*.img *hed);; TIFF (*.tif *.tiff);; PNG (*.png);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.hdr *.img);; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "mrc2d_mic_both":
			name = QFileDialog.getOpenFileName(self, "Select MRC micrograph/movie file", SXLookFeelConst.file_dialog_dir, "MRC micrograph & movie files (*.mrc *.mrcs);; MRC (*.mrc);; MRCS (*.mrcs)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "mic_both":
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			name = QFileDialog.getOpenFileName(self, "Select any micrograph/movie file", SXLookFeelConst.file_dialog_dir, "Typical micrograph & movie files (*.mrc *.mrcs *.tif *.tiff *.hdf *.bdb *.spi *.img);; MRC (*.mrc);; MRCS (*.mrcs);; TIFF (*.tif *.tiff);; HDF (*.hdf);; BDB (*.bdb);; Spider (*.spi);; Imagic (*.img);; PNG (*.png);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.img );; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "mrc2d_mic_one":
			name = QFileDialog.getOpenFileName(self, "Select MRC micrograph file", SXLookFeelConst.file_dialog_dir, "MRC micrograph files (*.mrc);; MRCS (*.mrcs)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "mic_one":
			# NOTE: Toshio Moriya 2018/01/25
			# Currently, the distinction between MRC and MRCS is not always used, and 
			# MRC can be also micrograph stack dependes on external programs (e.g. unblur, summovie)...
			# 
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# Only stack: ;; MRCS (*.mrcs)
			name = QFileDialog.getOpenFileName(self, "Select any micrograph file", SXLookFeelConst.file_dialog_dir, "Typical micrograph files (*.mrc *.mrcs *.tif *.tiff *.hdf *.bdb *.spi *.img);; MRC (*.mrc);; MRCS (*.mrcs);; TIFF (*.tif *.tiff);; HDF (*.hdf);; BDB (*.bdb);; Spider (*.spi);; Imagic (*.img);; PNG (*.png);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.img );; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "mrc2d_mic_one_list":
			# NOTE: Toshio Moriya 2018/01/25
			# Currently, the distinction between MRC and MRCS is not always used, and 
			# MRC can be also micrograph stack dependes on external programs (e.g. unblur, summovie)...
			# 
			name = QFileDialog.getOpenFileNames(self, "Select MRC micrograph files", SXLookFeelConst.file_dialog_dir, "MRC files (*.mrc);; MRCS (*.mrcs)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path_list = name[0]
			else:
				file_path_list = name
			# Use relative path.
			for a_file_path in file_path_list:
				file_path += str(SXLookFeelConst.format_path(a_file_path)) + " "
		elif file_format == "mic_one_list":
			# NOTE: Toshio Moriya 2018/01/25
			# Currently, the distinction between MRC and MRCS is not always used, and 
			# MRC can be also micrograph stack dependes on external programs (e.g. unblur, summovie)...
			# 
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# Only stack: ;; MRCS (*.mrcs)
			name = QFileDialog.getOpenFileNames(self, "Select any micrograph files", SXLookFeelConst.file_dialog_dir, "Typical micrograph files (*.mrc *.mrcs *.tif *.tiff *.hdf *.bdb *.spi *.img);; MRC (*.mrc);; MRCS (*.mrcs);; TIFF (*.tif *.tiff);; HDF (*.hdf);; BDB (*.bdb);; Spider (*.spi);; Imagic (*.img);; PNG (*.png);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.img );; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path_list = name[0]
			else:
				file_path_list = name
			# Use relative path.
			for a_file_path in file_path_list:
				a_file_path = str(SXLookFeelConst.format_path(a_file_path))
				try: # Check if the path is bdb
					a_file_path = translate_to_bdb_path(a_file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
				file_path += a_file_path + " "
		elif file_format == "mrc2d_mic_stack":
			name = QFileDialog.getOpenFileName(self, "Select MRC movie file", SXLookFeelConst.file_dialog_dir, "MRC movie files (*.mrcs);; MRC (*.mrc)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "mic_stack":
			# NOTE: Toshio Moriya 2018/01/25
			# Currently, the distinction between MRC and MRCS is not always used, and 
			# MRC can be also micrograph stack dependes on external programs (e.g. unblur, summovie)...
			# 
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# 2D image stack not supported: ;; Gatan (*.dm2 *.dm3);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; OMAP (*.omap);; PGM (*.pgm);; PNG (*.png);; SAL (*.hdr *.img);; SITUS (*.situs);; TIFF (*.tif *.tiff);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor)
			# Maybe only single 2D image: ;; MRC (*.mrc)
			name = QFileDialog.getOpenFileName(self, "Select any movie file", SXLookFeelConst.file_dialog_dir, "Typical movie files (*.mrcs *.mrc *.bdb *.hdf *.spi *.img );; MRCS (*.mrcs);; MRC (*.mrc);; BDB (*.bdb);; HDF (*.hdf);; Spider (*.spi);; Imagic (*.img *hed);; Gatan (*.dm4);; FEI (*.ser);; LST (*.lst);; LSTFAST (*.lsx *.lst);; PIF (*.pif);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "hdf2d_one":
			name = QFileDialog.getOpenFileName(self, "Select HDF image file", SXLookFeelConst.file_dialog_dir, "HDF image files (*.hdf)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "data2d_one":
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# Maybe only 2D image stack: ;; MRCS (*.mrcs)
			name = QFileDialog.getOpenFileName(self, "Select any image file", SXLookFeelConst.file_dialog_dir, "Typical image files (*.hdf *.bdb *.mrc *.spi *.img *.tif *.tiff *.png *.mrcs);; HDF (*.hdf);; BDB (*.bdb);; MRC (*.mrc *.mrcs);; Spider (*.spi);; Imagic (*.img);; TIFF (*.tif *.tiff);; PNG (*.png);; Gatan (*.dm2 *.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; SAL (*.img );; SITUS (*.situs);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "bdb2d_stack":
			name = QFileDialog.getOpenFileName(self, "Select BDB image stack file", SXLookFeelConst.file_dialog_dir, "BDB files (*.bdb)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				file_path = translate_to_bdb_path(file_path)
		elif file_format == "data2d_stack":
			# NOTE: Toshio Moriya 2018/01/25
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# 2D image stack not supported: ;; Gatan (*.dm2 *.dm3);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; OMAP (*.omap);; PGM (*.pgm);; PNG (*.png);; SAL (*.hdr *.img);; SITUS (*.situs);; TIFF (*.tif *.tiff);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor)
			# Maybe only single 2D image: ;; MRC (*.mrc)
			name = QFileDialog.getOpenFileName(self, "Select any image stack file", SXLookFeelConst.file_dialog_dir, "Typical image stack files (*.bdb *.hdf *.mrcs *.spi *.img );; BDB (*.bdb);; HDF (*.hdf);; MRCS (*.mrcs);; Spider (*.spi);; Imagic (*.img *hed);; Gatan (*.dm4);; FEI (*.ser);; LST (*.lst);; LSTFAST (*.lsx *.lst);; PIF (*.pif);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "hdf3d_one":
			name = QFileDialog.getOpenFileName(self, "Select HDF volume file", SXLookFeelConst.file_dialog_dir, "HDF volume files (*.hdf)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "data3d_one":
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# 3D volume not supported: ;; Gatan (*.dm2 *.dm3);; FEI (*.ser);; SAL (*.hdr *.img);; PGM (*.pgm);; PNG (*.png);; TIFF (*.tif *.tiff);; V4L (*.v4l)
			# Maybe only 3D volume stack: ;; MRCS (*.mrcs)
			name = QFileDialog.getOpenFileName(self, "Select any volume file", SXLookFeelConst.file_dialog_dir, "Typical volume files (*.hdf *.bdb *.mrc *.spi *.img);; HDF (*.hdf);; BDB (*.bdb);; MRC (*.mrc);; Spider (*.spi);; Imagic (*.img);; Gatan (*.dm4);; EM (*.em);; ICOS (*.icos);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PIF (*.pif);; SITUS (*.situs);; VTK (*.vtk);; XPLOR (*.xplor);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "hdf3d_stack":
			# NOTE: Toshio Moriya 2018/01/25
			# Currently, this case is not used.
			# 
			name = QFileDialog.getOpenFileName(self, "Select HDF volume stack file", SXLookFeelConst.file_dialog_dir, "HDF volume stack files (*.hdf)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "data3d_stack":
			# NOTE: Toshio Moriya 2018/01/25
			# Currently, this case is not used.
			# 
			# Read not supported: ;; JPEG (*.jpg *.jpeg)
			# 3D volume stack not supported: ;; Gatan (*.dm2)
			# Maybe 3D volume stack not supported: ;; Gatan (*.dm3 *.dm4);; FEI (*.ser);; EM (*.em);; ICOS (*.icos);; Spider (*.spi);; Amira (*.am);; DF3 (*.d3);; FITS (*.fts);; LST (*.lst);; LSTFAST (*.lsx *.lst);; OMAP (*.omap);; PGM (*.pgm);; PIF (*.pif);; PNG (*.png);; SAL (*.hdr *.img);; SITUS (*.situs);; TIFF (*.tif *.tiff);; V4L (*.v4l);; VTK (*.vtk);; XPLOR (*.xplor)
			# Maybe only sigle 3D volume: ;; MRC (*.mrc)
			name = QFileDialog.getOpenFileName(self, "Select any volume stack file", SXLookFeelConst.file_dialog_dir, "Typical volume stack files (*.hdf *.bdb *.mrcs *.img);; HDF (*.hdf);; BDB (*.bdb);; MRCS (*.mrcs);; Imagic (*.img *hed);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
				try: # Check if the path is bdb
					file_path = translate_to_bdb_path(file_path) # Convert the standard path to bdb key if possible.
				except ValueError:  # If the path is not bdb, we will receive this exception
					pass # This is not bdb path. Then, use standard path
		elif file_format == "select_mic_both":
			name = QFileDialog.getOpenFileName(self, "Select micrograph/movie selection file", SXLookFeelConst.file_dialog_dir, "Micrograph/Movie selection files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "select_mic_one":
			name = QFileDialog.getOpenFileName(self, "Select micrograph selection file", SXLookFeelConst.file_dialog_dir, "Micrograph selection files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "select_mic_stack":
			name = QFileDialog.getOpenFileName(self, "Select micrograph movie selection file", SXLookFeelConst.file_dialog_dir, "Micrograph movie selection files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "select_data2d_stack":
			name = QFileDialog.getOpenFileName(self, "Select image selection file", SXLookFeelConst.file_dialog_dir, "Image selection files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "select_drift_params":
			name = QFileDialog.getOpenFileName(self, "Select drift shift params selection file", SXLookFeelConst.file_dialog_dir, "Drift shift params selection files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_any_txt":
			name = QFileDialog.getOpenFileName(self, "Select parameters file", SXLookFeelConst.file_dialog_dir, "Parameters text files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_proj_txt":
			name = QFileDialog.getOpenFileName(self, "Select projection parameters file", SXLookFeelConst.file_dialog_dir, "Projection parameters files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_coords_box":
			name = QFileDialog.getOpenFileName(self, "Select EMAN BOX coordinates file", SXLookFeelConst.file_dialog_dir, "EMAN BOX coordinates files (*.box)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_coords_any":
			name = QFileDialog.getOpenFileName(self, "Select any coordinates file", SXLookFeelConst.file_dialog_dir, "Typical coordinates files (*.box *.json *.dat *.txt);; EMAN BOX (*.box);; EMAN2 JSON (*.json);; SPIDER DAT (*.dat);; SPHIRE TXT (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_cter_txt":
			name = QFileDialog.getOpenFileName(self, "Select CTER partres parameters file", SXLookFeelConst.file_dialog_dir, "CTER partres parameters files (*.txt)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_rebox_rbx":
			name = QFileDialog.getOpenFileName(self, "Select SPHIRE rebox file", SXLookFeelConst.file_dialog_dir, "SPHIRE rebox files (*.rbx)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_drift_txt":
			name = QFileDialog.getOpenFileName(self, "Select drift shift parameters file", SXLookFeelConst.file_dialog_dir, "MotionCor2|cisTEM Unblur (*.log);;Drift shift parameters files (*.txt);;All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "rot_matrix":
			name = QFileDialog.getOpenFileName(self, "Select rotational matrix file", SXLookFeelConst.file_dialog_dir, "Rotational matrix files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_relion_star":
			name = QFileDialog.getOpenFileName(self, "Select RELION STAR file", SXLookFeelConst.file_dialog_dir, "RELION STAR files (*.star);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_any_json":
			name = QFileDialog.getOpenFileName(self, "Select JSON file", SXLookFeelConst.file_dialog_dir, "JSON files (*.json);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "params_any_h5":
			name = QFileDialog.getOpenFileName(self, "Select h5 file", SXLookFeelConst.file_dialog_dir, "h5 files (*.h5);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			# Use relative path.
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "spectrum1d":
			name = QFileDialog.getOpenFileName(self, "Select 1D power spectrum file", SXLookFeelConst.file_dialog_dir, "1D power spectrum files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "mtf":
			name = QFileDialog.getOpenFileName(self, "Select MTF data file", SXLookFeelConst.file_dialog_dir, "MTF data files (*.txt);; All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "pdb":
			name = QFileDialog.getOpenFileName(self, "Select PDB data file", SXLookFeelConst.file_dialog_dir, "PDB data files (*.pdb *.pdb*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "exe":
			name = QFileDialog.getOpenFileName(self, "Select executable file", SXLookFeelConst.file_dialog_dir, "All files (*);; Python (*.py) ;; Executable files (*.exe)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)
		elif file_format == "py":
			name = QFileDialog.getOpenFileName(self, "Select Python script file", SXLookFeelConst.file_dialog_dir, "PY files (*.py)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use full path
###		elif file_format == "bdb":
###			file_path = QFileDialog.getOpenFileName(self, "Select BDB image file", SXLookFeelConst.file_dialog_dir, "BDB files (*.bdb)", options = QFileDialog.DontUseNativeDialog)
###			# Use relative path.
###			if file_path:
###				file_path = SXLookFeelConst.format_path(file_path)
###				file_path = translate_to_bdb_path(file_path)
###		elif file_format == "any_file_list" or file_format == "any_image_list":
###			file_path_list = QFileDialog.getOpenFileNames(self, "Select files", SXLookFeelConst.file_dialog_dir, "All files (*)", options = QFileDialog.DontUseNativeDialog)
###			# Use relative path.
###			for a_file_path in file_path_list:
###				file_path += SXLookFeelConst.format_path(str(a_file_path)) + " "
		elif file_format == 'submission_template':
			if 'SPHIRE_SUBMISSION_SCRIPT_TEMPLATE_FOLDER' in os.environ:
				template_folder = os.environ['SPHIRE_SUBMISSION_SCRIPT_TEMPLATE_FOLDER']
			else:
				template_folder = SXLookFeelConst.file_dialog_dir
			name = QFileDialog.getOpenFileName(self, "Select any file", template_folder, "All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
		else:
			if file_format:
				name = QFileDialog.getOpenFileName(self, "Select %s file" % (file_format.upper()), SXLookFeelConst.file_dialog_dir, "%s files (*.%s)"  % (file_format.upper(), file_format), options = QFileDialog.DontUseNativeDialog)
			else:
				name = QFileDialog.getOpenFileName(self, "Select any file", SXLookFeelConst.file_dialog_dir, "All files (*)", options = QFileDialog.DontUseNativeDialog)
			if isinstance(name, tuple):
				file_path = str(name[0])
			else:
				file_path = str(name)
			# Use relative path.
			if file_path:
				file_path = SXLookFeelConst.format_path(file_path)

		if file_path != "":
			target_widget.setText(file_path)

	def select_dir(self, target_widget):
		dir_path = QFileDialog.getExistingDirectory(self, "Select directory", SXLookFeelConst.file_dialog_dir, options = QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks | QFileDialog.DontUseNativeDialog)
		if dir_path != "":
			# Use relative path.
			target_widget.setText(SXLookFeelConst.format_path(dir_path))

	def quit_all_child_applications(self):
		# Quit all child applications
		for child_application in self.child_application_list:
			child_application.kill()
			# child_application.terminate() # This call ends up outputing "Program interrupted" Message and it is not pretty...

	"""
#	def show_output_info(self):
#		QMessageBox.information(self, "sx* output","outdir is the name of the output folder specified by the user. If it does not exist, the directory will be created. If it does exist, the program will crash and an error message will come up. Please change the name of directory and restart the program.")
	"""

class SXLineEdit(QtGui.QLineEdit):
	my_changed_state = QtCore.pyqtSignal(str, str)

	def __init__(self, dependency, parent=None):
		super(SXLineEdit, self).__init__(parent)
		self.dependency = dependency
		self.textChanged.connect(lambda x: self.my_changed_state.emit(self.dependency, x))


class SXCheckBox(QtGui.QCheckBox):
	my_changed_state = QtCore.pyqtSignal(str, int)

	def __init__(self, dependency, parent=None):
		super(SXCheckBox, self).__init__(parent)
		self.dependency = dependency
		self.stateChanged.connect(lambda x: self.my_changed_state.emit(self.dependency, x))


# ========================================================================================
class SXCmdTab(QWidget):
	def __init__(self, name, helical, parent=None):
		super(SXCmdTab, self).__init__(parent)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.name = name
		self.sxcmdwidget = parent

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# local constants
		required_cmd_token_restore_tooltip = "Please enter the value manually"
		locked_cmd_token_restore_tooltip = "This value is locked"
		const_cmd_token_restore_tooltip = "Retrieve the registed constant value for this parameter"
		default_cmd_token_restore_tooltip = "Retrieve this default value"

		# Setting for layout
		grid_row_origin = 0; grid_col_origin = 0
		title_row_span = 1; title_col_span = 2
		short_info_row_span = 1; short_info_col_span = 5
		func_btn_row_span = 1; func_btn_col_span = 2
		token_label_row_span = 1; token_label_col_span = 4
		token_widget_row_span = 1; token_widget_col_span = 1.5
		cmd_frame_row_span = 32; cmd_frame_col_span = 7

		title_label_min_width = 180 # title_label_min_width = 150
		title_label_min_height = 40 #title_label_min_height = 80
		short_info_min_width = 260 # short_info_min_width = 360
		short_info_min_height = 40 # short_info_min_height = 80
		func_btn_min_width = 150
		btn_min_width = 300
		token_label_min_width = 300 # token_label_min_width = 360
		token_widget_min_width = 120
		mpi_label_min_width = 100

		# Setup global layout
		global_layout = QVBoxLayout(self)
		global_layout.setContentsMargins(0,0,0,0)
		global_layout.setSpacing(0)
		# Setup scroll area and its widget
		scroll_area = QScrollArea()
		# scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
		# scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn) # MRK_DEBUG: Useful during designing layout
		scroll_area.setWidgetResizable(True)
		scroll_area_widget = QWidget(scroll_area)
		# Setup scroll widget and its background color
		scroll_area.setStyleSheet("QScrollArea {background-color:transparent;}");
		### scroll_area_widget.setStyleSheet("background-color:transparent;");
		scroll_area_widget.setAutoFillBackground(True)
		scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
		palette = QPalette()
		palette.setBrush(QPalette.Background, QBrush(SXLookFeelConst.sxcmd_tab_bg_color))
		scroll_area_widget.setPalette(palette)
		# Register the widget to scroll area
		scroll_area.setWidget(scroll_area_widget)
		# Register the scroll area to the global layout
		global_layout.addWidget(scroll_area)

		# Setup other layouts
		scroll_layout = QVBoxLayout(scroll_area_widget)
		scroll_layout.setContentsMargins(0,0,0,0)
		title_hbox = QHBoxLayout()
		title_layout = QGridLayout()
		title_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		title_layout.setSpacing(SXLookFeelConst.grid_spacing)
#		title_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span, token_widget_min_width)
#		title_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_min_width)
#		title_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_min_width)
#		title_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_min_width)
		grid_layout = QGridLayout()
		grid_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		grid_layout.setSpacing(SXLookFeelConst.grid_spacing)
		grid_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span, token_widget_min_width)
		grid_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_min_width)
		grid_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_min_width)
		grid_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_min_width)
		submit_layout = QGridLayout()
		submit_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		submit_layout.setSpacing(SXLookFeelConst.grid_spacing)
		submit_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span, token_widget_min_width)
		submit_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_min_width)
		submit_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_min_width)
		submit_layout.setColumnMinimumWidth(grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_min_width)
		title_hbox.addLayout(title_layout)
#		title_hbox.addStretch(1)
		title_layout.setColumnStretch(grid_row_origin + token_label_col_span, title_layout.columnStretch(grid_row_origin+token_label_col_span) + 1)
		scroll_layout.addLayout(title_hbox)
		scroll_layout.addLayout(grid_layout)
		scroll_layout.addLayout(submit_layout)
		scroll_layout.addStretch(1)
		# # Give the columns of token label a higher priority to stretch relative to the others
		# for col_span in xrange(token_label_col_span):
		# 	grid_layout.setColumnStretch(grid_row_origin + col_span, grid_layout.columnStretch(grid_row_origin+col_span) + 1)

		# Define the tab frame within the tab layout
		# tab_frame = QFrame()
		# grid_layout.addWidget(tab_frame, grid_row_origin, grid_col_origin, cmd_frame_row_span, cmd_frame_col_span)

		# Start add command token widgets to the grid layout
		grid_row = grid_row_origin

		tab_group = self.name.lower()
		if tab_group == "main":
			# Set a label and its position in this tab
			temp_label = QLabel("<b>%s</b>" % (self.sxcmdwidget.sxcmd.get_mode_name_for("human")))
			temp_label.setMinimumWidth(title_label_min_width)
			temp_label.setMinimumHeight(title_label_min_height)
#			temp_label.setFixedWidth(title_label_min_width)
#			temp_label.setFixedHeight(title_label_min_height)
			title_layout.addWidget(temp_label, grid_row, grid_col_origin, title_row_span, title_col_span)

			#
			# NOTE: 2015/11/17 Toshio Moriya
			# Necessary to separate "<b>%s</b>" from the information for avoiding to invoke the tag interpretations of string
			# e.g. < becomes the escape character
			#
			temp_label = QLabel("%s" % (self.sxcmdwidget.sxcmd.short_info))
			temp_label.setWordWrap(True)
			temp_label.setMinimumWidth(short_info_min_width)
			temp_label.setMinimumHeight(short_info_min_height)
#			temp_label.setFixedHeight(short_info_min_height)
			title_layout.addWidget(temp_label, grid_row, grid_col_origin + title_col_span, short_info_row_span, short_info_col_span)

			grid_row += short_info_row_span

		elif tab_group == "advanced":
			# Set a label and its position in this tab
			temp_label = QLabel("<b>%s</b>" % (self.sxcmdwidget.sxcmd.get_mode_name_for("human")))
			temp_label.setMinimumWidth(title_label_min_width)
			temp_label.setMinimumHeight(title_label_min_height)
#			temp_label.setFixedWidth(title_label_min_width)
#			temp_label.setFixedHeight(title_label_min_height)
			title_layout.addWidget(temp_label, grid_row, grid_col_origin, title_row_span, title_col_span)

			temp_label = QLabel("Set advanced parameters", self)
			temp_label.setWordWrap(True)
			temp_label.setMinimumWidth(short_info_min_width)
			temp_label.setMinimumHeight(short_info_min_height)
#			temp_label.setFixedHeight(short_info_min_height)
			title_layout.addWidget(temp_label, grid_row, grid_col_origin + title_col_span, short_info_row_span, short_info_col_span)

		# Add space
		grid_row += 2

		# Add widget for editing command args and options
		cmd_token_apix = None
		cmd_token_abs_freq_list = []
		for cmd_token in self.sxcmdwidget.sxcmd.token_list:
			# Keep some command tokens for further setting after the loop
			if cmd_token.type == "apix":
				cmd_token_apix = cmd_token
			elif cmd_token.type == "abs_freq":
				cmd_token_abs_freq_list.append(cmd_token)
			# else: # Do nothing
			curr_group = cmd_token.group
			if cmd_token.filament_tab and helical:
				curr_group = cmd_token.filament_tab

			if curr_group == tab_group:
				# First, handle very special cases
				if cmd_token.type == "user_func":
					n_widgets = 2 # function type has two line edit boxes
					cmd_token_restore_widget = [None] * n_widgets
					cmd_token_widget = [None] * n_widgets
					cmd_token_subwidget_left = None
					cmd_token_subwidget_right = None
					cmd_token_calculator_dialog = None

					# Define custom style for restore widgets
					custom_style = "QPushButton {color:gray; }"

					# Create widgets for user function name
					widget_index = 0
					temp_label = QLabel(cmd_token.label[widget_index])
					temp_label.setMinimumWidth(token_label_min_width)
					grid_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

					assert(cmd_token.is_required == False)
					restores = cmd_token.restore[widget_index][helical]
					if len(restores) == 1:
						cmd_token_restore_widget[widget_index] = QPushButton("%s" % restores[0])
						cmd_token_restore_widget[widget_index].setStyleSheet(custom_style)
						cmd_token_restore_widget[widget_index].setToolTip('<FONT>'+default_cmd_token_restore_tooltip+'</FONT>')
						grid_layout.addWidget(cmd_token_restore_widget[widget_index], grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

						cmd_token_restore_widget[widget_index].clicked.connect(partial(self.handle_restore_widget_event, cmd_token, widget_index))
					else:
						cmd_token_restore_widget[widget_index] = QComboBox()
						cmd_token_restore_widget[widget_index].addItems(restores)
						cmd_token_restore_widget[widget_index].setEditable(True)
						cmd_token_restore_widget[widget_index].lineEdit().setReadOnly(True)
						cmd_token_restore_widget[widget_index].lineEdit().setAlignment(QtCore.Qt.AlignCenter)
						cmd_token_restore_widget[widget_index].setStyleSheet(custom_style)
						cmd_token_restore_widget[widget_index].setToolTip('<FONT>'+default_cmd_token_restore_tooltip+'</FONT>')
						grid_layout.addWidget(cmd_token_restore_widget[widget_index], grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)
						cmd_token_restore_widget[widget_index].activated.connect(partial(self.handle_restore_widget_event, cmd_token, widget_index))

					# cmd_token_widget[widget_index] = SXLineEdit(self)
					cmd_token_widget[widget_index] = SXLineEdit(cmd_token.key_base)
					cmd_token_widget[widget_index].setText(cmd_token.restore[widget_index][helical][0])
					cmd_token_widget[widget_index].setToolTip('<FONT>'+cmd_token.help[widget_index]+'</FONT>')
					grid_layout.addWidget(cmd_token_widget[widget_index], grid_row, grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_row_span, token_widget_col_span)

					grid_row +=  1

					# Create widgets for external file path containing above user function
					widget_index = 1
					temp_label = QLabel(cmd_token.label[widget_index])
					grid_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

					assert(cmd_token.is_required == False)
					cmd_token_restore_widget[widget_index] = QPushButton("%s" % cmd_token.restore[widget_index][helical][0])
					cmd_token_restore_widget[widget_index].setStyleSheet(custom_style)
					cmd_token_restore_widget[widget_index].setToolTip('<FONT>'+default_cmd_token_restore_tooltip+'</FONT>')
					grid_layout.addWidget(cmd_token_restore_widget[widget_index], grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

					cmd_token_widget[widget_index] = SXLineEdit(cmd_token.key_base)
					cmd_token_widget[widget_index].setText(cmd_token.restore[widget_index][helical][0]) # Because default user functions is internal
					cmd_token_widget[widget_index].setToolTip('<FONT>'+cmd_token.help[widget_index]+'</FONT>')
					grid_layout.addWidget(cmd_token_widget[widget_index], grid_row, grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_row_span, token_widget_col_span)

					cmd_token_restore_widget[widget_index].clicked.connect(partial(self.handle_restore_widget_event, cmd_token, widget_index))

					file_format = "py"
					temp_btn = QPushButton("Select Python script")
					temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s python script file</FONT>" % file_format)
					grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
					temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget[widget_index], file_format))

					grid_row +=  1

#					temp_label = QLabel(cmd_token.help[widget_index])
#					grid_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)
#
#					grid_row +=  1

				# Then, handle the other cases
				else:
					# Create label widget
					temp_label = QLabel(cmd_token.label)
					temp_label.setMinimumWidth(token_label_min_width)
					grid_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

					# Create widget and associate it to this cmd_token
					cmd_token_restore_widget = None
					cmd_token_restore_tooltip = default_cmd_token_restore_tooltip
					cmd_token_widget = None
					cmd_token_subwidget_left = None
					cmd_token_subwidget_right = None
					cmd_token_calculator_dialog = None

					restores = cmd_token.restore[helical]
					if cmd_token.type in ("bool", "bool_ignore"):
						if len(restores) == 1:
							btn_name = "NO"
							is_btn_enable = True
							custom_style = "QPushButton {color:gray; }"
							if restores[0]:
								btn_name = "YES"
							if cmd_token.type in list(parent.sxconst_set.dict.keys()):
								custom_style = "QPushButton {color:green; }"
								cmd_token_restore_tooltip = const_cmd_token_restore_tooltip
							elif cmd_token.is_required:
								if cmd_token.is_locked:
									btn_name = "locked"
									custom_style = "QPushButton {color:blue; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = locked_cmd_token_restore_tooltip
								else:
									btn_name = "required"
									custom_style = "QPushButton {color:red; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = required_cmd_token_restore_tooltip
								
							cmd_token_restore_widget = QPushButton("%s" % btn_name)
							cmd_token_restore_widget.setStyleSheet(custom_style)
							cmd_token_restore_widget.setEnabled(is_btn_enable)
							grid_layout.addWidget(
								cmd_token_restore_widget,
								grid_row,
								grid_col_origin + token_label_col_span,
								token_widget_row_span,
								token_widget_col_span
								)
							cmd_token_restore_widget.clicked.connect(partial(self.handle_restore_widget_event, cmd_token))
						else:
							custom_style = "QComboBox {color:gray; }"
							is_btn_enable = True
							if cmd_token.type in list(parent.sxconst_set.dict.keys()):
								custom_style = "QComboBox {color:green; }"
								cmd_token_restore_tooltip = const_cmd_token_restore_tooltip
							elif cmd_token.is_required:
								if cmd_token.is_locked:
									btn_name = "locked"
									custom_style = "QComboBox {color:blue; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = locked_cmd_token_restore_tooltip
								else:
									btn_name = "required"
									custom_style = "QComboBox {color:red; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = required_cmd_token_restore_tooltip
							cmd_token_restore_widget = QComboBox()
							cmd_token_restore_widget.setEditable(True)
							cmd_token_restore_widget.lineEdit().setReadOnly(True)
							cmd_token_restore_widget.lineEdit().setAlignment(QtCore.Qt.AlignCenter)

							cmd_token_restore_widget.addItems(restores)
							cmd_token_restore_widget.setStyleSheet(custom_style)
							cmd_token_restore_widget.setEnabled(is_btn_enable)
							grid_layout.addWidget(
								cmd_token_restore_widget,
								grid_row,
								grid_col_origin + token_label_col_span,
								token_widget_row_span,
								token_widget_col_span
								)
							cmd_token_restore_widget.activated.connect(partial(self.handle_restore_widget_event, cmd_token))

						# construct new widget(s) for this command token
						cmd_token_widget = SXCheckBox(cmd_token.key_base)
						if restores[0] == True:
							cmd_token_widget.setCheckState(Qt.Checked)
						else:
							cmd_token_widget.setCheckState(Qt.Unchecked)
						cmd_token_widget.setEnabled(not cmd_token.is_locked)
						grid_layout.addWidget(cmd_token_widget, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_row_span, token_widget_col_span)


					else:

						if len(restores) == 1:
							btn_name = "%s" % restores[0]
							custom_style = "QPushButton {color:gray; }"
							is_btn_enable = True
							if cmd_token.type in list(parent.sxconst_set.dict.keys()):
								custom_style = "QPushButton {color:green; }"
								cmd_token_restore_tooltip = const_cmd_token_restore_tooltip
							elif cmd_token.is_required:
								if cmd_token.is_locked:
									btn_name = "locked"
									custom_style = "QPushButton {color:blue; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = locked_cmd_token_restore_tooltip
								else:
									btn_name = "required"
									custom_style = "QPushButton {color:red; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = required_cmd_token_restore_tooltip
							cmd_token_restore_widget = QPushButton("%s" % btn_name)
							cmd_token_restore_widget.setStyleSheet(custom_style)
							cmd_token_restore_widget.setEnabled(is_btn_enable)
							grid_layout.addWidget(
								cmd_token_restore_widget,
								grid_row,
								grid_col_origin + token_label_col_span,
								token_widget_row_span,
								token_widget_col_span
								)
							cmd_token_restore_widget.clicked.connect(partial(self.handle_restore_widget_event, cmd_token))
						else:
							custom_style = "QComboBox {color:gray; }"
							is_btn_enable = True
							if cmd_token.type in list(parent.sxconst_set.dict.keys()):
								custom_style = "QComboBox {color:green; }"
								cmd_token_restore_tooltip = const_cmd_token_restore_tooltip
							elif cmd_token.is_required:
								if cmd_token.is_locked:
									btn_name = "locked"
									custom_style = "QComboBox {color:blue; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = locked_cmd_token_restore_tooltip
								else:
									btn_name = "required"
									custom_style = "QComboBox {color:red; }"
									is_btn_enable = False
									cmd_token_restore_tooltip = required_cmd_token_restore_tooltip
							cmd_token_restore_widget = QComboBox()
							cmd_token_restore_widget.setEditable(True)
							cmd_token_restore_widget.lineEdit().setReadOnly(True)
							cmd_token_restore_widget.lineEdit().setAlignment(QtCore.Qt.AlignCenter)

							cmd_token_restore_widget.addItems(restores)
							cmd_token_restore_widget.setStyleSheet(custom_style)
							cmd_token_restore_widget.setEnabled(is_btn_enable)
							grid_layout.addWidget(
								cmd_token_restore_widget,
								grid_row,
								grid_col_origin + token_label_col_span,
								token_widget_row_span,
								token_widget_col_span
								)
							cmd_token_restore_widget.activated.connect(partial(self.handle_restore_widget_event, cmd_token))

						cmd_token_widget = SXLineEdit(cmd_token.key_base)
						cmd_token_widget.setText(str(restores[0]))
						cmd_token_widget.setEnabled(not cmd_token.is_locked)
						grid_layout.addWidget(cmd_token_widget, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_row_span, token_widget_col_span)


						if cmd_token.type == "displayable_list":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any displayables")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select displayable data files of any supported formats</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "data2d3d_both":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any image/volume")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select an image/volume file of any supported format</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "mic_both":
							file_format = "mrc2d_mic_both"
							temp_btn = QPushButton("Select MRC mic/movie")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a MRC format micrograph or movie file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any mic/movie")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph or movie file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "mic_one":
							file_format = "mrc2d_mic_one"
							temp_btn = QPushButton("Select MRC micrograph")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a MRC format micrograph file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any micrograph")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "mic_one_list":
							file_format = "mrc2d_mic_one_list"
							temp_btn = QPushButton("Select MRC micrographs")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select MRC format micrograph files</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any micrographs")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select micrograph files of any supported formats</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "mic_stack":
							file_format = "mrc2d_mic_stack"
							temp_btn = QPushButton("Select MRC movie")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a MRC format movie file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any movie")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a movie file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "data2d_one":
							file_format = "hdf2d_one"
							temp_btn = QPushButton("Select HDF image")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a HDF format image file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any image")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a image file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "bdb2d_stack" or cmd_token.type == "output_bdb2d_stack":
							file_format = "bdb2d_stack"
							temp_btn = QPushButton("Select BDB image stack")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a BDB format image stack file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "data2d_stack":
							file_format = "bdb2d_stack"
							temp_btn = QPushButton("Select BDB image stack")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a BDB format image stack file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any image stack")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a image stack file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "data3d_one":
							file_format = "hdf3d_one"
							temp_btn = QPushButton("Select HDF volume")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a HDF format volume file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any volume")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a volume file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "data3d_stack":
							file_format = "hdf3d_stack"
							temp_btn = QPushButton("Select HDF volume stack")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a HDF format volume stack file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any volume stack")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a volume stack file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "select_mic_both":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select mic/movie list")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph/movie selection file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "select_mic_one":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select micrograph list")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph selection file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "select_mic_one_ext":
							file_format = "select_mic_one"
							temp_btn = QPushButton("Select micrograph list")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph selection file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "mic_one"
							temp_btn = QPushButton("Select any micrograph")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "select_mic_stack":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select movie list")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a micrograph movie selection file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "select_data2d_stack":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select image list")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a image selection file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "select_drift_params":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select drift params list")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a drift shift parameters selection file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_any_txt":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select parameters text")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a parameters text file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_proj_txt":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select projection params")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a projection parameters file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_coords_any":
							file_format = "params_coords_box"
							temp_btn = QPushButton("Select BOX coordinates")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a EMAN BOX coordinates file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = cmd_token.type
							temp_btn = QPushButton("Select any coordinates")
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a coordinates file of any supported format</FONT>")
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
						elif cmd_token.type == "params_cter_txt":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select CTER partres")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a CTER partres parameters file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_rebox_rbx":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select SPHIRE rebox")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a SPHIRE rebox file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_drift_txt":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select drift shift params")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a drift shift parameters file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "rot_matrix":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select matrix file")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a rotational matrix file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_relion_star":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select RELION STAR file")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a RELION STAR file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_any_json":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select JSON file")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dailog to select a JSON file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "params_any_h5":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select h5 file")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dailog to select a h5 file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "spectrum1d":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select power spectrum")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a 1D power spectrum file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "mtf":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select MTF data")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a MTF data file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "pdb":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select PDB data")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a PDB data file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "exe":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select executable")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select an executable file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "py":
							file_format = cmd_token.type
							temp_btn = QPushButton("Select python file")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip(
								'<FONT>' + "Display open file dialog to select an executable file</FONT>")
							grid_layout.addWidget(temp_btn, grid_row,
												  grid_col_origin + token_label_col_span + token_widget_col_span * 2,
												  token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(
								partial(self.sxcmdwidget.select_file, cmd_token_widget,
										file_format))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>' + "This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet(
								'background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row,
												  grid_col_origin + token_label_col_span + token_widget_col_span * 3,
												  token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "dir" or cmd_token.type == "dir_list" or cmd_token.type == "output_continue":
							temp_btn = QPushButton("Select directory")
							temp_btn.setMinimumWidth(func_btn_min_width)
							temp_btn.setToolTip('<FONT>'+"Display select directory dialog"+'</FONT>')
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							temp_btn.clicked.connect(partial(self.sxcmdwidget.select_dir, cmd_token_widget))
							file_format = "INVISIBLE"
							temp_btn = QPushButton("%s" % file_format)
							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
							temp_btn.setEnabled(False)
							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
							temp_btn.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						elif cmd_token.type == "abs_freq":
							# Create button for resolution display. Here, I use button to keep the look & feel.
							cmd_token_subwidget_left = QPushButton("Label for Resolution [A]")
							cmd_token_subwidget_left.setEnabled(False)
							cmd_token_subwidget_left.setMinimumWidth(func_btn_min_width)
							cmd_token_subwidget_left.setToolTip('<FONT>'+"Resolution [A] corresponding to absolute frequency [1/Pixel]"+'</FONT>')
							grid_layout.addWidget(cmd_token_subwidget_left, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
							# Create button to show the associated calculator dialog.
							cmd_token_subwidget_right = QPushButton("Use resolution [A]")
							cmd_token_subwidget_right.setToolTip('<FONT>'+"Display calculator dialog to use the resolution [A] instead of absolute frequency [1/Pixel]. It calculates absolute frequency [1/Pixel] (abs_freq) From resolution [A] (ares) using a give pixel size [A/Pixel] (apix), where abs_freq = apix/ares. </FONT>")
							cmd_token_subwidget_right.setMinimumWidth(func_btn_min_width)
							grid_layout.addWidget(cmd_token_subwidget_right, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
							# Associated this subwidget to open the calculator dialog
							const_register_widget_apix = self.sxcmdwidget.sxconst_set.dict["apix"].register_widget
							cmd_token_calculator_dialog = SXDialogCalculator(const_register_widget_apix, cmd_token_widget, cmd_token_subwidget_left, self)
							cmd_token_calculator_dialog.reflect_external_local_update_abs_freq()
							# Connect the main widget "editing finished" event to the calculator dialog
							cmd_token_widget.editingFinished.connect(cmd_token_calculator_dialog.reflect_external_local_update_abs_freq)
							# Connect the right subwidget "clicked" event to the calculator dialog
							cmd_token_subwidget_right.clicked.connect(cmd_token_calculator_dialog.show)
###						elif cmd_token.type == "any_micrograph":
###							temp_btn = QPushButton("Select Image")
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select standard format image file (e.g. .hdf, .mrc)</FONT>")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget))
###							file_format = "txt"
###							temp_btn = QPushButton("Select text file")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a parameters text file</FONT>" )
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###						elif cmd_token.type == "any_image":
###							temp_btn = QPushButton("Select Image")
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select standard format image file (e.g. .hdf, .mrc)</FONT>")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###						elif cmd_token.type == "any_image_list":
###							temp_btn = QPushButton("Select Images")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select standard format image files (e.g. .hdf, .mrc)</FONT>")
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, cmd_token.type))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###						elif cmd_token.type == "any_file":
###							temp_btn = QPushButton("Select File")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select file (e.g. *.*)</FONT>")
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###						elif cmd_token.type == "any_file_list":
###							temp_btn = QPushButton("Select Files")
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select files (e.g. *.*)</FONT>")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, cmd_token.type))
###							file_format = "bdb"
###							temp_btn = QPushButton("Select .%s" % file_format)
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s format image file</FONT>" % file_format)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###						elif cmd_token.type == "image":
###							file_format = "hdf"
###							temp_btn = QPushButton("Select .%s" % file_format)
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s format image file</FONT>" % file_format)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###							file_format = "bdb"
###							temp_btn = QPushButton("Select .%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s format image file</FONT>" % file_format)
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###						elif cmd_token.type == "bdb":
###							file_format = "bdb"
###							temp_btn = QPushButton("Select .%s" % file_format)
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s format image file</FONT>" % file_format)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###						elif cmd_token.type == "hdf":
###							file_format = cmd_token.type
###							temp_btn = QPushButton("Select .%s" % file_format)
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s format image file</FONT>" % file_format)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###						elif cmd_token.type == "mrc":
###							file_format = cmd_token.type
###							temp_btn = QPushButton("Select .%s" % file_format)
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select .%s format image file</FONT>" % file_format)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
###						elif cmd_token.type == "txt":
###							file_format = cmd_token.type
###							temp_btn = QPushButton("Select text file")
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							temp_btn.setToolTip('<FONT>'+"Display open file dialog to select a parameters text file</FONT>")
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span)
###							self.connect(temp_btn, SIGNAL("clicked()"), partial(self.sxcmdwidget.select_file, cmd_token_widget, file_format))
###							file_format = "INVISIBLE"
###							temp_btn = QPushButton("%s" % file_format)
###							temp_btn.setToolTip('<FONT>'+"This is %s button</FONT>" % file_format)
###							temp_btn.setEnabled(False)
###							temp_btn.setStyleSheet('background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid')
###							temp_btn.setMinimumWidth(func_btn_min_width)
###							grid_layout.addWidget(temp_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 3, token_widget_row_span, token_widget_col_span)
						else:
							if cmd_token.type not in ["int", "float", "string", "output", "apix", "ctfwin", "box", "radius", "sym", "mass", "filament_width"]: ERROR("Logical Error: Encountered unsupported type (%s). Consult with the developer."  % cmd_token.type, "%s in %s" % (__name__, os.path.basename(__file__)))
						
						# if cmd_token.type in ["output", "output_continue", "output_bdb2d_stack"]:
						# 	# Need to add output info button in future

					cmd_token_widget.setToolTip('<FONT>'+cmd_token.help+'</FONT>')
					cmd_token_restore_widget.setToolTip('<FONT>'+cmd_token_restore_tooltip+'</FONT>')

					grid_row += 1
					
				# Register this widget
				cmd_token.restore_widget = cmd_token_restore_widget
				cmd_token.widget = cmd_token_widget
				cmd_token.subwidget_left = cmd_token_subwidget_left
				cmd_token.subwidget_right = cmd_token_subwidget_right
				cmd_token.calculator_dialog = cmd_token_calculator_dialog
		
		is_necessary_to_connet_now = True
		if len(cmd_token_abs_freq_list) == 0:
			is_necessary_to_connet_now = False
		if is_necessary_to_connet_now and len(cmd_token_abs_freq_list) > 0:
			for cmd_token_abs_freq in cmd_token_abs_freq_list:
				if cmd_token_abs_freq.calculator_dialog is None:
					is_necessary_to_connet_now = False
					break
		if is_necessary_to_connet_now and cmd_token_apix is None:
			is_necessary_to_connet_now = False
		if is_necessary_to_connet_now and cmd_token_apix.widget is None:
			is_necessary_to_connet_now = False
		# else: # Do nothing
		
		if is_necessary_to_connet_now:
			# Associated the widget of apix command token to the calculator dialog each other
			assert (cmd_token_apix.type == "apix")
			assert (cmd_token_apix.widget is not None)
			# Loop through all absolute frequency tokens of this command
			# print("MRK_DEBUG: ")
			# print("MRK_DEBUG: ----- SXConstSetWidget Constructor ----- ")
			# print("MRK_DEBUG: self.sxcmdwidget.sxcmd.name          := {}".format(self.sxcmdwidget.sxcmd.name))
			# print("MRK_DEBUG: self.sxcmdwidget.sxcmd.subname       := {}".format(self.sxcmdwidget.sxcmd.subname))
			# print("MRK_DEBUG: self.sxcmdwidget.sxcmd.mode          := {}".format(self.sxcmdwidget.sxcmd.mode))
			# print("MRK_DEBUG: self.sxcmdwidget.sxcmd.subset_config := {}".format(self.sxcmdwidget.sxcmd.subset_config))
			for cmd_token_abs_freq in cmd_token_abs_freq_list:
				assert (cmd_token_abs_freq.type == "abs_freq")
				# Register the calculator dialogs of all the other absolute frequency tokens
				# by looping through all the other absolute frequency tokens
				for other_cmd_token_abs_freq in cmd_token_abs_freq_list:
					assert (other_cmd_token_abs_freq.type == "abs_freq")
					# Exclude itself
					if cmd_token_abs_freq.key_base != other_cmd_token_abs_freq.key_base:
						cmd_token_abs_freq.other_dialog_list.append(other_cmd_token_abs_freq.calculator_dialog)
						cmd_token_abs_freq.calculator_dialog.sxcmd_token_other_dialog_list_abs_freq.append(other_cmd_token_abs_freq.calculator_dialog)
				# Register pixel size token widget to this absolut frequency calculator dialog
				# if cmd_token_abs_freq is None:
				# 	print("MRK_DEBUG: cmd_token_abs_freq is None")
				# if cmd_token_abs_freq.calculator_dialog is None:
				# 	print("MRK_DEBUG: cmd_token_abs_freq.calculator_dialog is None")
				# 	print("MRK_DEBUG: cmd_token_abs_freq.key_base := {}".format(cmd_token_abs_freq.key_base))
				# 	print("MRK_DEBUG: cmd_token_abs_freq.type     := {}".format(cmd_token_abs_freq.type))
				cmd_token_abs_freq.calculator_dialog.sxcmd_token_widget_apix = cmd_token_apix.widget
				# Register this absolut frequency calculator dialog to pixel size token
				cmd_token_apix.other_dialog_list.append(cmd_token_abs_freq.calculator_dialog)
			
			# print("MRK_DEBUG: len(cmd_token_apix.other_dialog_list) := \"{}\"".format(len(cmd_token_apix.other_dialog_list)))
			
			# Initialise pixel size of all calculate dialogs
			self.handle_apix_token_editing_finished_event(cmd_token_apix)
			# Connect the apix command token widget "editing finished" event to the calculator dialog
			cmd_token_apix.widget.editingFinished.connect(partial(self.handle_apix_token_editing_finished_event, cmd_token_apix))
		
		if tab_group == "main":
			# Add space
			grid_row += 1

			# Add gui components for MPI related parameters
			temp_label = QLabel("MPI processors")
			temp_label.setMinimumWidth(token_label_min_width)
			submit_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

			# self.mpi_nproc_edit = QLineEdit(self)
			self.mpi_nproc_edit = QLineEdit()
			self.mpi_nproc_edit.setToolTip('<FONT>'+"Number of processors to use. default is single processor mode"+'</FONT>')
			submit_layout.addWidget(self.mpi_nproc_edit, grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

			# Add save parameters button
			self.save_params_btn = QPushButton("Save parameters")
			self.save_params_btn.setMinimumWidth(btn_min_width)
			self.save_params_btn.setToolTip('<FONT>'+"Save gui parameter settings"+'</FONT>')
			self.save_params_btn.clicked.connect(self.sxcmdwidget.save_params)
			submit_layout.addWidget(self.save_params_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span*2)

			grid_row += 1

			temp_label = QLabel("MPI command line template")
			temp_label.setMinimumWidth(token_label_min_width)
			submit_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

			self.mpi_cmd_line_edit = QLineEdit()
			if 'SPHIRE_MPI_COMMAND_LINE_TEMPLATE' in os.environ:
				mpi_command_line_template = os.environ['SPHIRE_MPI_COMMAND_LINE_TEMPLATE']
			else:
				mpi_command_line_template = ""
			self.mpi_cmd_line_edit.setText(mpi_command_line_template)
			self.mpi_cmd_line_edit.setToolTip('<FONT>'+"Template of MPI command line (e.g. \"mpirun -np XXX_SXMPI_NPROC_XXX --host n0,n1,n2 XXX_SXCMD_LINE_XXX\"). if empty, use \"mpirun -np XXX_SXMPI_NPROC_XXX XXX_SXCMD_LINE_XXX\"</FONT>")
			submit_layout.addWidget(self.mpi_cmd_line_edit, grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

			# Add load parameters button
			self.load_params_btn = QPushButton("Load parameters")
			self.load_params_btn.setMinimumWidth(btn_min_width)
			self.load_params_btn.setToolTip('<FONT>'+"Load gui parameter settings to retrieve a previously-saved one"+'</FONT>')
			self.load_params_btn.clicked.connect(self.sxcmdwidget.load_params)
			submit_layout.addWidget(self.load_params_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span*2)

			grid_row += 1

			# If MPI is not supported, disable this widget
			self.set_text_entry_widget_enable_state(self.mpi_nproc_edit, self.sxcmdwidget.sxcmd.mpi_support)
			if not self.mpi_nproc_edit.isEnabled():
				self.mpi_nproc_edit.setText("1")
			elif 'SPHIRE_NPROC' in os.environ:
				self.mpi_nproc_edit.setText(os.environ['SPHIRE_NPROC'])
			else:
				self.mpi_nproc_edit.setText("1")
			self.set_text_entry_widget_enable_state(self.mpi_cmd_line_edit, self.sxcmdwidget.sxcmd.mpi_support)

			# Add gui components for queue submission (qsub)
			is_qsub_enabled = False
			temp_label = QLabel("Submit job to queue")
			temp_label.setMinimumWidth(token_label_min_width)
			submit_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

			self.qsub_enable_checkbox = QCheckBox("")
			if is_qsub_enabled == True:
				self.qsub_enable_checkbox.setCheckState(Qt.Checked)
			elif not self.sxcmdwidget.sxcmd.is_submittable:
				self.qsub_enable_checkbox.setCheckState(Qt.Unchecked)
			elif 'SPHIRE_ENABLE_QSUB' in os.environ: # assert(is_qsub_enabled == False)
				if os.environ['SPHIRE_ENABLE_QSUB'].lower() == 'true':
					self.qsub_enable_checkbox.setCheckState(Qt.Checked)
				else:
					self.qsub_enable_checkbox.setCheckState(Qt.Unchecked)
			else:
				self.qsub_enable_checkbox.setCheckState(Qt.Unchecked)
			self.qsub_enable_checkbox.setToolTip('<FONT>'+"Submit job to queue"+'</FONT>')
			self.qsub_enable_checkbox.stateChanged.connect(self.set_qsub_enable_state) # To control enable state of the following qsub related widgets
			self.qsub_enable_checkbox.setEnabled(self.sxcmdwidget.sxcmd.is_submittable)
			submit_layout.addWidget(self.qsub_enable_checkbox, grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

			grid_row += 1

			temp_label = QLabel("Job name")
			temp_label.setMinimumWidth(token_label_min_width)
			submit_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

			self.qsub_job_name_edit = QLineEdit()
			if self.sxcmdwidget.sxcmd.is_submittable == True:
				self.qsub_job_name_edit.setText(self.sxcmdwidget.sxcmd.get_mode_name_for("file_path"))
			else: # assert(self.sxcmdwidget.sxcmd.is_submittable == False)
				assert(self.sxcmdwidget.sxcmd.mpi_support == False)
				self.qsub_job_name_edit.setText("N/A")
			self.qsub_job_name_edit.setToolTip('<FONT>'+"Name of this job"+'</FONT>')
			submit_layout.addWidget(self.qsub_job_name_edit, grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

			grid_row += 1

			temp_label = QLabel("Submission command")
			temp_label.setMinimumWidth(token_label_min_width)
			submit_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

			self.qsub_cmd_edit = QLineEdit()
			if self.sxcmdwidget.sxcmd.is_submittable == True:
				if 'SPHIRE_SUBMISSION_COMMAND' in os.environ:
					submission_command = os.environ['SPHIRE_SUBMISSION_COMMAND']
				else:
					submission_command = "qsub"
				self.qsub_cmd_edit.setText(submission_command)
			else: # assert(self.sxcmdwidget.sxcmd.is_submittable == False)
				assert(self.sxcmdwidget.sxcmd.mpi_support == False)
				self.qsub_cmd_edit.setText("N/A")
			self.qsub_cmd_edit.setToolTip('<FONT>'+"Name of submission command to queue job"+'</FONT>')
			submit_layout.addWidget(self.qsub_cmd_edit, grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

			self.cmd_line_btn = QPushButton("Generate command line")
			self.cmd_line_btn.setMinimumWidth(btn_min_width)
			self.cmd_line_btn.setToolTip('<FONT>'+"Generate command line from gui parameter settings and automatically save settings"+'</FONT>')
			self.cmd_line_btn.clicked.connect(self.sxcmdwidget.print_cmd_line)
			submit_layout.addWidget(self.cmd_line_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span*2)

			grid_row += 1

			self.pipe_line_btn = QPushButton("Add to pipeline folder")
			self.pipe_line_btn.setMinimumWidth(btn_min_width)
			self.pipe_line_btn.setToolTip('<FONT>'+"Generate executable files that and add them to the queue folder."+'</FONT>')
			self.pipe_line_btn.clicked.connect(self.sxcmdwidget.add_to_pipeline)
			submit_layout.addWidget(self.pipe_line_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span*2)

			grid_row += 1

			temp_label = QLabel("Submission script template")
			temp_label.setMinimumWidth(token_label_min_width)
			submit_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)

			self.qsub_script_edit = QLineEdit()
			if self.sxcmdwidget.sxcmd.is_submittable == True:
				if 'SPHIRE_SUBMISSION_SCRIPT_TEMPLATE' in os.environ:
					submission_script_template = os.environ['SPHIRE_SUBMISSION_SCRIPT_TEMPLATE']
				else:
					submission_script_template = "msgui_qsub.sh"
				self.qsub_script_edit.setText(submission_script_template)
			else: # assert(self.sxcmdwidget.sxcmd.is_submittable == False)
				assert(self.sxcmdwidget.sxcmd.mpi_support == False)
				self.qsub_script_edit.setText("N/A")
			self.qsub_script_edit.setToolTip('<FONT>'+"File name of submission script template (e.g. $PROJECT_DIR/msgui_qsub.sh)"+'</FONT>')
			submit_layout.addWidget(self.qsub_script_edit, grid_row, grid_col_origin + token_label_col_span, token_widget_row_span, token_widget_col_span)

			self.qsub_script_open_btn = QPushButton("Select template")
			self.qsub_script_open_btn.setMinimumWidth(func_btn_min_width)
			self.qsub_script_open_btn.setToolTip('<FONT>'+"Display open file dialog to select job submission script template file"+'</FONT>')
			self.qsub_script_open_btn.clicked.connect(partial(self.sxcmdwidget.select_file, self.qsub_script_edit, 'submission_template'))
			submit_layout.addWidget(self.qsub_script_open_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span, token_widget_row_span, token_widget_col_span)

			# Add a run button
			# self.execute_btn = QPushButton("Run %s" % self.sxcmdwidget.sxcmd.get_mode_name_for("human"))
			self.execute_btn = QPushButton("Run command")
			# make 3D textured push button look
			custom_style = "QPushButton {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #8D0)} QPushButton:pressed {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #084)} QPushButton:focus {font: bold; color: #000;border: 2px solid #8D0;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #8D0)} QPushButton:disabled {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #ff0000)}"
			self.execute_btn.setStyleSheet(custom_style)
			self.execute_btn.setMinimumWidth(btn_min_width)
			self.execute_btn.setToolTip('<FONT>'+"Run %s and automatically save gui parameter settings</FONT>" % self.sxcmdwidget.sxcmd.get_mode_name_for("human"))
			self.execute_btn.clicked.connect(self.sxcmdwidget.execute_cmd_line)
			submit_layout.addWidget(self.execute_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span*2)

			grid_row += 1

###			# Add a debug button
###			sxoperand_set = self.construct_sxoperand_set()
###			self.res_calculator = SXDialogCalculator(sxoperand_set, self)
###			self.cmd_line_btn = QPushButton("Debug")
###			self.cmd_line_btn.setMinimumWidth(btn_min_width)
###			self.cmd_line_btn.setToolTip('<FONT>'+"Button for debug"+'</FONT>')
###			self.connect(self.cmd_line_btn, SIGNAL("clicked()"), self.res_calculator.show)
###			submit_layout.addWidget(self.cmd_line_btn, grid_row, grid_col_origin + token_label_col_span + token_widget_col_span * 2, token_widget_row_span, token_widget_col_span*2)
###
###			grid_row += 1

			# Initialize enable state of qsub related widgets
			self.set_qsub_enable_state()

	@staticmethod
	def set_text_entry_widget_enable_state(widget, is_enabled):
		# Set enable state and background color of text entry widget according to enable state
		default_palette = QPalette()
		bg_color = default_palette.color(QPalette.Inactive, QPalette.Base)
		if is_enabled == False:
			bg_color = default_palette.color(QPalette.Disabled, QPalette.Base)

		widget.setEnabled(is_enabled)
		palette = widget.palette()
		palette.setColor(widget.backgroundRole(), bg_color)
		widget.setPalette(palette)

	def set_qsub_enable_state(self):
		is_enabled = False
		if self.qsub_enable_checkbox.checkState() == Qt.Checked:
			is_enabled = True

		# Set enable state and background color of mpi related widgets
		if self.sxcmdwidget.sxcmd.mpi_support:
			self.set_text_entry_widget_enable_state(self.mpi_cmd_line_edit, not is_enabled)

		# Set enable state and background color of qsub related widgets
		self.set_text_entry_widget_enable_state(self.qsub_job_name_edit, is_enabled)
		self.set_text_entry_widget_enable_state(self.qsub_cmd_edit, is_enabled)
		self.set_text_entry_widget_enable_state(self.qsub_script_edit, is_enabled)
		self.qsub_script_open_btn.setEnabled(is_enabled)

	def handle_restore_widget_event(self, sxcmd_token, widget_index=0, restore_idx=0):
		assert(not sxcmd_token.is_locked)
		if sxcmd_token.type == "user_func":
			assert(len(sxcmd_token.widget) == 2 and len(sxcmd_token.restore) == 2 and widget_index < 2)
			try:
				text = str(self.sender().text())
			except AttributeError:
				if restore_idx == 0:
					text = str(self.sender().currentText())
				else:
					text = str(self.sender().itemText(restore_idx))
				text = text.strip()
				self.sender().blockSignals(True)
				self.sender().setCurrentIndex(0)
				self.sender().blockSignals(False)
			sxcmd_token.widget[widget_index].setText(text)
		else:
			if sxcmd_token.type in ("bool", "bool_ignore"):
				try:
					text = str(self.sender().text())
				except AttributeError:
					if restore_idx == 0:
						text = str(self.sender().currentText())
					else:
						text = str(self.sender().itemText(restore_idx))
					text = text.strip()
					self.sender().blockSignals(True)
					self.sender().setCurrentIndex(0)
					self.sender().blockSignals(False)
				if text == 'YES':
					sxcmd_token.widget.setChecked(Qt.Checked)
				else: # sxcmd_token.restore == False
					sxcmd_token.widget.setChecked(Qt.Unchecked)
			else:
				try:
					text = str(self.sender().text())
				except AttributeError:
					if restore_idx == 0:
						text = str(self.sender().currentText())
					else:
						text = str(self.sender().itemText(restore_idx))
					text = text.strip()
					self.sender().blockSignals(True)
					self.sender().setCurrentIndex(0)
					self.sender().blockSignals(False)
				sxcmd_token.widget.setText(text)
				if sxcmd_token.type == "abs_freq":
					sxcmd_token.calculator_dialog.reflect_external_local_update_abs_freq()
					for sxcmd_token_other_dialog in sxcmd_token.other_dialog_list:
						sxcmd_token_other_dialog.reflect_external_local_update_abs_freq()
				elif sxcmd_token.type == "apix":
					for sxcmd_token_other_dialog in sxcmd_token.other_dialog_list:
						sxcmd_token_other_dialog.reflect_external_local_update_apix()


	def handle_apix_token_editing_finished_event(self, sxcmd_token_apix):
		assert (sxcmd_token_apix.type == "apix")
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXCmdTab.handle_apix_token_editing_finished_event() ----- ")
		# print("MRK_DEBUG: len(sxcmd_token_apix.other_dialog_list) := \"{}\"".format(len(sxcmd_token_apix.other_dialog_list)))
		for cmd_token_calculator_dialog_abs_freq in sxcmd_token_apix.other_dialog_list:
			# print("MRK_DEBUG: cmd_token_calculator_dialog_abs_freq := \"{}\"".format(cmd_token_calculator_dialog_abs_freq))
			if cmd_token_calculator_dialog_abs_freq is not None:
				cmd_token_calculator_dialog_abs_freq.reflect_external_local_update_apix()

###	def handle_abs_freq_editing_finished_event(self, sxcmd_token_widget_abs_freq, sxcmd_token_subwidget_left_ares):
###		apix_str = self.sxcmdwidget.sxconst_set.dict["apix"].register
###		abs_freq_str = sxcmd_token_widget_abs_freq.text()
###		ares_str, is_valid_ares = SXDialogCalculator.convert_abs_freq_to_ares(apix_str, abs_freq_str)
###		print("MRK_DEBUG: ----- handle_abs_freq_editing_finished_event() ----- ")
###		print("MRK_DEBUG: Input Abs. Freq. [1/Pixel] string ; abs_freq_str := {}".format(abs_freq_str))
###		print("MRK_DEBUG: Input Pixel Size [A/Pixel] string ; apix_str := {}".format(apix_str))
###		print("MRK_DEBUG: Output Resolution [A] string      ; ares_str:= {}".format(ares_str))
###		if is_valid_ares:
###			sxcmd_token_subwidget_left_ares.setText("{}[A]@{}[A/Pix]".format(ares_str, apix_str))
###		else:
###			sxcmd_token_subwidget_left_ares.setText("Mode {}".format(ares_str))


# ========================================================================================
# Command Category Widget (opened by class SXMainWindow)
class SXCmdCategoryWidget(QWidget):
	def __init__(self, sxconst_set, sxcmd_category, helical=False, parent = None):
		super(SXCmdCategoryWidget, self).__init__(parent)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.sxconst_set = sxconst_set
		self.sxcmd_category = sxcmd_category
		self.cur_sxcmd = None

		# Layout constants
		self.sxcmd_btn_row_span = 1
		self.sxcmd_btn_col_span = 1

		self.sxcmd_btn_area_row_span = self.sxcmd_btn_row_span * SXLookFeelConst.expected_cmd_counts
		self.sxcmd_btn_area_col_span = self.sxcmd_btn_col_span

		self.sxcmd_widget_area_row_span = self.sxcmd_btn_area_row_span
		self.sxcmd_widget_area_col_span = 1

		self.grid_row_origin = 0
		self.grid_col_origin = 0

		# Layout variables
		self.grid_layout = None # grid layout

		self.grid_row = self.grid_row_origin # Keep current row
		self.grid_col = self.grid_col_origin # keep current column

		self.sxcmd_btn_group = None
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

		# --------------------------------------------------------------------------------
		# Setup Window Layout
		# --------------------------------------------------------------------------------
		self.setup_layout(QBrush(SXLookFeelConst.default_bg_color))

		# --------------------------------------------------------------------------------
		# Add SX Commands (sx*.py) associated widgets
		# --------------------------------------------------------------------------------
		self.add_sxcmd_widgets(helical)

#		# --------------------------------------------------------------------------------
#		# Load the previously saved parameter setting of this sx command
#		# Override the registration of project constant parameter settings with the previously-saved one
#		# --------------------------------------------------------------------------------
#		for sxcmd in self.sxcmd_category.cmd_list:
#			if os.path.exists(sxcmd.widget.gui_settings_file_path):
#				sxcmd.widget.read_params(sxcmd.widget.gui_settings_file_path)

		# --------------------------------------------------------------------------------
		# Alway select the 1st entry of the command list upon startup
		# --------------------------------------------------------------------------------
		self.handle_sxcmd_btn_event(self.sxcmd_category.cmd_list[0])

	def setup_layout(self, background_brush):
		# Setup background color of this widget
		self.setAutoFillBackground(True)
		palette = QPalette()
		palette.setBrush(QPalette.Background, background_brush)
		self.setPalette(palette)

		# Setup grid layout in the scroll area
		self.grid_layout = QGridLayout()
		self.grid_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		self.grid_layout.setSpacing(SXLookFeelConst.grid_spacing)
		self.grid_layout.setColumnMinimumWidth(0, SXLookFeelConst.sxcmd_btn_area_min_width)
		# self.grid_layout.setColumnMinimumWidth(1, SXLookFeelConst.sxcmd_widget_area_min_width)
		# Give the column of the command settings area a higher stretch priority so that the other area does not stretch horizontally
		# self.grid_layout.setColumnStretch(self.grid_col_origin + self.sxcmd_btn_area_col_span, self.grid_layout.columnStretch(self.grid_col_origin + self.sxcmd_btn_area_col_span) + 1)

	# Add Pipeline SX Commands (sx*.py) associated widgets
	def add_sxcmd_widgets(self, helical):
		self.sxcmd_btn_group = QButtonGroup()
		# self.sxcmd_btn_group.setExclusive(True) # NOTE: 2016/02/18 Toshio Moriya: Without QPushButton.setCheckable(True). This does not do anything. Let manually do this

		current_role = None
		self.stacked_layout = QStackedLayout()
		grid_box_layout = QVBoxLayout()
		grid_box_layout.addLayout(self.grid_layout)
		grid_box_layout.addStretch(1)
		global_layout = QHBoxLayout()
		global_layout.addLayout(grid_box_layout)
		global_layout.addLayout(self.stacked_layout, stretch=1)
		self.setLayout(global_layout)

		# Add SX Commands (sx*.py) associated widgets
		for sxcmd in self.sxcmd_category.cmd_list:
			if sxcmd.role != current_role:
				# Add title label and set position and font style
				label_text = ""
				if sxcmd.role == "sxr_pipe":
					label_text = "COMMANDS"
				elif sxcmd.role == "sxr_alt":
					label_text = "ALTERNATIVES"
				elif sxcmd.role == "sxr_util":
					label_text = "UTILITIES"
				else:
					label_text = "UNKNOWN"

				if current_role !=  None:
					self.grid_row += 1

				# title=QLabel("<span style=\'font-size:18pt; font-weight:600; color:#aa0000;\'><b>%s </b></span><span style=\'font-size:12pt; font-weight:60; color:#aa0000;\'>(shift-click for wiki)</span>" % label_text)
				title=QLabel("<span style=\'font-size:18pt; font-weight:600; color:#000000;\'><b>%s </b></span><span style=\'font-size:12pt; font-weight:60; color:#000000;\'>(shift-click for wiki)</span>" % label_text)
				self.grid_layout.addWidget(title, self.grid_row, self.grid_col_origin, self.sxcmd_btn_row_span, self.sxcmd_btn_col_span)

				self.grid_row += 1

				current_role = sxcmd.role

			# Add buttons for this sx*.py processe
			sxcmd.btn = QPushButton(sxcmd.label)
			# sxcmd.btn.setCheckable(True) # NOTE: 2016/02/18 Toshio Moriya: With this setting, we can not move the focus to the unchecked butttons... PyQt bug?
			sxcmd.btn.setToolTip('<FONT>'+sxcmd.short_info+'</FONT>')
			self.sxcmd_btn_group.addButton(sxcmd.btn)
			self.grid_layout.addWidget(sxcmd.btn, self.grid_row, self.grid_col_origin, self.sxcmd_btn_row_span, self.sxcmd_btn_col_span)

			# Create SXCmdWidget for this sx*.py processe
			sxcmd.widget = SXCmdWidget(self.sxconst_set, sxcmd, helical, self)
			self.stacked_layout.addWidget(sxcmd.widget)

			# connect widget signals
			sxcmd.btn.clicked.connect(partial(self.handle_sxcmd_btn_event, sxcmd))

			self.grid_row += 1

	def load_previous_session(self):
		for sxcmd in self.sxcmd_category.cmd_list:
			if os.path.exists(sxcmd.widget.gui_settings_file_path):
				sxcmd.widget.read_params(sxcmd.widget.gui_settings_file_path)

	def handle_sxcmd_btn_event(self, sxcmd):
		modifiers = QApplication.keyboardModifiers()
		if modifiers == Qt.ShiftModifier:
			# os.system("python -m webbrowser %s%s" % (SPARX_DOCUMENTATION_WEBSITE, sxcmd.name))
			# sxcmd_wiki_url = SXLookFeelConst.generate_sxcmd_wiki_url(sxcmd, wiki_type = "SPARX")
			sxcmd_wiki_url = SXLookFeelConst.generate_sxcmd_wiki_url(sxcmd)
			print("Opening Wiki Page ...")
			print(sxcmd_wiki_url)
			os.system("python -m webbrowser %s" % (sxcmd_wiki_url))
			return

		if self.cur_sxcmd == sxcmd: return

		if self.cur_sxcmd != None:
			custom_style = "QPushButton {font: normal; color:black; }" # custom_style = "QPushButton {color:#000; }"
			self.cur_sxcmd.btn.setStyleSheet(custom_style)

		self.cur_sxcmd = sxcmd

		if self.cur_sxcmd != None:
			self.stacked_layout.setCurrentWidget(self.cur_sxcmd.widget)
			custom_style = "QPushButton {font: bold; color:blue; }" # custom_style = "QPushButton {font: bold; color:#8D0; }"
			self.cur_sxcmd.btn.setStyleSheet(custom_style)

	def quit_all_child_applications(self):
		# Quit all child applications
		for sxcmd in self.sxcmd_category.cmd_list:
			sxcmd.widget.quit_all_child_applications()

# ========================================================================================
# Layout of the project constants parameters widget; owned by the main window
class SXConstSetWidget(QWidget):
	def __init__(self, sxconst_set, sxcmd_category_list, helical=False, parent=None):
		super(SXConstSetWidget, self).__init__(parent)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.parent = parent
		self.sxconst_set = sxconst_set
		self.sxcmd_category_list = sxcmd_category_list

		self.gui_settings_file_path = "%s/gui_settings_project_%d.txt" % (SXLookFeelConst.project_dir, int(helical))

		# Layout constants and variables
		global_row_origin = 0; global_col_origin = 0
		global_row_span = 4; global_col_span = 1

		header_row_origin = 0; header_col_origin = 0
		title_row_span = 1; title_col_span = 1
		short_info_row_span = 1; short_info_col_span = 1
		title_min_width = 300
		short_info_min_width = 300
		short_info_min_height = 80

		const_set_row_origin = 0; const_set_col_origin = 0
		const_label_row_span = 1; const_label_col_span = 1
		const_register_widget_row_span = 1; const_register_widget_col_span = 1
		const_widget_row_span = 1; const_widget_col_span = 1
		const_label_min_width = 150
		const_register_widget_min_width = const_label_min_width
		const_widget_min_width = const_label_min_width

		btn_row_origin = 0; btn_col_origin = 0
		func_btn_row_span = 1; func_btn_col_span = 1
		register_btn_row_span = 1; register_btn_col_span = 2
		func_btn_min_width = 50
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

		# Set the background color of this widget
		self.setAutoFillBackground(True)
		palette = QPalette()
		palette.setBrush(QPalette.Background, QBrush(SXLookFeelConst.default_bg_color))
		self.setPalette(palette)

		global_layout = QGridLayout()
		global_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		global_layout.setSpacing(SXLookFeelConst.grid_spacing)
		global_layout.setRowStretch(global_row_span - 1, global_layout.rowStretch(global_row_origin) + 1)

		header_layout = QGridLayout()
		header_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		header_layout.setSpacing(SXLookFeelConst.grid_spacing)

		const_set_layout = QGridLayout()
		const_set_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		const_set_layout.setSpacing(SXLookFeelConst.grid_spacing)

		btn_layout = QGridLayout()
		btn_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		btn_layout.setSpacing(SXLookFeelConst.grid_spacing * 2)

		global_grid_row = global_row_origin

		# Start add title widgets to the grid layout
		header_grid_row = header_row_origin

		# Set a label and its position in this tab
		temp_label = QLabel("<b>%s</b>" % (self.sxconst_set.label))
		temp_label.setMinimumWidth(title_min_width)
		header_layout.addWidget(temp_label, header_grid_row, header_col_origin, title_row_span, title_col_span)

		header_grid_row += 1

		# NOTE: 2015/11/17 Toshio Moriya
		# Necessary to separate "<b>%s</b>" from the information for avoiding to invoke the tag interpretations of string
		# e.g. < becomes the escape character
		temp_label = QLabel("%s" % (self.sxconst_set.short_info))
		temp_label.setWordWrap(True)
		temp_label.setMinimumWidth(short_info_min_width)
		temp_label.setMinimumHeight(short_info_min_height)
		header_layout.addWidget(temp_label, header_grid_row, header_col_origin, short_info_row_span, short_info_col_span)

		# Add const set grid layout to global layout
		global_layout.addLayout(header_layout, global_grid_row, global_col_origin)
		global_grid_row += 1

		# Start add project parameter constant widgets to the grid layout
		const_set_grid_row = const_set_row_origin

		# Add widget for editing command args and options
		for sxconst in self.sxconst_set.list:
			# Create widget associated to this project constant parameter
			temp_label = QLabel(sxconst.label)
			temp_label.setMinimumWidth(const_label_min_width)
			const_set_layout.addWidget(temp_label, const_set_grid_row, const_set_col_origin, const_label_row_span, const_label_col_span)

			sxconst_register_widget = QPushButton("%s" % sxconst.register)
			sxconst_register_widget.setMinimumWidth(const_register_widget_min_width)
			custom_style = "QPushButton {color:green; }"
			sxconst_register_widget.setStyleSheet(custom_style)
			const_set_layout.addWidget(sxconst_register_widget, const_set_grid_row, const_set_row_origin + const_label_col_span, const_register_widget_row_span, const_register_widget_col_span)
			sxconst_register_widget.setToolTip('<FONT>'+"Retrieve this registered value to edit box"+'</FONT>')
			sxconst_register_widget.clicked.connect(partial(self.handle_regster_widget_event, sxconst))

			sxconst_widget = QLineEdit()
			sxconst_widget.setMinimumWidth(const_widget_min_width)
			sxconst_widget.setText(sxconst.register)
			sxconst_widget.setToolTip('<FONT>'+sxconst.help+'</FONT>')
			const_set_layout.addWidget(sxconst_widget, const_set_grid_row, const_set_row_origin + const_label_col_span + const_register_widget_col_span, const_widget_row_span, const_widget_col_span)

			const_set_grid_row += 1

			# Register this widget
			sxconst.register_widget = sxconst_register_widget
			sxconst.widget = sxconst_widget

		# Add const set grid layout to global layout
		global_layout.addLayout(const_set_layout, global_grid_row, global_col_origin)
		# global_grid_row += 1

		# Start add buttons to the grid layout
		btn_grid_row = btn_row_origin

		# Add a register button
		self.execute_btn = QPushButton("Register settings")
		# make 3D textured push button look
		custom_style = "QPushButton {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #8D0);min-width:90px;margin:5px} QPushButton:pressed {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #084);min-width:90px;margin:5px}"
		self.execute_btn.setStyleSheet(custom_style)
		self.execute_btn.setMinimumWidth(func_btn_min_width * register_btn_col_span)
		self.execute_btn.setToolTip('<FONT>'+"Register project constant parameter settings to automatically set values to command arguments and options"+'</FONT>')
		self.execute_btn.clicked.connect(self.register_const_set)
		btn_layout.addWidget(self.execute_btn, btn_grid_row, btn_col_origin, register_btn_row_span, register_btn_col_span)

		btn_grid_row += 1

		# Add save project constant parameter settings button
		self.save_consts_btn = QPushButton("Save settings")
		self.save_consts_btn.setMinimumWidth(func_btn_min_width)
		self.save_consts_btn.setToolTip('<FONT>'+"Save project constant parameter settings"+'</FONT>')
		self.save_consts_btn.clicked.connect(self.save_consts)
		btn_layout.addWidget(self.save_consts_btn, btn_grid_row, btn_col_origin, func_btn_row_span, func_btn_col_span)

		# Add load project constant parameter settings button
		self.load_consts_btn = QPushButton("Load settings")
		self.load_consts_btn.setMinimumWidth(func_btn_min_width)
		self.load_consts_btn.setToolTip('<FONT>'+"Load project constant parameter settings to retrieve the previously-saved one"+'</FONT>')
		self.load_consts_btn.clicked.connect(self.load_consts)
		btn_layout.addWidget(self.load_consts_btn, btn_grid_row, btn_col_origin + func_btn_col_span, func_btn_row_span, func_btn_col_span)

		btn_grid_row += 1
		btn_layout.addWidget(QLabel(), btn_grid_row, btn_col_origin, register_btn_row_span, register_btn_col_span)

		btn_grid_row += 1

		# Add a register button
		custom_style = "QPushButton {color: #68a2c3}"
		if helical:
			self.helical_spa_btn = QPushButton("Switch to SPA GUI")
			self.helical_spa_btn.setToolTip('Switch to the SPA GUI')
			self.helical_spa_btn.setStyleSheet(custom_style)
		else:
			self.helical_spa_btn = QPushButton("Switch to helical GUI")
			self.helical_spa_btn.setToolTip('Switch to the helical GUI')
			self.helical_spa_btn.setStyleSheet(custom_style)
		# make 3D textured push button look
		self.helical_spa_btn.setMinimumWidth(func_btn_min_width * register_btn_col_span)
		self.helical_spa_btn.clicked.connect(self.switch_gui)
		btn_layout.addWidget(self.helical_spa_btn, btn_grid_row, btn_col_origin, register_btn_row_span, register_btn_col_span)

		btn_grid_row += 1

		# Add button grid layout to global layout
		# global_layout.addLayout(btn_layout, global_grid_row, global_col_origin) # Maybe later :)

		# Load the previously saved parameter setting of this sx command
		if os.path.exists(self.gui_settings_file_path):
			self.read_consts(self.gui_settings_file_path)

		# Layout for a constant size
		constant_height_layout = QVBoxLayout()
		constant_height_layout.addLayout(global_layout)
		constant_height_layout.addLayout(btn_layout)
		constant_height_layout.addStretch(1)
		constant_width_layout = QHBoxLayout(self)
		constant_width_layout.addLayout(constant_height_layout)
		constant_width_layout.addStretch(1)

	def switch_gui(self):
		self.parent.helical = not self.parent.helical
		self.parent.sxmenu_item_widget_stacked_layout_global.setCurrentIndex(self.parent.helical)
		self.parent.sxmenu_item_widget_stacked_layout[self.parent.helical].setCurrentWidget(self.parent.sxconst_set[self.parent.helical].widget)

	def handle_regster_widget_event(self, sxconst):
		sxconst.widget.setText(sxconst.register)

	def register_const_set(self):
		# Loop through all project constant parameters
		for sxconst in self.sxconst_set.list:
			sxconst.register = sxconst.widget.text()
			sxconst.register_widget.setText("%s" % sxconst.register)

		# Loop through all command categories
		for sxcmd_category in self.sxcmd_category_list:
			# Loop through all commands of this category
			for sxcmd in sxcmd_category.cmd_list:
				# Loop through all command tokens of this command
				for cmd_token in sxcmd.token_list:
					if not cmd_token.is_locked and cmd_token.type in list(self.sxconst_set.dict.keys()):
						sxconst = self.sxconst_set.dict[cmd_token.type]
						cmd_token.restore = sxconst.register
						try:
							cmd_token.restore_widget.setText("%s" % cmd_token.restore)
						except:
							cmd_token.restore_widget.setCurrentIndex(0)
						cmd_token.widget.setText(cmd_token.restore)
						# print "MRK_DEBUG: %s, %s, %s, %s, %s, %s" % (sxcmd.name, sxcmd.subname, cmd_token.key_base, cmd_token.type, cmd_token.default, cmd_token.restore)
					elif cmd_token.type == "abs_freq":
						assert("apix" in list(self.sxconst_set.dict.keys()))
###						print("MRK_DEBUG: ----- register_const_set() ----- ")
###						print("MRK_DEBUG: cmd_token.type = {}".format(cmd_token.type))
###						sxconst_apix = self.sxconst_set.dict["apix"]
###						print("MRK_DEBUG: sxconst_apix.type = {}".format(sxconst_apix.type))
###						print("MRK_DEBUG: sxconst_apix.register = {}".format(sxconst_apix.register))
###						assert (cmd_token.subwidget_left is not None)
###						assert (cmd_token.subwidget_right is not None)
						assert (cmd_token.calculator_dialog is not None)
###						sxoperand_apix = cmd_token.calculator_dialog.sxoperand_set.dict["apix"]
###						print("MRK_DEBUG: BEFORE sxoperand_apix.register = {}".format(sxoperand_apix.register))
###						print("MRK_DEBUG: BEFORE sxoperand_apix.register_widget.text() = {}".format(sxoperand_apix.register_widget.text()))
###						print("MRK_DEBUG: BEFORE sxoperand_apix.widget.text() = {}".format(sxoperand_apix.widget.text()))
###						sxoperand_apix.register = sxconst_apix.register
###						sxoperand_apix.register_widget.setText("%s" % sxoperand_apix.register)
###						sxoperand_apix.widget.setText(sxoperand_apix.register)
###						print("MRK_DEBUG: AFTER sxoperand_apix.register = {}".format(sxoperand_apix.register))
###						print("MRK_DEBUG: AFTER sxoperand_apix.register_widget.text() = {}".format(sxoperand_apix.register_widget.text()))
###						print("MRK_DEBUG: AFTER sxoperand_apix.widget.text() = {}".format(sxoperand_apix.widget.text()))
						cmd_token.calculator_dialog.reflect_external_global_update_apix()

		# Save the current state of GUI settings
		if os.path.exists(SXLookFeelConst.project_dir) == False:
			os.makedirs(SXLookFeelConst.project_dir)
		self.write_consts(self.gui_settings_file_path)

	def write_consts(self, file_path_out):
		file_out = open(file_path_out,"w")

		# Write script name for consistency check upon loading
		file_out.write("@@@@@ project settings gui settings - ")
		file_out.write(EMANVERSION + " (GITHUB: " + DATESTAMP +")" )
		file_out.write(" @@@@@ \n")

		# Loop through all project constant parameters
		for sxconst in self.sxconst_set.list:
			# The other type has only one line edit box
			val_str = str(sxconst.widget.text())
			file_out.write("<%s> %s (registered %s) == %s \n" % (sxconst.key, sxconst.label, sxconst.register, val_str))

		file_out.close()

	def read_consts(self, file_path_in):
		file_in = open(file_path_in,"r")

		# Check if this parameter file is for this sx script
		line_in = file_in.readline()
		if line_in.find("@@@@@ project settings gui settings") != -1:
			n_function_type_lines = 2
			function_type_line_counter = 0
			# loop through the rest of lines
			for line_in in file_in:
				# Extract label (which should be left of "=="). Also strip the ending spaces
				label_in = line_in.split("==")[0].strip()
				# Extract value (which should be right of "=="). Also strip all spaces
				val_str_in = line_in.split("==")[1].strip()

				# Extract key_base of this command token
				target_operator = "<"
				item_tail = label_in.find(target_operator)
				if item_tail != 0:
					QMessageBox.warning(self, "Invalid Project Settings File Format", "Project settings entry should start from \"%s\" for entry key in line (%s). The format of this file might be corrupted. Please save the project settings file again." % (target_operator, line_in))
				label_in = label_in[item_tail + len(target_operator):].strip() # Get the rest of line
				target_operator = ">"
				item_tail = label_in.find(target_operator)
				if item_tail == -1:
					QMessageBox.warning(self, "Invalid Project Settings File Format", "Project settings entry should have \"%s\" closing entry key in line (%s) The format of this file might be corrupted. Please save the project settings file again." % (target_operator, line_in))
				key = label_in[0:item_tail]
				# Get corresponding sxconst
				if key not in list(self.sxconst_set.dict.keys()):
					QMessageBox.warning(self, "Invalid Project Settings File Format", "Invalid entry key for project settings \"%s\" is found in line (%s). This project settings file might be incompatible with the current version. Please save the project settings file again." % (key, line_in))
				sxconst = self.sxconst_set.dict[key]
				sxconst.widget.setText(val_str_in)
		else:
			QMessageBox.warning(self, "Fail to load project settings", "The specified file is not project settings file.")

		file_in.close()

	def save_consts(self):
		name = QFileDialog.getSaveFileName(self, "Save settings", SXLookFeelConst.file_dialog_dir, options = QFileDialog.DontUseNativeDialog)
		if isinstance(name, tuple):
			file_path = str(name[0])
		else:
			file_path = str(name)
		if file_path != "":
			self.write_consts(file_path)

	def load_consts(self):
		name = QFileDialog.getOpenFileName(self, "Load settings", SXLookFeelConst.file_dialog_dir, options = QFileDialog.DontUseNativeDialog)
		if isinstance(name, tuple):
			file_path = str(name[0])
		else:
			file_path = str(name)
		if file_path != "":
			self.read_consts(file_path)

# ========================================================================================
# Layout of the information widget; owned by the main window
class SXInfoWidget(QWidget):
	def __init__(self, parent = None):
		super(SXInfoWidget, self).__init__(parent)

		self.setStyleSheet("background-color: {0}".format(SXLookFeelConst.default_bg_color_string))
		widget = QWidget(self)

		# Get the picture name
		pic_name = '{0}sxgui_info.png'.format(get_image_directory())
		# Import the picture as pixmap to get the right dimensions
		self.pixmap = QPixmap(pic_name)
		width = self.pixmap.width()
		height = self.pixmap.height()

		# Scrol widget
		scroll_widget = QWidget()
		scroll_widget.setStyleSheet('background-color: transparent')

		label1 = QLabel()
		label1.setFixedHeight(40)
		label2 = QLabel()
		label2.setFixedHeight(40)

		# Create a QLabel and show the picture
		self.label = QLabel()
		self.label.setFixedSize(width, height)
		self.label.setStyleSheet('border-image: url("{0}"); background-color: transparent'.format(pic_name))

		# Layout for the scroll widet vert
		label3 = QLabel()
		label3.setFixedWidth(40)
		label4 = QLabel()
		label4.setFixedWidth(40)

		# Layout for the scroll widget hor
		layout_vert = QHBoxLayout()
		layout_vert.addWidget(label3)
		layout_vert.addWidget(self.label)
		layout_vert.addWidget(label4)

		# Layout for the scroll widget hor
		layout = QVBoxLayout(scroll_widget)
		layout.addWidget(label1)
		layout.addLayout(layout_vert)
		layout.addWidget(label2)

		# Add a scroll area for vertical scrolling
		scroll_area = QScrollArea(widget)
		scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
		scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
		scroll_area.setWidget(scroll_widget)
		scroll_area.setStyleSheet("background-color: {0}".format(SXLookFeelConst.sxcmd_tab_bg_color_string))

		layout = QHBoxLayout(widget)
		layout.addWidget(scroll_area, stretch=1)

		layout = QHBoxLayout(self)
		layout.addWidget(widget)
		layout.setContentsMargins(0, 0, 0, 0)

"""
# ========================================================================================
class SXDialogCalculator(QDialog):
	def __init__(self, parent = None):
		super(QDialog, self).__init__(parent)
		
		self.setWindowModality(Qt.ApplicationModal)
		
		# self.setWindowTitle()
		self.setWindowTitle("Absolute Frequency Calculator")

		temp_label = QLabel("Calculate absolute frequency [1/Pixel] from resolution [A]", self)
		temp_label.move(50,50)
		
		# Create label widget
		temp_label = QLabel("Resolution [A]", self)
		# temp_label.setMinimumWidth(token_label_min_width)
		# grid_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)
		temp_label.move(50,100)
		self.edit_res = QLineEdit(self)
		self.edit_res.setText('Enter Resolution Here')
		self.edit_res.move(200,100)

		temp_label = QLabel("Pixel Size [A/Pixel]", self)
		# temp_label.setMinimumWidth(token_label_min_width)
		# grid_layout.addWidget(temp_label, grid_row, grid_col_origin, token_label_row_span, token_label_col_span)
		temp_label.move(50,200)
		self.edit_apix = QLineEdit(self)
		self.edit_apix.setText('Enter Pixel Size Here')
		self.edit_apix.move(200,200)
		
		self.btn_apply = QPushButton("Apply", self)
		self.btn_apply.move(50,300)
		self.btn_cancel = QPushButton("Cancel", self)
		self.btn_cancel.move(200,300)
		# self.connect(cmd_token_restore_widget[widget_index], SIGNAL("clicked()"), partial(self.handle_restore_widget_event, cmd_token, widget_index))
		
		### self.show()
"""

class SXDialogCalculator(QDialog):
	# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
	# static class variables
	# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
	precision_apix = 7
	precision_abs_freq = 7
	precision_ares = 2

	def __init__(self, sxconst_register_widget_apix, sxcmd_token_widget_abs_freq, sxcmd_token_subwidget_left_ares, parent = None):
		super(QDialog, self).__init__(parent)
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		self.sxoperand_set = self.construct_sxoperand_set()
		self.sxconst_register_widget_apix = sxconst_register_widget_apix # SXconst.register_widget of apix type
		self.sxcmd_token_widget_abs_freq = sxcmd_token_widget_abs_freq
		self.sxcmd_token_subwidget_left_ares = sxcmd_token_subwidget_left_ares
		# This have to be set upon the construction order...
		self.sxcmd_token_widget_apix = None                              # SXcmd_token.widget of apix type if this command has one
		self.sxcmd_token_other_dialog_list_abs_freq = []                 # SXcmd_token.other_dialog_list of the associated abs_freq type token if this command has more than one abs_freq type tokens
		
		# This should be a modal dialog
		self.setWindowModality(Qt.ApplicationModal)
		self.setWindowTitle("%s" % (self.sxoperand_set.label))

		# Layout constants and variables
		global_row_origin = 0; global_col_origin = 0
		global_row_span = 4; global_col_span = 1

		header_row_origin = 0; header_col_origin = 0
		title_row_span = 1; title_col_span = 1
		short_info_row_span = 1; short_info_col_span = 1
		title_min_width = 300
		short_info_min_width = 300
		short_info_min_height = 80

		operand_set_row_origin = 0; operand_set_col_origin = 0
		operand_label_row_span = 1; operand_label_col_span = 1
		operand_register_widget_row_span = 1; operand_register_widget_col_span = 1
		operand_widget_row_span = 1; operand_widget_col_span = 1
		operand_label_min_width = 150
		operand_register_widget_min_width = operand_label_min_width
		operand_widget_min_width = operand_label_min_width

		btn_row_origin = 0; btn_col_origin = 0
		func_btn_row_span = 1; func_btn_col_span = 1
		register_btn_row_span = 1; register_btn_col_span = 2
		func_btn_min_width = 50
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# background_image_file_path = '{0}sxgui_background.png'.format(get_image_directory())
		# self.setStyleSheet('background-image: url("{0}")'.format(background_image_file_path))

		# Set the background color of this widget
		self.setAutoFillBackground(True)
		palette = QPalette()
		palette.setBrush(QPalette.Background, QBrush(SXLookFeelConst.default_bg_color_calculator))
		self.setPalette(palette)

		global_layout = QGridLayout()
		global_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		global_layout.setSpacing(SXLookFeelConst.grid_spacing)
		global_layout.setRowStretch(global_row_span - 1, global_layout.rowStretch(global_row_origin) + 1)

#		header_layout = QGridLayout()
#		header_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
#		header_layout.setSpacing(SXLookFeelConst.grid_spacing)

		operand_layout = QGridLayout()
		operand_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		operand_layout.setSpacing(SXLookFeelConst.grid_spacing)

		btn_layout = QGridLayout()
		btn_layout.setContentsMargins(SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin, SXLookFeelConst.grid_margin)
		btn_layout.setSpacing(SXLookFeelConst.grid_spacing * 2)

		global_grid_row = global_row_origin

#		# Start add title widgets to the grid layout
#		header_grid_row = header_row_origin
#
#		# Set a label and its position in this dialog
#		temp_label = QLabel("<b>%s</b>" % (self.sxoperand_set.label))
#		temp_label.setMinimumWidth(title_min_width)
#		header_layout.addWidget(temp_label, header_grid_row, header_col_origin, title_row_span, title_col_span)
#
#		header_grid_row += 1

#		# NOTE: 2015/11/17 Toshio Moriya
#		# Necessary to separate "<b>%s</b>" from the information for avoiding to invoke the tag interpretations of string
#		# e.g. < becomes the escape character
#		temp_label = QLabel("%s" % (self.sxoperand_set.short_info))
#		temp_label.setWordWrap(True)
#		temp_label.setMinimumWidth(short_info_min_width)
#		temp_label.setMinimumHeight(short_info_min_height)
#		header_layout.addWidget(temp_label, header_grid_row, header_col_origin, short_info_row_span, short_info_col_span)
#
#		# Add const set grid layout to global layout
#		global_layout.addLayout(header_layout, global_grid_row, global_col_origin)
#		global_grid_row += 1

		# Start add project parameter constant widgets to the grid layout
		operand_grid_row = operand_set_row_origin

		# Add widget for editing command args and options
		for sxoperand in self.sxoperand_set.list:
			# Create widget associated to this project constant parameter
			temp_label = QLabel(sxoperand.label)
			temp_label.setMinimumWidth(operand_label_min_width)
			operand_layout.addWidget(temp_label, operand_grid_row, operand_set_col_origin, operand_label_row_span, operand_label_col_span)

			sxoperand_register_widget = QPushButton("%s" % sxoperand.register)
			sxoperand_register_widget.setMinimumWidth(operand_register_widget_min_width)
			custom_style = ""
			if sxoperand.is_input:
				if sxoperand.type in ["apix"]:
					custom_style = "QPushButton {color:green; }" # custom_style = "QPushButton {font: normal; color:black;}"
					sxoperand_register_widget.setStyleSheet(custom_style)
				sxoperand_register_widget.setToolTip('<FONT>'+"Retrieve this default value to edit box"+'</FONT>')
				sxoperand_register_widget.clicked.connect(partial(self.handle_regster_widget_event, sxoperand))
			else:
				assert (not sxoperand.is_input)
				custom_style = "QPushButton {background: rgba(0, 0, 0, 0); color: rgba(0, 0, 0, 0); border: 0px rgba(0, 0, 0, 0) solid;}"
				sxoperand_register_widget.setStyleSheet(custom_style)
				sxoperand_register_widget.setEnabled(False)
			operand_layout.addWidget(sxoperand_register_widget, operand_grid_row, operand_set_row_origin + operand_label_col_span, operand_register_widget_row_span, operand_register_widget_col_span)

			sxoperand_widget = QLineEdit()
			sxoperand_widget.setMinimumWidth(operand_widget_min_width)
			sxoperand_widget.setText(sxoperand.register)
			sxoperand_widget.setToolTip('<FONT>'+sxoperand.help+'</FONT>')
			operand_layout.addWidget(sxoperand_widget, operand_grid_row, operand_set_row_origin + operand_label_col_span + operand_register_widget_col_span, operand_widget_row_span, operand_widget_col_span)
			if not sxoperand.is_input:
				sxoperand_widget.setEnabled(False)
				
			operand_grid_row += 1

			# Register this widget
			sxoperand.register_widget = sxoperand_register_widget
			sxoperand.widget = sxoperand_widget

		# Add const set grid layout to global layout
		global_layout.addLayout(operand_layout, global_grid_row, global_col_origin)
		# global_grid_row += 1

		# Start add buttons to the grid layout
		btn_grid_row = btn_row_origin

		# Add save project constant parameter settings button
		self.convert_units_btn = QPushButton("Convert units")
		self.convert_units_btn.setMinimumWidth(func_btn_min_width)
		self.convert_units_btn.setToolTip('<FONT>'+"Calculate unit conversion"+'</FONT>')
		self.convert_units_btn.setDefault(True);
		self.convert_units_btn.clicked.connect(self.handle_convert_units)
		btn_layout.addWidget(self.convert_units_btn, btn_grid_row, btn_col_origin, func_btn_row_span, func_btn_col_span)

		# Add load project constant parameter settings button
		self.cancel_btn = QPushButton("Cancel")
		self.cancel_btn.setMinimumWidth(func_btn_min_width)
		self.cancel_btn.setToolTip('<FONT>'+"Cancel the unit convertion. Close the calculator dialog without applying the converted value to the associated edit box"+'</FONT>')
		self.cancel_btn.clicked.connect(self.close)
		btn_layout.addWidget(self.cancel_btn, btn_grid_row, btn_col_origin + func_btn_col_span, func_btn_row_span, func_btn_col_span)

		btn_grid_row += 1

###		# For now, initialise with Nyquist resolution of 1.0 [A] pixel size
###		self.apix_str = self.sxconst_register_widget_apix.text()
###		self.abs_freq_str = self.sxcmd_token_widget_abs_freq.text()
###		self.convert_abs_freq_to_ares()
###		sxoperand_apix = self.sxoperand_set.dict["apix"]
###		sxoperand_apix.register = apix_str
###		sxoperand_apix.widget.setText(sxoperand_apix.register)
###		sxoperand_ares = self.sxoperand_set.dict["ares"]
###		sxoperand_ares.register = ares_str
###		sxoperand_ares.widget.setText(sxoperand_ares.register)
		
		# Add a apply button
		self.apply_btn = QPushButton("Apply")
		# make 3D textured push button look
		custom_style = "QPushButton {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #8D0)} QPushButton:pressed {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #084)} QPushButton:focus {font: bold; color: #000;border: 2px solid #8D0;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #8D0)} QPushButton:disabled {font: bold; color: #000;border: 1px solid #333;border-radius: 11px;padding: 2px;background: qradialgradient(cx: 0, cy: 0,fx: 0.5, fy:0.5,radius: 1, stop: 0 #fff, stop: 1 #ff0000)}"
		self.apply_btn.setStyleSheet(custom_style)
		self.apply_btn.setMinimumWidth(func_btn_min_width * register_btn_col_span)
		self.apply_btn.setToolTip('<FONT>'+"Apply converted value to the corresponding command arguments and options"+'</FONT>')
		self.apply_btn.clicked.connect(self.handle_apply_unit_conversion)
		btn_layout.addWidget(self.apply_btn, btn_grid_row, btn_col_origin, register_btn_row_span, register_btn_col_span)

		btn_grid_row += 1

		# Add button grid layout to global layout
		# global_layout.addLayout(btn_layout, global_grid_row, global_col_origin) # Maybe later :)

#		# Load the previously saved parameter setting of this sx command
#		if os.path.exists(self.gui_settings_file_path):
#			self.read_consts(self.gui_settings_file_path)

		# Layout for a constant size
		constant_height_layout = QVBoxLayout()
		constant_height_layout.addLayout(global_layout)
		constant_height_layout.addLayout(btn_layout)
		constant_height_layout.addStretch(1)
		constant_width_layout = QHBoxLayout(self)
		constant_width_layout.addLayout(constant_height_layout)
		constant_width_layout.addStretch(1)
		
		# Synchronise the initial status of operands in this dialog
		self.handle_convert_units(is_enable_message = False)
		
	def construct_sxoperand_set(self):
		sxoperand_set = SXoperand_set(); sxoperand_set.name = "sxc_calculator"; sxoperand_set.label = "Absolute Frequency Calculator"; sxoperand_set.short_info = "From resolution [A] (ares), calculate absolute frequency [1/Pixel] (abs_freq) using a give pixel size [A/Pixel] (apix), where abs_freq = apix/ares."
		sxoperand = SXoperand(); sxoperand.key = "ares"; sxoperand.label = "Resolution [A]"; sxoperand.help = "Resolution [A] to calculate the corresponding absolute frequency [1/Pixel] using a given pixel size [A/Pixel]."; sxoperand.register = " -9999.9"; sxoperand.type = "float"; sxoperand_set.list.append(sxoperand); sxoperand_set.dict[sxoperand.key] = sxoperand
		sxoperand = SXoperand(); sxoperand.key = "apix"; sxoperand.label = "Pixel size [A/Pixel]"; sxoperand.help = "Pixel size of dataset [A/Pixel]."; sxoperand.register = " -9999.9"; sxoperand.type = "apix"; sxoperand_set.list.append(sxoperand); sxoperand_set.dict[sxoperand.key] = sxoperand
		sxoperand = SXoperand(); sxoperand.key = "abs_freq"; sxoperand.label = "Absolute Frequency [1/Pixel]"; sxoperand.help = "Absolute frquency [1/Pixel] corresponding to the given resolution [A/Pixel]."; sxoperand.register = " -9999.9"; sxoperand.type = "abs_freq"; sxoperand.is_input = False; sxoperand_set.list.append(sxoperand); sxoperand_set.dict[sxoperand.key] = sxoperand

		# Store the project constant parameter set as a class data member
		return sxoperand_set
	
	def handle_regster_widget_event(self, sxoperand):
		sxoperand.widget.setText(sxoperand.register)
		self.handle_convert_units()

###	def handle_show_event(self):
###		# Use current value of resolution [A] widget as a register value of resolution [A] operand upon showing this dialog
###		sxoperand_ares = self.sxoperand_set.dict["ares"]
###		sxoperand_ares.register = self.sxcmd_token_subwidget_left_ares.text()
###		sxoperand_ares.register_widget.setText("%s" % sxoperand_ares.register)
###		sxoperand_ares.widget.setText(sxoperand_ares.register)
###		
###		# Show this dialog
###		self.show()

###	def handle_abs_freq_editing_finished_event(self):
###		apix_str = self.sxconst_register_widget_apix.text()
###		abs_freq_str = sxcmd_token_widget_abs_freq.text()
###		ares_str, is_valid_ares = selft.convert_abs_freq_to_ares(apix_str, abs_freq_str)
###		sxoperand_
###		print("MRK_DEBUG: ----- handle_abs_freq_editing_finished_event() ----- ")
###		print("MRK_DEBUG: Input Abs. Freq. [1/Pixel] string ; abs_freq_str := {}".format(abs_freq_str))
###		print("MRK_DEBUG: Input Pixel Size [A/Pixel] string ; apix_str := {}".format(apix_str))
###		print("MRK_DEBUG: Output Resolution [A] string      ; ares_str:= {}".format(ares_str))
###		if is_valid_ares:
###			sxcmd_token_subwidget_left_ares.setText("{}[A]@{}[A/Pix]".format(ares_str, apix_str))
###		else:
###			sxcmd_token_subwidget_left_ares.setText("Mode {}".format(ares_str))
	
###	def convert_abs_freq_to_ares():
###		# Initialise resolution [A] string to absolute frequency [1/Pixel] string
###		self.ares_str = self.abs_freq_str
###		self.is_valid_ares = False
###		
###		# Initialise resolution [A] and pixel size [A/Pixel] with invalid values
###		self.abs_freq_str abs_freq = 0.0
###		self.abs_freq_strapix = 0.0
###		# Check data type of arguments
###		try:
###			self.apix = float(self.apix_str)
###		except ValueError:
###			# The text string can not be converted to float. This must be special value of this option
###			assert (self.ares_str == self.abs_freq_str)
###			assert (not self.is_valid_ares)
###			return
###		try:
###			self.abs_freq = float(self.abs_freq_str)
###		except ValueError:
###			# The text string can not be converted to float. This must be special value of this option
###			assert (self.ares_str == self.abs_freq_str)
###			assert (not self.is_valid_ares)
###			return
###		# Check invalid values of arguments
###		if self.apix <= 0.0:
###			# The valid range of pixel size [A] is apix > 0.0. The valid value must have not been set yet.
###			assert (self.ares_str == self.abs_freq_str)
###			assert (not self.is_valid_ares)
###			return ares_str, self.is_valid_ares
###		assert (apix > 0.0)
###		if abs_freq <= 0.0 or abs_freq > 0.5:
###			# The valid range of absolut frequency is 0.0 < abs_freq <= 0.5. This must be special value of this option
###			assert (ares_str == abs_freq_str)
###			assert (not is_valid_ares)
###			return ares_str, is_valid_ares
###		assert (abs_freq > 0.0 or abs_freq <= 0.5)
###		
###		ares = round(apix/abs_freq, SXDialogCalculator.ares_precision)
###		ares_str = "{}".format(ares)
###		is_valid_ares = True
###		
###		print("MRK_DEBUG: ----- convert_abs_freq_to_ares() ----- ")
###		print("MRK_DEBUG: Input Abs. Freq. [1/Pixel]   ; abs_freq := {}".format(abs_freq))
###		print("MRK_DEBUG: Input Pixel Size [A/Pixel]   ; apix := {}".format(apix))
###		print("MRK_DEBUG: Output Resolution [A]        ; ares:= {}".format(ares))
###		print("MRK_DEBUG: Output Resolution [A] string ; ares_str:= {}".format(ares_str))
###		
###		return ares_str, is_valid_ares

	def handle_convert_units(self, is_enable_message = True):
		sxoperand_apix = self.sxoperand_set.dict["apix"]
		apix_str = sxoperand_apix.widget.text()
		sxoperand_ares = self.sxoperand_set.dict["ares"]
		ares_str = sxoperand_ares.widget.text()
		sxoperand_abs_freq = self.sxoperand_set.dict["abs_freq"]
		
		apix = self.convert_str_to_float_apix(apix_str)
		
		if apix is None:
			sxoperand_apix.validated = None
			sxoperand_ares.validated = None
			sxoperand_abs_freq.validated = None
			sxoperand_abs_freq.widget.setText("Invalid Pixel Size {}".format(apix_str))
			self.apply_btn.setEnabled(False)
			if is_enable_message:
				QMessageBox.warning(self, "Invalid Pixel Size [A/Pixel]", "Invalid Value {} for Pixel Size [A/Pixel] is provided. It must be a non-zero positive float value...".format(apix_str))
		else:
			assert (apix is not None)
			sxoperand_apix.validated = apix
			nyquist_res = 2.0 * apix
			nyquist_res_str = "{}".format(nyquist_res)
			ares = self.convert_str_to_float_ares(ares_str, nyquist_res)
			if ares is None:
				sxoperand_ares.validated = None
				sxoperand_abs_freq.validated = None
				sxoperand_abs_freq.widget.setText("Invalid Resolution {}".format(ares_str))
				self.apply_btn.setEnabled(False)
				if is_enable_message:
					QMessageBox.warning(self, "Invalid Resolution [A]", "Invalid Value {} for Resolution [A] is provided. It must be a float value, and larger than or equal to Nyquist resolution {} [A] (2.0 * Pixel Size [A/Pixel])...".format(ares_str, nyquist_res_str))
			else:
				assert (ares >= nyquist_res)
				sxoperand_ares.validated = ares
				abs_freq = round(apix/ares, self.precision_abs_freq)
				# The valid range of absolute frequency [1/Pixel] is 0.0 < abs_freq <= 0.5
				# If both pixel size and resolution values are valid, the absolute frequency must be always valid.
				assert (abs_freq > 0.0 or abs_freq <= 0.5)
				sxoperand_abs_freq.validated = abs_freq
				abs_freq_str = "{}".format(abs_freq)
				# Update widget associating to absolute frequency [1/Pixel] in this dialog
				sxoperand_abs_freq.widget.setText(abs_freq_str)
				self.apply_btn.setEnabled(True)
		
		
#		if sxoperand_apix.validated is None:
#			sxoperand_abs_freq.validated = None
#			sxoperand_abs_freq.widget.setText("Invalid Pixel Size {}".format(apix_str))
#			self.apply_btn.setEnabled(False)
#			if is_enable_message:
#				QMessageBox.warning(self, "Invalid Pixel Size [A/Pixel]", "Invalid Value {} for Pixel Size [A/Pixel] is provided. It must be a non-zero positive float value...".format(apix_str))
#		else:
#			assert (sxoperand_apix.validated is not None)
#			assert (apix > 0.0)
#			nyquist_res = 2.0 * apix
#			nyquist_res_str = "{}".format(nyquist_res)
#			if sxoperand_ares.validated is None:
#				sxoperand_abs_freq.validated = None
#				sxoperand_abs_freq.widget.setText("Invalid Resolution {}".format(ares_str))
#				self.apply_btn.setEnabled(False)
#				if is_enable_message:
#					QMessageBox.warning(self, "Invalid Resolution [A]", "Invalid Value {} for Resolution [A] is provided. It must be a float value, and larger than or equal to Nyquist resolution {} [A] (2.0 * Pixel Size [A/Pixel])...".format(ares_str, nyquist_res_str))
#			else:
#				assert (sxoperand_apix.validated is not None)
#				assert (nyquist_res * 2.0 )
#				abs_freq = round(apix/ares, self.precision_abs_freq)
#				# The valid range of absolute frequency [1/Pixel] is 0.0 < abs_freq <= 0.5
#				# If both pixel size and resolution values are valid, the absolute frequency must be always valid.
#				assert (abs_freq > 0.0 or abs_freq <= 0.5)
#				sxoperand_abs_freq.validated = abs_freq
#				abs_freq_str = "{}".format(abs_freq)
#				# Update widget associating to absolute frequency [1/Pixel] in this dialog
#				sxoperand_abs_freq.widget.setText(abs_freq_str)
#				self.apply_btn.setEnabled(True)
		
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.handle_convert_units() ----- ")
		# print("MRK_DEBUG: Edit Widget Resolution [A]              ; sxoperand_ares.widget.text()              := \"{}\"".format(sxoperand_ares.widget.text()))
		# print("MRK_DEBUG: Edit Validated Resolution [A]           ; sxoperand_ares.validated                  := \"{}\"".format(sxoperand_ares.validated))
		# print("MRK_DEBUG: Register Widget Resolution [A]          ; sxoperand_ares.register_widget.text()     := \"{}\"".format(sxoperand_ares.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Resolution [A]       ; sxoperand_ares.validated_register         := \"{}\"".format(sxoperand_ares.validated_register))
		# print("MRK_DEBUG: Register Resolution [A]                 ; sxoperand_ares.register                   := \"{}\"".format(sxoperand_ares.register))
		# 
		# print("MRK_DEBUG: Edit Widget Pixel Size [A/Pixel]        ; sxoperand_apix.widget.text()              := \"{}\"".format(sxoperand_apix.widget.text()))
		# print("MRK_DEBUG: Edit Validated Abs. Freq. [1/Pixel]     ; sxoperand_apix.validated                  := \"{}\"".format(sxoperand_apix.validated))
		# print("MRK_DEBUG: Register Widget Pixel Size [A/Pixel]    ; sxoperand_apix.register_widget.text()     := \"{}\"".format(sxoperand_apix.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Pixel Size [A/Pixel] ; sxoperand_apix.validated_register         := \"{}\"".format(sxoperand_apix.validated_register))
		# print("MRK_DEBUG: Register Pixel Size [A/Pixel]           ; sxoperand_apix.register                   := \"{}\"".format(sxoperand_apix.register))
		# 
		# print("MRK_DEBUG: Edit Widget bs. Freq. [1/Pixel]         ; sxoperand_abs_freq.widget.text()          := \"{}\"".format(sxoperand_abs_freq.widget.text()))
		# print("MRK_DEBUG: Edit Validated Abs. Freq. [1/Pixel]     ; sxoperand_abs_freq.validated              := \"{}\"".format(sxoperand_abs_freq.validated))
		# print("MRK_DEBUG: Register Widget Abs. Freq. [1/Pixel]    ; sxoperand_abs_freq.register_widget.text() := \"{}\"".format(sxoperand_abs_freq.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Abs. Freq. [1/Pixel] ; sxoperand_abs_freq.validated_register     := \"{}\"".format(sxoperand_abs_freq.validated_register))
		# print("MRK_DEBUG: Register Abs. Freq. [1/Pixel]           ; sxoperand_abs_freq.register               := \"{}\"".format(sxoperand_abs_freq.register))

	def handle_apply_unit_conversion(self):
		# Do the unit conversion first
		self.handle_convert_units()
		sxoperand_abs_freq = self.sxoperand_set.dict["abs_freq"]
		sxoperand_ares = self.sxoperand_set.dict["ares"]
		sxoperand_apix = self.sxoperand_set.dict["apix"]
		
		abs_freq = sxoperand_abs_freq.validated
		ares = sxoperand_ares.validated
		apix = sxoperand_apix.validated
		# This button should have been disabled if all operands are valid
		if abs_freq is None:
			ERROR("Logical Error: Encountered unexpected condition in SXDialogCalculator.handle_apply_unit_conversion(). Consult with the developer.", "%s in %s" % (__name__, os.path.basename(__file__)))
			self.sxcmd_token_widget_abs_freq.setText("Logical Error")
			self.sxcmd_token_subwidget_left_ares.setText("Logical Error")
		else:
			assert (abs_freq is not None and ares is not None and apix is not None)
			# Update register, validated_register, and register_widget of absolute frequency [1/Pixel]
			sxoperand_abs_freq.validated_register = abs_freq
			sxoperand_abs_freq.register = "{}".format(sxoperand_abs_freq.validated_register)
			sxoperand_abs_freq.register_widget.setText(sxoperand_abs_freq.register)

			# Update register, validated_register, and register_widget of resolution [A]
			sxoperand_ares.validated_register = ares
			sxoperand_ares.register = "{}".format(sxoperand_ares.validated_register)
			sxoperand_ares.register_widget.setText(sxoperand_ares.register)

			# DO NOT update register, validated_register, and register_widget of pixel size [A/Pixel]
			# These should be updated through only project constants settings
			
			# Update command token subwidget associating to resolution [A] accordingly
			self.sxcmd_token_widget_abs_freq.setText(sxoperand_abs_freq.register)
			self.sxcmd_token_subwidget_left_ares.setText("{}[A]@{}[A/Pix]".format(ares, apix))
			
			# Update the associated pixel size token of this command if there is any
			if self.sxcmd_token_widget_apix is not None:
				self.sxcmd_token_widget_apix.setText("{}".format(apix))
			for sxcmd_token_other_dialog_abs_freq in self.sxcmd_token_other_dialog_list_abs_freq:
				assert (sxcmd_token_other_dialog_abs_freq is not self)
				sxcmd_token_other_dialog_abs_freq.reflect_external_local_update_apix()
		
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.handle_apply_unit_conversion() ----- ")
		# print("MRK_DEBUG: Edit Widget Resolution [A]              ; sxoperand_ares.widget.text()              := \"{}\"".format(sxoperand_ares.widget.text()))
		# print("MRK_DEBUG: Edit Validated Resolution [A]           ; sxoperand_ares.validated                  := \"{}\"".format(sxoperand_ares.validated))
		# print("MRK_DEBUG: Register Widget Resolution [A]          ; sxoperand_ares.register_widget.text()     := \"{}\"".format(sxoperand_ares.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Resolution [A]       ; sxoperand_ares.validated_register         := \"{}\"".format(sxoperand_ares.validated_register))
		# print("MRK_DEBUG: Register Resolution [A]                 ; sxoperand_ares.register                   := \"{}\"".format(sxoperand_ares.register))
		# 
		# print("MRK_DEBUG: Edit Widget Pixel Size [A/Pixel]        ; sxoperand_apix.widget.text()              := \"{}\"".format(sxoperand_apix.widget.text()))
		# print("MRK_DEBUG: Edit Validated Abs. Freq. [1/Pixel]     ; sxoperand_apix.validated                  := \"{}\"".format(sxoperand_apix.validated))
		# print("MRK_DEBUG: Register Widget Pixel Size [A/Pixel]    ; sxoperand_apix.register_widget.text()     := \"{}\"".format(sxoperand_apix.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Pixel Size [A/Pixel] ; sxoperand_apix.validated_register         := \"{}\"".format(sxoperand_apix.validated_register))
		# print("MRK_DEBUG: Register Pixel Size [A/Pixel]           ; sxoperand_apix.register                   := \"{}\"".format(sxoperand_apix.register))
		# 
		# print("MRK_DEBUG: Edit Widget bs. Freq. [1/Pixel]         ; sxoperand_abs_freq.widget.text()          := \"{}\"".format(sxoperand_abs_freq.widget.text()))
		# print("MRK_DEBUG: Edit Validated Abs. Freq. [1/Pixel]     ; sxoperand_abs_freq.validated              := \"{}\"".format(sxoperand_abs_freq.validated))
		# print("MRK_DEBUG: Register Widget Abs. Freq. [1/Pixel]    ; sxoperand_abs_freq.register_widget.text() := \"{}\"".format(sxoperand_abs_freq.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Abs. Freq. [1/Pixel] ; sxoperand_abs_freq.validated_register     := \"{}\"".format(sxoperand_abs_freq.validated_register))
		# print("MRK_DEBUG: Register Abs. Freq. [1/Pixel]           ; sxoperand_abs_freq.register               := \"{}\"".format(sxoperand_abs_freq.register))
		
		self.close()

	def convert_str_to_float_apix(self, apix_str):
		# Initialise with easy-to-recognise invalid value, which won't be used.
		apix = -9999.9 
		# Check data type of string
		try:
			apix = float(apix_str)
		except ValueError:
			# The text string can not be converted to float.
			return None
		# round by predefined precision so that calculated resolution [A] is likely to be a simple float value (e.g. 3.0[A])
		apix = round(apix, self.precision_apix)
		# Check value validity. The valid range of pixel size [A] is apix > 0.0.
		if apix <= 0.0:
			return None
		assert (apix > 0.0)
		return apix

	def register_operand_apix_str(self, apix_str):
		# Validated the operand value
		sxoperand_apix = self.sxoperand_set.dict["apix"]
		sxoperand_apix.register = apix_str
		apix = self.convert_str_to_float_apix(sxoperand_apix.register)
		if apix is None:
			sxoperand_apix.validated_register = None
		else:
			sxoperand_apix.validated_register = apix
			assert (sxoperand_apix.validated_register is not None)
		sxoperand_apix.validated = sxoperand_apix.validated_register
		# Set the string regardless of the validity
		# Update widgets associating to pixel size [A/Pixel] in this dialog
		sxoperand_apix.register_widget.setText(sxoperand_apix.register)
		sxoperand_apix.widget.setText(sxoperand_apix.register)

	def edit_operand_apix_str(self, apix_str):
		# Validated the operand value
		sxoperand_apix = self.sxoperand_set.dict["apix"]
		apix = self.convert_str_to_float_apix(apix_str)
		if apix is None:
			sxoperand_apix.validated = None
		else:
			sxoperand_apix.validated = apix
			assert (sxoperand_apix.validated is not None)
		# Set the string regardless of the validity
		# Update widgets associating to pixel size [A/Pixel] in this dialog
		sxoperand_apix.widget.setText(apix_str)

	def convert_str_to_float_abs_freq(self, abs_freq_str):
		# Initialise with easy-to-recognise invalid value, which won't be used.
		abs_freq = -9999.9 
		# Check data type of arguments
		try:
			abs_freq = float(abs_freq_str)
		except ValueError:
			# The text string can not be converted to float.
			return None
		# round by predefined precision so that calculated resolution [A] is likely to be a simple float value (e.g. 3.0[A])
		abs_freq = round(abs_freq, self.precision_abs_freq)
		# Check value validity. The valid range of absolut frequency is 0.0 < abs_freq <= 0.5. 
		if abs_freq <= 0.0 or abs_freq > 0.5:
			return None
		assert (abs_freq > 0.0 or abs_freq <= 0.5)
		return abs_freq

	def register_operand_abs_freq_str(self, abs_freq_str):
		# Validated the operand value
		sxoperand_abs_freq = self.sxoperand_set.dict["abs_freq"]
		sxoperand_abs_freq.register = abs_freq_str
		abs_freq = self.convert_str_to_float_abs_freq(sxoperand_abs_freq.register)
		if abs_freq is None:
			sxoperand_abs_freq.validated_register = None
		else:
			sxoperand_abs_freq.validated_register = abs_freq
			assert (sxoperand_abs_freq.validated_register is not None)
		sxoperand_abs_freq.validated = sxoperand_abs_freq.validated_register
		# Set the string regardless of the validity
		# Update widgets associating to absolute frequency [1/Pixel] in this dialog
		sxoperand_abs_freq.register_widget.setText(sxoperand_abs_freq.register)
		sxoperand_abs_freq.widget.setText(sxoperand_abs_freq.register)

	def convert_str_to_float_ares(self, ares_str, nyquist_res = 0.0):
		# Initialise with easy-to-recognise invalid value, which won't be used.
		ares = -9999.9 
		# Check data type of arguments
		try:
			ares = float(ares_str)
		except ValueError:
			# The text string can not be converted to float. This must be special value of this option
			return None
		# Check value validity. The valid range of resolution is 0.0 < ares
		# Here, we ignore Nyquist with pixel size.
		ares = round(ares, self.precision_ares)
		if ares < nyquist_res:
			return None
		assert (ares >= nyquist_res)
		return ares

	# def rigister_operand_ares_str(self, ares_str, nyquist_res = 0.0):
	# 	sxoperand_ares = self.sxoperand_set.dict["ares"]
	# 	sxoperand_ares.register = ares_str
	# 	ares = self.convert_str_to_float_ares(sxoperand_ares.register, nyquist_res)
	# 	if ares is None:
	# 		sxoperand_ares.validated_register = None
	# 		return
	# 	sxoperand_ares.validated_register = ares
	# 	assert (sxoperand_ares.validated_register is not None)
	# 	return
	
	def synchronize_external_update_to_ares(self):
		# print("MRK_DEBUG: ----- synchronize_external_update_to_ares ----- ")
		sxoperand_apix = self.sxoperand_set.dict["apix"]
		sxoperand_abs_freq = self.sxoperand_set.dict["abs_freq"]
		sxoperand_ares = self.sxoperand_set.dict["ares"]
		if sxoperand_abs_freq.validated is None:
			# Use registered string of absolute frequency. This can be special value of this option (e.g. indicate lpf modes)
			sxoperand_abs_freq.widget.setText("Mode {}".format(sxoperand_abs_freq.register))
			# sxoperand_ares.register = sxoperand_abs_freq.widget.text()
			sxoperand_ares.register = "No Default"
			sxoperand_ares.validated_register = None
		else:
			assert (sxoperand_abs_freq.validated is not None)
			if sxoperand_apix.validated is None:
				# Invalid value to indicate the invalid pixel size but invalid absolute frequency value...
				sxoperand_ares.validated_register = None
				sxoperand_ares.register = "Invalid Pixel Size {}".format(sxoperand_apix.widget.text())
				sxoperand_abs_freq.widget.setText(sxoperand_ares.register)
			else:
				assert (sxoperand_apix.validated is not None)
				apix = sxoperand_apix.validated
				abs_freq = sxoperand_abs_freq.validated
				ares = round(apix/abs_freq, self.precision_ares)
				# The valid range of resolution [A] is ares <= apix * 2.0 (Nyquist resolution)
				# If both pixel size and absolute frequency values are valid, this must be always true.
				assert (ares >= apix * 2.0 ) 
				sxoperand_ares.validated_register = ares
				sxoperand_ares.register = "{}".format(sxoperand_ares.validated_register)
		sxoperand_ares.validated = sxoperand_ares.validated_register
		
		# Update widgets associating to resolution [A] in this dialog
		# and command token subwidget associating to resolution [A]
		if sxoperand_ares.validated_register is not None:
			self.apply_btn.setEnabled(True)
			sxoperand_ares.register_widget.setEnabled(True)
			sxoperand_ares.register_widget.setText(sxoperand_ares.register)
			sxoperand_ares.widget.setText(sxoperand_ares.register)
			# Update widget of this command token for absolute frequency [1/Pixel] associating to this dialog
			self.sxcmd_token_widget_abs_freq.setText("{}".format(sxoperand_abs_freq.register))
			# Update left subwidget of this command token for absolute frequency [1/Pixel] associating to this dialog
			self.sxcmd_token_subwidget_left_ares.setText("{}[A]@{}[A/Pix]".format(sxoperand_ares.validated, sxoperand_apix.validated))
		else:
			assert (sxoperand_ares.validated_register is None)
			self.apply_btn.setEnabled(False)
			sxoperand_ares.register_widget.setEnabled(False)
			# sxoperand_ares.register_widget.setText("No Default")
			sxoperand_ares.register_widget.setText(sxoperand_ares.register)
			sxoperand_ares.widget.setText("")
			if sxoperand_abs_freq.validated is None:
				self.sxcmd_token_subwidget_left_ares.setText("Mode {}".format(sxoperand_abs_freq.register))
			else:
				assert (sxoperand_abs_freq.validated is not None)
				self.sxcmd_token_subwidget_left_ares.setText("{}".format(sxoperand_ares.register))
		
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.synchronize_external_update_to_ares() ----- ")
		# print("MRK_DEBUG: Edit Widget Resolution [A]              ; sxoperand_ares.widget.text()              := \"{}\"".format(sxoperand_ares.widget.text()))
		# print("MRK_DEBUG: Edit Validated Resolution [A]           ; sxoperand_ares.validated                  := \"{}\"".format(sxoperand_ares.validated))
		# print("MRK_DEBUG: Register Widget Resolution [A]          ; sxoperand_ares.register_widget.text()     := \"{}\"".format(sxoperand_ares.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Resolution [A]       ; sxoperand_ares.validated_register         := \"{}\"".format(sxoperand_ares.validated_register))
		# print("MRK_DEBUG: Register Resolution [A]                 ; sxoperand_ares.register                   := \"{}\"".format(sxoperand_ares.register))
		# 
		# print("MRK_DEBUG: Edit Widget Pixel Size [A/Pixel]        ; sxoperand_apix.widget.text()              := \"{}\"".format(sxoperand_apix.widget.text()))
		# print("MRK_DEBUG: Edit Validated Abs. Freq. [1/Pixel]     ; sxoperand_apix.validated                  := \"{}\"".format(sxoperand_apix.validated))
		# print("MRK_DEBUG: Register Widget Pixel Size [A/Pixel]    ; sxoperand_apix.register_widget.text()     := \"{}\"".format(sxoperand_apix.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Pixel Size [A/Pixel] ; sxoperand_apix.validated_register         := \"{}\"".format(sxoperand_apix.validated_register))
		# print("MRK_DEBUG: Register Pixel Size [A/Pixel]           ; sxoperand_apix.register                   := \"{}\"".format(sxoperand_apix.register))
		# 
		# print("MRK_DEBUG: Edit Widget bs. Freq. [1/Pixel]         ; sxoperand_abs_freq.widget.text()          := \"{}\"".format(sxoperand_abs_freq.widget.text()))
		# print("MRK_DEBUG: Edit Validated Abs. Freq. [1/Pixel]     ; sxoperand_abs_freq.validated              := \"{}\"".format(sxoperand_abs_freq.validated))
		# print("MRK_DEBUG: Register Widget Abs. Freq. [1/Pixel]    ; sxoperand_abs_freq.register_widget.text() := \"{}\"".format(sxoperand_abs_freq.register_widget.text()))
		# print("MRK_DEBUG: Register Validated Abs. Freq. [1/Pixel] ; sxoperand_abs_freq.validated_register     := \"{}\"".format(sxoperand_abs_freq.validated_register))
		# print("MRK_DEBUG: Register Abs. Freq. [1/Pixel]           ; sxoperand_abs_freq.register               := \"{}\"".format(sxoperand_abs_freq.register))
		
	def reflect_external_global_update_apix(self):
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.reflect_external_global_update_apix() ----- ")
		# print("MRK_DEBUG: self.sxconst_register_widget_apix.text() := \"{}\"".format(self.sxconst_register_widget_apix.text()))
		apix_str = self.sxconst_register_widget_apix.text()
		self.register_operand_apix_str(apix_str)
		self.synchronize_external_update_to_ares()

	def reflect_external_local_update_apix(self):
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.reflect_external_local_update_apix() ----- ")
		# print("MRK_DEBUG: self.sxcmd_token_widget_apix.text() := \"{}\"".format(self.sxcmd_token_widget_apix.text()))
		apix_str = self.sxcmd_token_widget_apix.text()
		self.edit_operand_apix_str(apix_str)
		self.synchronize_external_update_to_ares()

	def reflect_external_local_update_abs_freq(self):
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.reflect_external_local_update_abs_freq() ----- ")
		# print("MRK_DEBUG: self.sxcmd_token_widget_abs_freq.text() := \"{}\"".format(self.sxcmd_token_widget_abs_freq.text()))
		abs_freq_str = self.sxcmd_token_widget_abs_freq.text()
		self.register_operand_abs_freq_str(abs_freq_str)
		self.synchronize_external_update_to_ares()

	def reflect_external_local_update_apix_and_abs_freq(self):
		# print("MRK_DEBUG: ")
		# print("MRK_DEBUG: ----- SXDialogCalculator.reflect_external_local_update_apix_and_abs_freq() ----- ")
		# print("MRK_DEBUG: self.sxcmd_token_widget_apix.text() := \"{}\"".format(self.sxcmd_token_widget_apix.text()))
		# print("MRK_DEBUG: self.sxcmd_token_widget_abs_freq.text() := \"{}\"".format(self.sxcmd_token_widget_abs_freq.text()))
		apix_str = self.sxcmd_token_widget_apix.text()
		self.edit_operand_apix_str(apix_str)
		abs_freq_str = self.sxcmd_token_widget_abs_freq.text()
		self.register_operand_abs_freq_str(abs_freq_str)
		self.synchronize_external_update_to_ares()

###	@staticmethod
###	def convert_ares_to_abs_freq(apix_str, ares_str):
###		# Initialise resolution [A] string to absolute frequency [1/Pixel] string
###		abs_freq_str = ares_str
###		is_valid_abs_freq = False
###		
###		# Initialise resolution [A] and pixel size [A/Pixel] with invalid values
###		ares = 0.0
###		apix = 0.0
###		# Check data type of arguments
###		try:
###			apix = float(apix_str)
###		except ValueError:
###			# The text string can not be converted to float. This must be special value of this option
###			assert (abs_freq_str == ares_str)
###			assert (not is_valid_abs_freq)
###			return abs_freq_str, is_valid_abs_freq
###		try:
###			ares = float(ares_str)
###		except ValueError:
###			# The text string can not be converted to float. This must be special value of this option
###			assert (abs_freq_str == ares_str)
###			assert (not is_valid_abs_freq)
###			return abs_freq_str, is_valid_abs_freq
###		if apix <= 0.0:
###			# The valid range of pixel size [A] is apix > 0.0. The valid value must have not been set yet.
###			assert (abs_freq_str == ares_str)
###			assert (not is_valid_abs_freq)
###			return abs_freq_str, is_valid_abs_freq
###		assert (apix > 0.0)
###		nyquist_res = apix * 2.0
###		# Check invalid values of arguments
###		if ares < nyquist_res:
###			# The valid range of absolut frequency is 0.0 < abs_freq <= 0.5. This must be special value of this option
###			assert (abs_freq_str == ares_str)
###			assert (not is_valid_abs_freq)
###			return abs_freq_str, is_valid_abs_freq
###		assert (ares >= nyquist_res)
###		
###		abs_freq = round(apix/ares, SXDialogCalculator.abs_freq_precision)
###		abs_freq_str = "{}".format(abs_freq)
###		is_valid_abs_freq = True
###		
###		print("MRK_DEBUG: ----- convert_ares_to_abs_freq() ----- ")
###		print("MRK_DEBUG: Input Abs. Freq. [1/Pixel]   ; abs_freq := {}".format(abs_freq))
###		print("MRK_DEBUG: Input Pixel Size [A/Pixel]   ; apix := {}".format(apix))
###		print("MRK_DEBUG: Output Resolution [A]        ; ares:= {}".format(ares))
###		print("MRK_DEBUG: Output Resolution [A] string ; ares_str:= {}".format(ares_str))
###		
###		return abs_freq_str, is_valid_abs_freq

###	def convert_units(self):
###		# Initialise resolution [A] and pixel size [A/Pixel] with invalid values
###		apix = 0.0
###		ares = 0.0
###		# Get resolution [A] and pixel size [A/Pixel] values from the edit boxes
###		apix_str = self.sxoperand_set.dict["apix"].widget.text()
###		ares_str = self.sxoperand_set.dict["ares"].widget.text()
###		try:
###			apix = float(apix_str)
###		except ValueError:
###			# Check invalid values make sure the text string can be converted to float
###			QMessageBox.warning(self, "Invalid Value for Pixel Size [A/Pixel]", "Invalid Value {} for Pixel Size [A/Pixel] is provided. It must be a non-zero positive float value...".format(apix_str))
###			return
###		try:
###			ares = float(ares_str)
###		except ValueError:
###			# Check invalid values make sure the text string can be converted to float
###			QMessageBox.warning(self, "Invalid Value for Resolution [A]", "Invalid Value {} for Resolution [A] is provided. It must be a non-zero positive float value...".format(ares_str))
###			return
###		# Check invalid values
###		if apix <= 0.0:
###			QMessageBox.warning(self, "Invalid Value for Pixel Size [A/Pixel]", "Invalid Value {} for Pixel Size [A/Pixel] is provided. It must be a non-zero positive float value...".format(apix))
###			return
###		assert (apix > 0.0)
###		nyquist_res = apix * 2.0
###		if ares < nyquist_res:
###			QMessageBox.warning(self, "Invalid Value for Resolution [A]", "Invalid Value {} for Resolution [A] is provided. It must be larger than or equal to the Nyquest resolution {} [A] of the provided pixel size {} [A/Pixel]...".format(ares, nyquist_res, apix))
###			return
###		assert (ares >= nyquist_res)
###		
###		self.abs_freq = round(apix/ares, self.abs_freq_precision)
###		self.apply_btn.setText("Apply Abs. Freq. {} [1/Pixel]".format(self.abs_freq))
###		
###		print("MRK_DEBUG: ----- convert_units() ----- ")
###		print("MRK_DEBUG: Input Resolution [A]       ; ares := {}".format(ares))
###		print("MRK_DEBUG: Input Pixel Size [A/Pixel] ; apix := {}".format(apix))
###		print("MRK_DEBUG: Output Abs. Freq. [1/Pixel]; self.abs_freq:= {}".format(self.abs_freq))
###		print("MRK_DEBUG: Output Resolution [A]      ; self.abs_freq:= {}".format(round(apix/self.abs_freq, self.ares_precision)))

# ========================================================================================
# Main Window (started by class SXApplication)
class SXMainWindow(QMainWindow): # class SXMainWindow(QWidget):

	def __init__(self, helical, parent = None):
		super(SXMainWindow, self).__init__(parent)

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		# class variables
		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><
		self.helical = helical
		self.sxinfo = []
		self.sxconst_set = []
		self.sxcmd_category_list = []

		self.cur_sxmenu_item = None
		self.sxmenu_item_widget_stacked_layout = []

		# ><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><

		# --------------------------------------------------------------------------------
		# Construct menu items
		# --------------------------------------------------------------------------------
		self.sxinfo.append(self.construct_sxinfo())              # Construct application information
		self.sxinfo.append(self.construct_sxinfo())              # Construct application information
		self.sxconst_set.append(self.construct_sxconst_set(0))         # Construct project constant set for project settings
		self.sxconst_set.append(self.construct_sxconst_set(1))         # Construct project constant set for project settings
		self.sxcmd_category_list.append(self.construct_sxcmd_category_list()) # Construct list of categorised sxscript objects (extracted from associated wiki documents)
		self.sxcmd_category_list.append(self.construct_sxcmd_category_list()) # Construct list of categorised sxscript objects (extracted from associated wiki documents)

		# --------------------------------------------------------------------------------
		# Setup Window Layout
		# --------------------------------------------------------------------------------
		background_image_file_path = '{0}sxgui_background.png'.format(get_image_directory())

		central_widget_global = QWidget(self)
		central_widget_global.setObjectName('central')
		central_widget_global.setStyleSheet(
			'QWidget#central {{background-image: url("{0}")}}'.format(background_image_file_path)
			)
		self.setCentralWidget(central_widget_global)
		self.sxmenu_item_widget_stacked_layout_global = QStackedLayout()
		central_widget_global.setLayout(self.sxmenu_item_widget_stacked_layout_global)

		for idx in range(2):
			# Central widget
			central_widget = QWidget(self)
			central_widget.setObjectName('central')
			central_widget.setStyleSheet(
				'QWidget#central {{background-image: url("{0}")}}'.format(background_image_file_path)
				)

			# Layout for central widget
			central_layout = QHBoxLayout(central_widget)
			central_widget.setLayout(central_layout)

			# --------------------------------------------------------------------------------
			# Construct and add a widget for menu item button area (containing all menu item buttons)
			# --------------------------------------------------------------------------------
			sxmenu_item_btn_area_widget = SXMenuItemBtnAreaWidget(self.sxconst_set[idx], self.sxcmd_category_list[idx], self.sxinfo[idx], idx, central_widget)
			central_layout.addWidget(sxmenu_item_btn_area_widget)

			# --------------------------------------------------------------------------------
			# Construct and add widgets for menu item widget area (containing all menu item widgets)
			# --------------------------------------------------------------------------------
			# Stacked layout for sx menu item widgets area
			self.sxmenu_item_widget_stacked_layout.append(QStackedLayout())
			central_layout.addLayout(self.sxmenu_item_widget_stacked_layout[idx], stretch = 1)

			# Construct and add a widget for project constants settings
			self.sxconst_set[idx].widget = SXConstSetWidget(self.sxconst_set[idx], self.sxcmd_category_list[idx], helical=idx, parent=self)
			self.sxmenu_item_widget_stacked_layout[idx].addWidget(self.sxconst_set[idx].widget)

			# Construct and add widgets for sx command categories
			for sxcmd_category in self.sxcmd_category_list[idx]:
				# Create SXCmdCategoryWidget for this command category
				sxcmd_category.widget = SXCmdCategoryWidget(self.sxconst_set[idx], sxcmd_category, idx)
				self.sxmenu_item_widget_stacked_layout[idx].addWidget(sxcmd_category.widget)

			# Construct and add a widget for GUI application information
			self.sxinfo[idx].widget = SXInfoWidget()
			self.sxmenu_item_widget_stacked_layout[idx].addWidget(self.sxinfo[idx].widget)

			# --------------------------------------------------------------------------------
			# Set up event handler of all menu item buttons
			# --------------------------------------------------------------------------------
			for i, sxcmd_category in enumerate(self.sxcmd_category_list[idx]):
				sxcmd_category.btn.clicked.connect(partial(self.handle_sxmenu_item_btn_event, sxcmd_category, i))
			self.sxconst_set[idx].btn.clicked.connect(partial(self.handle_sxmenu_item_btn_event, self.sxconst_set[idx]))
			self.sxinfo[idx].btn.clicked.connect(partial(self.handle_sxmenu_item_btn_event, self.sxinfo[idx]))

			# --------------------------------------------------------------------------------
			# Register project constant parameter settings upon initialization
			# --------------------------------------------------------------------------------
			self.sxconst_set[idx].widget.register_const_set()

			# --------------------------------------------------------------------------------
			# Load the previously saved parameter setting of all sx commands
			# Override the registration of project constant parameter settings with the previously-saved one
			# --------------------------------------------------------------------------------
			for sxcmd_category in self.sxcmd_category_list[idx]:
				sxcmd_category.widget.load_previous_session()

			# --------------------------------------------------------------------------------
			# Start widget
			# --------------------------------------------------------------------------------
			start_widget = QtGui.QWidget()
			logo_container = QtGui.QWidget()
			layout_start_widget = QtGui.QHBoxLayout()
			layout_logo_container = QtGui.QVBoxLayout()
			logo_container.setStyleSheet('border-image: url("{0}sxgui_pictograph_info.png")'.format(get_image_directory()))
			logo_container.setFixedSize(100, 100)
			layout_start_widget.setContentsMargins(0, 0, 0, 20)

			layout_logo_container.addStretch(1)
			layout_logo_container.addWidget(logo_container)
			layout_start_widget.addLayout(layout_logo_container)
			layout_start_widget.addStretch(1)
			start_widget.setLayout(layout_start_widget)
			self.sxmenu_item_widget_stacked_layout[idx].addWidget(start_widget)

			# --------------------------------------------------------------------------------
			# Display application information upon startup
			# --------------------------------------------------------------------------------
			self.sxmenu_item_widget_stacked_layout[idx].setCurrentWidget(start_widget)
			self.sxmenu_item_widget_stacked_layout_global.addWidget(central_widget)
		self.sxmenu_item_widget_stacked_layout_global.setCurrentIndex(self.helical)

		# --------------------------------------------------------------------------------
		# Get focus to main window
		# --------------------------------------------------------------------------------
		self.setFocus()

	def construct_sxinfo(self):
###		sxinfo = SXmenu_item(); sxinfo.name = "GUI Information"; sxinfo.label = "GUI Appliation Information"; sxinfo.short_info = "DUMMY STRING"
		sxinfo = SXmenu_item(); sxinfo.name = "sxc_gui_info"; sxinfo.label = "GUI Appliation Information"; sxinfo.short_info = "DUMMY STRING"

		# Store GUI application information as a class data member
		return sxinfo

	def construct_sxconst_set(self, helical):
		sxconst_set = SXconst_set(); sxconst_set.name = "sxc_project"; sxconst_set.label = "Project Settings"; sxconst_set.short_info = "Set constant parameter values for this project. These constants will be used as default values of associated arguments and options in command settings. However, the project settings here are not required to run commands."
		sxconst = SXconst(); sxconst.key = "protein"; sxconst.label = "Protein name"; sxconst.help = "a valid string for file names on your OS."; sxconst.register = "MY_PROTEIN"; sxconst.type = "string"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "apix"; sxconst.label = "Micrograph pixel size [A]"; sxconst.help = ""; sxconst.register = "1.0"; sxconst.type = "float"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "ctfwin"; sxconst.label = "CTF window size [pixels]"; sxconst.help = "it should be slightly larger than particle box size"; sxconst.register = "512"; sxconst.type = "int"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "box"; sxconst.label = "Particle box size [pixels]" ; sxconst.help = ""; sxconst.register = "-1"; sxconst.type = "int"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "radius"; sxconst.label = "Protein particle radius [pixels]"; sxconst.help = ""; sxconst.register = "-1"; sxconst.type = "int"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "sym"; sxconst.label = "Point-group symmetry"; sxconst.help = "e.g. c1, c4, d5"; sxconst.register = "c1"; sxconst.type = "string"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "mass"; sxconst.label = "Protein molecular mass [kDa]"; sxconst.help = ""; sxconst.register = "-1.0"; sxconst.type = "float"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		if helical:
			sxconst = SXconst(); sxconst.key = "filament_width"; sxconst.label = "Filament width [pixels]"; sxconst.help = "Width of the filament in case of filamentous processing."; sxconst.register = "-1"; sxconst.type = "int"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst
		sxconst = SXconst(); sxconst.key = "config"; sxconst.label = "Imaging configurations"; sxconst.help = "a free-style string for your record. please use it to describe the set of imaging configurations used in this project (e.g. types of microscope, detector, enegy filter, abbration corrector, phase plate, and etc."; sxconst.register = "MY_MICROSCOPE"; sxconst.type = "int"; sxconst_set.list.append(sxconst); sxconst_set.dict[sxconst.key] = sxconst

		# Store the project constant parameter set as a class data member
		return sxconst_set

	def construct_sxcmd_category_list(self):
		sxcmd_category_list = []
		sxcmd_list = []           # Used only within this function
		sxcmd_category_dict = {}  # Used only within this function

		# Actual configurations of all sx command categories and sx commands are inserted into the following section by wikiparser.py
		# as sxcmd_category_list and sxcmd_list
		# @@@@@ START_INSERTION @@@@@
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_cter"; sxcmd_category.label = "CTF"; sxcmd_category.short_info = "CTF Estimation and CTF Assessment"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_window"; sxcmd_category.label = "Particle Stack"; sxcmd_category.short_info = "Particle Coordinates and Particle Extraction"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_isac"; sxcmd_category.label = "2D Clustering"; sxcmd_category.short_info = "ISAC2 2D Clustering and Beautifier"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_viper"; sxcmd_category.label = "Initial 3D Modeling"; sxcmd_category.short_info = "Initial 3D modeling with VIPER/RVIPER"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_meridien"; sxcmd_category.label = "3D Refinement"; sxcmd_category.short_info = "MERIDIEN 3d Refinement and PostRefiner"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_sort3d"; sxcmd_category.label = "3D Clustering"; sxcmd_category.short_info = "3D Variability and SROT3D_DEPTH 3D Clustering"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_subtract"; sxcmd_category.label = "Signal Subtraction"; sxcmd_category.short_info = "Signal Subtraction"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_localres"; sxcmd_category.label = "Local Resolution"; sxcmd_category.short_info = "Local Resolution, and Local Filtering"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_movie"; sxcmd_category.label = "Movie Micrograph"; sxcmd_category.short_info = "Micrograph Movie Alignemnt and Drift Assessment"
		sxcmd_category_list.append(sxcmd_category)
		sxcmd_category = SXcmd_category(); sxcmd_category.name = "sxc_utilities"; sxcmd_category.label = "Utilities"; sxcmd_category.short_info = "Miscellaneous Utlitity Commands"
		sxcmd_category_list.append(sxcmd_category)

		sxcmd = SXcmd(); sxcmd.name = "sp_cter"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "CTF Estimation"; sxcmd.short_info = "Automated estimation of CTF parameters with error assessment, including Volta phase shift."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_image_path"; token.key_prefix = ""; token.label = "Input micrograph path pattern"; token.help = "Specify input micrographs path pattern with a wild card (*) for any of Micrograph Modes. Images in bdb format cannot be used as input micrographs. In an advanced option, a particle stack file path can also be given using --stack_mode. However, Stack Mode is not supported by sp_gui. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The CTF parameters (partres file), rotationally averaged power spectra (rotinf), and micrograph thumbnails (thumb files) will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = "--"; token.label = "Micrograph selection file"; token.help = "Specify path of a micrograph selection list text file for Selected Micrographs Mode. The file extension must be '.txt'. Alternatively, the file name of a single micrograph can be specified for Single Micrograph Mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_mic_one_ext"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "The pixel size of input micrograph(s) or images in input particle stack. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "Cs"; token.key_prefix = "--"; token.label = "Microscope spherical aberration (Cs) [mm]"; token.help = "The spherical aberration (Cs) of microscope used for imaging. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2.0"; token.restore = [['2.0'], ['2.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "voltage"; token.key_prefix = "--"; token.label = "Microscope voltage [kV]"; token.help = "The acceleration voltage of microscope used for imaging. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "300.0"; token.restore = [['300.0'], ['300.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ac"; token.key_prefix = "--"; token.label = "Amplitude contrast [%]"; token.help = "The amplitude contrast is in the range of 7% - 14%. The value  depends on the thickness of the ice embedding the particles, among other factors. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "10.0"; token.restore = [['10.0'], ['10.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "f_start"; token.key_prefix = "--"; token.label = "Lowest resolution [A]"; token.help = "Lowest resolution used in the CTF estimation. Determined automatically if not given. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "f_stop"; token.key_prefix = "--"; token.label = "Highest resolution [A]"; token.help = "Highest resolution used in the CTF estimation. Determined automatically if not given. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "wn"; token.key_prefix = "--"; token.label = "CTF window size [Pixels]"; token.help = "The window size should be slightly larger than particle box size. This will be ignored in Stack Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "512"; token.restore = [['512'], ['512']]; token.type = "ctfwin"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "kboot"; token.key_prefix = "--"; token.label = "Number of CTF estimates per micrograph"; token.help = "Used for error assessment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "16"; token.restore = [['16'], ['16']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overlap_x"; token.key_prefix = "--"; token.label = "X overlap [%]"; token.help = "Overlap between micrograph windows in the x direction. This will be ignored in Stack Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "50"; token.restore = [['50'], ['50']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overlap_y"; token.key_prefix = "--"; token.label = "Y overlap [%]"; token.help = "Overlap between micrograph windows in the y direction. This will be ignored in Stack Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "50"; token.restore = [['50'], ['50']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_x"; token.key_prefix = "--"; token.label = "Edge x [pixels]"; token.help = "Specifies micrograph exclusion margin in the x direction. This will be ignored in Stack Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_y"; token.key_prefix = "--"; token.label = "Edge y [pixels]"; token.help = "Specifies micrograph exclusion margin in the y direction. This will be ignored in Stack Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "check_consistency"; token.key_prefix = "--"; token.label = "Check consistency of inputs"; token.help = "Create a text file containing the list of inconsistent Micrograph ID entries (i.e. inconsist_mic_list_file.txt). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "debug_mode"; token.key_prefix = "--"; token.label = "Enable debug mode"; token.help = "Print out debug information. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "vpp"; token.key_prefix = "--"; token.label = "Volta Phase Plate Dataset"; token.help = "UNDER DEVELOPMENT! Estimate phase shift. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "defocus_min"; token.key_prefix = "--"; token.label = "Minimum defocus search [um]"; token.help = "UNDER DEVELOPMENT! This is applicable only with --vpp option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['vpp', 'True', 'False']]; token.default = "0.3"; token.restore = [['0.3'], ['0.3']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('vpp', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "defocus_max"; token.key_prefix = "--"; token.label = "Maximum defocus search [um]"; token.help = "UNDER DEVELOPMENT! This is applicable only with --vpp option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['vpp', 'True', 'False']]; token.default = "9.0"; token.restore = [['9.0'], ['9.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('vpp', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "defocus_step"; token.key_prefix = "--"; token.label = "Defocus search step [um]"; token.help = "UNDER DEVELOPMENT! This is applicable only with --vpp option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['vpp', 'True', 'False']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('vpp', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "phase_min"; token.key_prefix = "--"; token.label = "Minimum phase search [degrees]"; token.help = "UNDER DEVELOPMENT! This is applicable only with --vpp option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['vpp', 'True', 'False']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('vpp', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "phase_max"; token.key_prefix = "--"; token.label = "Maximum phase search [degrees]"; token.help = "UNDER DEVELOPMENT! This is applicable only with --vpp option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['vpp', 'True', 'False']]; token.default = "175.0"; token.restore = [['175.0'], ['175.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('vpp', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "phase_step"; token.key_prefix = "--"; token.label = "Phase search step [degrees]"; token.help = "UNDER DEVELOPMENT! This is applicable only with --vpp option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['vpp', 'True', 'False']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('vpp', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "pap"; token.key_prefix = "--"; token.label = "Use PW spectrum"; token.help = "UNDER DEVELOPMENT! Use power spectrum for CTF parameter search instead of amplitude. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_gui_cter"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "CTF Assessment"; sxcmd.short_info = "GUI tool to assess and sort micrographs according to their CTF parameters estimated by sp_cter."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "cter_ctf_file"; token.key_prefix = ""; token.label = "File containing CTF parameters"; token.help = "This file is produced by sp_cter and normally called partres.txt. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "params_cter_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ctffind"; token.key_prefix = "--"; token.label = "ctffind"; token.help = "Use CTFFIND outputs (e.g., PWROT_DIR/*_avrot.txt, POWER2D_DIR/*.mrc). If using this option, you may need to specify the advanced parameters pwrot_dir and power2d_dir "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pwrot_dir"; token.key_prefix = "--"; token.label = "1D profile directory"; token.help = "Directory for 1D profiles "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "pwrot"; token.restore = [['pwrot'], ['pwrot']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "power2d_dir"; token.key_prefix = "--"; token.label = "2D power-spectrum directory"; token.help = "Directory for 2D power spectra "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "power2d"; token.restore = [['power2d'], ['power2d']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "micthumb_dir"; token.key_prefix = "--"; token.label = "2D power-spectrum directory"; token.help = "Directory for 2D power spectra "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "micthumb"; token.restore = [['micthumb'], ['micthumb']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "resample_micrographs"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Resample Micrographs"; sxcmd.short_info = "Resample micrographs in input directory specified by input micrograph pattern with user-specified ratio. This operation changes the image dimensitions and the pixel size. Mainly, it is designed to reduce the demensions and pixel size of micrographs taken with the super resolution mode of the K2 direct electron detector."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_micrograph_pattern"; token.key_prefix = ""; token.label = "Input micrograph path pattern"; token.help = "Specify path pattern of input micrographs with a wild card (*).. The path pattern must be enclosed by single quotes (\') or double quotes (\'). (Note: sp_gui.py automatically adds single quotes (\')). The substring at the variable part must be same between the associated pair of input micrograph and coordinates file. bdb files cannot be selected as input micrographs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Resampling ratio"; token.help = "Specify ratio between new and original pixel size. Use a value between 0.0 and 1.0 (exclusive both ends). "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = "--"; token.label = "Micrograph selecting list"; token.help = "Specify a name of micrograph selection list text file for Selected Micrographs Mode. The file extension must be .txt. Alternatively, the file name of a single micrograph can be specified for Single Micrograph Mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_mic_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "check_consistency"; token.key_prefix = "--"; token.label = "Check consistency of dataset"; token.help = "Create a text file containing the list of Micrograph ID entries might have inconsistency among the provided dataset. (i.e. mic_consistency_check_info_TIMESTAMP.txt). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "organize_micrographs"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Organize Micrographs/Movies"; sxcmd.short_info = "Organize micrographs/movies by moving micrographs/movies listed in a selecting file from a source directory (specified by source micrographs/movies pattern) to a destination directory."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "source_micrograph_pattern"; token.key_prefix = ""; token.label = "Source micrograph/movies path pattern"; token.help = "Specify path pattern of source micrographs/movies with a wild card (*). The path pattern must be enclosed by single quotes (\') or double quotes (\'). (Note: sp_gui.py automatically adds single quotes (\')). The substring at the variable part must be same between each associated pair of micrograph/movie names. bdb files cannot be selected as source micrographs/movies. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_both"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = ""; token.label = "Micrograph/Movie selection file"; token.help = "Specify a path of text file containing a list of selected micrograph/movie names or paths. The file extension must be '.txt'. The directory path of each entry will be ignored if there are any. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "select_mic_both"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "destination_directory"; token.key_prefix = ""; token.label = "Destination directory"; token.help = "The micrographs/movies in selecting list will be moved to this directory. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "reverse"; token.key_prefix = "--"; token.label = "Reverse operation"; token.help = "Move back micrographs/movies from the destination directory to the source directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "check_consistency"; token.key_prefix = "--"; token.label = "Check consistency of dataset"; token.help = "Create a text file containing the list of micrograph/movie ID entries might have inconsistency among the provided dataset. (i.e. mic_consistency_check_info.txt). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_batch"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Batch Pipeline Execution"; sxcmd.short_info = "Run jobs that wait with the execution on each other."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "submission_command"; token.key_prefix = ""; token.label = "Submission command"; token.help = "Submission command, e.g., qsub, qsub -V, sbatch, bash "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "Pipeline directory"; token.help = "Directory containin the pipeline submission files "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hold_flag"; token.key_prefix = "--"; token.label = "Hold flag"; token.help = "Hold flag for the submission command, e.g. -hold_jid=, --wait=, --dependency=afterany:; Default is None and should be used in combination with a local execution with bash "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "first_hold_number"; token.key_prefix = "--"; token.label = "First hold number"; token.help = "Wait number for the first job that is submitted. By default, the first job will not wait for others "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_ctf_refine"; sxcmd.subname = "meridien"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "CTF refine (Meridien)"; sxcmd.short_info = "Refine the defocus per particle"; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "path_to_input_stack"; token.key_prefix = ""; token.label = "Input stack path"; token.help = "Path to input stack. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Folder to output files etc. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "refinement_dir"; token.key_prefix = ""; token.label = "Meridien directory"; token.help = "Path to Meridien directory.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "Path to mask"; token.help = "Path to mask used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "range"; token.key_prefix = "--"; token.label = "Defocus search range [um]"; token.help = "Search range +- around the current defocus value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.15"; token.restore = [['0.15'], ['0.15']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Defocus search delta [um]"; token.help = "Finest search delta. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0025"; token.restore = [['0.0025'], ['0.0025']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resolution"; token.key_prefix = "--"; token.label = "Nominal resolution [A]"; token.help = "Nominal resolution of the reconstruction. Low-pass Gaussian filter will be applied at this resolution. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size in Angstroms. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['resolution', 'None', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('resolution', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "number_part"; token.key_prefix = "--"; token.label = "Number of particles"; token.help = "Number of particles to process. Mainly for debugging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_ctf_refine"; sxcmd.subname = "manual"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "CTF refine (Stack)"; sxcmd.short_info = "Refine the defocus per particle"; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "path_to_input_stack"; token.key_prefix = ""; token.label = "Input stack path"; token.help = "Path to input stack. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Folder to output files etc. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "volume_path"; token.key_prefix = ""; token.label = "Path to volume"; token.help = "Path to volume.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "params_file"; token.key_prefix = ""; token.label = "Params file"; token.help = "Path to params file "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "Path to mask"; token.help = "Path to mask used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "range"; token.key_prefix = "--"; token.label = "Defocus search range [um]"; token.help = "Search range +- around the current defocus value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.15"; token.restore = [['0.15'], ['0.15']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Defocus search delta [um]"; token.help = "Finest search delta. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0025"; token.restore = [['0.0025'], ['0.0025']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resolution"; token.key_prefix = "--"; token.label = "Nominal resolution [A]"; token.help = "Nominal resolution of the reconstruction. Low-pass Gaussian filter will be applied at this resolution. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size in Angstroms. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['resolution', 'None', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('resolution', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "number_part"; token.key_prefix = "--"; token.label = "Number of particles"; token.help = "Number of particles to process. Mainly for debugging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chunk"; token.key_prefix = "--"; token.label = "Path to chunk file"; token.help = "Path to chunk file "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "volume2"; token.key_prefix = "--"; token.label = "Path to second volume"; token.help = "Path to second half volume. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['chunk', 'None', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('chunk', []).append([token.key_base, 'None', 'True'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "transphire"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "TranSPHIRE GUI"; sxcmd.short_info = "Automated data pre-processing. TranSPHIRE needs to be installed seperately and the transphire executable needs to be present in the PATH: https:github.com/MPI-Dortmund/transphire"; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_cter"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_cryolo_predict"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "crYOLO - predict"; sxcmd.short_info = "Prediction with crYOLO, a deep learning high accuracy particle picking procedure."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "cryolo_predict_path"; token.key_prefix = "--"; token.label = "crYOLO predict executable"; token.help = "Path to the cryolo_predict.py in your crYOLO environment "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "py"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "config_path"; token.key_prefix = ""; token.label = "Config file"; token.help = "Path of the crYOLO config file.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_json"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_dir"; token.key_prefix = ""; token.label = "Image directory"; token.help = "Folder which contain all images. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "model_path"; token.key_prefix = ""; token.label = "Model path"; token.help = "Path to the trained model. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_h5"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Folder to write the box files. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "confidence_threshold"; token.key_prefix = "--"; token.label = "Confidence threshold [0-1]"; token.help = "Confidence threshold for picking. Particles with a confidence threshold lower than this value will be discarded. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.3"; token.restore = [['0.3'], ['0.3']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gpu"; token.key_prefix = "--"; token.label = "GPUs"; token.help = "List of GPUs to use, separated by commas if more than one. If no GPU is present, be sure to supply the non-GPU executable. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gpu_fraction"; token.key_prefix = "--"; token.label = "GPU memory fraction"; token.help = "Specify the fraction of memory per GPU used by crYOLO during prediction. Only values between 0.0 and 1.0 are allowed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "num_cpu"; token.key_prefix = "--"; token.label = "Number of CPUs"; token.help = "Number of CPUs used during prediction. By default it will use half of the available CPUs. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "otf"; token.key_prefix = "--"; token.label = "On the fly filtering"; token.help = "If checked, the images are filtered on the fly and not saved to disk. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "min_distance"; token.key_prefix = "--"; token.label = "Minimum distance"; token.help = "Particles with a distance less than this value (in pixel) will be removed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_mode"; token.key_prefix = "--"; token.label = "Filament mode [Yes/No]"; token.help = "Check if you want to use the filament mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [Pixel]"; token.help = "Width of your filament in pixel. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['filament_mode', 'True', 'False']]; token.default = "100"; token.restore = [['100'], ['100']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_mode', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "min_box_per_filament"; token.key_prefix = "--"; token.label = "Minimum number of boxes per filament"; token.help = "Specifies the minimum number of boxes per filament. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['filament_mode', 'True', 'False']]; token.default = "6"; token.restore = [['6'], ['6']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_mode', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "box_distance"; token.key_prefix = "--"; token.label = "Box distance"; token.help = "Distance between two filament boxes in pixels. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['filament_mode', 'True', 'False']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_mode', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "nosplit"; token.key_prefix = "--"; token.label = "Don't split curved filaments"; token.help = "If checked, the filament mode does not split curved filaments. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['filament_mode', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_mode', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "nomerging"; token.key_prefix = "--"; token.label = "Don't merge filaments"; token.help = "If checked, the filament mode does not merge filaments. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['filament_mode', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_mode', []).append([token.key_base, 'True', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_window"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Particle Extraction"; sxcmd.short_info = "Window particles from micrographs using the particle coordinates."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_micrograph_pattern"; token.key_prefix = ""; token.label = "Input micrograph path pattern"; token.help = "Specify path pattern of input micrographs with a wild card (*).. The path pattern must be enclosed by single quotes (') or double quotes ('). (Note: sp_gui.py automatically adds single quotes (')). The substring at the variable part must be same between the associated pair of input micrograph and coordinates file. bdb files cannot be selected as input micrographs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_coordinates_pattern"; token.key_prefix = ""; token.label = "Input coordinates path pattern"; token.help = "Specify path pattern of input coordinates files with a wild card (*).. The path pattern must be enclosed by single quotes (') or double quotes ('). (Note: sp_gui.py automatically adds single quotes (')). The substring at the variable part must be same between the associated pair of input micrograph and coordinates file. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_coords_any"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_ctf_params_source"; token.key_prefix = ""; token.label = "CTF parameters source"; token.help = "Specify the file produced by sp_cter and normally called partres.txt for cryo data. For negative staining data, enter pixel size [A/Pixels]. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_cter_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = "--"; token.label = "Micrograph selection file"; token.help = "Specify a name of micrograph selection list text file for Selected Micrographs Mode. The file extension must be '.txt'. Alternatively, the file name of a single micrograph can be specified for Single Micrograph Mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_mic_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "coordinates_format"; token.key_prefix = "--"; token.label = "Coordinate file format"; token.help = "Allowed values are 'sphire', 'eman1', 'eman2', 'cryolo,'cryolo_helical_segmented' or 'spider'. The sphire, eman2, and spider formats use the particle center as coordinates. The eman1 format uses the lower left corner of the box as coordinates. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cryolo"; token.restore = [['cryolo', 'eman1', 'eman2', 'cryolo_helical_segmented'], ['cryolo', 'eman1', 'eman2', 'cryolo_helical_segmented']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Particle box size [Pixels]"; token.help = "The x and y dimensions of square area to be windowed. The box size after resampling is assumed when resample_ratio &lt; 1.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "256"; token.restore = [['256'], ['256']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_invert"; token.key_prefix = "--"; token.label = "Invert image contrast"; token.help = "Indicate if image contrast should be inverted or not. Do not invert for negative staining data. By default, the image contrast will be inverted for cryo data. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "limit_ctf"; token.key_prefix = "--"; token.label = "Use CTF limit filter"; token.help = "Frequencies where CTF oscillations cannot be properly modeled with the resampled pixel size will be discarded in the images with the appropriate low-pass filter. This flag has no effect when the CTER CTF File is not specified by the CTF paramters source argument. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "astigmatism_error"; token.key_prefix = "--"; token.label = "Astigmatism error limit [Degrees]"; token.help = "Set astigmatism to zero for all micrographs where the angular error computed by sp_cter is larger than the desired value. This parameter has no effect when the CTER CTF File is not specified by the CTF paramters source argument. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "360.0"; token.restore = [['360.0'], ['360.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Image size reduction factor (<1)"; token.help = "Use a value between 0.0 and 1.0 (excluding 0.0). The new pixel size will be automatically recalculated and stored in CTF paramers when resample_ratio &lt; 1.0 is used. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "check_consistency"; token.key_prefix = "--"; token.label = "Check consistency of inputs"; token.help = "Create a text file containing the list of inconsistent Micrograph ID entries (i.e. inconsist_mic_list_file.txt). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [Pixels]"; token.help = "Filament width for the creation of the rectangular mask. If -1 is selected, no mask will be applied. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2bdb"; sxcmd.subname = ""; sxcmd.mode = "makevstack"; sxcmd.subset_config = "fullset"; sxcmd.label = "Particle Stack"; sxcmd.short_info = "Make a 'virtual' bdb image stack with the specified name from one or more other stacks. "; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "makevstack"; token.key_prefix = "--"; token.label = "Output virtual image stack"; token.help = "Make a 'virtual' bdb image stack with the specified name from one or more other stacks. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_bdb_stack_pattern"; token.key_prefix = ""; token.label = "Input BDB image stack pattern"; token.help = "Specify file path pattern of stack subsets created in particle extraction using a wild card /'*/' (e.g. /'//sp_window_output_dir//*/'). The stack subsets are located in the sp_window output directory."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_cryolo_train"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "crYOLO - training"; sxcmd.short_info = "Training of crYOLO, a deep learning high accuracy particle picking procedure."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "cryolo_train_path"; token.key_prefix = "--"; token.label = "crYOLO train executeable"; token.help = "Path to the cryolo_train.py in your crYOLO environment "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "py"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "particle_diameter"; token.key_prefix = ""; token.label = "Particle diameter [Pixel]"; token.help = "Particle diameter in pixels. This size will be used for as box size for picking. Should be as small as possible. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "training_dir"; token.key_prefix = ""; token.label = "Training image directory"; token.help = "Folder which contains all images. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "annot_dir"; token.key_prefix = ""; token.label = "Annotation directory"; token.help = "Box or star files used for training. Files should have the same name as the images, with the appropriate extension. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "architecture"; token.key_prefix = "--"; token.label = "Network architecture"; token.help = "Type of network that is trained.  "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "PhosaurusNet"; token.restore = [['PhosaurusNet'], ['PhosaurusNet']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_size"; token.key_prefix = "--"; token.label = "Input image dimension [Pixel]"; token.help = "Dimension of the image used as input by the network. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1024"; token.restore = [['1024'], ['1024']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "num_patches"; token.key_prefix = "--"; token.label = "Number of patches"; token.help = "The number of patches (e.g 2x2) the image is divided into and classified separately. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overlap_patches"; token.key_prefix = "--"; token.label = "Patch overlap [Pixel]"; token.help = "The amount of overlap the patches will overlap "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "train_times"; token.key_prefix = "--"; token.label = "Repeat images"; token.help = "How often a images is augmented and repeated in one epoch. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "10"; token.restore = [['10'], ['10']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pretrained_weights_name"; token.key_prefix = "--"; token.label = "Pretrained weights name"; token.help = "Name of the pretrained model "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cryolo_model.h5"; token.restore = [['cryolo_model.h5'], ['cryolo_model.h5']]; token.type = "params_any_h5"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "saved_weights_name"; token.key_prefix = "--"; token.label = "Saved weights name"; token.help = "Name of the model to save "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cryolo_model.h5"; token.restore = [['cryolo_model.h5'], ['cryolo_model.h5']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "batch_size"; token.key_prefix = "--"; token.label = "Batch size"; token.help = "How many patches are processed in parallel. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fine_tune"; token.key_prefix = "--"; token.label = "Fine tune mode"; token.help = "Set it to true if you only want to use the fine tune mode. Don't forget to choose an appropriate pretrained model (like the general model) that is refined. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "warmup"; token.key_prefix = "--"; token.label = "Warm up epochs"; token.help = "Number of warmup epochs. If you fine tune a model, set it to zero. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "learning_rate"; token.key_prefix = "--"; token.label = "Learning rate"; token.help = "Learning rate used during training. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0001"; token.restore = [['0.0001'], ['0.0001']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "np_epoch"; token.key_prefix = "--"; token.label = "Number of epochs"; token.help = "Maximum number of epochs. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "100"; token.restore = [['100'], ['100']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "object_scale"; token.key_prefix = "--"; token.label = "Object loss scale"; token.help = "Loss scale for object. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "no_object_scale"; token.key_prefix = "--"; token.label = "Background loss scale"; token.help = "Loss scale for background. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "coord_scale"; token.key_prefix = "--"; token.label = "Coordinates loss scale"; token.help = "Loss scale for coordinates. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "valid_image_dir"; token.key_prefix = "--"; token.label = "Path to validation images"; token.help = "Images used "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "valid_annot_dir"; token.key_prefix = "--"; token.label = "Path to validation annotations"; token.help = "Path to the validation box files "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skiplowpass"; token.key_prefix = "--"; token.label = "Skip low pass filtering"; token.help = "Set it to true if you want to skip the low pass filter "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "cutoff"; token.key_prefix = "--"; token.label = "Low pass cutoff"; token.help = "Cut off for low pass filter. Should be between 0 and 0.5. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['skiplowpass', 'False', 'False']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('skiplowpass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "filtered_dir"; token.key_prefix = "--"; token.label = "Filtering directory"; token.help = "Path to write filtered images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cryolo_filtered_micrographs"; token.restore = [['cryolo_filtered_micrographs'], ['cryolo_filtered_micrographs']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "usejanni"; token.key_prefix = "--"; token.label = "Use JANNI for denoising"; token.help = "Set it to true if you want to use JANNI for denoising. Low pass filtering has to be skipped. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['skiplowpass', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('skiplowpass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "janni_model"; token.key_prefix = "--"; token.label = "JANNI model"; token.help = "Path to JANNI model "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['skiplowpass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "params_any_h5"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('skiplowpass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "janni_overlap"; token.key_prefix = "--"; token.label = "JANNI patch overlap"; token.help = "Overlap of patches in pixel "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['skiplowpass', 'True', 'False']]; token.default = "24"; token.restore = [['24'], ['24']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('skiplowpass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "janni_batches"; token.key_prefix = "--"; token.label = "JANNI number batches"; token.help = "Number of batches when using JANNI "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['skiplowpass', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('skiplowpass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "gpu"; token.key_prefix = "--"; token.label = "GPUs"; token.help = "List of GPUs to use, separated by commas. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gpu_fraction"; token.key_prefix = "--"; token.label = "GPU memory fraction"; token.help = "Specify the fraction of memory per GPU used by crYOLO during training. Only values between 0.0 and 1.0 are allowed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "num_cpu"; token.key_prefix = "--"; token.label = "Number of CPUs"; token.help = "Number of CPUs used during training. By default it will use half of the available CPUs. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "restacking"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Restacking"; sxcmd.short_info = "Generate all necessary information to restack the input stack (i.e., particle image ID list, CTF parameters list, projection parameters list) while applying micrograph selection list. Optionally, the command can directly output the virtual stack.  In addition, this command can be used to generate all parameters files for reboxing (i.e. original/centered particle coordinates list files, CTF parameters list, original/centered projection parameters list as well as micrograph selection file). Optionally, user can provided a 3D shift to recenter the projection parameters and so the particle coordinates."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_bdb_stack_path"; token.key_prefix = ""; token.label = "Input bdb image stack"; token.help = "Specify the input bdb image stack. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = "--"; token.label = "Micrograph/Movie selection file"; token.help = "Specify path to text file containing a list of selected micrograph/movie names or paths. The particles associated with the micrographs/movies in this list will be processed. The file extension must be .txt. The directory path of each entry will be ignored if there is any. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_mic_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_x"; token.key_prefix = "--"; token.label = "3D x-shift [Pixels]"; token.help = "3D x-shift value. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_y"; token.key_prefix = "--"; token.label = "3D y-shift [Pixels]"; token.help = "3D y-shift value. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_z"; token.key_prefix = "--"; token.label = "3D z-shift [Pixels]"; token.help = "3D z-shift value. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "save_vstack"; token.key_prefix = "--"; token.label = "Save virtual stack"; token.help = "Use this option to save the virtual stack. By default, the virtual stack will not be generated. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sv_vstack_basename"; token.key_prefix = "--"; token.label = "Virtual stack basename"; token.help = "Specify the basename of output virtual stack file. It cannot be empty string or only white spaces. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['save_vstack', 'True', 'False']]; token.default = "vstack"; token.restore = [['vstack'], ['vstack']]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('save_vstack', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "reboxing"; token.key_prefix = "--"; token.label = "Generate reboxing information"; token.help = "Prepare reboxing by extracting coordinates from the input stack headers, then center them according to projection parameters in the header and user-provided 3D shift. If the headers do not contain projection parameters, the program assumes the projection parameters are all zeros (null alignment). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rb_box_size"; token.key_prefix = "--"; token.label = "Particle box size [Pixels]"; token.help = "For --reboxing option, specify the x and y dimensions of square area to be windowed. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['reboxing', 'True', 'False']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('reboxing', []).append([token.key_base, 'True', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_rewindow"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Particle Reextraction"; sxcmd.short_info = "Rewindow particles from micrographs using the information stored in rebox files."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_micrograph_pattern"; token.key_prefix = ""; token.label = "Input micrograph path pattern"; token.help = "Specify path pattern of input micrographs with a wild card (*).. The path pattern must be enclosed by single quotes (') or double quotes ('). (Note: sp_gui.py automatically adds single quotes (')). The substring at the variable part must be same between the associated pair of input micrograph and rebox file. bdb files cannot be selected as input micrographs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_rebox_pattern"; token.key_prefix = ""; token.label = "Input rebox path pattern"; token.help = "Specify path pattern of input rebox files with a wild card (*).. The path pattern must be enclosed by single quotes (') or double quotes ('). (Note: sp_gui.py automatically adds single quotes (')). The substring at the variable part must be same between the associated input micrograph. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_rebox_rbx"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = "--"; token.label = "Micrograph selection file"; token.help = "Specify a name of micrograph selection list text file for Selected Micrographs Mode. The file extension must be '.txt'. Alternatively, the file name of a single micrograph can be specified for Single Micrograph Mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_mic_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Particle box size [Pixels]"; token.help = "The x and y dimensions of square area to be windowed. The box size after resampling is assumed when mic_resample_ratio &lt; 1.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "256"; token.restore = [['256'], ['256']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_invert"; token.key_prefix = "--"; token.label = "Invert image contrast"; token.help = "Indicate if image contrast should be inverted or not. Do not invert for negative staining data. By default, the image contrast will be inverted for cryo data. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mic_resample_ratio"; token.key_prefix = "--"; token.label = "Image size reduction factor (<1)"; token.help = "Use a value between 0.0 and 1.0 (excluding 0.0). The new pixel size will be automatically recalculated and stored in CTF paramers when mic_resample_ratio &lt; 1.0 is used. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "swap_ctf_params"; token.key_prefix = "--"; token.label = "Swap CTF parameters"; token.help = "Swaps CTF parameters by setting the CTF parameters in the specified CTER partres file while ignoring the CTF parameters in the input rebox parameters file. Typically, specify the file produced by sp_cter and normally called partres.txt. Alternatively, enter pixel size [A/Pixels] to simulate ideal CTF. By default, the program uses the CTF parameters in the input rebox parameters file. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "params_cter_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "check_consistency"; token.key_prefix = "--"; token.label = "Check consistency of dataset"; token.help = "Create a text file containing the list of Micrograph ID entries might have inconsistency among the provided dataset. (i.e. mic_consistency_check_info_TIMESTAMP.txt). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2boxer_old"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Particle Coordinates"; sxcmd.short_info = "Generate files containing particle coordinates for all input micrographs by picking particles manual and/or automatically."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_micrograph_list"; token.key_prefix = ""; token.label = "Input micrographs"; token.help = "Wild cards (e.g. *) can be used to specify a list of micrographs. Not recommended if their number is very large. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_one_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "boxsize"; token.key_prefix = "--"; token.label = "Box size [Pixels]"; token.help = "Box size for extraction of particle images. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "invert"; token.key_prefix = "--"; token.label = "Invert contrast"; token.help = "Invert contrast of micrographs. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Verbose level. Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2boxer"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Particle Coordinates (NEW)"; sxcmd.short_info = "Generate files containing particle coordinates for all input micrographs by picking particles manual and/or automatically."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_micrograph_list"; token.key_prefix = ""; token.label = "Input micrograph list"; token.help = "Wild cards (e.g. *) can be used to specify a list of micrographs. Not recommended if their number is very large. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_one_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Angstroms per pixel for all images. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "boxsize"; token.key_prefix = "--"; token.label = "Box size [Pixels]"; token.help = "Box size in pixels. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ptclsize"; token.key_prefix = "--"; token.label = "Particle diameter [Pixels]"; token.help = "Longest axis of particle in pixels (diameter, not radius). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "invert"; token.key_prefix = "--"; token.label = "Invert input contrast"; token.help = "Preferably, particles should be bright on a dark background. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "no_ctf"; token.key_prefix = "--"; token.label = "Disable CTF estimation"; token.help = "Disable CTF estimation. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gui"; token.key_prefix = "--"; token.label = "Interactive GUI mode"; token.help = "Use interactive GUI mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "write_dbbox"; token.key_prefix = "--"; token.label = "Export EMAN1 box files"; token.help = "Export EMAN1 box files (.box extension). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "unboxedonly"; token.key_prefix = "--"; token.label = "Include only unboxed micrographs"; token.help = "Only include image files without existing box locations. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "write_ptcls"; token.key_prefix = "--"; token.label = "Save selected particle"; token.help = "Extract selected particles from micrographs and write to disk. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "allmicrographs"; token.key_prefix = "--"; token.label = "Include all micrographs in a directory"; token.help = "Add all images from micrographs folder. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "suffix"; token.key_prefix = "--"; token.label = "Micrograph suffix"; token.help = "Suffix of the micrographs used for particle picking (i.e. suffix=goodali will use micrographs end with _goodali.hdf). It is only useful when --allmicrographs option is True. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['allmicrographs', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('allmicrographs', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "voltage"; token.key_prefix = "--"; token.label = "Microscope voltage [kV]"; token.help = "The acceleration voltage of microscope used for imaging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "cs"; token.key_prefix = "--"; token.label = "Microscope spherical aberration (Cs) [mm]"; token.help = "The spherical aberration (Cs) of microscope used for imaging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ac"; token.key_prefix = "--"; token.label = "Amplitude contrast [%]"; token.help = "The typical amplitude contrast is in the range of 7% - 14%. The value mainly depends on the thickness of the ice embedding the particles. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "10.0"; token.restore = [['10.0'], ['10.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "autopick"; token.key_prefix = "--"; token.label = "Perform automatic particle picking"; token.help = "Provide mode and parameter string (eg - auto_local:threshold=5.5). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "threads"; token.key_prefix = "--"; token.label = "Number of threads"; token.help = "Number of threads to run in parallel on a single computer. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "organize_micrographs"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Organize Micrographs/Movies"; sxcmd.short_info = "Organize micrographs/movies by moving micrographs/movies listed in a selecting file from a source directory (specified by source micrographs/movies pattern) to a destination directory."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "source_micrograph_pattern"; token.key_prefix = ""; token.label = "Source micrograph/movies path pattern"; token.help = "Specify path pattern of source micrographs/movies with a wild card (*). The path pattern must be enclosed by single quotes (\') or double quotes (\'). (Note: sp_gui.py automatically adds single quotes (\')). The substring at the variable part must be same between each associated pair of micrograph/movie names. bdb files cannot be selected as source micrographs/movies. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mic_both"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = ""; token.label = "Micrograph/Movie selection file"; token.help = "Specify a path of text file containing a list of selected micrograph/movie names or paths. The file extension must be '.txt'. The directory path of each entry will be ignored if there are any. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "select_mic_both"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "destination_directory"; token.key_prefix = ""; token.label = "Destination directory"; token.help = "The micrographs/movies in selecting list will be moved to this directory. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "reverse"; token.key_prefix = "--"; token.label = "Reverse operation"; token.help = "Move back micrographs/movies from the destination directory to the source directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "check_consistency"; token.key_prefix = "--"; token.label = "Check consistency of dataset"; token.help = "Create a text file containing the list of micrograph/movie ID entries might have inconsistency among the provided dataset. (i.e. mic_consistency_check_info.txt). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_cryolo_boxmanager"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "crYOLO - boxmanager"; sxcmd.short_info = "Displays boxfiles on images. Allows creation of new training data for crYOLO."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "cryolo_bm_path"; token.key_prefix = "--"; token.label = "crYOLO boxmanager executable"; token.help = "Path to crYOLO boxmanager executable "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "py"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_dir"; token.key_prefix = "--"; token.label = "Input image directory"; token.help = "Path to input images "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_dir"; token.key_prefix = "--"; token.label = "Annotation directory"; token.help = "Path to annotation data like .box or .star files "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_batch"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Batch Pipeline Execution"; sxcmd.short_info = "Run jobs that wait with the execution on each other."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_window"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "submission_command"; token.key_prefix = ""; token.label = "Submission command"; token.help = "Submission command, e.g., qsub, qsub -V, sbatch, bash "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "Pipeline directory"; token.help = "Directory containin the pipeline submission files "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hold_flag"; token.key_prefix = "--"; token.label = "Hold flag"; token.help = "Hold flag for the submission command, e.g. -hold_jid=, --wait=, --dependency=afterany:; Default is None and should be used in combination with a local execution with bash "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "first_hold_number"; token.key_prefix = "--"; token.label = "First hold number"; token.help = "Wait number for the first job that is submitted. By default, the first job will not wait for others "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_isac2"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "ISAC2 - 2D Clustering"; sxcmd.short_info = "Iterative Stable Alignment and Clustering (ISAC) of a 2D image stack."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack_file"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Images must to be square (nx=ny). The stack can be either in .bdb or in .hdf format. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "General ISAC output directory to store all results. If the directory already exists ISAC will only run in continuation mode (see advanced parameter restart). "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Radius of the particle in pixels. ISAC cannot offer a default here since the value will depend on the particle im question. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp"; token.key_prefix = "--"; token.label = "Images per class"; token.help = "Ideally the number of images per class. In practice this value determines the number of classes K = N / img_per_grp, where N is the total number of images in the input stack. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "200"; token.restore = [['200'], ['200']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "CTF"; token.key_prefix = "--"; token.label = "CTF phase flipping"; token.help = "Use for cryo datasets. If set to True the data will be phase-flipped using CTF information included in the image headers. Cannot be used together with the VPP option. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['VPP', 'False', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('VPP', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "VPP"; token.key_prefix = "--"; token.label = "Phase Plate data"; token.help = "Use this option if the dataset is taken with a phase plate. Cannot be used together with the CTF option. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['CTF', 'False', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('CTF', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Translation search range [Pixels]"; token.help = "The translational search range. Change with care; higher values will incur significantly higher processing costs. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "thld_err"; token.key_prefix = "--"; token.label = "Pixel error threshold [Pixels]"; token.help = "Used as a threshold value when checking cluster stability. The pixel error is defined as the root mean square of distances between corresponding pixels from set of found transformations and their average transformation; it depends linearly on square of radius (parameter ou). units - pixels. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.7"; token.restore = [['0.7'], ['0.7']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_radius"; token.key_prefix = "--"; token.label = "Target particle radius [Pixels]"; token.help = "Particle radius used by ISAC2 to process the data. All particle images will be re-scaled to match their particle radius with this radius. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "29"; token.restore = [['29'], ['29']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_nx"; token.key_prefix = "--"; token.label = "Target particle image size [Pixels]"; token.help = "Image size used by ISAC2 to process the data. particle images will first be resized according to target particle radius (see above) and then cropped or padded to achieve the target image size. When xr &gt; 0, the final image size for ISAC2 processing is target_nx + xr - 1  "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "76"; token.restore = [['76'], ['76']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ir"; token.key_prefix = "--"; token.label = "Inner ring [Pixels]"; token.help = "Radius of the inner-most ring when resampling images to polar coordinates. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rs"; token.key_prefix = "--"; token.label = "Ring step [Pixels]"; token.help = "Radius step size when resampling images to polar coordinates. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step [Pixels]"; token.help = "Translational search step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxit"; token.key_prefix = "--"; token.label = "Reference-free alignment iterations"; token.help = "The number of iterations for reference-free alignments. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "30"; token.restore = [['30'], ['30']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "center_method"; token.key_prefix = "--"; token.label = "Centering method"; token.help = "Method to center global 2D average during the initial prealignment of the data (0: no centering; -1: average shift method; please see function center_2D in sp_utilities.py for methods 1-7). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "dst"; token.key_prefix = "--"; token.label = "Discrete angle used for within-group alignment"; token.help = "Discrete angle used for within-group alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "90.0"; token.restore = [['90.0'], ['90.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "FL"; token.key_prefix = "--"; token.label = "Lowest filter frequency [1/Pixel]"; token.help = "Lowest frequency used for the tangent filter. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.2"; token.restore = [['0.2'], ['0.2']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "FH"; token.key_prefix = "--"; token.label = "Highest filter frequency [1/Pixel]"; token.help = "Highest frequency used for the tangent filter. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.45"; token.restore = [['0.45'], ['0.45']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "FF"; token.key_prefix = "--"; token.label = "Tangent filter fall-off"; token.help = "The fall-off of the tangent filter. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.2"; token.restore = [['0.2'], ['0.2']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "init_iter"; token.key_prefix = "--"; token.label = "Maximum generations"; token.help = "Maximum number of generation iterations performed for a given subset. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "7"; token.restore = [['7'], ['7']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "iter_reali"; token.key_prefix = "--"; token.label = "SAC stability check interval"; token.help = "Defines every how many iterations the SAC stability checking is performed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stab_ali"; token.key_prefix = "--"; token.label = "Number of alignments for stability check"; token.help = "The number of alignment runs when checking stability. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "minimum_grp_size"; token.key_prefix = "--"; token.label = "Minimum size of reproducible classes"; token.help = "Minimum size of reproducible classes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "60"; token.restore = [['60'], ['60']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rand_seed"; token.key_prefix = "--"; token.label = "Seed"; token.help = "Random seed set before calculations. Useful for testing purposes. By default, ISAC2 sets a random seed number. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_prealignment"; token.key_prefix = "--"; token.label = "Do pre-alignment"; token.help = "Indicate if pre-alignment should be used or not. Do not use pre-alignment if images are already centered. The 2dalignment directory will still be generated but the parameters will be zero. By default, do pre-alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "restart"; token.key_prefix = "--"; token.label = "Restart run"; token.help = "0: Restart ISAC2 after the last completed main iteration (i.e. the directory must contain finished file); k: Restart ISAC2 after k-th main iteration, it has to be completed (i.e. the directory must contain finished file), and higer iterations will be removed; Default: Do not restart. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_ordering"; token.key_prefix = "--"; token.label = "Skip ordered class averages"; token.help = "Skip the creation of the ordered class averages. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [Pixels]"; token.help = "When this is set to a non-default value, ISAC assumes helical data, in which case particle images will be subjected to rectangular masking of the given this value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_mask_ignore"; token.key_prefix = "--"; token.label = "Ignore filament masking (filament use only)"; token.help = "ONLY RELEVANT IF parameter filament_width is set to a non-default value. When processing helical particle images rectangular masking is used (A) to normalize and (B) to mask the actual particle images. The latter can be disabled by setting this flag to True. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_isac2_gpu"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "GPU ISAC - 2D Clustering"; sxcmd.short_info = "Iterative Stable Alignment and Clustering (ISAC) of a 2D image stack, using the GPU."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack_file"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Images must to be square (nx=ny). The stack can be either in .bdb or in .hdf format. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "General ISAC output directory to store all results. If the directory already exists ISAC will only run in continuation mode (see advanced parameter restart). "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Radius of the particle in pixels. ISAC cannot offer a default here since the value will depend on the particle im question. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp"; token.key_prefix = "--"; token.label = "Images per class"; token.help = "also defines number of classes K=(total number of images)/img_per_grp. If not specified, the value will be set to yield 200 classes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "CTF"; token.key_prefix = "--"; token.label = "CTF phase flipping"; token.help = "Use for cryo datasets. If set to True the data will be phase-flipped using CTF information included in the image headers. Cannot be used together with the VPP option. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['VPP', 'False', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('VPP', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "VPP"; token.key_prefix = "--"; token.label = "Phase Plate data"; token.help = "Use this option if the dataset is taken with a phase plate. Cannot be used together with the CTF option. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['CTF', 'False', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('CTF', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "gpu_devices"; token.key_prefix = "--"; token.label = "GPU devices"; token.help = "Specify the GPUs to be used (e.g. --gpu_devices=0, or --gpu_devices=0,1 for one or two GPUs, respectively). Using nvidia-smi in the terminal will print out what GPUs are available. For a more detailed printout you can also use --gpu_info here in ISAC. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gpu_memory_use"; token.key_prefix = "--"; token.label = "GPU memory use"; token.help = "Specify how much memory on the chosen GPUs ISAC is allowed to use. A value of 0.9 results in using 90% of the available memory (this is the default; higher percentages should be used with caution). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.9"; token.restore = [['0.9'], ['0.9']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Translation search range [Pixels]"; token.help = "The translational search range. Change with care; higher values will incur significantly higher processing costs. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "thld_err"; token.key_prefix = "--"; token.label = "Pixel error threshold [Pixels]"; token.help = "Used as a threshold value when checking cluster stability. The pixel error is defined as the root mean square of distances between corresponding pixels from set of found transformations and their average transformation; it depends linearly on square of radius (parameter ou). units - pixels. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.7"; token.restore = [['0.7'], ['0.7']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_radius"; token.key_prefix = "--"; token.label = "Target particle radius [Pixels]"; token.help = "Particle radius used by ISAC2 to process the data. All particle images will be re-scaled to match their particle radius with this radius. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "29"; token.restore = [['29'], ['29']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_nx"; token.key_prefix = "--"; token.label = "Target particle image size [Pixels]"; token.help = "Image size used by ISAC2 to process the data. particle images will first be resized according to target particle radius (see above) and then cropped or padded to achieve the target image size. When xr &gt; 0, the final image size for ISAC2 processing is target_nx + xr - 1  "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "76"; token.restore = [['76'], ['76']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ir"; token.key_prefix = "--"; token.label = "Inner ring [Pixels]"; token.help = "Radius of the inner-most ring when resampling images to polar coordinates. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rs"; token.key_prefix = "--"; token.label = "Ring step [Pixels]"; token.help = "Radius step size when resampling images to polar coordinates. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step [Pixels]"; token.help = "Translational search step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxit"; token.key_prefix = "--"; token.label = "Reference-free alignment iterations"; token.help = "The number of iterations for reference-free alignments. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "30"; token.restore = [['30'], ['30']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "center_method"; token.key_prefix = "--"; token.label = "Centering method"; token.help = "Method for centering of global 2D averages during the initial prealignment of the data (0 : average centering; -1 : average shift method; please see center_2D in sp_utilities.py for methods 1-7). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "dst"; token.key_prefix = "--"; token.label = "Discrete angle used for within-group alignment"; token.help = "Discrete angle used for within-group alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "90.0"; token.restore = [['90.0'], ['90.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "FL"; token.key_prefix = "--"; token.label = "Lowest filter frequency [1/Pixel]"; token.help = "Lowest frequency used for the tangent filter. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.2"; token.restore = [['0.2'], ['0.2']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "FH"; token.key_prefix = "--"; token.label = "Highest filter frequency [1/Pixel]"; token.help = "Highest frequency used for the tangent filter. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.45"; token.restore = [['0.45'], ['0.45']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "FF"; token.key_prefix = "--"; token.label = "Tangent filter fall-off"; token.help = "The fall-off of the tangent filter. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.2"; token.restore = [['0.2'], ['0.2']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "init_iter"; token.key_prefix = "--"; token.label = "Maximum generations"; token.help = "Maximum number of generation iterations performed for a given subset. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "7"; token.restore = [['7'], ['7']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "main_iter_limit"; token.key_prefix = "--"; token.label = "Main iteration limit"; token.help = "If set to a positive value N, ISAC execution is halted after N main iterations. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "iter_reali"; token.key_prefix = "--"; token.label = "SAC stability check interval"; token.help = "Defines every how many iterations the SAC stability checking is performed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stab_ali"; token.key_prefix = "--"; token.label = "Number of alignments for stability check"; token.help = "The number of alignment runs when checking stability. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "minimum_grp_size"; token.key_prefix = "--"; token.label = "Minimum group size"; token.help = "Minimum size of reproducible classes. If not specified, this value will be set to 60% of the 'img_per_group' value (see above). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rand_seed"; token.key_prefix = "--"; token.label = "Seed"; token.help = "Random seed set before calculations. Useful for testing purposes. By default, ISAC2 sets a random seed number. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_prealignment"; token.key_prefix = "--"; token.label = "Do pre-alignment"; token.help = "Indicate if pre-alignment should be used or not. Do not use pre-alignment if images are already centered. The 2dalignment directory will still be generated but the parameters will be zero. By default, do pre-alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "restart"; token.key_prefix = "--"; token.label = "Restart run"; token.help = "0: Restart ISAC2 after the last completed main iteration (i.e. the directory must contain finished file); k: Restart ISAC2 after k-th main iteration, it has to be completed (i.e. the directory must contain finished file), and higer iterations will be removed; Default: Do not restart. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_ordering"; token.key_prefix = "--"; token.label = "Skip ordered class averages"; token.help = "Skip the creation of the ordered class averages. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [Pixels]"; token.help = "When this is set to a non-default value, ISAC assumes helical data, in which case particle images will be subjected to rectangular masking of the given this value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_mask_ignore"; token.key_prefix = "--"; token.label = "Ignore filament masking (filament use only)"; token.help = "ONLY RELEVANT IF parameter filament_width is set to a non-default value. When processing helical particle images rectangular masking is used (A) to normalize and (B) to mask the actual particle images. The latter can be disabled by setting this flag to True. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gpu_info"; token.key_prefix = "--"; token.label = "GPU info"; token.help = "Print detailed information about the selected GPUs, including the class limit which is relevant when using the --gpu_class_limit parameter. Use --gpu_devices to specify what GPUs you want to know about. NOTE: ISAC will stop after printing this information, so don't use this parameter if you intend to actually process any data. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_compute_isac_avg"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Beautifier"; sxcmd.short_info = "Perform local 2D alignment of ISAC2 2D clustering results using the original pixel size and full CTF correction."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = "--"; token.label = "Original image stack"; token.help = "Data stack that used for the associated ISAC2 run. The particle images in this stack are used to create the full-sized class averages. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "isac_dir"; token.key_prefix = "--"; token.label = "ISAC2 run directory"; token.help = "Path to the output directory of the associated ISAC2 run. This is the input directory for this command. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "This automatically-created output directory will contain results. By default, the program uses sharpen_DATA_AND_TIME for the name. If this is the same as ISAC2 run directory, the program automatically creates sharpen subdirectory under the ISAC2 run directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "The pixel size of particle images in input particle stack for the associated ISAC2 run. Use 1.0 in case of negative stain data. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "There is no default radius. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "noctf"; token.key_prefix = "--"; token.label = "CTF correction"; token.help = "Indicate if full CTF correction should be applied or not. Always use the CTF correction for cryo data, but not for negative stained data. By default, do full CTF correction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "local_alignment"; token.key_prefix = "--"; token.label = "Local alignment"; token.help = "Indicate if local alignment should be applied or not. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = True; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency [1/Pixel]"; token.help = "Cutoff frequency of low-pass filter. =-1.0, do not apply the low-pass filter; =0.0, apply low pass filter to initial ISAC2 resolution; =1.0, to resolution after local alignment; else use user-provided cutoff in absolute frequency (&gt;0.0 and &lt;=0.45). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "abs_freq"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pw_adjustment"; token.key_prefix = "--"; token.label = "Power spectrum adjustment method"; token.help = "Specify the method for the power spectrum (PWS) adjustment of 2-D averages to enhance averages. ='analytical_model' adjusts PWS to an analytic model; ='bfactor' adjusts PWS using B-factor; ='FILE_PATH' adjusts PWS to rotationally averaged 1D power spectrum stored in the text file; ='no_adjustment' skips adjustment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "analytical_model"; token.restore = [['analytical_model'], ['analytical_model']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "B_start"; token.key_prefix = "--"; token.label = "Lower bound for B-factor estimation [A]"; token.help = "Lower resolution bound of power spectrum for B-factor estimation. Specific to adjust to B-factor method. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "45.0"; token.restore = [['45.0'], ['45.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "Bfactor"; token.key_prefix = "--"; token.label = "Use ad-hoc B-factor [A^2]"; token.help = "Skip the automatic estimation and use user-provided ad-hoc B-factor (e.g. 25.0[A^2]) for the enhancement. By default, the program automatically estimates B-factor. Specific to adjust to B-factor method. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Local search range [Pixels]"; token.help = "Translational search range for local alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['local_alignment', 'True', 'False']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('local_alignment', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Local search step [Pixels]"; token.help = "Translational search step for local alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['local_alignment', 'True', 'False']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('local_alignment', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "fh"; token.key_prefix = "--"; token.label = "High frequency search limit [1/Pixel]"; token.help = "High frequency search limit in absolute frequency for local alignment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['local_alignment', 'True', 'False']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "abs_freq"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('local_alignment', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "maxit"; token.key_prefix = "--"; token.label = "Local alignment iterations"; token.help = "The number of iterations for local aligment. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['local_alignment', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('local_alignment', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "navg"; token.key_prefix = "--"; token.label = "Number of averages"; token.help = "The number of averages to be process, starting from the first image. By default, uses all ISAC2 average images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1000000"; token.restore = [['1000000'], ['1000000']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "isac_substack"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "ISAC2 Stack Subset"; sxcmd.short_info = "Create a virtual subset stack consisting of particles acounted for by ISAC2 by retrieving particle numbers associated with the ISAC2 or Beautifier class averages. The command also saves a selection file containing the retrieved original image numbers and 2D alignment parameters. In addition, it stores the 2D alignment parameters to the stack header."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_bdb_stack_path"; token.key_prefix = ""; token.label = "Input bdb image stack"; token.help = "Specify the same bdb image stack used for the associated ISAC2 run. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "ISAC2 or Beautifier run output directory"; token.help = "Specify output directory of an ISAC2 or Beautifier run as an input to this command. From this directory, the program extracts the shrink ratio and 2D alignment parameters of the ISAC2 run or local 2D alignment parameters of the Beautifier run. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "isac_class_avgs_path"; token.key_prefix = "--"; token.label = "ISAC2 or Beautifier class averages path"; token.help = "Specify path to a file containg ISAC2 or Beautifier class averages. The class averages can be fullset or selected subset, as long as they are associated with the input bdb image stack and contain class member information stored in the headers. By default, the program uses the same default name of ordered class averages in ISAC2 or Beautifier (i.e. ordered_class_averages.hdf). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "substack_basename"; token.key_prefix = "--"; token.label = "Stack subset basename"; token.help = "Specify the basename of ISAC2 stack subset file. It cannot be empty string or only white spaces. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "isac_substack"; token.restore = [['isac_substack'], ['isac_substack']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2bdb"; sxcmd.subname = ""; sxcmd.mode = "makevstack"; sxcmd.subset_config = ""; sxcmd.label = "Create Virtual Stack"; sxcmd.short_info = "Make a 'virtual' bdb image stack with the specified name from one or more other stacks. "; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "makevstack"; token.key_prefix = "--"; token.label = "Output virtual image stack"; token.help = "Make a 'virtual' bdb image stack with the specified name from one or more other stacks. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_bdb_stack_file"; token.key_prefix = ""; token.label = "Input BDB image stack"; token.help = "Specify path to input BDB stack file. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "list"; token.key_prefix = "--"; token.label = "Image selection file"; token.help = "Input selection text file containing a list of selected image IDs (or indexes of the data subset) to create a new virtual bdb image stack from an existed stack or virtual stack. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['exlist', 'none', 'False'], ['step', '0,1', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('exlist', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('step', []).append([token.key_base, '0,1', 'False'])
		token = SXcmd_token(); token.key_base = "exlist"; token.key_prefix = "--"; token.label = "Image exclusion file"; token.help = "Input exclusion text file containing a list of excluded image IDs (or indexes of the data subset) to create a new virtual bdb image stacks from an existed stack or virtual stack. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['list', 'none', 'False'], ['step', '0,1', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('list', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('step', []).append([token.key_base, '0,1', 'False'])
		token = SXcmd_token(); token.key_base = "step"; token.key_prefix = "--"; token.label = "Processes only subset"; token.help = "Specify &lt;init&gt;,&lt;step&gt;[,&lt;max&gt;]. Process only a subset of the input data. For example, 0,2 would process only the even numbered particles. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['list', 'none', 'False'], ['exlist', 'none', 'False']]; token.default = "0,1"; token.restore = [['0,1'], ['0,1']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('list', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('exlist', []).append([token.key_base, 'none', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_eval_isac"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Evaluate ISAC classes"; sxcmd.short_info = "Separates stacks of particle images into stacks for each class."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_class_avgs"; token.key_prefix = ""; token.label = "Input class averages"; token.help = "Set of 2D class averages, with particle-membership information in header. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Directory where outputs will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "particles"; token.key_prefix = "--"; token.label = "Input particle stack"; token.help = "Required if particles will be processed (i.e., everything except simple class seperation or bandpass filtration).  "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "align_isac_dir"; token.key_prefix = "--"; token.label = "ISAC/Beautifier direrctory"; token.help = "ISAC or beautifier directory, from which alignment parameters will be applied. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filtrad"; token.key_prefix = "--"; token.label = "Low-pass filter radius"; token.help = "Low-pass filter radius. If pixel size is provided, then units will be Angstroms.  If pixel size is not is not specified, program will assume units of absolute frequency (0..0.5). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'True']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size"; token.help = "Pixel size, in Angstroms. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shrink"; token.key_prefix = "--"; token.label = "Downsampling factor"; token.help = "Factor by which images will be downsampled. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'True']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "ctf"; token.key_prefix = "--"; token.label = "CTF-correction mode"; token.help = "Allowed options are 'flip' (phase-flipping) and 'wiener' (phase and amplitude, like with beautifier). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'True']]; token.default = "None"; token.restore = [['None', 'flip', 'wiener'], ['None', 'flip', 'wiener']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "chains_radius"; token.key_prefix = "--"; token.label = "Chains radius"; token.help = "Alignment radius for generating ordered class averages internally. Units are pixels, on the scale of the input class averages.  In other words, for ISAC averages, this value will typically be 29. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'True']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "nvec"; token.key_prefix = "--"; token.label = "Number of eigenimages"; token.help = "Number of eigenimages to compute using principal component analysis. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'True']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "pca_radius"; token.key_prefix = "--"; token.label = "PCA radius"; token.help = "Radius for principal component analysis (PCA). Only pixels within this radius will be examined using PCA. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nvec', '0', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nvec', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "mask_binary"; token.key_prefix = "--"; token.label = "Binary mask file"; token.help = "A binary mask file to use for principal component analysis. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nvec', '0', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nvec', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "mask_drawn"; token.key_prefix = "--"; token.label = "Drawn mask file"; token.help = "A drawn mask means that, using e2display.py or a similar program, a mask was drawn on top of a class average. The mask will be binarized according to the maximum of the class average, and that binary mask will be used for PCA. When running sp_eval_isac.py, the screen output will report the maximum pixel value among the class averages. Use a value exceeding that value for the pen intensity when drawing the mask using e2display.py. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nvec', '0', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nvec', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "bandpass_width"; token.key_prefix = "--"; token.label = "Bandpass width"; token.help = "Width of bandpass filter, in units of absolute frequency (i.e., 0..0.5). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['bandpass_radius', 'None', 'True']]; token.default = "0.03"; token.restore = [['0.03'], ['0.03']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('bandpass_radius', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "write_centered"; token.key_prefix = "--"; token.label = "Apply centering"; token.help = "Applies centering parameters from sp_center_2d3d.py. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "applyparams"; token.key_prefix = "--"; token.label = "Centering options"; token.help = "Allowed centering options are  'combined' (shifts and rotation, both floating point) and 'intshifts' (integer shifts only, to avoid interpolation). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['write_centered', 'True', 'False']]; token.default = "combined"; token.restore = [['combined', 'intshifts'], ['combined', 'intshifts']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('write_centered', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "debug"; token.key_prefix = "--"; token.label = "Debug centering"; token.help = "For use with centering option 'intshifts', to make sure that correct shifts are applied, rotation is applied, and averages are computed. If this flag is not activated, no averages will be generated. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['applyparams', 'intshifts', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('applyparams', []).append([token.key_base, 'intshifts', 'False'])
		token = SXcmd_token(); token.key_base = "bandpass_radius"; token.key_prefix = "--"; token.label = "Bandpass radius"; token.help = "Applies a bandpass filter to class averages.  Bandpass radius assumed to be in units of Angstroms if pixel size is provided. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['particles', 'None', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('particles', []).append([token.key_base, 'None', 'False'])
		token = SXcmd_token(); token.key_base = "verbosity"; token.key_prefix = "--"; token.label = "Verbosity"; token.help = "Controls the amount of information written to the screen, ranging from 0..6. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_compare2d"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Compare 2D images"; sxcmd.short_info = "Find best match between two sets of 2D images."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "image_stack_1"; token.key_prefix = ""; token.label = "Input stack #1"; token.help = "To each imagine in this stack, all of the images in the second input stack will be compared.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "image_stack_2"; token.key_prefix = ""; token.label = "Input stack #2"; token.help = "Each image from this stack will be aligned to each image from the first input stack. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Directory where output files will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outterradius"; token.key_prefix = "--"; token.label = "Outer radius"; token.help = "Outer radius in pixels. If not specified, the maximum allowed from the image dimension and maximum shift will be used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "normalize"; token.key_prefix = "--"; token.label = "Normalization mode"; token.help = "Methods for displaying the images from the two inputs stacks. If both comes from the same source, uses 'None'. Other options: 'minmax' (sets the minimum and maximum for each image to constants), 'rops' (sets 1D rotational power spectra equal to each other), and 'sigmaone' (sets the average to 0 and sigma to 1). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None', 'minmax', 'rops', 'sigmaone'], ['None', 'minmax', 'rops', 'sigmaone']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxshift"; token.key_prefix = "--"; token.label = "Maximum shift"; token.help = "Maximum shift allowed during alignment. Alignment will be slowed significantly as the maximum shift increases. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ringstep"; token.key_prefix = "--"; token.label = "Ring step"; token.help = "Alignments will be computed at this radial increment, in pixels. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbosity"; token.key_prefix = "--"; token.label = "Verbosity level"; token.help = "Controls how much information will be written to the screen, from 0..2. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_batch"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Batch Pipeline Execution"; sxcmd.short_info = "Run jobs that wait with the execution on each other."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "submission_command"; token.key_prefix = ""; token.label = "Submission command"; token.help = "Submission command, e.g., qsub, qsub -V, sbatch, bash "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "Pipeline directory"; token.help = "Directory containin the pipeline submission files "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hold_flag"; token.key_prefix = "--"; token.label = "Hold flag"; token.help = "Hold flag for the submission command, e.g. -hold_jid=, --wait=, --dependency=afterany:; Default is None and should be used in combination with a local execution with bash "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "first_hold_number"; token.key_prefix = "--"; token.label = "First hold number"; token.help = "Wait number for the first job that is submitted. By default, the first job will not wait for others "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_cinderella_pred"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Automatic 2D class selection"; sxcmd.short_info = "Classify classes into good and bad."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_isac"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "cinderella_path"; token.key_prefix = ""; token.label = "Cinderella predict executeable"; token.help = "Path to the sp_cinderella_predict.py in your Cinderella environment "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "py"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_stack_cinderella"; token.key_prefix = ""; token.label = "Input stack"; token.help = "Path to your class stack (.mrcs/.hdf). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Folder to write results. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "model_path"; token.key_prefix = ""; token.label = "Model path"; token.help = "Specifiy the path to your model file. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_h5"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "confidence_threshold"; token.key_prefix = "--"; token.label = "Confidence threshold"; token.help = "Classes with a confidence higher as that threshold are classified as good.  "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "gpu"; token.key_prefix = "--"; token.label = "GPU ID"; token.help = "ID of the GPU that should be used. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "batch_size"; token.key_prefix = "--"; token.label = "Number of batches"; token.help = "Number of mini-batches during prediction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "32"; token.restore = [['32'], ['32']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_rviper"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Initial 3D Model - RVIPER"; sxcmd.short_info = "Reproducible ab initio 3D structure determination. The program determines a validated initial intermediate resolution structure using a subset of class averages produced by ISAC2."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input images stack"; token.help = "Subset of class averages, e.g., produced by ISAC2. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The automatically-created output directory will contain results. If the directory already exists, results will be written there, possibly overwriting previous runs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Use the same value as in ISAC2. It has to be less than half the box size. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "29"; token.restore = [['29'], ['29']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the particle. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "moon_elimination"; token.key_prefix = "--"; token.label = "Eliminate disconnected regions"; token.help = "Used to removed disconnected pieces from the model. As an argument it requires a comma-separated string with the mass in KDa and the pixel size in [A]. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "iteration_start"; token.key_prefix = "--"; token.label = "Restarting iteration"; token.help = "Iteration from which to restart the program. 0 means go to the most recent one. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "n_rv_runs"; token.key_prefix = "--"; token.label = "RVIPER iterations"; token.help = "Corresponds to main### output directory. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "10"; token.restore = [['10'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "n_v_runs"; token.key_prefix = "--"; token.label = "Minimum number of VIPER runs per RVIPER iterations"; token.help = "Corresponds to run### output directory. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "criterion_name"; token.key_prefix = "--"; token.label = "Stable projection criterion"; token.help = "Used to decide if the solution is stable, i.e., whether projection images adopt similar orientations in independent runs of the program. Valid options are: '80th percentile',  or 'fastest increase in the last quartile'. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "80th percentile"; token.restore = [['80th percentile'], ['80th percentile']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_index_threshold_method"; token.key_prefix = "--"; token.label = "Outlier selection method"; token.help = "Used to decide which images to keep. Valid options are: 'discontinuity_in_derivative', 'percentile', or 'angle_measure'. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "discontinuity_in_derivative"; token.restore = [['discontinuity_in_derivative'], ['discontinuity_in_derivative']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "angle_threshold"; token.key_prefix = "--"; token.label = "Angle threshold"; token.help = "Threshold used to remove projections if 'angle_measure' is used to decide the outliers. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "30"; token.restore = [['30'], ['30']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_percentile"; token.key_prefix = "--"; token.label = "Percentile for outlier"; token.help = "Threshold above which images are considered outliers and removed if 'percentile' is used as outlier selection method. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "95.0"; token.restore = [['95.0'], ['95.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ir"; token.key_prefix = "--"; token.label = "Inner rotational search radius [Pixels]"; token.help = "Inner rotational search radius [Pixels]. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rs"; token.key_prefix = "--"; token.label = "Ring step size [Pixels]"; token.help = "Step between rings used for the rotational search. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "X search range [Pixels]"; token.help = "The translational search range in the x direction. Search will +/-xr range in steps of ts. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "yr"; token.key_prefix = "--"; token.label = "Y search range [Pixels]"; token.help = "The translational search range in the y direction. If omitted it will be equal to the x search range. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Translational search step [Pixels]"; token.help = "The search will be performed in -xr, -xr+ts, 0, xr-ts, xr, can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Projection angular step [Degrees]"; token.help = "Projection angular step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "2.0"; token.restore = [['2.0'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "center"; token.key_prefix = "--"; token.label = "Center 3D template"; token.help = "-1: center of coordinates, 0: no centering; 1: center of gravity "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxit1"; token.key_prefix = "--"; token.label = "Maximum iterations - GA step"; token.help = "Maximum number of iterations for GA step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "400"; token.restore = [['400'], ['400']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxit2"; token.key_prefix = "--"; token.label = "Maximum iterations - Finish step"; token.help = "Maximum iterations number of for Finish step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "50"; token.restore = [['50'], ['50']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask"; token.help = "Path to 3D mask file. By default, a spherical mask will be used. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['filament_width', '-1', 'False']]; token.default = "sphere"; token.restore = [['sphere'], ['sphere']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_width', []).append([token.key_base, '-1', 'False'])
		token = SXcmd_token(); token.key_base = "L2threshold"; token.key_prefix = "--"; token.label = "GA stop threshold"; token.help = "Defines the maximum relative dispersion of structures' L2 norms. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.03"; token.restore = [['0.03'], ['0.03']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ref_a"; token.key_prefix = "--"; token.label = "Projection generation method"; token.help = "Method for generating the quasi-uniformly distributed projection directions. S - Saff algorithm, M - Markus 2019 algorithm or P - Penczek 1994 algorithm. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'P', 'M'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "n_shc_runs"; token.key_prefix = "--"; token.label = "GA population size"; token.help = "This defines the number of quasi-independent structures generated. (same as --nruns parameter from sp_viper). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "doga"; token.key_prefix = "--"; token.label = "Threshold to start GA"; token.help = "Do GA when the fraction of orientation that changes less than 1.0 degrees is at least this fraction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency [1/Pixels]"; token.help = "Using a hyperbolic tangent low-pass filter. Specify with absolute frequency. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.25"; token.restore = [['0.25'], ['0.25']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Fall-off of for the hyperbolic tangent low-pass filter. Specify with absolute frequency. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pwreference"; token.key_prefix = "--"; token.label = "Power spectrum reference"; token.help = "Text file containing a 1D reference power spectrum. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "spectrum1d"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "npad"; token.key_prefix = "--"; token.label = "Image padding factor"; token.help = "Used by 3D reconstruction algorithm in the program. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta1"; token.key_prefix = "--"; token.label = "Minimum theta"; token.help = "Lower bound for out-of-plane tilt angle. No tilt corresponds to 90 degrees. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta2"; token.key_prefix = "--"; token.label = "Maximum theta"; token.help = "Upper bound for out-of-plane tilt angle. No tilt corresponds to 90 degrees. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "dpi"; token.key_prefix = "--"; token.label = "BILD resolution"; token.help = "Resolution in dpi for angular distribution plot. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "72"; token.restore = [['72'], ['72']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [Pixels]"; token.help = "When this is set to a non-default value helical data is assumed in which case the input images will be aligned to a mask with this width. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['mask3D', 'sphere', 'False']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mask3D', []).append([token.key_base, 'sphere', 'False'])
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Resample ratio"; token.help = "Specify a value larger than 0.0. By default, the program does not resample the input map (i.e. resample ratio is 1.0). Use this option maily to restore the original dimensions or pixel size of VIPER or R-VIPER model. Alternatively, specify the path to the output directory of an ISAC2 run. The program automatically extracts the resampling ratio used by the ISAC2 run. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['filament_width', '-1', 'True'], ['mask3D', 'sphere', 'False']]; token.default = "'1.0'"; token.restore = [["'1.0'"], ["'1.0'"]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('filament_width', []).append([token.key_base, '-1', 'True']); sxcmd.dependency_dict.setdefault('mask3D', []).append([token.key_base, 'sphere', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_proj_compare"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Compare Re-projections"; sxcmd.short_info = "Compare re-projections to class averages."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Set of 2D images to be compared, i.e., class averages. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_volume"; token.key_prefix = ""; token.label = "Input volume"; token.help = "Reconstruction for which re-projections will be computed. In RVIPER, this file is of the form main003/run002/rotated_volume.hdf. This volume must have the same dimensions as the input images. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outdir"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Directory where outputs will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mode"; token.key_prefix = "--"; token.label = "Comparison method"; token.help = "Choices are: viper, projmatch, and meridien. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "projmatch"; token.restore = [['projmatch', 'viper', 'meridien'], ['projmatch', 'viper', 'meridien']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "classangles"; token.key_prefix = "--"; token.label = "VIPER - Projection parameter file"; token.help = "Parameter file containing projection angles. Not required if the projection parameters are stored in the header of the input images, which for ISAC2 is generally not the case. In RVIPER, this file is of the form main003/run002/rotated_reduced_params.txt. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'viper', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "params_proj_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'viper', 'False'])
		token = SXcmd_token(); token.key_base = "classselect"; token.key_prefix = "--"; token.label = "VIPER - Image selection file"; token.help = "Input selection containing list of images to be included from the input stack.  For instance, RVIPER, if it finds a stable solution, may exclude some images, and thus their projection angles will be excluded from the parameters file.  The file containing the list of included images will have a name like main003/index_keep_image.txt. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'viper', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'viper', 'False'])
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "ProjMatch - Sampling angle"; token.help = "Angular-sampling for reference projections. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'projmatch', 'False']]; token.default = "7.5"; token.restore = [['7.5', '15', '3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['7.5', '15', '3.75', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'projmatch', 'False'])
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "ProjMatch - Symmetry"; token.help = "To limit angular projections. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'projmatch', 'False']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'projmatch', 'False'])
		token = SXcmd_token(); token.key_base = "partangles"; token.key_prefix = "--"; token.label = "MERIDIEN - Alignment parameter file"; token.help = "Input refinement parameter file, e.g., Refine3D/final_params_037.txt. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "params_proj_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])
		token = SXcmd_token(); token.key_base = "partselect"; token.key_prefix = "--"; token.label = "MERIDIEN - Particle selection file"; token.help = "Input substack selection file if particles removed before refinement, e.g., Substack/isac_substack_particle_id_list.txt. This file is used to map the particle number before classification to particle number in refinement. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])
		token = SXcmd_token(); token.key_base = "outliers"; token.key_prefix = "--"; token.label = "MERIDIEN - Outlier angle"; token.help = "Particles differing from average Euler angle by more than this threshold (in degrees) will be excluded from average calculation, by default keeps all. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])
		token = SXcmd_token(); token.key_base = "prjmethod"; token.key_prefix = "--"; token.label = "Interpolation method"; token.help = "Valid choices are trilinear, gridding, and nn. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "trilinear"; token.restore = [['trilinear', 'gridding', 'nn'], ['trilinear', 'gridding', 'nn']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "display"; token.key_prefix = "--"; token.label = "e2display"; token.help = "Automatically pops up a window with the output montage. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "matchshift"; token.key_prefix = "--"; token.label = "ProjMatch - Maximum shift"; token.help = "Maximum shift to allow during translation alignment, pixels. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'projmatch', 'False']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'projmatch', 'False'])
		token = SXcmd_token(); token.key_base = "matchrad"; token.key_prefix = "--"; token.label = "ProjMatch - Outer radius"; token.help = "Outer alignment radius, defaults to automatically-determined. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'projmatch', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'projmatch', 'False'])
		token = SXcmd_token(); token.key_base = "matchstep"; token.key_prefix = "--"; token.label = "ProjMatch - Radius step size"; token.help = "Alignment radius step size. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'projmatch', 'False']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'projmatch', 'False'])
		token = SXcmd_token(); token.key_base = "refinerad"; token.key_prefix = "--"; token.label = "MERIDIEN - Outer radius"; token.help = "Outer alignment radius, defaults to automatically-determined. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])
		token = SXcmd_token(); token.key_base = "refineshift"; token.key_prefix = "--"; token.label = "MERIDIEN - Maximum shift"; token.help = "Maximum shift to allow during translation alignment, pixels. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])
		token = SXcmd_token(); token.key_base = "refinestep"; token.key_prefix = "--"; token.label = "MERIDIEN - Radius step size"; token.help = "Alignment radius step size. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])
		token = SXcmd_token(); token.key_base = "align"; token.key_prefix = "--"; token.label = "MERIDIEN - Alignment method"; token.help = "Valid choices are apsh and scf. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mode', 'meridien', 'False']]; token.default = "apsh"; token.restore = [['apsh', 'scf'], ['apsh', 'scf']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mode', []).append([token.key_base, 'meridien', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "moon_eliminator"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Volume Adjustment"; sxcmd.short_info = "Eliminate moons or remove dust from the background of a 3D density map based on the expected molecular mass."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume_path"; token.key_prefix = ""; token.label = "Input volume path"; token.help = "Path to input volume file containing the 3D density map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Output pixel size [A]"; token.help = "The original pixel size of dataset. This must be the pixel size after resampling when resample_ratio != 1.0. That is, it will be the pixel size of the output map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_density_threshold', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_density_threshold', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "use_density_threshold"; token.key_prefix = "--"; token.label = "Use ad-hoc density threshold"; token.help = "Use user-provided ad-hoc density threshold, instead of computing the value from the molecular mass. Below this density value, the data is assumed not to belong to the main body of the particle density. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "moon_distance"; token.key_prefix = "--"; token.label = "Distance to the nearest moon [Pixels]"; token.help = "The moons further than this distance from the density surface will be elminated. The value smaller than the default is not recommended because it is difficult to avoid the stair-like gray level change at the edge of the density surface. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.0"; token.restore = [['3.0'], ['3.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Resample ratio"; token.help = "Specify a value larger than 0.0. By default, the program does not resample the input map (i.e. resample ratio is 1.0). Use this option maily to restore the original dimensions or pixel size of VIPER or R-VIPER model. Alternatively, specify the path to the output directory of an ISAC2 run. The program automatically extracts the resampling ratio used by the ISAC2 run. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "'1.0'"; token.restore = [["'1.0'"], ["'1.0'"]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Output box size [Pixels]"; token.help = "The x, y, and z dimensions of cubic area to be windowed from input 3D volume for output 3D volumes. This must be the box size after resampling when resample_ratio != 1.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "invert_handedness"; token.key_prefix = "--"; token.label = "Invert handedness"; token.help = "Invert the handedness of the 3D map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter resolution [A]"; token.help = "&gt;0.0: low-pass filter to the value in Angstrom; =-1.0: no low-pass filter. The program applies this low-pass filter before the moon elimination. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_volume_path_2nd"; token.key_prefix = ""; token.label = "Second input volume path"; token.help = "Path to second input volume file containing the 3D density map. Use this option to create a mask from the sum of two MERIDIEN half-set maps. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. By default, it is set to half of --moon_distance so that the voxels with 1.0 values in the mask are same as the hard-edged molecular-mass binary volume. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) 'cosine' for cosine soft-edged (used in PostRefiner) and (2) 'gauss' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outputs_root"; token.key_prefix = "--"; token.label = "Root name of outputs"; token.help = "Specify the root name of all outputs. It cannot be empty string or only white spaces. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol3d"; token.restore = [['vol3d'], ['vol3d']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resampled_shift3d"; token.key_prefix = "--"; token.label = "Providing resampled 3D shifts"; token.help = "Use this option when you are providing the resampled 3D shifts (using pixel size of outputs) when --resample_ratio!=1.0. By default, the program assums the provided shifts are not resampled. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_x"; token.key_prefix = "--"; token.label = "3D x-shift [Pixels]"; token.help = "3D x-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_y"; token.key_prefix = "--"; token.label = "3D y-shift [Pixels]"; token.help = "3D y-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_z"; token.key_prefix = "--"; token.label = "3D z-shift [Pixels]"; token.help = "3D z-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off in absolute frequency. The program applies this low-pass filter before the moon elimination. Effective only when --fl &gt; 0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "debug"; token.key_prefix = "--"; token.label = "Run with debug mode"; token.help = "Mainly for developers. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_mask"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Masking"; sxcmd.short_info = "Mask creation tool for 2D or 3D masks."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume"; token.key_prefix = ""; token.label = "Input image"; token.help = "Path to the 2D image or 3D Volume "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output direcory path "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "Output prefix"; token.help = "Prefix of the produced files "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "sp_mask"; token.restore = [['sp_mask'], ['sp_mask']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overwrite"; token.key_prefix = "--"; token.label = "Overwrite outputs"; token.help = "Overwrite the output mask in case it exists already. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A/px]"; token.help = "Pixel size of the volume. Used for filtering and the molcular mask threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['nsigma', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. This is used to calculate the binarization threshold automatically. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One cycle of dilation will add about 2 pixels to the mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking. If the width is 0, a binary mask is returned. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "allow_disconnected"; token.key_prefix = "--"; token.label = "Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fill_mask"; token.key_prefix = "--"; token.label = "Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nerosion"; token.key_prefix = "--"; token.label = "Number of erosions"; token.help = "Number of times to erode binarized volume. One cycle of erosion will remove about 2 pixels from the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "do_old"; token.key_prefix = "--"; token.label = "Old behaviour"; token.help = "Restore the old masking behaviour, which is a bit less smooth. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "low_pass_filter_resolution"; token.key_prefix = "--"; token.label = "Low pass filter resolution [A]"; token.help = "Low pass filter resolution in angstroms. If set, the volume will be filtered prior to create a mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "low_pass_filter_falloff"; token.key_prefix = "--"; token.label = "Low pass filter falloff [1/Pixel]"; token.help = "Low pass filter falloff in absolute frequencies. If set, the volume will be filtered prior to create a mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['low_pass_filter_resolution', 'none', 'True']]; token.default = "0.01"; token.restore = [['0.01'], ['0.01']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('low_pass_filter_resolution', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "use_second_mask"; token.key_prefix = "--"; token.label = "Use a second mask"; token.help = "ONLY A GUI OPTION. The second mask can be used to mask the first one after creation. This is useful to create soft edged regions of the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [True]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "second_mask"; token.key_prefix = "--"; token.label = "Second mask path"; token.help = "Path to the input second mask used for masking the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask_shape', 'none', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "second_mask_shape"; token.key_prefix = "--"; token.label = "Second mask shape"; token.help = "Shape of the second mask. Possible values: sphere, cylinder, cube. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'False']]; token.default = "none"; token.restore = [['none', 'sphere', 'cylinder', 'cube'], ['cylinder', 'none', 'sphere', 'cube']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "s_radius"; token.key_prefix = "--"; token.label = "Second - Radius of the mask"; token.help = "Radius of the sphere or cylinder for the second mask in pixels. Radius must be less than half of the volume dimension. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['second_mask_shape', 'cube', 'True'], ['use_second_mask', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'cube', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_nx"; token.key_prefix = "--"; token.label = "Second - X dimension of the mask"; token.help = "X Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_ny"; token.key_prefix = "--"; token.label = "Second - Y dimension of the mask"; token.help = "Y Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. If not provided, a squared volume with s_nx is assumed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False'], ['s_nx', 'none;Main', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_nx', []).append([token.key_base, 'none;Main', 'True'])
		token = SXcmd_token(); token.key_base = "s_nz"; token.key_prefix = "--"; token.label = "Second - Z dimension of the mask"; token.help = "Z Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. If not provided, a squared volume with s_nx is assumed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False'], ['s_nx', 'none;Main', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_nx', []).append([token.key_base, 'none;Main', 'True'])
		token = SXcmd_token(); token.key_base = "s_pixel_size"; token.key_prefix = "--"; token.label = "Second - Pixel size [A/px]"; token.help = "Pixel size of the second volume. Used for the molecular mass threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_use_mol_mass"; token.key_prefix = "--"; token.label = "Second - Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_threshold', 'none', 'False'], ['s_nsigma', 'none', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_mol_mass"; token.key_prefix = "--"; token.label = "Second - Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. This is used to calculate the binarization threshold automatically. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_use_mol_mass', 'True', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_threshold"; token.key_prefix = "--"; token.label = "Second - Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. If the value is lower-equal than the default, the option will be ignored and the threshold will be set according to nsigma method above. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_nsigma', 'none', 'False'], ['s_use_mol_mass', 'False', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_nsigma"; token.key_prefix = "--"; token.label = "Second - Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_threshold', 'none', 'False'], ['s_use_mol_mass', 'False', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_ndilation"; token.key_prefix = "--"; token.label = "Second - Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One round of erosion will add about 2 pixels to the mask "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_nerosion"; token.key_prefix = "--"; token.label = "Second - Number of erosions"; token.help = "Number of times to erode binarized volume. One round of erosion will remove about 2 pixels from the mask "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_edge_width"; token.key_prefix = "--"; token.label = "Second - Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_edge_type"; token.key_prefix = "--"; token.label = "Second - Soft-edge type"; token.help = "The type of soft-edge for the 3D mask. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['s_edge_width', '0', 'True']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "s_do_old"; token.key_prefix = "--"; token.label = "Second - Old behaviour"; token.help = "Restore the old masking behaviour, which is a bit less smooth. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['s_edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "s_allow_disconnected"; token.key_prefix = "--"; token.label = "Second - Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_fill_mask"; token.key_prefix = "--"; token.label = "Second - Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "s_invert"; token.key_prefix = "--"; token.label = "Second - Invert masking"; token.help = "If True, the mask will remove everything that is inside instead of leaving it. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_viper"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Initial 3D Model - VIPER"; sxcmd.short_info = "ab initio 3D structure determination using Validation of Individual Parameter Reproducibility (VIPER). Determines a validated initial model using a small set of class averages produced by ISAC2."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input images stack"; token.help = "A small subset of class averages produced by ISAC2. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data2d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The automatically-created output directory will contain results. If the directory already exists, results will be written there, possibly overwriting previous runs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Use the same value as in ISAC2. It has to be less than half the box size. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "29"; token.restore = [['29'], ['29']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the particle. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "moon_elimination"; token.key_prefix = "--"; token.label = "Eliminate disconnected regions"; token.help = "Used to removed disconnected pieces from the model. As an argument it requires a comma-separated string with the mass in KDa and the pixel size in [A]. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ir"; token.key_prefix = "--"; token.label = "Inner rotational search radius [Pixels]"; token.help = "Inner rotational search radius [Pixels]. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "rs"; token.key_prefix = "--"; token.label = "Ring step size [Pixels]"; token.help = "Step between rings used for the rotational search. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "X search range [Pixels]"; token.help = "The translational search range in the x direction. Search will be +/-xr in steps of ts. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "yr"; token.key_prefix = "--"; token.label = "Y search range [Pixels]"; token.help = "The translational search range in the y direction. If omitted, it will be set to xr. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Translational search step [Pixels]"; token.help = "The search will be performed in -xr, -xr+ts, 0, xr-ts, xr, can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Projection angular step [Degrees]"; token.help = "Projection angular step, degrees. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2.0"; token.restore = [['2.0'], ['2.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "center"; token.key_prefix = "--"; token.label = "Centering method"; token.help = "For 3D template, -1: center of coordinates, 0: no centering; 1: center of gravity "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxit1"; token.key_prefix = "--"; token.label = "Maximum iterations - GA step"; token.help = "Maximum number of iterations for GA step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "400"; token.restore = [['400'], ['400']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maxit2"; token.key_prefix = "--"; token.label = "Maximum iterations - Finish step"; token.help = "Maximum iterations number of for Finish step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "50"; token.restore = [['50'], ['50']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask"; token.help = "Path to 3D mask file. By default, a spherical mask will be used. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "sphere"; token.restore = [['sphere'], ['sphere']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "L2threshold"; token.key_prefix = "--"; token.label = "GA stop threshold"; token.help = "Defines the maximum relative dispersion of structures' L2 norms. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.03"; token.restore = [['0.03'], ['0.03']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ref_a"; token.key_prefix = "--"; token.label = "Projection generation method"; token.help = "Method for generating the quasi-uniformly distributed projection directions. S - Saff algorithm, or P - Penczek 1994 algorithm. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S'], ['S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nruns"; token.key_prefix = "--"; token.label = "Number of runs"; token.help = "GA population size, i.e., the number of quasi-independent structures generated. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "6"; token.restore = [['6'], ['6']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "doga"; token.key_prefix = "--"; token.label = "Threshold to start GA"; token.help = "Do GA when the fraction of orientation that changes less than 1.0 degrees is at least this fraction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency [1/Pixels]"; token.help = "Using a hyperbolic tangent low-pass filter. Specify with absolute frequency. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.25"; token.restore = [['0.25'], ['0.25']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Fall-off of for the hyperbolic tangent low-pass filter. Specify with absolute frequency. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pwreference"; token.key_prefix = "--"; token.label = "Power spectrum reference"; token.help = "Text file containing a 1D reference power spectrum. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "spectrum1d"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "debug"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Print debug info. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pdb2em"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "PDB File Conversion"; sxcmd.short_info = "Converts an atomic model stored in a PDB file into a simulated electron density map."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_pdb"; token.key_prefix = ""; token.label = "Input PDB file"; token.help = "Starting atomic coordinates. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "pdb"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_hdf"; token.key_prefix = ""; token.label = "Output map"; token.help = "Specify file path for output map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size of output map [A]"; token.help = "Pixel size of the output map [A]. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box"; token.key_prefix = "--"; token.label = "Output box size [Voxels]"; token.help = "Specify string of a single value (e.g. '256') to get a cubic box. Alternatively, use 'x,y,z' format to specify demensions of x,y,z-axis (e.g. '128,64,256'). If not given, the program will find the minimum box size fitting the structure. Be aware that this will most likely result in a rectangular box. Note that GUI does not support the default mode. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "het"; token.key_prefix = "--"; token.label = "Include hetero atoms"; token.help = "Otherwise, the HETATM entries in the PDB file are ignored. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chains"; token.key_prefix = "--"; token.label = "Chain identifiers"; token.help = "A string list of chain identifiers to include (e.g. 'ABEFG'). By default, all chains will be included. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "center"; token.key_prefix = "--"; token.label = "Center model at the origin"; token.help = "Specifies whether the atomic model should be moved to the origin before generating density map. Available options are: 'c' - Use the geometrical center of atoms; 'a' - Use the center of mass (recommended); 'x,y,z' - Vector to be subtracted from all PDB coordinates. 'n' - No centering, in which case (0,0,0) in the PDB space will map to the center of the EM volume. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "a"; token.restore = [['a'], ['a']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "O"; token.key_prefix = "--"; token.label = "Apply additional rotation"; token.help = "This can be used to modify the orientation of the atomic model by using O system of coordinates. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "tr0"; token.key_prefix = "--"; token.label = "Rotational matrix file"; token.help = "This file must contain the 3x4 transformation matrix to be applied to the PDB coordinates after centering. The translation vector (last column of the matrix) must be specified in Angstrom. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "rot_matrix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "set_apix_value"; token.key_prefix = "--"; token.label = "Set header pixel size"; token.help = "Set pixel size in header of the ouput map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "quiet"; token.key_prefix = "--"; token.label = "Silent mode"; token.help = "Does not print any information to the monitor. Verbose is the default. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "angular_distribution"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Angular Distribution"; sxcmd.short_info = "Generate a chimera .bild file for the visual representation of the resulting projection parameters."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "params_file"; token.key_prefix = ""; token.label = "Projection parameters"; token.help = "File containing the 3D projection parameters "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_folder"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output folder name "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "File prefix"; token.help = "Prefix for the output files - None uses the same name as the params file - Existing files will be overwritten "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Symmetry - c0 creates full sphere distribution; XXX_full, e.g. c1_full, c4_full, icos_full, creates a histogram for both hemispheres. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Particle box size [Pixels]"; token.help = "Box size "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "256"; token.restore = [['256'], ['256']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size of the project "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "particle_radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Particle radius "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "120"; token.restore = [['120'], ['120']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Angular sampling step [Degrees]"; token.help = "Angular step size in degree - Low deltas combined with low symmetry might crash chimera session "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "method"; token.key_prefix = "--"; token.label = "Distribution method"; token.help = "Method used to create the reference angles (S or P or M) "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'M', 'P'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "dpi"; token.key_prefix = "--"; token.label = "Plot DPI"; token.help = "Dpi for the legend plot "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "144"; token.restore = [['144'], ['144']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nth_percentile"; token.key_prefix = "--"; token.label = "Nth percentil"; token.help = "Use the value of the nth percentile of the radius distribution for normalization "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "99"; token.restore = [['99'], ['99']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "old"; token.key_prefix = "--"; token.label = "Use old"; token.help = "Use the old color scheme and normalization. The old style was normalizing the maximum length instead of the nth percentile and used a blue to green instead of a blue over green to yellow color scheme. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_batch"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Batch Pipeline Execution"; sxcmd.short_info = "Run jobs that wait with the execution on each other."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_viper"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "submission_command"; token.key_prefix = ""; token.label = "Submission command"; token.help = "Submission command, e.g., qsub, qsub -V, sbatch, bash "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "Pipeline directory"; token.help = "Directory containin the pipeline submission files "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hold_flag"; token.key_prefix = "--"; token.label = "Hold flag"; token.help = "Hold flag for the submission command, e.g. -hold_jid=, --wait=, --dependency=afterany:; Default is None and should be used in combination with a local execution with bash "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "first_hold_number"; token.key_prefix = "--"; token.label = "First hold number"; token.help = "Wait number for the first job that is submitted. By default, the first job will not wait for others "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "3D Refinement"; sxcmd.short_info = "Performs 3D structure refinement using a quasi-Maximum Likelihood approach."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Input image stack. Required for new runs, not for restarts. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. If not given, the program will use name master_DATA_AND_TIME. For standard continuation run, local refinement from iteration, and final reconstruction only, the directory must exist and further iterations will be written to this directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "initial_volume"; token.key_prefix = ""; token.label = "Initial 3D reference"; token.help = "The 3D reference used in the initial iteration of 3D refinement. Required only for new runs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "initialshifts"; token.key_prefix = "--"; token.label = "Read shifts from header"; token.help = "Start refinement using translation parameters located in the input file header to jumpstart the procedure. Specific to standard run mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = True; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_prealignment"; token.key_prefix = "--"; token.label = "Skip the 2D pre-alignment step"; token.help = "Indicate if pre-alignment should be omitted.  By default it is performed in order to find initial translation parameters. This accelerates convergence of the program. Do not use 2D pre-alignment if images are already centered. Specific to standard run modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['initialshifts', 'False', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('initialshifts', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "center_method"; token.key_prefix = "--"; token.label = "Centering method"; token.help = "Method for centering averages during initial 2D prealignment of data (0: no centering; -1: average shift method; For 1-7, see center_2D in utilities.py). Specific to standard run modes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_radius"; token.key_prefix = "--"; token.label = "Target particle radius [Pixels]"; token.help = "For 2D prealignment, images will be shrunk or enlarged to this radius. Specific to standard run modes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "29"; token.restore = [['29'], ['29']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial map used to start the refinement. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "25.0"; token.restore = [['25.0'], ['25.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "7.5"; token.restore = [['7.5', '15', '3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_gui_meridien"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "3D Refinement Assessment"; sxcmd.short_info = "GUI tool to assess 3D Refinement based on outputs of sp_meridien."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = False

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sxresolution"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Compute mFSC"; sxcmd.short_info = "Compute overall and local resolution measures using a pair of half-maps."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = True; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "firstvolume"; token.key_prefix = ""; token.label = "Volume #1"; token.help = "First unfiltered half-map.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "secondvolume"; token.key_prefix = ""; token.label = "Volume #2"; token.help = "Second unfiltered half-map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maskfile"; token.key_prefix = ""; token.label = "3D mask"; token.help = "Defines the region within which FSCM will be computed. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outputdir"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Directory where output files will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Mask radius"; token.help = "If there is no maskfile, sphere with r=radius will be used. By default, the radius is nx/2-wn. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['maskfile', 'None', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('maskfile', []).append([token.key_base, 'None', 'False'])
		token = SXcmd_token(); token.key_base = "wn"; token.key_prefix = "--"; token.label = "Window size"; token.help = "Size of window within which local real-space FSC is computed. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "15"; token.restore = [['15'], ['15']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "local_fsc"; token.key_prefix = "--"; token.label = "Compute local resolution"; token.help = "Set to 1 to compute local resolution volume. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0', '1'], ['0', '1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sigmag"; token.key_prefix = "--"; token.label = "Sigma of Gaussian window"; token.help = "Sigma of the Fourier space Gaussian window in pixels. Local resolution values are computed within small windowed areas (size wn^15). Due to small sample size the values are inaccurate and outcome tends to be noisy. It is thus suggested to use broader Gaussian window when local resolution is computed, say sigmag=3.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "step"; token.key_prefix = "--"; token.label = "Shell step"; token.help = "Shell step in Fourier size in pixels (integer). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "lfi"; token.key_prefix = "--"; token.label = "Inner radius"; token.help = "First Fourier index from which to begin calculation (in Fourier pixels) "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hfi"; token.key_prefix = "--"; token.label = "Outer radius"; token.help = "Last Fourier index to end calculation (in Fourier pixels). Default radius is nx2-2. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "significance"; token.key_prefix = "--"; token.label = "Significance level"; token.help = "Significance level for the upper confidence interval "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.99"; token.restore = [['0.99'], ['0.99']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ndf_reduce"; token.key_prefix = "--"; token.label = "Number of asymmetric units"; token.help = "Reduction of number of degrees of freedom due to point group symmetry, for example for D3 set to 6. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "cutoff"; token.key_prefix = "--"; token.label = "FSC criterion"; token.help = "Resolution cut-off for FSCM confidence interval. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.143"; token.restore = [['0.143'], ['0.143']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nthreads"; token.key_prefix = "--"; token.label = "Number of threads"; token.help = "Number of threads (mainly for 3D FFT). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_process"; sxcmd.subname = ""; sxcmd.mode = "combinemaps"; sxcmd.subset_config = "halfset maps"; sxcmd.label = "PostRefiner"; sxcmd.short_info = "Post-refine a pair of unfiltered odd & even halfset maps by combining them, then enhancing the high frequencies (Halfset Maps Mode). B-factor can be automatically estimated from the unfiltered halfset maps. This mode requires two arguments; use unfiltered halfset maps produced by MERIDIEN."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "combinemaps"; token.key_prefix = "--"; token.label = "Post-refine halfset volumes"; token.help = "Post-refine a pair of unfiltered odd & even halfset maps by combining them, then enhancing the high frequencies (Halfset Maps Mode). B-factor can be automatically estimated from the unfiltered halfset maps. This mode requires two arguments; use unfiltered halfset maps produced by MERIDIEN."; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "firstvolume"; token.key_prefix = ""; token.label = "First unfiltered halfset map"; token.help = "As generated by sxmeridien."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "secondvolume"; token.key_prefix = ""; token.label = "Second unfiltered halfset map"; token.help = "As generated by sxmeridien."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Specify path to the output directory for PostRefiner process. By default, the program uses the current directory. However, GUI requires the path other than the current directory. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size of input data. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "Path to user-provided mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "do_adaptive_mask"; token.key_prefix = "--"; token.label = "Apply adaptive mask"; token.help = "Program creates mask adaptively with given density threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mask', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mask', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', '-9999.0', 'False'], ['nsigma', '1.0', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, '-9999.0', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, '1.0', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. If the value changed from the default, the option will be ignored and the threshold will be set according to nsigma method under the Advanced tab. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', '1.0', 'False'], ['use_mol_mass', 'False', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = "-9999.0"; token.restore = [['-9999.0'], ['-9999.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, '1.0', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is larger than -9999.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', '-9999.0', 'False'], ['use_mol_mass', 'False', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, '-9999.0', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gauss\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "do_approx"; token.key_prefix = "--"; token.label = "Approximate values"; token.help = "Approximate the values of the soft edge area instead of using the exact values. This will lead to a less smoothened mask, but will mirror the previous behaviour. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "mtf"; token.key_prefix = "--"; token.label = "MTF file"; token.help = "Path to file contains the MTF (modulation transfer function) of the detector used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mtf"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fsc_adj"; token.key_prefix = "--"; token.label = "Apply FSC-based low-pass filter"; token.help = "Applies an FSC-based low-pass filter to the merged map before the B-factor estimation. Effective only in Halfset Volumes Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "B_enhance"; token.key_prefix = "--"; token.label = "B-factor enhancement"; token.help = "0.0: program automatically estimates B-factor using power spectrum at frequencies from B_start (usually 10 Angstrom) to the resolution determined by FSC143 (valid only in Halfset Volumes Mode; Non-zero positive value: program use the given value [A^2] to enhance map); -1.0: B-factor is not applied. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "B_start"; token.key_prefix = "--"; token.label = "B-factor estimation lower limit [A]"; token.help = "Frequency in Angstrom defining lower boundary of B-factor estimation. Effective only in Halfset Volumes Mode with --B_enhance=0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['B_enhance', '0.0', 'False']]; token.default = "10.0"; token.restore = [['10.0'], ['10.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('B_enhance', []).append([token.key_base, '0.0', 'False'])
		token = SXcmd_token(); token.key_base = "B_stop"; token.key_prefix = "--"; token.label = "B-factor estimation upper limit [A]"; token.help = "Frequency in Angstrom defining upper boundary of B-factor estimation. Recommended to set the upper boundary to the frequency where FSC is smaller than 0.0. Effective only in Halfset Volumes Mode with --B_enhance=0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['B_enhance', '0.0', 'False']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('B_enhance', []).append([token.key_base, '0.0', 'False'])
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency [A]"; token.help = "0.0: low-pass filter to resolution (valid only in Halfset Volumes Mode); A value larger than 0.5: low-pass filter to the value in Angstrom; -1.0: no low-pass filter. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off. Effective only when --fl option is not -1.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.01"; token.restore = [['0.01'], ['0.01']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "output"; token.key_prefix = "--"; token.label = "Output file name"; token.help = "File name of output final post-refined structure. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol_combined.hdf"; token.restore = [['vol_combined.hdf'], ['vol_combined.hdf']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sxresolution"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Compute mFSC"; sxcmd.short_info = "Compute overall and local resolution measures using a pair of half-maps."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = True; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "firstvolume"; token.key_prefix = ""; token.label = "Volume #1"; token.help = "First unfiltered half-map.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "secondvolume"; token.key_prefix = ""; token.label = "Volume #2"; token.help = "Second unfiltered half-map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "maskfile"; token.key_prefix = ""; token.label = "3D mask"; token.help = "Defines the region within which FSCM will be computed. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outputdir"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Directory where output files will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Mask radius"; token.help = "If there is no maskfile, sphere with r=radius will be used. By default, the radius is nx/2-wn. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['maskfile', 'None', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('maskfile', []).append([token.key_base, 'None', 'False'])
		token = SXcmd_token(); token.key_base = "wn"; token.key_prefix = "--"; token.label = "Window size"; token.help = "Size of window within which local real-space FSC is computed. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "15"; token.restore = [['15'], ['15']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "local_fsc"; token.key_prefix = "--"; token.label = "Compute local resolution"; token.help = "Set to 1 to compute local resolution volume. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0', '1'], ['0', '1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sigmag"; token.key_prefix = "--"; token.label = "Sigma of Gaussian window"; token.help = "Sigma of the Fourier space Gaussian window in pixels. Local resolution values are computed within small windowed areas (size wn^15). Due to small sample size the values are inaccurate and outcome tends to be noisy. It is thus suggested to use broader Gaussian window when local resolution is computed, say sigmag=3.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "step"; token.key_prefix = "--"; token.label = "Shell step"; token.help = "Shell step in Fourier size in pixels (integer). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "lfi"; token.key_prefix = "--"; token.label = "Inner radius"; token.help = "First Fourier index from which to begin calculation (in Fourier pixels) "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hfi"; token.key_prefix = "--"; token.label = "Outer radius"; token.help = "Last Fourier index to end calculation (in Fourier pixels). Default radius is nx2-2. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "significance"; token.key_prefix = "--"; token.label = "Significance level"; token.help = "Significance level for the upper confidence interval "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.99"; token.restore = [['0.99'], ['0.99']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ndf_reduce"; token.key_prefix = "--"; token.label = "Number of asymmetric units"; token.help = "Reduction of number of degrees of freedom due to point group symmetry, for example for D3 set to 6. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "cutoff"; token.key_prefix = "--"; token.label = "FSC criterion"; token.help = "Resolution cut-off for FSCM confidence interval. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.143"; token.restore = [['0.143'], ['0.143']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nthreads"; token.key_prefix = "--"; token.label = "Number of threads"; token.help = "Number of threads (mainly for 3D FFT). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_header"; sxcmd.subname = ""; sxcmd.mode = "import"; sxcmd.subset_config = ""; sxcmd.label = "Import Projection Parameters"; sxcmd.short_info = "Import projection orientation parameters to the header of the input stack. (Five columns: phi, theta, pshi, sx, sy). These parameters are required by 'Local Refinement from Stack' mode."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "import"; token.key_prefix = "--"; token.label = "Import parameters"; token.help = "Import parameters from file. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Path to input image stack. The stack can be either bdb or hdf. However, the GUI supports only bdb. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "params"; token.key_prefix = "--"; token.label = "Target parameters"; token.help = "List of parameters names (i.e. image header entry keys) to perform operations on (e.g. 'parm1 parm2 ...'). "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [['xform.projection'], ['xform.projection']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = "local_refinement"; sxcmd.subset_config = "stack"; sxcmd.label = "Local Refinement from Stack"; sxcmd.short_info = "Perform local refinement in which the restricted search begins from the user-provided orientation parameters stored in image headers. Note delta has to be less than or equal to 3.75[A]."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "local_refinement"; token.key_prefix = "--"; token.label = "Perform local refinement"; token.help = "Perform local refinement starting from user-provided orientation parameters stored in the header of input image stack. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "The stack must have 3D orientation parameters (xform.projection) stored in image headers. They can be imported using sp_header.py."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. If not given, the program will use name master_DATA_AND_TIME. For standard continuation run, local refinement from iteration, and final reconstruction only, the directory must exist and further iterations will be written to this directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial volume. For local refinement, the program automatically calculates the initial resolution using provided orientation parameters."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [[-1.0], [-1.0]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. For local refinement, the value has to be less than or equal to 3.75."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = "local_refinement"; sxcmd.subset_config = "iteration"; sxcmd.label = "Restart Local Refinement"; sxcmd.short_info = "Restart local refinement after the last fully finished iteration of meridien run or local refinement run. One can change some parameters, but MPI settings have to be the same as in the original run."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "local_refinement"; token.key_prefix = "--"; token.label = "Perform local refinement"; token.help = "Perform local refinement starting from user-provided orientation parameters stored in the header of input image stack. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Directory"; token.help = "This directory must exist. The results will be written there."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial volume. For local refinement, the program automatically calculates the initial resolution using provided orientation parameters."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [[-1.0], [-1.0]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. For local refinement, the value has to be less than or equal to 3.75."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = "restart"; sxcmd.label = "3D Refinement Restart"; sxcmd.short_info = "Restart 3D refinement after the last fully finished iteration of meridien run or local refinement run. One can change some parameters, but MPI settings have to be the same."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Output Directory"; token.help = "This directory must exist. The results will be written here."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial map used to start the refinement. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "25.0"; token.restore = [['25.0'], ['25.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "7.5"; token.restore = [['7.5', '15', '3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = "do_final"; sxcmd.subset_config = ""; sxcmd.label = "Final 3D Reconstruction Only"; sxcmd.short_info = "Compute a final 3D reconstruction using either select or best resolution iteration of meridien."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "do_final"; token.key_prefix = "--"; token.label = "Iteration number"; token.help = "Specify the iteration which to perform final reconstruction. By setting to 0, program searches for the iteration which had best resolution, then performs correponding final reconstruction. Value must be zero or positive. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Output Directory"; token.help = "This directory must exist. In this mode information is read from files in this directory and the results will be written there."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien_alpha"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = "beta"; sxcmd.label = "3D Refinement (Alpha)"; sxcmd.short_info = "Performs 3D structure refinement for helical specimen using a quasi-Maximum Likelihood approach."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Input image stack. Required for new runs, not for restarts. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. If not given, the program will use name master_DATA_AND_TIME. For standard continuation run, local refinement from iteration, and final reconstruction only, the directory must exist and further iterations will be written to this directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "initial_volume"; token.key_prefix = ""; token.label = "Initial 3D reference"; token.help = "The 3D reference used in the initial iteration of 3D refinement. Required only for new runs. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "initialshifts"; token.key_prefix = "--"; token.label = "Read shifts from header"; token.help = "Start refinement using translation parameters located in the input file header to jumpstart the procedure. Specific to standard run mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = True; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "skip_prealignment"; token.key_prefix = "--"; token.label = "Skip the 2D pre-alignment step"; token.help = "Indicate if pre-alignment should be omitted.  By default it is performed in order to find initial translation parameters. This accelerates convergence of the program. Do not use 2D pre-alignment if images are already centered. Specific to standard run modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['initialshifts', 'False', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('initialshifts', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "center_method"; token.key_prefix = "--"; token.label = "Centering method"; token.help = "Method for centering averages during initial 2D prealignment of data (0: no centering; -1: average shift method; For 1-7, see center_2D in utilities.py). Specific to standard run modes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_radius"; token.key_prefix = "--"; token.label = "Target particle radius [Pixels]"; token.help = "For 2D prealignment, images will be shrunk or enlarged to this radius. Specific to standard run modes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "29"; token.restore = [['29'], ['29']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial map used to start the refinement. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "25.0"; token.restore = [['25.0'], ['25.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "7.5"; token.restore = [['7.5', '15', '3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta_min"; token.key_prefix = "--"; token.label = "Theta min [degree]"; token.help = "Minimum out-of-plane rotation value to use for the reference projection angles. Default is the full range from 0 to 180. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta_max"; token.key_prefix = "--"; token.label = "Theta max [degree]"; token.help = "Maximum out-of-plane rotation value to use for the reference projection angles. Default is the full range from 0 to 180. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "angle_method"; token.key_prefix = "--"; token.label = "Even angle method"; token.help = "Method to use for even angle creation (S, M, P). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'M', 'P'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chunk_by"; token.key_prefix = "--"; token.label = "Group name for chunks"; token.help = "Group the particles by the specified header name before splitting them into chunks. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "ptcl_source_image"; token.restore = [['ptcl_source_image'], ['filament_id']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_by"; token.key_prefix = "--"; token.label = "Outlier group ID"; token.help = "Group the particles by the header name for outlier detection. By default do not perform outlier detection. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['filament_id']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [px]"; token.help = "This is used to normalize the particles in case of filaments. A rectangular mask will be used instead of a circular one. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "helical_rise"; token.key_prefix = "--"; token.label = "Helical rise [A]"; token.help = "Helical rise used to limit the shift along the helical axis to +-rise/2 "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "plot_ang_dist"; token.key_prefix = "--"; token.label = "Plot angular distribution"; token.help = "Plot the angular distribution in every iteration. This will take some time for high symmetries."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_tracker"; token.key_prefix = "--"; token.label = "Outlier Tracker"; token.help = "Tracker from a previous MERIDIEN directory that ran with the same stack to skip the OUTLIER finding. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "params_any_json"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "howmany"; token.key_prefix = "--"; token.label = "Howmany value"; token.help = "Howmany neighbors shouls be taken into account for the exhaustive step of the refinementHowmany neighbors shouls be taken into account for the exhaustive step of the refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien_alpha"; sxcmd.subname = ""; sxcmd.mode = "local_refinement"; sxcmd.subset_config = "stack_beta"; sxcmd.label = "Local Refinement from Stack (Beta)"; sxcmd.short_info = "Perform local refinement in which the restricted search begins from the user-provided orientation parameters stored in image headers. Note delta has to be less than or equal to 3.75[A]."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "local_refinement"; token.key_prefix = "--"; token.label = "Perform local refinement"; token.help = "Perform local refinement starting from user-provided orientation parameters stored in the header of input image stack. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "The stack must have 3D orientation parameters (xform.projection) stored in image headers. They can be imported using sp_header.py."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. If not given, the program will use name master_DATA_AND_TIME. For standard continuation run, local refinement from iteration, and final reconstruction only, the directory must exist and further iterations will be written to this directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial volume. For local refinement, the program automatically calculates the initial resolution using provided orientation parameters."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [[-1.0], [-1.0]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. For local refinement, the value has to be less than or equal to 3.75."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta_min"; token.key_prefix = "--"; token.label = "Theta min [degree]"; token.help = "Minimum out-of-plane rotation value to use for the reference projection angles. Default is the full range from 0 to 180. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta_max"; token.key_prefix = "--"; token.label = "Theta max [degree]"; token.help = "Maximum out-of-plane rotation value to use for the reference projection angles. Default is the full range from 0 to 180. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "angle_method"; token.key_prefix = "--"; token.label = "Even angle method"; token.help = "Method to use for even angle creation (S, M, P). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'M', 'P'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chunk_by"; token.key_prefix = "--"; token.label = "Group name for chunks"; token.help = "Group the particles by the specified header name before splitting them into chunks. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "ptcl_source_image"; token.restore = [['ptcl_source_image'], ['filament_id']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_by"; token.key_prefix = "--"; token.label = "Outlier group ID"; token.help = "Group the particles by the header name for outlier detection. By default do not perform outlier detection. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['filament_id']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [px]"; token.help = "This is used to normalize the particles in case of filaments. A rectangular mask will be used instead of a circular one. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "helical_rise"; token.key_prefix = "--"; token.label = "Helical rise [A]"; token.help = "Helical rise used to limit the shift along the helical axis to +-rise/2 "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "plot_ang_dist"; token.key_prefix = "--"; token.label = "Plot angular distribution"; token.help = "Plot the angular distribution in every iteration. This will take some time for high symmetries."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_tracker"; token.key_prefix = "--"; token.label = "Outlier Tracker"; token.help = "Tracker from a previous MERIDIEN directory that ran with the same stack to skip the OUTLIER finding. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "params_any_json"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "howmany"; token.key_prefix = "--"; token.label = "Howmany value"; token.help = "Howmany neighbors shouls be taken into account for the exhaustive step of the refinementHowmany neighbors shouls be taken into account for the exhaustive step of the refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien_alpha"; sxcmd.subname = ""; sxcmd.mode = "local_refinement"; sxcmd.subset_config = "iteration_beta"; sxcmd.label = "Restart Local Refinement (Beta)"; sxcmd.short_info = "Restart local refinement after the last fully finished iteration of meridien run or local refinement run. One can change some parameters, but MPI settings have to be the same as in the original run."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "local_refinement"; token.key_prefix = "--"; token.label = "Perform local refinement"; token.help = "Perform local refinement starting from user-provided orientation parameters stored in the header of input image stack. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Directory"; token.help = "This directory must exist. The results will be written there."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial volume. For local refinement, the program automatically calculates the initial resolution using provided orientation parameters."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [[-1.0], [-1.0]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. For local refinement, the value has to be less than or equal to 3.75."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta_min"; token.key_prefix = "--"; token.label = "Theta min [degree]"; token.help = "Minimum out-of-plane rotation value to use for the reference projection angles. Default is the full range from 0 to 180. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "theta_max"; token.key_prefix = "--"; token.label = "Theta max [degree]"; token.help = "Maximum out-of-plane rotation value to use for the reference projection angles. Default is the full range from 0 to 180. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['90']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "angle_method"; token.key_prefix = "--"; token.label = "Even angle method"; token.help = "Method to use for even angle creation (S, M, P). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'M', 'P'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chunk_by"; token.key_prefix = "--"; token.label = "Group name for chunks"; token.help = "Group the particles by the specified header name before splitting them into chunks. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "ptcl_source_image"; token.restore = [['ptcl_source_image'], ['filament_id']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_by"; token.key_prefix = "--"; token.label = "Outlier group ID"; token.help = "Group the particles by the header name for outlier detection. By default do not perform outlier detection. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['filament_id']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filament_width"; token.key_prefix = "--"; token.label = "Filament width [px]"; token.help = "This is used to normalize the particles in case of filaments. A rectangular mask will be used instead of a circular one. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "filament_width"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "helical_rise"; token.key_prefix = "--"; token.label = "Helical rise [A]"; token.help = "Helical rise used to limit the shift along the helical axis to +-rise/2 "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "plot_ang_dist"; token.key_prefix = "--"; token.label = "Plot angular distribution"; token.help = "Plot the angular distribution in every iteration. This will take some time for high symmetries."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outlier_tracker"; token.key_prefix = "--"; token.label = "Outlier Tracker"; token.help = "Tracker from a previous MERIDIEN directory that ran with the same stack to skip the OUTLIER finding. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "params_any_json"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "howmany"; token.key_prefix = "--"; token.label = "Howmany value"; token.help = "Howmany neighbors shouls be taken into account for the exhaustive step of the refinementHowmany neighbors shouls be taken into account for the exhaustive step of the refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4"; token.restore = [['4'], ['4']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien_alpha"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = "restart_beta"; sxcmd.label = "3D Refinement Restart (Beta)"; sxcmd.short_info = "Restart 3D refinement after the last fully finished iteration of meridien run or local refinement run. One can change some parameters, but MPI settings have to be the same."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Output Directory"; token.help = "This directory must exist. The results will be written here."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial map used to start the refinement. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "25.0"; token.restore = [['25.0'], ['25.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "7.5"; token.restore = [['7.5', '15', '3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "plot_ang_dist"; token.key_prefix = "--"; token.label = "Plot angular distribution"; token.help = "Plot the angular distribution in every iteration. This will take some time for high symmetries."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien_alpha"; sxcmd.subname = ""; sxcmd.mode = "do_final"; sxcmd.subset_config = ""; sxcmd.label = "Final 3D Reconstruction Only (Beta)"; sxcmd.short_info = "Compute a final 3D reconstruction using either select or best resolution iteration of meridien."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "do_final"; token.key_prefix = "--"; token.label = "Iteration number"; token.help = "Specify the iteration which to perform final reconstruction. By setting to 0, program searches for the iteration which had best resolution, then performs correponding final reconstruction. Value must be zero or positive. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Output Directory"; token.help = "This directory must exist. In this mode information is read from files in this directory and the results will be written there."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "moon_eliminator"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Volume Adjustment"; sxcmd.short_info = "Eliminate moons or remove dust from the background of a 3D density map based on the expected molecular mass."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume_path"; token.key_prefix = ""; token.label = "Input volume path"; token.help = "Path to input volume file containing the 3D density map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Output pixel size [A]"; token.help = "The original pixel size of dataset. This must be the pixel size after resampling when resample_ratio != 1.0. That is, it will be the pixel size of the output map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_density_threshold', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_density_threshold', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "use_density_threshold"; token.key_prefix = "--"; token.label = "Use ad-hoc density threshold"; token.help = "Use user-provided ad-hoc density threshold, instead of computing the value from the molecular mass. Below this density value, the data is assumed not to belong to the main body of the particle density. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "moon_distance"; token.key_prefix = "--"; token.label = "Distance to the nearest moon [Pixels]"; token.help = "The moons further than this distance from the density surface will be elminated. The value smaller than the default is not recommended because it is difficult to avoid the stair-like gray level change at the edge of the density surface. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.0"; token.restore = [['3.0'], ['3.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Resample ratio"; token.help = "Specify a value larger than 0.0. By default, the program does not resample the input map (i.e. resample ratio is 1.0). Use this option maily to restore the original dimensions or pixel size of VIPER or R-VIPER model. Alternatively, specify the path to the output directory of an ISAC2 run. The program automatically extracts the resampling ratio used by the ISAC2 run. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "'1.0'"; token.restore = [["'1.0'"], ["'1.0'"]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Output box size [Pixels]"; token.help = "The x, y, and z dimensions of cubic area to be windowed from input 3D volume for output 3D volumes. This must be the box size after resampling when resample_ratio != 1.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "invert_handedness"; token.key_prefix = "--"; token.label = "Invert handedness"; token.help = "Invert the handedness of the 3D map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter resolution [A]"; token.help = "&gt;0.0: low-pass filter to the value in Angstrom; =-1.0: no low-pass filter. The program applies this low-pass filter before the moon elimination. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_volume_path_2nd"; token.key_prefix = ""; token.label = "Second input volume path"; token.help = "Path to second input volume file containing the 3D density map. Use this option to create a mask from the sum of two MERIDIEN half-set maps. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. By default, it is set to half of --moon_distance so that the voxels with 1.0 values in the mask are same as the hard-edged molecular-mass binary volume. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) 'cosine' for cosine soft-edged (used in PostRefiner) and (2) 'gauss' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outputs_root"; token.key_prefix = "--"; token.label = "Root name of outputs"; token.help = "Specify the root name of all outputs. It cannot be empty string or only white spaces. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol3d"; token.restore = [['vol3d'], ['vol3d']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resampled_shift3d"; token.key_prefix = "--"; token.label = "Providing resampled 3D shifts"; token.help = "Use this option when you are providing the resampled 3D shifts (using pixel size of outputs) when --resample_ratio!=1.0. By default, the program assums the provided shifts are not resampled. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_x"; token.key_prefix = "--"; token.label = "3D x-shift [Pixels]"; token.help = "3D x-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_y"; token.key_prefix = "--"; token.label = "3D y-shift [Pixels]"; token.help = "3D y-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_z"; token.key_prefix = "--"; token.label = "3D z-shift [Pixels]"; token.help = "3D z-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off in absolute frequency. The program applies this low-pass filter before the moon elimination. Effective only when --fl &gt; 0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "debug"; token.key_prefix = "--"; token.label = "Run with debug mode"; token.help = "Mainly for developers. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_mask"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Masking"; sxcmd.short_info = "Mask creation tool for 2D or 3D masks."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume"; token.key_prefix = ""; token.label = "Input image"; token.help = "Path to the 2D image or 3D Volume "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output direcory path "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "Output prefix"; token.help = "Prefix of the produced files "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "sp_mask"; token.restore = [['sp_mask'], ['sp_mask']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overwrite"; token.key_prefix = "--"; token.label = "Overwrite outputs"; token.help = "Overwrite the output mask in case it exists already. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A/px]"; token.help = "Pixel size of the volume. Used for filtering and the molcular mask threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['nsigma', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. This is used to calculate the binarization threshold automatically. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One cycle of dilation will add about 2 pixels to the mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking. If the width is 0, a binary mask is returned. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "allow_disconnected"; token.key_prefix = "--"; token.label = "Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fill_mask"; token.key_prefix = "--"; token.label = "Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nerosion"; token.key_prefix = "--"; token.label = "Number of erosions"; token.help = "Number of times to erode binarized volume. One cycle of erosion will remove about 2 pixels from the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "do_old"; token.key_prefix = "--"; token.label = "Old behaviour"; token.help = "Restore the old masking behaviour, which is a bit less smooth. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "low_pass_filter_resolution"; token.key_prefix = "--"; token.label = "Low pass filter resolution [A]"; token.help = "Low pass filter resolution in angstroms. If set, the volume will be filtered prior to create a mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "low_pass_filter_falloff"; token.key_prefix = "--"; token.label = "Low pass filter falloff [1/Pixel]"; token.help = "Low pass filter falloff in absolute frequencies. If set, the volume will be filtered prior to create a mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['low_pass_filter_resolution', 'none', 'True']]; token.default = "0.01"; token.restore = [['0.01'], ['0.01']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('low_pass_filter_resolution', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "use_second_mask"; token.key_prefix = "--"; token.label = "Use a second mask"; token.help = "ONLY A GUI OPTION. The second mask can be used to mask the first one after creation. This is useful to create soft edged regions of the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [True]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "second_mask"; token.key_prefix = "--"; token.label = "Second mask path"; token.help = "Path to the input second mask used for masking the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask_shape', 'none', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "second_mask_shape"; token.key_prefix = "--"; token.label = "Second mask shape"; token.help = "Shape of the second mask. Possible values: sphere, cylinder, cube. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'False']]; token.default = "none"; token.restore = [['none', 'sphere', 'cylinder', 'cube'], ['cylinder', 'none', 'sphere', 'cube']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "s_radius"; token.key_prefix = "--"; token.label = "Second - Radius of the mask"; token.help = "Radius of the sphere or cylinder for the second mask in pixels. Radius must be less than half of the volume dimension. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['second_mask_shape', 'cube', 'True'], ['use_second_mask', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'cube', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_nx"; token.key_prefix = "--"; token.label = "Second - X dimension of the mask"; token.help = "X Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_ny"; token.key_prefix = "--"; token.label = "Second - Y dimension of the mask"; token.help = "Y Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. If not provided, a squared volume with s_nx is assumed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False'], ['s_nx', 'none;Main', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_nx', []).append([token.key_base, 'none;Main', 'True'])
		token = SXcmd_token(); token.key_base = "s_nz"; token.key_prefix = "--"; token.label = "Second - Z dimension of the mask"; token.help = "Z Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. If not provided, a squared volume with s_nx is assumed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False'], ['s_nx', 'none;Main', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_nx', []).append([token.key_base, 'none;Main', 'True'])
		token = SXcmd_token(); token.key_base = "s_pixel_size"; token.key_prefix = "--"; token.label = "Second - Pixel size [A/px]"; token.help = "Pixel size of the second volume. Used for the molecular mass threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_use_mol_mass"; token.key_prefix = "--"; token.label = "Second - Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_threshold', 'none', 'False'], ['s_nsigma', 'none', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_mol_mass"; token.key_prefix = "--"; token.label = "Second - Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. This is used to calculate the binarization threshold automatically. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_use_mol_mass', 'True', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_threshold"; token.key_prefix = "--"; token.label = "Second - Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. If the value is lower-equal than the default, the option will be ignored and the threshold will be set according to nsigma method above. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_nsigma', 'none', 'False'], ['s_use_mol_mass', 'False', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_nsigma"; token.key_prefix = "--"; token.label = "Second - Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_threshold', 'none', 'False'], ['s_use_mol_mass', 'False', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_ndilation"; token.key_prefix = "--"; token.label = "Second - Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One round of erosion will add about 2 pixels to the mask "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_nerosion"; token.key_prefix = "--"; token.label = "Second - Number of erosions"; token.help = "Number of times to erode binarized volume. One round of erosion will remove about 2 pixels from the mask "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_edge_width"; token.key_prefix = "--"; token.label = "Second - Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_edge_type"; token.key_prefix = "--"; token.label = "Second - Soft-edge type"; token.help = "The type of soft-edge for the 3D mask. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['s_edge_width', '0', 'True']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "s_do_old"; token.key_prefix = "--"; token.label = "Second - Old behaviour"; token.help = "Restore the old masking behaviour, which is a bit less smooth. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['s_edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "s_allow_disconnected"; token.key_prefix = "--"; token.label = "Second - Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_fill_mask"; token.key_prefix = "--"; token.label = "Second - Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "s_invert"; token.key_prefix = "--"; token.label = "Second - Invert masking"; token.help = "If True, the mask will remove everything that is inside instead of leaving it. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "angular_distribution"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Angular Distribution"; sxcmd.short_info = "Generate a chimera .bild file for the visual representation of the resulting projection parameters."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "params_file"; token.key_prefix = ""; token.label = "Projection parameters"; token.help = "File containing the 3D projection parameters "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_folder"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output folder name "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "File prefix"; token.help = "Prefix for the output files - None uses the same name as the params file - Existing files will be overwritten "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Symmetry - c0 creates full sphere distribution; XXX_full, e.g. c1_full, c4_full, icos_full, creates a histogram for both hemispheres. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Particle box size [Pixels]"; token.help = "Box size "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "256"; token.restore = [['256'], ['256']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size of the project "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "particle_radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Particle radius "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "120"; token.restore = [['120'], ['120']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Angular sampling step [Degrees]"; token.help = "Angular step size in degree - Low deltas combined with low symmetry might crash chimera session "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "method"; token.key_prefix = "--"; token.label = "Distribution method"; token.help = "Method used to create the reference angles (S or P or M) "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'M', 'P'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "dpi"; token.key_prefix = "--"; token.label = "Plot DPI"; token.help = "Dpi for the legend plot "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "144"; token.restore = [['144'], ['144']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nth_percentile"; token.key_prefix = "--"; token.label = "Nth percentil"; token.help = "Use the value of the nth percentile of the radius distribution for normalization "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "99"; token.restore = [['99'], ['99']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "old"; token.key_prefix = "--"; token.label = "Use old"; token.help = "Use the old color scheme and normalization. The old style was normalizing the maximum length instead of the nth percentile and used a blue to green instead of a blue over green to yellow color scheme. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_process"; sxcmd.subname = ""; sxcmd.mode = "balance_angular_distribution"; sxcmd.subset_config = ""; sxcmd.label = "Balance Angular Distribution"; sxcmd.short_info = "Balance angular distribution of projection directions by removing excess particles, as determined by their angular histogram on a coarser grid, as specified by the angular_step option. "; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "balance_angular_distribution"; token.key_prefix = "--"; token.label = "Balance Angular Distribution"; token.help = "Balance angular distribution of projection directions by removing excess particles, as determined by their angular histogram on a coarser grid, as specified by the angular_step option. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "params_any_txt"; token.key_prefix = ""; token.label = "Projection parameters"; token.help = "Typically from MERIDIEN with a filename in the form of Refine3D/final_params_0##.txt"; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "selection_list"; token.key_prefix = ""; token.label = "Output selection list"; token.help = "Text file with a list of retained particle images"; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "max_occupy"; token.key_prefix = "--"; token.label = "Maximum orientations per reference angle"; token.help = "Maximum number of particles per reference angle. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "angstep"; token.key_prefix = "--"; token.label = "Angular increment"; token.help = "Angular step of reference angles. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "15"; token.restore = [['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "angular step of reference angles, i.e., number of bins of angular histogram. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_batch"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Batch Pipeline Execution"; sxcmd.short_info = "Run jobs that wait with the execution on each other."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "submission_command"; token.key_prefix = ""; token.label = "Submission command"; token.help = "Submission command, e.g., qsub, qsub -V, sbatch, bash "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "Pipeline directory"; token.help = "Directory containin the pipeline submission files "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hold_flag"; token.key_prefix = "--"; token.label = "Hold flag"; token.help = "Hold flag for the submission command, e.g. -hold_jid=, --wait=, --dependency=afterany:; Default is None and should be used in combination with a local execution with bash "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "first_hold_number"; token.key_prefix = "--"; token.label = "First hold number"; token.help = "Wait number for the first job that is submitted. By default, the first job will not wait for others "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_ctf_refine"; sxcmd.subname = "meridien"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "CTF refine (Meridien)"; sxcmd.short_info = "Refine the defocus per particle"; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "path_to_input_stack"; token.key_prefix = ""; token.label = "Input stack path"; token.help = "Path to input stack. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Folder to output files etc. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "refinement_dir"; token.key_prefix = ""; token.label = "Meridien directory"; token.help = "Path to Meridien directory.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "Path to mask"; token.help = "Path to mask used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "range"; token.key_prefix = "--"; token.label = "Defocus search range [um]"; token.help = "Search range +- around the current defocus value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.15"; token.restore = [['0.15'], ['0.15']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Defocus search delta [um]"; token.help = "Finest search delta. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0025"; token.restore = [['0.0025'], ['0.0025']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resolution"; token.key_prefix = "--"; token.label = "Nominal resolution [A]"; token.help = "Nominal resolution of the reconstruction. Low-pass Gaussian filter will be applied at this resolution. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size in Angstroms. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['resolution', 'None', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('resolution', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "number_part"; token.key_prefix = "--"; token.label = "Number of particles"; token.help = "Number of particles to process. Mainly for debugging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_ctf_refine"; sxcmd.subname = "manual"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "CTF refine (Stack)"; sxcmd.short_info = "Refine the defocus per particle"; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_meridien"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "path_to_input_stack"; token.key_prefix = ""; token.label = "Input stack path"; token.help = "Path to input stack. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Folder to output files etc. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "volume_path"; token.key_prefix = ""; token.label = "Path to volume"; token.help = "Path to volume.  "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "params_file"; token.key_prefix = ""; token.label = "Params file"; token.help = "Path to params file "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "Path to mask"; token.help = "Path to mask used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "range"; token.key_prefix = "--"; token.label = "Defocus search range [um]"; token.help = "Search range +- around the current defocus value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.15"; token.restore = [['0.15'], ['0.15']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Defocus search delta [um]"; token.help = "Finest search delta. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0025"; token.restore = [['0.0025'], ['0.0025']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resolution"; token.key_prefix = "--"; token.label = "Nominal resolution [A]"; token.help = "Nominal resolution of the reconstruction. Low-pass Gaussian filter will be applied at this resolution. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size in Angstroms. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['resolution', 'None', 'True']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('resolution', []).append([token.key_base, 'None', 'True'])
		token = SXcmd_token(); token.key_base = "number_part"; token.key_prefix = "--"; token.label = "Number of particles"; token.help = "Number of particles to process. Mainly for debugging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chunk"; token.key_prefix = "--"; token.label = "Path to chunk file"; token.help = "Path to chunk file "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "volume2"; token.key_prefix = "--"; token.label = "Path to second volume"; token.help = "Path to second half volume. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['chunk', 'None', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('chunk', []).append([token.key_base, 'None', 'True'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_header"; sxcmd.subname = ""; sxcmd.mode = "import"; sxcmd.subset_config = ""; sxcmd.label = "Import Projection Parameters"; sxcmd.short_info = "Import projection orientation parameters from a file (for example created by sxmeridien) to header of the input stack; they are required by 3DVARIABILITY.py and SORT3D_DEPTH Stack Mode."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "import"; token.key_prefix = "--"; token.label = "Import parameters"; token.help = "Import parameters from file. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "Path to input image stack. The stack can be either bdb or hdf. However, the GUI supports only bdb. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "params"; token.key_prefix = "--"; token.label = "Target parameters"; token.help = "List of parameters names (i.e. image header entry keys) to perform operations on (e.g. 'parm1 parm2 ...'). "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [['xform.projection'], ['xform.projection']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_3dvariability"; sxcmd.subname = ""; sxcmd.mode = "symmetrize"; sxcmd.subset_config = ""; sxcmd.label = "3D Variability Preprocess"; sxcmd.short_info = "Prepare input stack for handling symmetry. Please skip this preparation step if the structure is asymmetrical (i.e. c1), since it is required only when the structure has internal symmetry. Notice this step can be run with only one CPU and there is no MPI version for it. "; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "symmetrize"; token.key_prefix = "--"; token.label = "Symmetrise input stack"; token.help = "Prepare input stack for handling symmetry. Please skip this preparation step if the structure is asymmetrical (i.e. c1), since it is required only when the structure has internal symmetry. Notice this step can be run with only one CPU and there is no MPI version for it. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prj_stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "The images must contain the 3D orientation parameters in headers and optionally also CTF information. The output image stack is bdb:sdata. Please use it as an input image stack of sx3dvariability."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Specify path to the output directory of 3D Variability. By default, the program uses the current directory. However, GUI requires the path other than the current directory. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "If the structure has symmetry higher than c1, the command requires symmetrization of the dataset, using --symmetrize option, before computing 3D variability. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_3dvariability"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "3D Variability Estimation"; sxcmd.short_info = "Calculate 3D variability using a set of aligned projection images as an input."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "prj_stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "The images must containt the 3D orientation parameters in the header and optionally CTF information. If the structure has a symmetry higher than c1, please specify the image stack which is prepared by the symmetrization using --symmetrize option. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Specify path to the output directory of 3D Variability. By default, the program uses the current directory. However, GUI requires the path other than the current directory. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "var3D"; token.key_prefix = "--"; token.label = "Output 3D variability"; token.help = "Specify a file name to indicate if the program should write the reconstructed 3D variability map to the disk. The 3D volume will contain, for each voxel, a measure of the variability in the dataset. Careful, time consuming! "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ave3D"; token.key_prefix = "--"; token.label = "Output 3D average"; token.help = "Specify a file name to indicate if the program should write the reconstructed 3D average map to the disk. The 3D map will be reconstructed from projections averaged within respective angular neighbourhood. It should be used to assess the resolvability and possible artifacts of the variability map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp"; token.key_prefix = "--"; token.label = "Number of projections"; token.help = "Specify the number of images from the angular neighbourhood that will be used to estimate 2D variance for each projection data. The larger the number the less noisy the estimate, but the lower the resolution. Usage of large number also results in rotational artifacts in variances that will be visible in 3D variability map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "100"; token.restore = [['100'], ['100']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "If the structure has symmetry higher than c1, the command requires symmetrization of the dataset, using --symmetrize option, before computing 3D variability. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "CTF"; token.key_prefix = "--"; token.label = "Use CTF correction"; token.help = "If set to true, CTF correction will be applied using the parameters found in the image headers. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = True; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overhead"; token.key_prefix = "--"; token.label = "Python overhead memory"; token.help = "User estimates python overhead memory per CPU. The default value is 0.5 GB per CPU. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "var2D"; token.key_prefix = "--"; token.label = "Output 2D variances"; token.help = "Specify a file name to indicate if the program should write the stack of computed 2D variances to the disk. Useful for debugging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ave2D"; token.key_prefix = "--"; token.label = "Output 2D averages"; token.help = "Specify a file name to indicate if the program should write the stack of computed 2D averages to the disk. Useful for debugging. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "VAR"; token.key_prefix = "--"; token.label = "Input stack consists of 2D variances"; token.help = "Stack on input consists of 2D variances. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "no_norm"; token.key_prefix = "--"; token.label = "Apply normalization"; token.help = "Indicate if normalization should be applied or not. By default, apply normalization. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "npad"; token.key_prefix = "--"; token.label = "Image padding factor"; token.help = "The number of time to pad the original images. The images will be padded to achieve the original size times this option. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "decimate"; token.key_prefix = "--"; token.label = "Image decimate factor"; token.help = "Reduce images by this factor and change the pixel size. Specify a non-zero positive float value smaller than 1.0. By default, it does not change size of images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.25"; token.restore = [['0.25'], ['0.25']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency"; token.help = "Stop-band frequency of the low-pass filter to be applied to the images prior to variability calculation but after decimation. Specify it in absolute frequency (&gt; 0.0 and &lt;= 0.5). By default, no filtering. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "abs_freq"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off"; token.help = "Fall-off width of the low-pass filter to be applied to the images prior to variability calculation but after decimation. Specify it in absolute frequency (&gt; 0.0 and &lt;= 0.5). 0.01 works in most of cases. Effective only with --fl &gt; 0.0 and --aa &gt; 0.0 has to be specified. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '0.0', 'True']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '0.0', 'True'])
		token = SXcmd_token(); token.key_base = "window"; token.key_prefix = "--"; token.label = "Target image size [Pixels]"; token.help = "Window (or clip) images using the specified size without changing pixel size. It is relative to the orignal window size. The target window size cannot be larger than the orignal image size. By default, use the original particle image size. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nvec"; token.key_prefix = "--"; token.label = "Number of eigenvectors"; token.help = "By default, no PCA will be calculated. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_sort3d_depth"; sxcmd.subname = ""; sxcmd.mode = "refinement_dir"; sxcmd.subset_config = ""; sxcmd.label = "3D Clustering from Iteration - SORT3D_DEPTH"; sxcmd.short_info = "Initialize from a given iteration of meridien run using the associated parameters, i.e., full set of orientation parameters per image, including orientation probabilities, normalizations and so on."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "refinement_dir"; token.key_prefix = "--"; token.label = "Meridien refinement directory"; token.help = "A string denotes meridien 3D refinement directory. Sorting switches to meridien iteration mode when specified. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "A string denotes output directory for 3D sorting. It can be either existing or non-existing. By default, the program uses sort3d_DATA_AND_TIME for the name. Here, you can find a log.txt that describes the sequences of computations in the program. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "niter_for_sorting"; token.key_prefix = "--"; token.label = "Iteration ID of 3D refinement for importing data"; token.help = "By default, the program uses the iteration at which refinement achieved the best resolution. Option is valid only for meridien iteration mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask"; token.help = "A string denotes file path of the global 3D mask for clustering. Imported from 3D refinement unless user wishes a different one in meridien iteration mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "focus"; token.key_prefix = "--"; token.label = "Focus 3D mask"; token.help = "A string denotes file path of a binary 3D mask for focused clustering. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Estimated particle radius [Pixels]"; token.help = "A integer value that is smaller than half of the box size. Imported from refinement unless user wishes a different one in meridien iteration mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "A string denotes point group symmetry of the macromolecular structure. Imported from refinement unless the user wishes a different one in meridien iteration mode. Require specification in stack mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp"; token.key_prefix = "--"; token.label = "Number of images per group"; token.help = "User expected group size in integer. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1000"; token.restore = [['1000'], ['1000']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp_split_rate"; token.key_prefix = "--"; token.label = "Group splitting rate"; token.help = "An integer value denotes split rate of the group size(--img_per_grp). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "minimum_grp_size"; token.key_prefix = "--"; token.label = "Minimum size of reproducible class"; token.help = "The minimum size of selected or accounted clusters as well as the minimum group size constraint in MGSKmeans. However this value must be smaller than the number of images per a group (img_per_grp). By default, the program uses half number of the images per group.  "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "do_swap_au"; token.key_prefix = "--"; token.label = "Swap flag"; token.help = "A boolean flag to control random swapping a certain number of accounted elements per cluster with the unaccounted elements. If the processing with the default values are extremely slow or stalled, please use this --do_swap_au option and set --swap_ratio to a large value (15.0[%] is a good start point). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "swap_ratio"; token.key_prefix = "--"; token.label = "Swap percentage [%]"; token.help = "the percentage of images for swapping ranges between 0.0[%] and 50.0[%]. Option valid only with --do_swap_au. Without --do_swap_au, the program automatically sets --swap_ratio to 0.0. If the processing with the default values are extremely slow or stalled, please use --do_swap_au and set this --swap_ratio option to a large value (15.0[%] is a good start point). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). It will be used to evaluate the number of CPUs per node from user-provided MPI setting. By default, it uses 2GB * (number of CPUs per node). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "depth_order"; token.key_prefix = "--"; token.label = "Depth order"; token.help = "An integer value defines the number of initial independent MGSKmeans runs (2^depth_order). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stop_mgskmeans_percentage"; token.key_prefix = "--"; token.label = "Image assignment percentage to stop MGSKmeans [%]"; token.help = "A floating number denotes particle assignment change percentage that serves as the converge criteria of minimum group size K-means. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "10.0"; token.restore = [['10.0'], ['10.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nsmear"; token.key_prefix = "--"; token.label = "Number of smears for sorting"; token.help = "Fill it with 1 if user does not want to use all smears. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "orientation_groups"; token.key_prefix = "--"; token.label = "Number of orientation groups"; token.help = "Number of orientation groups in an asymmetric unit. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "100"; token.restore = [['100'], ['100']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "not_include_unaccounted"; token.key_prefix = "--"; token.label = "Do unaccounted reconstruction"; token.help = "Do not reconstruct unaccounted elements in each generation. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "notapplybckgnoise"; token.key_prefix = "--"; token.label = "Use background noise flag"; token.help = "Flag to turn off background noise. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "random_group_elimination_threshold"; token.key_prefix = "--"; token.label = "Random group elimination threshold"; token.help = "A floating value denotes the random group reproducibility standard deviation for eliminating random groups. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2.0"; token.restore = [['2.0'], ['2.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2bdb"; sxcmd.subname = ""; sxcmd.mode = "makevstack"; sxcmd.subset_config = "subset"; sxcmd.label = "SORT3D_DEPTH Stack Subset"; sxcmd.short_info = "Make a 'virtual' bdb image stack with the specified name from one or more other stacks. "; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "makevstack"; token.key_prefix = "--"; token.label = "Output stack subset"; token.help = "Make a 'virtual' bdb image stack with the specified name from one or more other stacks. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_bdb_stack_file"; token.key_prefix = ""; token.label = "Original image stack"; token.help = "Specify path to input BDB stack file used for the associated SORT3D_DEPTH run."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "list"; token.key_prefix = "--"; token.label = "Image selection file"; token.help = "Input selection text file containing a list of selected image IDs (or indexes of the data subset) to create a new virtual BDB image stack from an existed stack or virtual stack. Typically, Cluster#.txt created by sxsort3d_depth (e.g. Cluster1.txt)."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['exlist', 'none', 'False'], ['step', '0,1', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('exlist', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('step', []).append([token.key_base, '0,1', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = "local_refinement"; sxcmd.subset_config = "stack"; sxcmd.label = "Local Refinement from Stack"; sxcmd.short_info = "Perform local refinement in which the restricted search begins from the user-provided orientation parameters stored in image headers. Note delta has to be less than or equal to 3.75[A]."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "local_refinement"; token.key_prefix = "--"; token.label = "Perform local refinement"; token.help = "Perform local refinement starting from user-provided orientation parameters stored in the header of input image stack. "; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stack"; token.key_prefix = ""; token.label = "Input image stack"; token.help = "The stack must have 3D orientation parameters (xform.projection) stored in image headers. They can be imported using sp_header.py."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. If not given, the program will use name master_DATA_AND_TIME. For standard continuation run, local refinement from iteration, and final reconstruction only, the directory must exist and further iterations will be written to this directory. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "inires"; token.key_prefix = "--"; token.label = "Starting resolution [A]"; token.help = "Resolution of the initial volume. For local refinement, the program automatically calculates the initial resolution using provided orientation parameters."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [[-1.0], [-1.0]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Initial angular sampling step [Degrees]"; token.help = "Initial angular sampling step. For local refinement, the value has to be less than or equal to 3.75."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Outer particle radius in pixels &lt; int(boxsize/2)-1. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "A mask applied to half-map during iterations of the program. If not given, a hard sphere of radius boxsize/2-1 will be used. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Point-group symmetry of the refined structure. Supported point groups symmetries are: cn and dn, where n is multiplicity, oct, tet, and icos. Ignored in final reconstruction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xr"; token.key_prefix = "--"; token.label = "Search range [Pixels]"; token.help = "Range for translation search in both directions. Search is +/-xr. It can be fractional. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5.0"; token.restore = [['5.0'], ['5.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ts"; token.key_prefix = "--"; token.label = "Search step size [Pixels]"; token.help = "Step size of translation search in both directions. Search is within a circle of radius xr on a grid with steps ts. It can be fractional. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "an"; token.key_prefix = "--"; token.label = "Angular neighborhood"; token.help = "Angular neighborhood for local search. Used only in Local Refinement mode. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shake"; token.key_prefix = "--"; token.label = "Shake"; token.help = "Shake randomizes grid searches by a factor of (shake x grid step). Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.5"; token.restore = [['0.5'], ['0.5']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "small_memory"; token.key_prefix = "--"; token.label = "Keep data in memory"; token.help = "Indicate if data should be kept in memory or not. By default, data will be kept in memory. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ccfpercentage"; token.key_prefix = "--"; token.label = "Correlation peaks to be included [%]"; token.help = "Percentage of correlation peaks to be included. 0.0 corresponds to hard matching. Ignored in final reconstruction. This value will be adjusted during helical refinement. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = "99.9"; token.restore = [['99.9'], ['10']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Apply image norm correction"; token.help = "Indicate if image norm correction should be applied or not. By default, apply image norm correction. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "function"; token.key_prefix = "--"; token.label = "Reference preparation function"; token.help = "Specify name of function that the program should use to prepare the reference structure after each iteration. Ignored in final reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "do_volume_mask"; token.restore = [['do_volume_mask'], ['do_volume_mask']]; token.type = "user_func"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_process"; sxcmd.subname = ""; sxcmd.mode = "combinemaps"; sxcmd.subset_config = "halfset maps"; sxcmd.label = "PostRefiner"; sxcmd.short_info = "Post-refine a pair of unfiltered odd & even halfset maps by combining them, then enhancing the high frequencies (Halfset Maps Mode). B-factor can be automatically estimated from the unfiltered halfset maps. This mode requires two arguments; use unfiltered halfset maps produced by MERIDIEN."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "combinemaps"; token.key_prefix = "--"; token.label = "Post-refine halfset volumes"; token.help = "Post-refine a pair of unfiltered odd & even halfset maps by combining them, then enhancing the high frequencies (Halfset Maps Mode). B-factor can be automatically estimated from the unfiltered halfset maps. This mode requires two arguments; use unfiltered halfset maps produced by MERIDIEN."; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "firstvolume"; token.key_prefix = ""; token.label = "First unfiltered halfset map"; token.help = "As generated by sxmeridien."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "secondvolume"; token.key_prefix = ""; token.label = "Second unfiltered halfset map"; token.help = "As generated by sxmeridien."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Specify path to the output directory for PostRefiner process. By default, the program uses the current directory. However, GUI requires the path other than the current directory. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size of input data. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "Path to user-provided mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "do_adaptive_mask"; token.key_prefix = "--"; token.label = "Apply adaptive mask"; token.help = "Program creates mask adaptively with given density threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mask', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mask', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', '-9999.0', 'False'], ['nsigma', '1.0', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, '-9999.0', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, '1.0', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. If the value changed from the default, the option will be ignored and the threshold will be set according to nsigma method under the Advanced tab. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', '1.0', 'False'], ['use_mol_mass', 'False', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = "-9999.0"; token.restore = [['-9999.0'], ['-9999.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, '1.0', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is larger than -9999.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', '-9999.0', 'False'], ['use_mol_mass', 'False', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, '-9999.0', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gauss\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "do_approx"; token.key_prefix = "--"; token.label = "Approximate values"; token.help = "Approximate the values of the soft edge area instead of using the exact values. This will lead to a less smoothened mask, but will mirror the previous behaviour. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "mtf"; token.key_prefix = "--"; token.label = "MTF file"; token.help = "Path to file contains the MTF (modulation transfer function) of the detector used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mtf"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fsc_adj"; token.key_prefix = "--"; token.label = "Apply FSC-based low-pass filter"; token.help = "Applies an FSC-based low-pass filter to the merged map before the B-factor estimation. Effective only in Halfset Volumes Mode. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "B_enhance"; token.key_prefix = "--"; token.label = "B-factor enhancement"; token.help = "0.0: program automatically estimates B-factor using power spectrum at frequencies from B_start (usually 10 Angstrom) to the resolution determined by FSC143 (valid only in Halfset Volumes Mode; Non-zero positive value: program use the given value [A^2] to enhance map); -1.0: B-factor is not applied. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "B_start"; token.key_prefix = "--"; token.label = "B-factor estimation lower limit [A]"; token.help = "Frequency in Angstrom defining lower boundary of B-factor estimation. Effective only in Halfset Volumes Mode with --B_enhance=0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['B_enhance', '0.0', 'False']]; token.default = "10.0"; token.restore = [['10.0'], ['10.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('B_enhance', []).append([token.key_base, '0.0', 'False'])
		token = SXcmd_token(); token.key_base = "B_stop"; token.key_prefix = "--"; token.label = "B-factor estimation upper limit [A]"; token.help = "Frequency in Angstrom defining upper boundary of B-factor estimation. Recommended to set the upper boundary to the frequency where FSC is smaller than 0.0. Effective only in Halfset Volumes Mode with --B_enhance=0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['B_enhance', '0.0', 'False']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('B_enhance', []).append([token.key_base, '0.0', 'False'])
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency [A]"; token.help = "0.0: low-pass filter to resolution (valid only in Halfset Volumes Mode); A value larger than 0.5: low-pass filter to the value in Angstrom; -1.0: no low-pass filter. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off. Effective only when --fl option is not -1.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.01"; token.restore = [['0.01'], ['0.01']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "output"; token.key_prefix = "--"; token.label = "Output file name"; token.help = "File name of output final post-refined structure. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol_combined.hdf"; token.restore = [['vol_combined.hdf'], ['vol_combined.hdf']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_sort3d_depth"; sxcmd.subname = ""; sxcmd.mode = "instack"; sxcmd.subset_config = ""; sxcmd.label = "3D Clustering from Stack - SORT3D_DEPTH"; sxcmd.short_info = "Run from user-provided orientation parameters stored in stack header.  This mode uses only highest probability orientation parameters per image, ths it should be avoided.  This program is used only for testing."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "instack"; token.key_prefix = "--"; token.label = "Input images stack"; token.help = "A string denotes file path of input particle stack for sorting. Sorting switches to stack mode when option is specified. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "A string denotes output directory for 3D sorting. It can be either existing or non-existing. By default, the program uses sort3d_DATA_AND_TIME for the name. Here, you can find a log.txt that describes the sequences of computations in the program. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nxinit"; token.key_prefix = "--"; token.label = "Initial image size"; token.help = "Image size used for MGSKmeans in case of starting sorting from a data stack. By default, the program determines window size. Option is valid only for stack mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask3D"; token.key_prefix = "--"; token.label = "3D mask"; token.help = "A string denotes file path of the global 3D mask for clustering. Imported from 3D refinement unless user wishes a different one in meridien iteration mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "focus"; token.key_prefix = "--"; token.label = "Focus 3D mask"; token.help = "A string denotes file path of a binary 3D mask for focused clustering. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "radius"; token.key_prefix = "--"; token.label = "Estimated particle radius [Pixels]"; token.help = "A integer value that is smaller than half of the box size. Imported from refinement unless user wishes a different one in meridien iteration mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "radius"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "A string denotes point group symmetry of the macromolecular structure. Imported from refinement unless the user wishes a different one in meridien iteration mode. Require specification in stack mode. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp"; token.key_prefix = "--"; token.label = "Number of images per group"; token.help = "User expected group size in integer. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1000"; token.restore = [['1000'], ['1000']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "img_per_grp_split_rate"; token.key_prefix = "--"; token.label = "Group splitting rate"; token.help = "An integer value denotes split rate of the group size(--img_per_grp). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "minimum_grp_size"; token.key_prefix = "--"; token.label = "Minimum size of reproducible class"; token.help = "The minimum size of selected or accounted clusters as well as the minimum group size constraint in MGSKmeans. However this value must be smaller than the number of images per a group (img_per_grp). By default, the program uses half number of the images per group.  "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "do_swap_au"; token.key_prefix = "--"; token.label = "Swap flag"; token.help = "A boolean flag to control random swapping a certain number of accounted elements per cluster with the unaccounted elements. If the processing with the default values are extremely slow or stalled, please use this --do_swap_au option and set --swap_ratio to a large value (15.0[%] is a good start point). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "swap_ratio"; token.key_prefix = "--"; token.label = "Swap percentage [%]"; token.help = "the percentage of images for swapping ranges between 0.0[%] and 50.0[%]. Option valid only with --do_swap_au. Without --do_swap_au, the program automatically sets --swap_ratio to 0.0. If the processing with the default values are extremely slow or stalled, please use --do_swap_au and set this --swap_ratio option to a large value (15.0[%] is a good start point). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). It will be used to evaluate the number of CPUs per node from user-provided MPI setting. By default, it uses 2GB * (number of CPUs per node). "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "depth_order"; token.key_prefix = "--"; token.label = "Depth order"; token.help = "An integer value defines the number of initial independent MGSKmeans runs (2^depth_order). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "stop_mgskmeans_percentage"; token.key_prefix = "--"; token.label = "Image assignment percentage to stop MGSKmeans [%]"; token.help = "A floating number denotes particle assignment change percentage that serves as the converge criteria of minimum group size K-means. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "10.0"; token.restore = [['10.0'], ['10.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nsmear"; token.key_prefix = "--"; token.label = "Number of smears for sorting"; token.help = "Fill it with 1 if user does not want to use all smears. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "orientation_groups"; token.key_prefix = "--"; token.label = "Number of orientation groups"; token.help = "Number of orientation groups in an asymmetric unit. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "100"; token.restore = [['100'], ['100']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "not_include_unaccounted"; token.key_prefix = "--"; token.label = "Do unaccounted reconstruction"; token.help = "Do not reconstruct unaccounted elements in each generation. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "notapplybckgnoise"; token.key_prefix = "--"; token.label = "Use background noise flag"; token.help = "Flag to turn off background noise. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "random_group_elimination_threshold"; token.key_prefix = "--"; token.label = "Random group elimination threshold"; token.help = "A floating value denotes the random group reproducibility standard deviation for eliminating random groups. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2.0"; token.restore = [['2.0'], ['2.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_process"; sxcmd.subname = ""; sxcmd.mode = "combinemaps"; sxcmd.subset_config = "single map"; sxcmd.label = "PostRefiner (Single Map)"; sxcmd.short_info = "Post-refine a single map by enhancing the high frequencies (Single Map Mode). Only ad-hoc low-pass filter cutoff and B-factor can be used. This mode requires one argument; path pattern with wild card '*' to specify a list of  volumes or a path to a volume (without wild card '*')."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "combinemaps"; token.key_prefix = "--"; token.label = "Post-refine single map"; token.help = "Post-refine a single map by enhancing the high frequencies (Single Map Mode). Only ad-hoc low-pass filter cutoff and B-factor can be used. This mode requires one argument; path pattern with wild card '*' to specify a list of  volumes or a path to a volume (without wild card '*')."; token.group = "main"; token.is_required = True; token.is_locked = True; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = True; token.restore = [[True], [True]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_volume_pattern"; token.key_prefix = ""; token.label = "Input volume pattern"; token.help = "Specify path pattern of input single volumes with a wild card '*' or path to single volume file (without wild card '*'). Use the wild card to indicate the place of variable part of the file names. The path pattern must be enclosed by single quotes (') or double quotes (') (Note: sxgui.py automatically adds single quotes ('))."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_dir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Specify path to the output directory for PostRefiner process. By default, the program uses the current directory. However, GUI requires the path other than the current directory. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size of input data. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mask"; token.key_prefix = "--"; token.label = "3D mask file"; token.help = "Path to user-provided mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "do_adaptive_mask"; token.key_prefix = "--"; token.label = "Apply adaptive mask"; token.help = "Program creates mask adaptively with given density threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['mask', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('mask', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', '-9999.0', 'False'], ['nsigma', '1.0', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, '-9999.0', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, '1.0', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. If the value changed from the default, the option will be ignored and the threshold will be set according to nsigma method under the Advanced tab. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', '1.0', 'False'], ['use_mol_mass', 'False', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = "-9999.0"; token.restore = [['-9999.0'], ['-9999.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, '1.0', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is larger than -9999.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', '-9999.0', 'False'], ['use_mol_mass', 'False', 'False'], ['do_adaptive_mask', 'True', 'False']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, '-9999.0', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gauss\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['do_adaptive_mask', 'True', 'False']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('do_adaptive_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "do_approx"; token.key_prefix = "--"; token.label = "Approximate values"; token.help = "Approximate the values of the soft edge area instead of using the exact values. This will lead to a less smoothened mask, but will mirror the previous behaviour. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "mtf"; token.key_prefix = "--"; token.label = "MTF file"; token.help = "Path to file contains the MTF (modulation transfer function) of the detector used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mtf"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "B_enhance"; token.key_prefix = "--"; token.label = "B-factor enhancement"; token.help = "Non-zero positive value: program use provided B-factor [A^2] to enhance the map; -1.0: B-factor is not applied."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter frequency [A]"; token.help = "A value larger than 0.5: low-pass filter to the resolution in Angstrom; -1.0: no low-pass filter."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off. Effective only when --fl option is not -1.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.01"; token.restore = [['0.01'], ['0.01']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "output"; token.key_prefix = "--"; token.label = "Output file name"; token.help = "File name of output final post-refined structure. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol_combined.hdf"; token.restore = [['vol_combined.hdf'], ['vol_combined.hdf']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = "do_final"; sxcmd.subset_config = ""; sxcmd.label = "Final 3D Reconstruction Only"; sxcmd.short_info = "Compute a final 3D reconstruction using either select or best resolution iteration of meridien."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_alt"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "do_final"; token.key_prefix = "--"; token.label = "Iteration number"; token.help = "Specify the iteration which to perform final reconstruction. By setting to 0, program searches for the iteration which had best resolution, then performs correponding final reconstruction. Value must be zero or positive. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Output Directory"; token.help = "This directory must exist. In this mode information is read from files in this directory and the results will be written there."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "moon_eliminator"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Volume Adjustment"; sxcmd.short_info = "Eliminate moons or remove dust from the background of a 3D density map based on the expected molecular mass."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume_path"; token.key_prefix = ""; token.label = "Input volume path"; token.help = "Path to input volume file containing the 3D density map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Output pixel size [A]"; token.help = "The original pixel size of dataset. This must be the pixel size after resampling when resample_ratio != 1.0. That is, it will be the pixel size of the output map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_density_threshold', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_density_threshold', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "use_density_threshold"; token.key_prefix = "--"; token.label = "Use ad-hoc density threshold"; token.help = "Use user-provided ad-hoc density threshold, instead of computing the value from the molecular mass. Below this density value, the data is assumed not to belong to the main body of the particle density. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "moon_distance"; token.key_prefix = "--"; token.label = "Distance to the nearest moon [Pixels]"; token.help = "The moons further than this distance from the density surface will be elminated. The value smaller than the default is not recommended because it is difficult to avoid the stair-like gray level change at the edge of the density surface. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.0"; token.restore = [['3.0'], ['3.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Resample ratio"; token.help = "Specify a value larger than 0.0. By default, the program does not resample the input map (i.e. resample ratio is 1.0). Use this option maily to restore the original dimensions or pixel size of VIPER or R-VIPER model. Alternatively, specify the path to the output directory of an ISAC2 run. The program automatically extracts the resampling ratio used by the ISAC2 run. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "'1.0'"; token.restore = [["'1.0'"], ["'1.0'"]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Output box size [Pixels]"; token.help = "The x, y, and z dimensions of cubic area to be windowed from input 3D volume for output 3D volumes. This must be the box size after resampling when resample_ratio != 1.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "invert_handedness"; token.key_prefix = "--"; token.label = "Invert handedness"; token.help = "Invert the handedness of the 3D map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter resolution [A]"; token.help = "&gt;0.0: low-pass filter to the value in Angstrom; =-1.0: no low-pass filter. The program applies this low-pass filter before the moon elimination. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_volume_path_2nd"; token.key_prefix = ""; token.label = "Second input volume path"; token.help = "Path to second input volume file containing the 3D density map. Use this option to create a mask from the sum of two MERIDIEN half-set maps. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. By default, it is set to half of --moon_distance so that the voxels with 1.0 values in the mask are same as the hard-edged molecular-mass binary volume. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) 'cosine' for cosine soft-edged (used in PostRefiner) and (2) 'gauss' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outputs_root"; token.key_prefix = "--"; token.label = "Root name of outputs"; token.help = "Specify the root name of all outputs. It cannot be empty string or only white spaces. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol3d"; token.restore = [['vol3d'], ['vol3d']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resampled_shift3d"; token.key_prefix = "--"; token.label = "Providing resampled 3D shifts"; token.help = "Use this option when you are providing the resampled 3D shifts (using pixel size of outputs) when --resample_ratio!=1.0. By default, the program assums the provided shifts are not resampled. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_x"; token.key_prefix = "--"; token.label = "3D x-shift [Pixels]"; token.help = "3D x-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_y"; token.key_prefix = "--"; token.label = "3D y-shift [Pixels]"; token.help = "3D y-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_z"; token.key_prefix = "--"; token.label = "3D z-shift [Pixels]"; token.help = "3D z-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off in absolute frequency. The program applies this low-pass filter before the moon elimination. Effective only when --fl &gt; 0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "debug"; token.key_prefix = "--"; token.label = "Run with debug mode"; token.help = "Mainly for developers. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_mask"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Masking"; sxcmd.short_info = "Mask creation tool for 2D or 3D masks."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume"; token.key_prefix = ""; token.label = "Input image"; token.help = "Path to the 2D image or 3D Volume "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output direcory path "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "Output prefix"; token.help = "Prefix of the produced files "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "sp_mask"; token.restore = [['sp_mask'], ['sp_mask']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overwrite"; token.key_prefix = "--"; token.label = "Overwrite outputs"; token.help = "Overwrite the output mask in case it exists already. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A/px]"; token.help = "Pixel size of the volume. Used for filtering and the molcular mask threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['nsigma', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. This is used to calculate the binarization threshold automatically. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One cycle of dilation will add about 2 pixels to the mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking. If the width is 0, a binary mask is returned. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "allow_disconnected"; token.key_prefix = "--"; token.label = "Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fill_mask"; token.key_prefix = "--"; token.label = "Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nerosion"; token.key_prefix = "--"; token.label = "Number of erosions"; token.help = "Number of times to erode binarized volume. One cycle of erosion will remove about 2 pixels from the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "do_old"; token.key_prefix = "--"; token.label = "Old behaviour"; token.help = "Restore the old masking behaviour, which is a bit less smooth. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "low_pass_filter_resolution"; token.key_prefix = "--"; token.label = "Low pass filter resolution [A]"; token.help = "Low pass filter resolution in angstroms. If set, the volume will be filtered prior to create a mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "low_pass_filter_falloff"; token.key_prefix = "--"; token.label = "Low pass filter falloff [1/Pixel]"; token.help = "Low pass filter falloff in absolute frequencies. If set, the volume will be filtered prior to create a mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['low_pass_filter_resolution', 'none', 'True']]; token.default = "0.01"; token.restore = [['0.01'], ['0.01']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('low_pass_filter_resolution', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "use_second_mask"; token.key_prefix = "--"; token.label = "Use a second mask"; token.help = "ONLY A GUI OPTION. The second mask can be used to mask the first one after creation. This is useful to create soft edged regions of the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [True]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "second_mask"; token.key_prefix = "--"; token.label = "Second mask path"; token.help = "Path to the input second mask used for masking the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask_shape', 'none', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "second_mask_shape"; token.key_prefix = "--"; token.label = "Second mask shape"; token.help = "Shape of the second mask. Possible values: sphere, cylinder, cube. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'False']]; token.default = "none"; token.restore = [['none', 'sphere', 'cylinder', 'cube'], ['cylinder', 'none', 'sphere', 'cube']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "s_radius"; token.key_prefix = "--"; token.label = "Second - Radius of the mask"; token.help = "Radius of the sphere or cylinder for the second mask in pixels. Radius must be less than half of the volume dimension. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['second_mask_shape', 'cube', 'True'], ['use_second_mask', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'cube', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_nx"; token.key_prefix = "--"; token.label = "Second - X dimension of the mask"; token.help = "X Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_ny"; token.key_prefix = "--"; token.label = "Second - Y dimension of the mask"; token.help = "Y Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. If not provided, a squared volume with s_nx is assumed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False'], ['s_nx', 'none;Main', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_nx', []).append([token.key_base, 'none;Main', 'True'])
		token = SXcmd_token(); token.key_base = "s_nz"; token.key_prefix = "--"; token.label = "Second - Z dimension of the mask"; token.help = "Z Dimensions of the second mask. The mask is clipped to the input volume size afterwards. Therefore this values can be used to generate a percentage mask for helical processing. If not provided, a squared volume with s_nx is assumed. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['second_mask_shape', 'none', 'True'], ['use_second_mask', 'True', 'False'], ['s_nx', 'none;Main', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('second_mask_shape', []).append([token.key_base, 'none', 'True']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_nx', []).append([token.key_base, 'none;Main', 'True'])
		token = SXcmd_token(); token.key_base = "s_pixel_size"; token.key_prefix = "--"; token.label = "Second - Pixel size [A/px]"; token.help = "Pixel size of the second volume. Used for the molecular mass threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_use_mol_mass"; token.key_prefix = "--"; token.label = "Second - Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_threshold', 'none', 'False'], ['s_nsigma', 'none', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_mol_mass"; token.key_prefix = "--"; token.label = "Second - Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. This is used to calculate the binarization threshold automatically. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_use_mol_mass', 'True', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_threshold"; token.key_prefix = "--"; token.label = "Second - Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. If the value is lower-equal than the default, the option will be ignored and the threshold will be set according to nsigma method above. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_nsigma', 'none', 'False'], ['s_use_mol_mass', 'False', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_nsigma"; token.key_prefix = "--"; token.label = "Second - Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['s_threshold', 'none', 'False'], ['s_use_mol_mass', 'False', 'False'], ['use_second_mask', 'True', 'False'], ['second_mask', 'none', 'True']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('s_threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('s_use_mol_mass', []).append([token.key_base, 'False', 'False']); sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('second_mask', []).append([token.key_base, 'none', 'True'])
		token = SXcmd_token(); token.key_base = "s_ndilation"; token.key_prefix = "--"; token.label = "Second - Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One round of erosion will add about 2 pixels to the mask "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_nerosion"; token.key_prefix = "--"; token.label = "Second - Number of erosions"; token.help = "Number of times to erode binarized volume. One round of erosion will remove about 2 pixels from the mask "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_edge_width"; token.key_prefix = "--"; token.label = "Second - Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "main"; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_edge_type"; token.key_prefix = "--"; token.label = "Second - Soft-edge type"; token.help = "The type of soft-edge for the 3D mask. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['s_edge_width', '0', 'True']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "s_do_old"; token.key_prefix = "--"; token.label = "Second - Old behaviour"; token.help = "Restore the old masking behaviour, which is a bit less smooth. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False'], ['s_edge_width', '0', 'True']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False']); sxcmd.dependency_dict.setdefault('s_edge_width', []).append([token.key_base, '0', 'True'])
		token = SXcmd_token(); token.key_base = "s_allow_disconnected"; token.key_prefix = "--"; token.label = "Second - Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "s_fill_mask"; token.key_prefix = "--"; token.label = "Second - Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "s_invert"; token.key_prefix = "--"; token.label = "Second - Invert masking"; token.help = "If True, the mask will remove everything that is inside instead of leaving it. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_second_mask', 'True', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_second_mask', []).append([token.key_base, 'True', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "angular_distribution"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Angular Distribution"; sxcmd.short_info = "Generate a chimera .bild file for the visual representation of the resulting projection parameters."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "params_file"; token.key_prefix = ""; token.label = "Projection parameters"; token.help = "File containing the 3D projection parameters "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "params_any_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_folder"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output folder name "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "File prefix"; token.help = "Prefix for the output files - None uses the same name as the params file - Existing files will be overwritten "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "symmetry"; token.key_prefix = "--"; token.label = "Point-group symmetry"; token.help = "Symmetry - c0 creates full sphere distribution; XXX_full, e.g. c1_full, c4_full, icos_full, creates a histogram for both hemispheres. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Particle box size [Pixels]"; token.help = "Box size "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "256"; token.restore = [['256'], ['256']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A]"; token.help = "Pixel size of the project "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "particle_radius"; token.key_prefix = "--"; token.label = "Particle radius [Pixels]"; token.help = "Particle radius "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "120"; token.restore = [['120'], ['120']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "delta"; token.key_prefix = "--"; token.label = "Angular sampling step [Degrees]"; token.help = "Angular step size in degree - Low deltas combined with low symmetry might crash chimera session "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.75"; token.restore = [['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375'], ['3.75', '15', '7.5', '1.875', '0.9375', '0.46875', '0.234375']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "method"; token.key_prefix = "--"; token.label = "Distribution method"; token.help = "Method used to create the reference angles (S or P or M) "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = "advanced"; token.dependency_group = [['', '', '']]; token.default = "S"; token.restore = [['S', 'M', 'P'], ['M', 'P', 'S']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "dpi"; token.key_prefix = "--"; token.label = "Plot DPI"; token.help = "Dpi for the legend plot "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "144"; token.restore = [['144'], ['144']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nth_percentile"; token.key_prefix = "--"; token.label = "Nth percentil"; token.help = "Use the value of the nth percentile of the radius distribution for normalization "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "99"; token.restore = [['99'], ['99']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "old"; token.key_prefix = "--"; token.label = "Use old"; token.help = "Use the old color scheme and normalization. The old style was normalizing the maximum length instead of the nth percentile and used a blue to green instead of a blue over green to yellow color scheme. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_recons3d_n"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "3D Reconstruction"; sxcmd.short_info = "3D Reconstruction using nearest-neighbor interpolation."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = True; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "prj_stack"; token.key_prefix = ""; token.label = "Input stack"; token.help = "Stack of projections "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "bdb2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_volume"; token.key_prefix = ""; token.label = "Output volume"; token.help = "Output reconstructed volume file "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "sym"; token.key_prefix = "--"; token.label = "Symmetry"; token.help = "Symmetry. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "c1"; token.restore = [['c1'], ['c1']]; token.type = "sym"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "CTF"; token.key_prefix = "--"; token.label = "Apply CTF"; token.help = "Apply CTF correction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "list"; token.key_prefix = "--"; token.label = "Selection list"; token.help = "File with list of images to be used in the first column. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbosity"; token.help = "Enter 0 for no verbosity, 1 for verbose output. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "begin"; token.key_prefix = ""; token.label = "First image"; token.help = "First image for reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "end"; token.key_prefix = ""; token.label = "Last image"; token.help = "Last image for reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "step"; token.key_prefix = ""; token.label = "Image step"; token.help = "Increment used for image list. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "snr"; token.key_prefix = "--"; token.label = "SNR"; token.help = "Signal-to-Noise Ratio. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "group"; token.key_prefix = "--"; token.label = "Group number"; token.help = "Perform reconstruction using images for a given group number (group is attribute in the header). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "npad"; token.key_prefix = "--"; token.label = "Padding"; token.help = "Number of times padding "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "xysize"; token.key_prefix = "--"; token.label = "X,Y-dimension"; token.help = "Expected size in xy-plane. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "zsize"; token.key_prefix = "--"; token.label = "Z-dimension"; token.help = "Expected size in z. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "smearstep"; token.key_prefix = "--"; token.label = "Smear step"; token.help = "Rotational smear step. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0.0"; token.restore = [['0.0'], ['0.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "interpolation_method"; token.key_prefix = "--"; token.label = "Interpolation method"; token.help = "Interpolation methods: nearest neighbor (4nn), or trilinear interpolation (tril). "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "4nn"; token.restore = [['4nn'], ['4nn']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "niter"; token.key_prefix = "--"; token.label = "Iteration number"; token.help = "NNumber of iterations for iterative reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "10"; token.restore = [['10'], ['10']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "upweighted"; token.key_prefix = "--"; token.label = "Upweight"; token.help = "Apply background noise. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "compensate"; token.key_prefix = "--"; token.label = "Compensate"; token.help = "Compensate in reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "chunk_id"; token.key_prefix = "--"; token.label = "Chunk ID"; token.help = "Reconstruct both odd and even groups of particles. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "select_data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "target_window_size"; token.key_prefix = "--"; token.label = "Target window size"; token.help = "Size of the targeted reconstruction. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1"; token.restore = [['-1'], ['-1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_batch"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Batch Pipeline Execution"; sxcmd.short_info = "Run jobs that wait with the execution on each other."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_sort3d"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "submission_command"; token.key_prefix = ""; token.label = "Submission command"; token.help = "Submission command, e.g., qsub, qsub -V, sbatch, bash "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_run_dir"; token.key_prefix = ""; token.label = "Pipeline directory"; token.help = "Directory containin the pipeline submission files "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "hold_flag"; token.key_prefix = "--"; token.label = "Hold flag"; token.help = "Hold flag for the submission command, e.g. -hold_jid=, --wait=, --dependency=afterany:; Default is None and should be used in combination with a local execution with bash "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "first_hold_number"; token.key_prefix = "--"; token.label = "First hold number"; token.help = "Wait number for the first job that is submitted. By default, the first job will not wait for others "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_signalsubtract"; sxcmd.subname = "avgfilt"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Average and Filter"; sxcmd.short_info = "Average and low-pass filter a map for segmentation."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "avol1"; token.key_prefix = "--"; token.label = "Map #1 to average"; token.help = "First map to average. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'avgfilt', 'False']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'avgfilt', 'False'])
		token = SXcmd_token(); token.key_base = "avol2"; token.key_prefix = "--"; token.label = "Map #2 to average"; token.help = "Second map to average. If not provided, first map will be used. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'avgfilt', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'avgfilt', 'False'])
		token = SXcmd_token(); token.key_base = "outdir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Directory where outputs will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "filtrad"; token.key_prefix = "--"; token.label = "Filter radius"; token.help = "Low-pass filter radius. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'avgfilt', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'avgfilt', 'False'])
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size"; token.help = "Pixel size in Angstroms. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbosity"; token.key_prefix = "--"; token.label = "Verbosity level"; token.help = "Controls how much information will be written to the screen. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_signalsubtract"; sxcmd.subname = "sp_mask"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Generate Mask"; sxcmd.short_info = "Mask a soft mask of the region to be excluded."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "map2mask"; token.key_prefix = "--"; token.label = "Map to mask"; token.help = "Map from which a mask will be generated. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'sp_mask', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'sp_mask', 'False'])
		token = SXcmd_token(); token.key_base = "fullmap"; token.key_prefix = "--"; token.label = "Full map"; token.help = "Map which will be multiplied by the mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'sp_mask', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'sp_mask', 'False'])
		token = SXcmd_token(); token.key_base = "outdir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Directory where outputs will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "mapthresh"; token.key_prefix = "--"; token.label = "Map threshold"; token.help = "Full map will be initially binarized at this threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'sp_mask', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'sp_mask', 'False'])
		token = SXcmd_token(); token.key_base = "verbosity"; token.key_prefix = "--"; token.label = "Verbosity level"; token.help = "Controls how much information will be written to the screen. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_signalsubtract"; sxcmd.subname = "projsubtract"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Project and Subtract"; sxcmd.short_info = "Compute re-projections of map to be subtracted, and subtract them from the original images."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "origparts"; token.key_prefix = "--"; token.label = "Particle stack"; token.help = "Original particle stack before signal subtraction. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "map2subtract"; token.key_prefix = "--"; token.label = "Map to subtract"; token.help = "Map whose projections will be subtracted from the original images. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "projparams"; token.key_prefix = "--"; token.label = "Projection parameters"; token.help = "Angles and shifts, from Meridien. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "params_proj_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "outdir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Directory where outputs will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbosity"; token.key_prefix = "--"; token.label = "Verbosity level"; token.help = "Controls how much information will be written to the screen. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nmontage"; token.key_prefix = "--"; token.label = "Number of examples"; token.help = "This number of original, projections, and subtracted images will be written to disk. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "inmem"; token.key_prefix = "--"; token.label = "In memory?"; token.help = "Flag to store projections in memory. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "saveprojs"; token.key_prefix = "--"; token.label = "Save projections?"; token.help = "Flag to save intermediate projections. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "stats"; token.key_prefix = "--"; token.label = "Save stats?"; token.help = "Flag to save normalization statistics.  "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])
		token = SXcmd_token(); token.key_base = "nonorm"; token.key_prefix = "--"; token.label = "Skip normalization?"; token.help = "Flag to skip normalization. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'projsubtract', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'projsubtract', 'False'])

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_meridien"; sxcmd.subname = ""; sxcmd.mode = "do_final"; sxcmd.subset_config = ""; sxcmd.label = "Compute 3D Reconstruction"; sxcmd.short_info = "Compute a final 3D reconstruction from iteration number used for signal-subtraction."; sxcmd.mpi_support = True; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "do_final"; token.key_prefix = "--"; token.label = "Iteration number"; token.help = "Specify the iteration which to perform final reconstruction. By setting to 0, program searches for the iteration which had best resolution, then performs correponding final reconstruction. Value must be zero or positive. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Meridien Directory"; token.help = "This directory must exist. In this mode information is read from files in this directory."; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output_continue"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "voldir"; token.key_prefix = "--"; token.label = "Output Reconstruction Directory"; token.help = "Directory in which the output reconstructions will be written."; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "memory_per_node"; token.key_prefix = "--"; token.label = "Memory per node [GB]"; token.help = "User provided information about memory per node in GB (NOT per CPU). By default, it uses 2GB * (number of CPUs per node). Used in all modes. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_signalsubtract"; sxcmd.subname = "centershift"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Center Map"; sxcmd.short_info = "Center map of remaining density."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_pipe"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "cvol1"; token.key_prefix = "--"; token.label = "Center map #1"; token.help = "First map to average and center. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'centershift', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'centershift', 'False'])
		token = SXcmd_token(); token.key_base = "cvol2"; token.key_prefix = "--"; token.label = "Center map #2"; token.help = "Second map to average and center. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'centershift', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'centershift', 'False'])
		token = SXcmd_token(); token.key_base = "shiftparams"; token.key_prefix = "--"; token.label = "Shift parameters"; token.help = "Meridien parameters to combine with centering parameters. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'centershift', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "params_proj_txt"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'centershift', 'False'])
		token = SXcmd_token(); token.key_base = "diffimgs"; token.key_prefix = "--"; token.label = "Subtracted images"; token.help = "Images from signal subtraction to reconstruct. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'centershift', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "data2d_stack"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'centershift', 'False'])
		token = SXcmd_token(); token.key_base = "outdir"; token.key_prefix = "--"; token.label = "Output directory"; token.help = "Directory where outputs will be written. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "volradius"; token.key_prefix = "--"; token.label = "Structure radius"; token.help = "Radius to use for centering reconstruction. If the structure is not globular, try the shortest dimension. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['subtraction_mode', 'centershift', 'False']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('subtraction_mode', []).append([token.key_base, 'centershift', 'False'])
		token = SXcmd_token(); token.key_base = "apix"; token.key_prefix = "--"; token.label = "Pixel size"; token.help = "Pixel size in Angstroms. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "None"; token.restore = [['None'], ['None']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbosity"; token.key_prefix = "--"; token.label = "Verbosity level"; token.help = "Controls how much information will be written to the screen. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "2"; token.restore = [['2'], ['2']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "e2display"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Display Data"; sxcmd.short_info = "Displays images, volumes, or 1D plots."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = False
		token = SXcmd_token(); token.key_base = "input_data_list"; token.key_prefix = ""; token.label = "Input files"; token.help = "List of input images, volumes, plots. Wild cards (e.g *) can be used to select a list of files. Not recommended when the list is large. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "displayable_list"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "singleimage"; token.key_prefix = "--"; token.label = "Single image view"; token.help = "Display a stack one image at a time. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fullrange"; token.key_prefix = "--"; token.label = "Use full range of pixel values"; token.help = "Instead of default auto-contrast, use full range of pixel values for the display of particles stacks and 2D images. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "verbose"; token.key_prefix = "--"; token.label = "Verbose"; token.help = "Accepted values 0-9. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_pipe"; sxcmd.subname = "moon_eliminator"; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Volume Adjustment"; sxcmd.short_info = "Eliminate moons or remove dust from the background of a 3D density map based on the expected molecular mass."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume_path"; token.key_prefix = ""; token.label = "Input volume path"; token.help = "Path to input volume file containing the 3D density map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "The results will be written here. It cannot be an existing one. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Output pixel size [A]"; token.help = "The original pixel size of dataset. This must be the pixel size after resampling when resample_ratio != 1.0. That is, it will be the pixel size of the output map. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_density_threshold', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_density_threshold', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodalton. "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = ""; token.restore = [[""], [""]]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "use_density_threshold"; token.key_prefix = "--"; token.label = "Use ad-hoc density threshold"; token.help = "Use user-provided ad-hoc density threshold, instead of computing the value from the molecular mass. Below this density value, the data is assumed not to belong to the main body of the particle density. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "moon_distance"; token.key_prefix = "--"; token.label = "Distance to the nearest moon [Pixels]"; token.help = "The moons further than this distance from the density surface will be elminated. The value smaller than the default is not recommended because it is difficult to avoid the stair-like gray level change at the edge of the density surface. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3.0"; token.restore = [['3.0'], ['3.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resample_ratio"; token.key_prefix = "--"; token.label = "Resample ratio"; token.help = "Specify a value larger than 0.0. By default, the program does not resample the input map (i.e. resample ratio is 1.0). Use this option maily to restore the original dimensions or pixel size of VIPER or R-VIPER model. Alternatively, specify the path to the output directory of an ISAC2 run. The program automatically extracts the resampling ratio used by the ISAC2 run. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "'1.0'"; token.restore = [["'1.0'"], ["'1.0'"]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "box_size"; token.key_prefix = "--"; token.label = "Output box size [Pixels]"; token.help = "The x, y, and z dimensions of cubic area to be windowed from input 3D volume for output 3D volumes. This must be the box size after resampling when resample_ratio != 1.0. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "box"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "invert_handedness"; token.key_prefix = "--"; token.label = "Invert handedness"; token.help = "Invert the handedness of the 3D map. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fl"; token.key_prefix = "--"; token.label = "Low-pass filter resolution [A]"; token.help = "&gt;0.0: low-pass filter to the value in Angstrom; =-1.0: no low-pass filter. The program applies this low-pass filter before the moon elimination. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "input_volume_path_2nd"; token.key_prefix = ""; token.label = "Second input volume path"; token.help = "Path to second input volume file containing the 3D density map. Use this option to create a mask from the sum of two MERIDIEN half-set maps. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Dilation width [Pixels]"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. By default, it is set to half of --moon_distance so that the voxels with 1.0 values in the mask are same as the hard-edged molecular-mass binary volume. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "-1.0"; token.restore = [['-1.0'], ['-1.0']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking."; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1"; token.restore = [['1'], ['1']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge for moon-eliminator 3D mask and a moon-eliminated soft-edged 3D mask. Available methods are (1) 'cosine' for cosine soft-edged (used in PostRefiner) and (2) 'gauss' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "cosine"; token.restore = [['cosine'], ['cosine']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "outputs_root"; token.key_prefix = "--"; token.label = "Root name of outputs"; token.help = "Specify the root name of all outputs. It cannot be empty string or only white spaces. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "vol3d"; token.restore = [['vol3d'], ['vol3d']]; token.type = "output"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "resampled_shift3d"; token.key_prefix = "--"; token.label = "Providing resampled 3D shifts"; token.help = "Use this option when you are providing the resampled 3D shifts (using pixel size of outputs) when --resample_ratio!=1.0. By default, the program assums the provided shifts are not resampled. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_x"; token.key_prefix = "--"; token.label = "3D x-shift [Pixels]"; token.help = "3D x-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_y"; token.key_prefix = "--"; token.label = "3D y-shift [Pixels]"; token.help = "3D y-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "shift3d_z"; token.key_prefix = "--"; token.label = "3D z-shift [Pixels]"; token.help = "3D z-shift value. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "aa"; token.key_prefix = "--"; token.label = "Low-pass filter fall-off [1/Pixels]"; token.help = "Low-pass filter fall-off in absolute frequency. The program applies this low-pass filter before the moon elimination. Effective only when --fl &gt; 0.0. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['fl', '-1.0', 'True']]; token.default = "0.1"; token.restore = [['0.1'], ['0.1']]; token.type = "float"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('fl', []).append([token.key_base, '-1.0', 'True'])
		token = SXcmd_token(); token.key_base = "debug"; token.key_prefix = "--"; token.label = "Run with debug mode"; token.help = "Mainly for developers. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token

		sxcmd_list.append(sxcmd)

		sxcmd = SXcmd(); sxcmd.name = "sp_mask"; sxcmd.subname = ""; sxcmd.mode = ""; sxcmd.subset_config = ""; sxcmd.label = "Masking"; sxcmd.short_info = "Mask creation tool for 2D or 3D masks."; sxcmd.mpi_support = False; sxcmd.mpi_add_flag = False; sxcmd.category = "sxc_subtract"; sxcmd.role = "sxr_util"; sxcmd.is_submittable = True
		token = SXcmd_token(); token.key_base = "input_volume"; token.key_prefix = ""; token.label = "Input image"; token.help = "Path to the 2D image or 3D Volume "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "data3d_one"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "output_directory"; token.key_prefix = ""; token.label = "Output directory"; token.help = "Output direcory path "; token.group = "main"; token.is_required = True; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = ""; token.restore = [[""], [""]]; token.type = "dir"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "prefix"; token.key_prefix = "--"; token.label = "Output prefix"; token.help = "Prefix of the produced files "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "sp_mask"; token.restore = [['sp_mask'], ['sp_mask']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "overwrite"; token.key_prefix = "--"; token.label = "Overwrite outputs"; token.help = "Overwrite the output mask in case it exists already. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "pixel_size"; token.key_prefix = "--"; token.label = "Pixel size [A/px]"; token.help = "Pixel size of the volume. Used for filtering and the molcular mask threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "1.0"; token.restore = [['1.0'], ['1.0']]; token.type = "apix"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "use_mol_mass"; token.key_prefix = "--"; token.label = "Use molecular mass"; token.help = "GUI OPTION ONLY - Define if one want to use the molecular mass option as a masking threshold. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['nsigma', 'none', 'False']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool_ignore"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False'])
		token = SXcmd_token(); token.key_base = "mol_mass"; token.key_prefix = "--"; token.label = "Molecular mass [kDa]"; token.help = "The estimated molecular mass of the target particle in kilodaltons. This is used to calculate the binarization threshold automatically. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['use_mol_mass', 'True', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "mass"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'True', 'False'])
		token = SXcmd_token(); token.key_base = "threshold"; token.key_prefix = "--"; token.label = "Binarization threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the input structure. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['nsigma', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('nsigma', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "nsigma"; token.key_prefix = "--"; token.label = "Density standard deviation threshold"; token.help = "Defines the threshold used in the first step of the processing to generate a binary version of the structure. The threshold is set to &lt;= mean + (nsigma x standard deviations). This option will not be used if the option threshold is none. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', ''], ['threshold', 'none', 'False'], ['use_mol_mass', 'False', 'False']]; token.default = "none"; token.restore = [['none'], ['none']]; token.type = "string"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token; sxcmd.dependency_dict.setdefault('threshold', []).append([token.key_base, 'none', 'False']); sxcmd.dependency_dict.setdefault('use_mol_mass', []).append([token.key_base, 'False', 'False'])
		token = SXcmd_token(); token.key_base = "ndilation"; token.key_prefix = "--"; token.label = "Number of dilations"; token.help = "The pixel width to dilate the 3D binary volume corresponding to the specified molecular mass or density threshold prior to softening the edge. One cycle of dilation will add about 2 pixels to the mask. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "3"; token.restore = [['3'], ['3']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_width"; token.key_prefix = "--"; token.label = "Soft-edge width [Pixels]"; token.help = "The pixel width of transition area for soft-edged masking. If the width is 0, a binary mask is returned. "; token.group = "main"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "5"; token.restore = [['5'], ['5']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "allow_disconnected"; token.key_prefix = "--"; token.label = "Allow disconnected regions"; token.help = "Allow disconnected region in the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "fill_mask"; token.key_prefix = "--"; token.label = "Fill mask"; token.help = "Fills empty spaces inside a map. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = False; token.restore = [[False], [False]]; token.type = "bool"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "nerosion"; token.key_prefix = "--"; token.label = "Number of erosions"; token.help = "Number of times to erode binarized volume. One cycle of erosion will remove about 2 pixels from the mask. "; token.group = "advanced"; token.is_required = False; token.is_locked = False; token.is_reversed = False; token.filament_tab = ""; token.dependency_group = [['', '', '']]; token.default = "0"; token.restore = [['0'], ['0']]; token.type = "int"; sxcmd.token_list.append(token); sxcmd.token_dict[token.key_base] = token
		token = SXcmd_token(); token.key_base = "edge_type"; token.key_prefix = "--"; token.label = "Soft-edge type"; token.help = "The type of soft-edge. Available methods are (1) \'cosine\' for cosine soft-edged (used in PostRefiner) and (2) \'gaussian\' for gaussian soft-edge. "; token.group = "advanced"; token.is_required = False; token.is_locked