"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplicaProvider = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const path = require("path");
class ReplicaProvider extends core_1.NestedStack {
    constructor(scope, id) {
        super(scope, id);
        const code = lambda.Code.fromAsset(path.join(__dirname, 'replica-handler'));
        // Issues UpdateTable API calls
        this.onEventHandler = new lambda.Function(this, 'OnEventHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.onEventHandler',
            timeout: core_1.Duration.minutes(5),
        });
        // Checks if table is back to `ACTIVE` state
        this.isCompleteHandler = new lambda.Function(this, 'IsCompleteHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.isCompleteHandler',
            timeout: core_1.Duration.seconds(30),
        });
        // Allows the creation of the `AWSServiceRoleForDynamoDBReplication` service linked role
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'iam',
                    region: '',
                    resource: 'role',
                    resourceName: 'aws-service-role/replication.dynamodb.amazonaws.com/AWSServiceRoleForDynamoDBReplication',
                })],
        }));
        // Required for replica table creation
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:DescribeLimits'],
            resources: ['*'],
        }));
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: this.onEventHandler,
            isCompleteHandler: this.isCompleteHandler,
            queryInterval: core_1.Duration.seconds(10),
        });
    }
    /**
     * Creates a stack-singleton resource provider nested stack.
     */
    static getOrCreate(scope) {
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-dynamodb.ReplicaProvider';
        return stack.node.tryFindChild(uid) || new ReplicaProvider(stack, uid);
    }
}
exports.ReplicaProvider = ReplicaProvider;
//# sourceMappingURL=data:application/json;base64,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