"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachineCustomResourceProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
/**
 * A state machine custom resource provider
 */
class StateMachineCustomResourceProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const cfnResponseSuccessFn = this.createCfnResponseFn('Success');
        const cfnResponseFailedFn = this.createCfnResponseFn('Failed');
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [cfnResponseSuccessFn.functionArn, cfnResponseFailedFn.functionArn],
        }));
        // https://docs.aws.amazon.com/step-functions/latest/dg/stepfunctions-iam.html
        // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution'],
            resources: [props.stateMachine.stateMachineArn],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution', 'states:StopExecution'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'states',
                    resource: 'execution',
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: `${aws_cdk_lib_1.Stack.of(this).splitArn(props.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                })],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'events',
                    resource: 'rule',
                    resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                })],
        }));
        const definition = aws_cdk_lib_1.Stack.of(this).toJsonString({
            StartAt: 'StartExecution',
            States: {
                StartExecution: {
                    Type: 'Task',
                    Resource: 'arn:aws:states:::states:startExecution.sync:2',
                    Parameters: {
                        'Input.$': '$',
                        'StateMachineArn': props.stateMachine.stateMachineArn,
                    },
                    TimeoutSeconds: (props.timeout ?? aws_cdk_lib_1.Duration.minutes(30)).toSeconds(),
                    Next: 'CfnResponseSuccess',
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'CfnResponseFailed',
                        }],
                },
                CfnResponseSuccess: {
                    Type: 'Task',
                    Resource: cfnResponseSuccessFn.functionArn,
                    End: true,
                },
                CfnResponseFailed: {
                    Type: 'Task',
                    Resource: cfnResponseFailedFn.functionArn,
                    End: true,
                },
            },
        });
        const stateMachine = new aws_cdk_lib_1.CfnResource(this, 'StateMachine', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        stateMachine.node.addDependency(role);
        const startExecution = new lambda.Function(this, 'StartExecution', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            handler: 'index.startExecution',
            runtime: lambda.Runtime.NODEJS_14_X,
        });
        startExecution.addToRolePolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution'],
            resources: [stateMachine.ref],
        }));
        startExecution.addEnvironment('STATE_MACHINE_ARN', stateMachine.ref);
        this.serviceToken = startExecution.functionArn;
    }
    createCfnResponseFn(status) {
        return new lambda.Function(this, `CfnResponse${status}`, {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            handler: `index.cfnResponse${status}`,
            runtime: lambda.Runtime.NODEJS_14_X,
        });
    }
}
exports.StateMachineCustomResourceProvider = StateMachineCustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
StateMachineCustomResourceProvider[_a] = { fqn: "cloudstructs.StateMachineCustomResourceProvider", version: "0.6.3" };
//# sourceMappingURL=data:application/json;base64,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