# AUTOGENERATED FROM: babilim/model/layers/convolution.ipynb

# Cell: 0
"""doc
# babilim.model.layers.convolution

> Convolution for 1d and 2d.
"""

# Cell: 1
from typing import Optional, Any, Tuple
from babilim.core.annotations import RunOnlyOnce
from babilim.core.module_native import ModuleNative
from babilim.model.layers.activation import Activation


# Cell: 2
class Conv1D(ModuleNative):
    def __init__(self, filters: int, kernel_size: int, padding: Optional[str] = None, stride: int = 1, dilation_rate: int = 1, kernel_initializer: Optional[Any] = None, activation=None):
        """
        A 1d convolution layer.
    
        :param filters: The number of filters in the convolution. Defines the number of output channels.
        :param kernel_size: The kernel size of the convolution. Defines the area over which is convolved. Typically 1, 3 or 5 are recommended.
        :param padding: What type of padding should be applied. The string "none" means no padding is applied, None or "same" means the input is padded in a way that the output stays the same size if no stride is applied.
        :param stride: The offset between two convolutions that are applied. Typically 1. Stride affects also the resolution of the output feature map. A stride 2 halves the resolution, since convolutions are only applied every odd pixel.
        :param dilation_rate: The dilation rate for a convolution.
        :param kernel_initializer: A kernel initializer function. By default orthonormal weight initialization is used.
        :param activation: The activation function that should be added after the dense layer.
        """
        super().__init__()
        self.filters = filters
        self.kernel_size = kernel_size
        self.padding = padding
        self.dilation = dilation_rate
        self.stride = stride
        self.kernel_initializer = kernel_initializer
        self.activation = Activation(activation)
        
    @RunOnlyOnce
    def _build_pytorch(self, features):
        import torch
        from torch.nn import Conv1d as _Conv1d
        if self.kernel_initializer is None:
            from torch.nn.init import orthogonal_
            self.kernel_initializer = orthogonal_
        if self.padding == "same" or self.padding is None:
            self.padding = int((self.kernel_size - 1) / 2)
        elif self.padding == "none":
            self.padding = 0
        else:
            raise NotImplementedError("Padding {} is not implemented.".format(padding))
        in_channels = features.shape[1]
        self.conv = _Conv1d(in_channels, self.filters, self.kernel_size, self.stride, self.padding, self.dilation)
        self.conv.weight.data = self.kernel_initializer(self.conv.weight.data)
        if torch.cuda.is_available():
            self.conv = self.conv.to(torch.device("cuda"))  # TODO move to correct device
        from babilim.core.tensor_pt import Tensor as _Tensor
        self.weight = _Tensor(data=None, trainable=True, native=self.conv.weight)
        self.bias = _Tensor(data=None, trainable=True, native=self.conv.bias)
        
    def _call_pytorch(self, features):
        return self.activation(self.conv(features))
    
    @RunOnlyOnce
    def _build_tf(self, features):
        #TODO Implement
        raise NotImplementedError()
        
    def _call_tf(self, features):
        #TODO Implement
        raise NotImplementedError()


# Cell: 3
class Conv2D(ModuleNative):
    def __init__(self, filters: int, kernel_size: Tuple[int, int], padding: Optional[str] = None, strides: Tuple[int, int] = (1, 1), dilation_rate: Tuple[int, int] = (1, 1), kernel_initializer: Optional[Any] = None, activation=None):
        """
        A 2d convolution layer.
    
        :param filters: The number of filters in the convolution. Defines the number of output channels.
        :param kernel_size: The kernel size of the convolution. Defines the area over which is convolved. Typically (1,1) (3,3) or (5,5) are recommended.
        :param padding: What type of padding should be applied. The string "none" means no padding is applied, None or "same" means the input is padded in a way that the output stays the same size if no stride is applied.
        :param stride: The offset between two convolutions that are applied. Typically (1, 1). Stride affects also the resolution of the output feature map. A stride 2 halves the resolution, since convolutions are only applied every odd pixel.
        :param dilation_rate: The dilation rate for a convolution.
        :param kernel_initializer: A kernel initializer function. By default orthonormal weight initialization is used.
        :param activation: The activation function that should be added after the dense layer.
        """
        super().__init__()
        self.filters = filters
        self.kernel_size = kernel_size
        self.padding = padding
        self.dilation = dilation_rate
        self.stride = strides
        self.kernel_initializer = kernel_initializer
        self.activation = Activation(activation)
        
    @RunOnlyOnce
    def _build_pytorch(self, features):
        import torch
        from torch.nn import Conv2d as _Conv2d
        if self.kernel_initializer is None:
            from torch.nn.init import orthogonal_
            self.kernel_initializer = orthogonal_
        if self.padding == "same" or self.padding is None:
            px = int((self.kernel_size[0] - 1) / 2)
            py = int((self.kernel_size[1] - 1) / 2)
            self.padding = (px, py)
        elif self.padding == "none":
            self.padding = (0, 0)
        else:
            raise NotImplementedError("Padding {} is not implemented.".format(padding))
        in_channels = features.shape[1]
        self.conv = _Conv2d(in_channels, self.filters, self.kernel_size, self.stride, self.padding, self.dilation)
        self.conv.weight.data = self.kernel_initializer(self.conv.weight.data)
        if torch.cuda.is_available():
            self.conv = self.conv.to(torch.device("cuda"))  # TODO move to correct device
        from babilim.core.tensor_pt import Tensor as _Tensor
        self.weight = _Tensor(data=None, trainable=True, native=self.conv.weight)
        self.bias = _Tensor(data=None, trainable=True, native=self.conv.bias)
        
    def _call_pytorch(self, features):
        return self.activation(self.conv(features))
    
    @RunOnlyOnce
    def _build_tf(self, features):
        from tensorflow.keras.layers import Conv2D as _Conv2D
        if self.kernel_initializer is None:
            from tensorflow.keras.initializers import Orthogonal
            self.kernel_initializer = Orthogonal()
        if self.padding is None:
            self.padding = "same"
        self.conv = _Conv2D(filters=self.filters, kernel_size=self.kernel_size, strides=self.stride, dilation_rate=self.dilation_rate, padding=self.padding, activation=None, kernel_initializer=self.kernel_initializer)
        self.conv.build(features.shape)
        from babilim.core.tensor_tf import Tensor as _Tensor
        self.weight = _Tensor(data=None, trainable=True, native=self.conv.kernel)
        self.bias = _Tensor(data=None, trainable=True, native=self.conv.bias)

    def _call_tf(self, features):
        raise self.activation(self.conv(features))
