import os

import requests
import tldextract
from dotenv import load_dotenv

load_dotenv("/isearch/.env", override=True)
address = os.getenv("address")
def getPath(user_name):
    """
        环境变量设置
        windows
        setx address "192.168.12.249"

        linux
        export address = "192.168.12.249"
        source /etc/profile
    :param port:
    :return:
    """

    # address = os.environ.get("address", "192.168.12.249")

    url = f"{address}/client/getPath"
    print(url)
    headers = {
        'Content-Type': 'application/json',  # 说明请求体是 JSON 格式
    }
    # 请求体的数据
    data = {
        'user_name': user_name
    }
    print(data)
    response = requests.post(url, headers=headers, json=data)
    if response.status_code != 200:
        print("请求失败，状态码：", response.status_code)
        return "failure"
    return response.json()


def upload_file(file_path, dest_file, user_name):
    """
    通知客户端上传文件
    :param file_path:待上传文件
    :param dest_file:上传目标目录
    :param port:
    :return:
    """

    # address = os.environ.get("address", "192.168.12.249")
    url = f"{address}/client/noticeUpload"
    print(url)
    headers = {
        'Content-Type': 'application/json',  # 说明请求体是 JSON 格式
    }
    # 请求体的数据
    data = {
        'file_path': file_path,
        'dest_file': dest_file,
        'user_name': user_name
    }
    print(data)
    requests.post(url, headers=headers, json=data)


def vCode(image: str, code_type, apiKey, secretKey):
    """
    ocr 识别图片验证码
    :param image: 图片base64
    :param code_type: 7000:问答题、智能回答题；8000:不定长度英文数字汉字混合；9000:坐标选一（滑动）；9001:坐标多选，返回1~4个坐标；9002:坐标多选，返回3~5个坐标；9003:点击两个相同的字
    :param apiKey:
    :param secretKey:
    :return:
    """
    url = "https://ai.i-search.com.cn/ocr/v2/vCode"
    print(url)
    headers = {
        'Content-Type': 'application/json',  # 说明请求体是 JSON 格式
    }

    # 请求体的数据
    data = {
        'image': image,
        'code_type': code_type,
        'apiKey': apiKey,
        'secretKey': secretKey
    }
    print(data)
    response = requests.post(url, headers=headers, json=data)
    status_code = response.status_code
    if status_code != 200:
        print("请求失败，状态码：", status_code)
        return {"error_msg": "failure", "error_code": status_code}
    return response.json()


def save_file(url, cookies, file_path):
    """
    playwright保存文件
    """
    extracted = tldextract.extract(url)
    top_level_domain = f"{extracted.domain}.{extracted.suffix}"
    cookie = {}
    for item in cookies:
        if top_level_domain in item.get("domain"):
            cookie[item["name"]] = item["value"]
    response = requests.get(url, cookies=cookie)
    with open(file_path, 'wb') as file:
        file.write(response.content)


# {'error_msg': 'Image recognize error', 'error_code': -1}
if __name__ == '__main__':
    print(vCode(
        "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",
        9000, "19a18b361a324a39a3c7b6d8e7f5d301", "2c4f629a48d64642944164ec05a28b1f"))
