"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHost = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The database is located in an isolated subnet, meaning that it is not accessible from the public internet. As such, to interact with the database directly, a user must tunnel through a bastion host.
 *
 * ### Configuring
 *
 * This codebase controls _who_ is allowed to connect to the bastion host. This requires two steps:
 *
 * 1. Adding the IP address from which you are connecting to the `ipv4Allowlist` array
 * 1. Creating a bastion host system user by adding the user's configuration inform to `userdata.yaml`
 *
 * #### Adding an IP address to the `ipv4Allowlist` array
 *
 * The `BastionHost` construct takes in an `ipv4Allowlist` array as an argument. Find your IP address (eg `curl api.ipify.org`) and add that to the array along with the trailing CIDR block (likely `/32` to indicate that you are adding a single IP address).
 *
 * #### Creating a user via `userdata.yaml`
 *
 * Add an entry to the `users` array with a username (likely matching your local systems username, which you can get by running the `whoami` command in your terminal) and a public key (likely your default public key, which you can get by running `cat ~/.ssh/id_*.pub` in your terminal).
 *
 * #### Tips & Tricks when using the Bastion Host
 *
 * **Connecting to RDS Instance via SSM**
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID \
 * --document-name AWS-StartPortForwardingSessionToRemoteHost \
 * --parameters '{
 * "host": [
 * "example-db.c5abcdefghij.us-west-2.rds.amazonaws.com"
 * ],
 * "portNumber": [
 * "5432"
 * ],
 * "localPortNumber": [
 * "9999"
 * ]
 * }' \
 * --profile $AWS_PROFILE
 * ```
 *
 * ```sh
 * psql -h localhost -p 9999 # continue adding username (-U) and db (-d) here...
 * ```
 *
 * Connect directly to Bastion Host:
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID --profile $AWS_PROFILE
 * ```
 *
 * **Setting up an SSH tunnel**
 *
 * In your `~/.ssh/config` file, add an entry like:
 *
 * ```
 * Host db-tunnel
 * Hostname {the-bastion-host-address}
 * LocalForward 54322 {the-db-hostname}:5432
 * ```
 *
 * Then a tunnel can be opened via:
 *
 * ```
 * ssh -N db-tunnel
 * ```
 *
 * And a connection to the DB can be made via:
 *
 * ```
 * psql -h 127.0.0.1 -p 5433 -U {username} -d {database}
 * ```
 *
 * **Handling `REMOTE HOST IDENTIFICATION HAS CHANGED!` error**
 *
 * If you've redeployed a bastion host that you've previously connected to, you may see an error like:
 *
 * ```
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * @    WARNING: REMOTE HOST IDENTIFICATION HAS CHANGED!     @
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY!
 * Someone could be eavesdropping on you right now (man-in-the-middle attack)!
 * It is also possible that a host key has just been changed.
 * The fingerprint for the ECDSA key sent by the remote host is
 * SHA256:mPnxAOXTpb06PFgI1Qc8TMQ2e9b7goU8y2NdS5hzIr8.
 * Please contact your system administrator.
 * Add correct host key in /Users/username/.ssh/known_hosts to get rid of this message.
 * Offending ECDSA key in /Users/username/.ssh/known_hosts:28
 * ECDSA host key for ec2-12-34-56-789.us-west-2.compute.amazonaws.com has changed and you have requested strict checking.
 * Host key verification failed.
 * ```
 *
 * This is due to the server's fingerprint changing. We can scrub the fingerprint from our system with a command like:
 *
 * ```
 * ssh-keygen -R 12.34.56.789
 * ```
 *
 */
class BastionHost extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { stackName } = aws_cdk_lib_1.Stack.of(this);
        // Build ec2 instance
        this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, "bastion-host", {
            vpc: props.vpc,
            vpcSubnets: { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC },
            instanceName: `${stackName} bastion host`,
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE4_GRAVITON, aws_cdk_lib_1.aws_ec2.InstanceSize.NANO),
            machineImage: aws_cdk_lib_1.aws_ec2.MachineImage.latestAmazonLinux({
                generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.ARM_64,
            }),
            userData: props.userData,
            userDataCausesReplacement: true,
        });
        // Assign elastic IP
        if (props.createElasticIp ?? true) {
            new aws_cdk_lib_1.aws_ec2.CfnEIP(this, "IP", {
                instanceId: this.instance.instanceId,
                tags: [{ key: "Name", value: stackName }],
            });
        }
        // Allow bastion host to connect to db
        this.instance.connections.allowTo(props.db.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connection from bastion host");
        // Allow IP access to bastion host
        for (const ipv4 of props.ipv4Allowlist) {
            this.instance.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4(ipv4), aws_cdk_lib_1.aws_ec2.Port.tcp(props.sshPort || 22), "SSH Access");
        }
        // Integrate with SSM
        this.instance.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: [
                "ssmmessages:*",
                "ssm:UpdateInstanceInformation",
                "ec2messages:*",
            ],
            resources: ["*"],
        }));
        new aws_cdk_lib_1.CfnOutput(this, "instance-id-output", {
            value: this.instance.instanceId,
            exportName: `${stackName}-instance-id`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-ip-output", {
            value: this.instance.instancePublicIp,
            exportName: `${stackName}-instance-public-ip`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-dns-name-output", {
            value: this.instance.instancePublicDnsName,
            exportName: `${stackName}-public-dns-name`,
        });
    }
}
exports.BastionHost = BastionHost;
_a = JSII_RTTI_SYMBOL_1;
BastionHost[_a] = { fqn: "cdk-pgstac.BastionHost", version: "2.6.0" };
//# sourceMappingURL=data:application/json;base64,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