# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nopdb']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'nopdb',
    'version': '0.1.0rc1',
    'description': 'Non-interactive Python debugger',
    'long_description': 'NoPdb: Non-interactive Python Debugger\n======================================\n|pypi-package| |docs-status| |test-status| |lint-status|\n\nNoPdb is a **programmatic** (non-interactive) **debugger** for Python. This means it gives you access to\n**debugger-like superpowers** directly from your code. With NoPdb, you can:\n\n* **capture function calls**, including arguments, local variables, return values and stack traces\n* **set "breakpoints"** that trigger user-defined actions when hit, namely:\n\n  * **evaluate expressions** to retrieve their values later\n  * **execute arbitrary code**, including modifying local variables\n  * **enter an interactive debugger** like `pdb`\n\nNoPdb is also a convenient tool for inspecting **machine learning model internals**. For example, `this notebook <https://colab.research.google.com/github/cifkao/nopdb/blob/main/docs/pytorch_tutorial.ipynb>`_ shows how to\nuse it to visualize Transformer attention in PyTorch.\n\nNoPdb should run at least under CPython and PyPy. Most features should work under any implementation\nas long as it has :code:`sys.settrace()`.\n\n**Note:** This project is in its early development stage. Contributions and improvement ideas are welcome.\n\nInstallation\n------------\n\n:code:`pip install nopdb`\n\nCapturing function calls\n------------------------\n\nThe functions :code:`capture_call()` and :code:`capture_calls()` allow\ncapturing useful information about calls to a given function.\nThey are typically used as context managers, e.g.:\n\n.. code-block:: python\n\n    with nopdb.capture_calls(fn) as calls:\n        some_code_that_calls_fn()\n\n        print(calls)  # see details about how fn() was called\n\nThe information we can retrieve includes the function\'s arguments, return value, local variables and stack trace. For example:\n\n.. code-block:: python\n\n    >>> with nopdb.capture_call(f) as call:\n    ...     g(1)\n    >>> call\n    CallCapture(name=\'f\', args=OrderedDict(x=1, y=1), return_value=4)\n    >>> call.print_stack()\n      File "<stdin>", line 2, in <module>\n      File "<stdin>", line 2, in g\n      File "<stdin>", line 1, in f\n    >>> call.args[\'x\']\n    1\n    >>> call.return_value\n    4\n    >>> call.locals\n    {\'y\': 1, \'x\': 1, \'z\': 2}\n\nSetting breakpoints\n-------------------\n\nLike conventional debuggers, NoPdb can set breakpoints. However, because NoPdb is a\n*non-interactive* debugger, its breakpoints do not actually stop the execution of the\nprogram. Instead, they allow executing actions scheduled in advance, such as\nevaluating expressions.\n\nTo set a breakpoint, call the :code:`breakpoint()` function. A breakpoint object\nis returned, allowing to schedule actions using its methods such as\n:code:`eval()` and :code:`exec()`. For example:\n\n.. code-block:: python\n\n   # Break at line 3 of the file or notebook cell where f is defined\n   with nopdb.breakpoint(function=f, line=3) as bp:\n       x = bp.eval("x")             # Schedule an expression\n       type_y = bp.eval("type(y)")  # Another one\n       bp.exec("print(y)")          # Schedule a print statement\n\n       some_code_that_calls_f()\n\n   print(x, type_y)  # Retrieve the captured values\n\nThere are other ways to specify the breakpoint location. For example:\n\n.. code-block:: python\n\n   # Break at any line with the given source code in the given file\n   with nopdb.breakpoint(file="pathlib.py", line="return obj") as bp:\n       ...\n\n   # Break as soon as any function with the given name is called\n   with nopdb.breakpoint(function="myfunc") as bp:\n       ...\n\nNot only can we capture values, we can also modify them!\n\n.. code-block:: python\n\n    >>> with nopdb.breakpoint(function=f, line=3) as bp:\n    ...     # Get the value of x, then increment it, then get the new value\n    ...     x_before = bp.eval(\'x\')\n    ...     bp.exec(\'x += 1\')\n    ...     x_after = bp.eval(\'x\')\n    ...\n    ...     some_code_that_calls_f()\n    >>> x_before\n    [2]\n    >>> x_after\n    [3]\n\n.. |pypi-package| image:: https://badge.fury.io/py/nopdb.svg?\n   :target: https://pypi.org/project/nopdb/\n   :alt: PyPI Package\n.. |docs-status| image:: https://readthedocs.org/projects/nopdb/badge/?version=latest\n   :target: https://nopdb.readthedocs.io/en/latest/?badge=latest\n   :alt: Documentation Status\n.. |test-status| image:: https://github.com/cifkao/nopdb/actions/workflows/test.yml/badge.svg\n   :target: https://github.com/cifkao/nopdb/actions/workflows/test.yml\n   :alt: Lint Status\n.. |lint-status| image:: https://github.com/cifkao/nopdb/actions/workflows/lint.yml/badge.svg\n   :target: https://github.com/cifkao/nopdb/actions/workflows/lint.yml\n   :alt: Lint Status\n',
    'author': 'Ondřej Cífka',
    'author_email': 'cifkao@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cifkao/nopdb',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
