"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    /**
     * The ARN of the key.
     *
     * @stability stable
     */
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    /**
     * The ID of the key (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     *
     * @stability stable
     */
    get keyId() {
        return this.aliasName;
    }
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement, allowNoOp) {
        return this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management Service (AWS KMS).
 *
 * Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @stability stable
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias.
     * @stability stable
     */
    static fromAliasAttributes(scope, id, attrs) {
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app, by the alias name.
     *
     * This method should be used
     * instead of 'fromAliasAttributes' when the underlying KMS Key ARN is not available.
     * This Alias will not have a direct reference to the KMS Key, so addAlias and grant* methods are not supported.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param aliasName The full name of the KMS Alias (e.g., 'alias/aws/s3', 'alias/myKeyAlias').
     * @stability stable
     */
    static fromAliasName(scope, id, aliasName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.keyArn = core_1.Stack.of(this).formatArn({ service: 'kms', resource: aliasName });
                this.keyId = aliasName;
                this.aliasName = aliasName;
            }
            get aliasTargetKey() { throw new Error('Cannot access aliasTargetKey on an Alias imported by Alias.fromAliasName().'); }
            addAlias(_alias) { throw new Error('Cannot call addAlias on an Alias imported by Alias.fromAliasName().'); }
            addToResourcePolicy(_statement, _allowNoOp) {
                return { statementAdded: false };
            }
            grant(grantee, ..._actions) { return iam.Grant.drop(grantee, ''); }
            grantDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncryptDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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