import numpy as np
import pandas as pd
from typing import List, Union

from sklearn.preprocessing import StandardScaler
from sklearn.decomposition import PCA
from sklearn.covariance import EmpiricalCovariance

from cytominer_eval.transform import metric_melt
from cytominer_eval.transform.util import (
    set_pair_ids,
    check_grit_replicate_summary_method,
)


def assign_replicates(
    similarity_melted_df: pd.DataFrame,
    replicate_groups: List[str],
) -> pd.DataFrame:
    r"""Determine which profiles should be considered replicates.

    Given an elongated pairwise correlation matrix with metadata annotations, determine
    how to assign replicate information.

    Parameters
    ----------
    similarity_melted_df : pandas.DataFrame
        Long pandas DataFrame of annotated pairwise correlations output from
        :py:func:`cytominer_eval.transform.transform.metric_melt`.
    replicate_groups : list
        a list of metadata column names in the original profile dataframe used to
        indicate replicate profiles.

    Returns
    -------
    pd.DataFrame
        A similarity_melted_df but with added columns indicating whether or not the
        pairwise similarity metric is comparing replicates or not. Used in most eval
        operations.
    """
    pair_ids = set_pair_ids()
    replicate_col_names = {x: "{x}_replicate".format(x=x) for x in replicate_groups}

    compare_dfs = []
    for replicate_col in replicate_groups:
        replicate_cols_with_suffix = [
            "{col}{suf}".format(col=replicate_col, suf=pair_ids[x]["suffix"])
            for x in pair_ids
        ]

        assert all(
            [x in similarity_melted_df.columns for x in replicate_cols_with_suffix]
        ), "replicate_group not found in melted dataframe columns"

        replicate_col_name = replicate_col_names[replicate_col]

        compare_df = similarity_melted_df.loc[:, replicate_cols_with_suffix]
        compare_df.loc[:, replicate_col_name] = False

        compare_df.loc[
            np.where(compare_df.iloc[:, 0] == compare_df.iloc[:, 1])[0],
            replicate_col_name,
        ] = True
        compare_dfs.append(compare_df)

    compare_df = pd.concat(compare_dfs, axis="columns").reset_index(drop=True)
    compare_df = compare_df.assign(
        group_replicate=compare_df.loc[:, replicate_col_names.values()].min(
            axis="columns"
        )
    ).loc[:, list(replicate_col_names.values()) + ["group_replicate"]]

    similarity_melted_df = similarity_melted_df.merge(
        compare_df, left_index=True, right_index=True
    )
    return similarity_melted_df


def calculate_precision_recall(replicate_group_df: pd.DataFrame, k: int) -> pd.Series:
    """Given an elongated pairwise correlation dataframe of replicate groups, calculate
    precision and recall.

    Usage: Designed to be called within a pandas.DataFrame().groupby().apply(). See
    :py:func:`cytominer_eval.operations.precision_recall.precision_recall`.

    Parameters
    ----------
    replicate_group_df : pandas.DataFrame
        An elongated dataframe storing pairwise correlations of all profiles to a single
        replicate group.
    k : int
        an integer indicating how many pairwise comparisons to threshold.

    Returns
    -------
    dict
        A return bundle of identifiers (k) and results (precision and recall at k).
        The dictionary has keys ("k", "precision", "recall").
    """
    assert (
        "group_replicate" in replicate_group_df.columns
    ), "'group_replicate' not found in dataframe; remember to call assign_replicates()."

    recall_denom__total_relevant_items = sum(replicate_group_df.group_replicate)
    precision_denom__num_recommended_items = k

    num_recommended_items_at_k = sum(
        replicate_group_df.iloc[
            :k,
        ].group_replicate
    )

    precision_at_k = num_recommended_items_at_k / precision_denom__num_recommended_items
    recall_at_k = num_recommended_items_at_k / recall_denom__total_relevant_items

    return_bundle = {"k": k, "precision": precision_at_k, "recall": recall_at_k}

    return pd.Series(return_bundle)


def calculate_grit(
    replicate_group_df: pd.DataFrame,
    control_perts: List[str],
    column_id_info: dict,
    replicate_summary_method: str = "mean",
) -> pd.Series:
    """Given an elongated pairwise correlation dataframe of replicate groups,
    calculate grit.

    Usage: Designed to be called within a pandas.DataFrame().groupby().apply(). See
    :py:func:`cytominer_eval.operations.grit.grit`.

    Parameters
    ----------
    replicate_group_df : pandas.DataFrame
        An elongated dataframe storing pairwise correlations of all profiles to a single
        replicate group.
    control_perts : list
        The profile_ids that should be considered controls (the reference)
    column_id_info: dict
        A dictionary of column identifiers noting profile and replicate group ids. This
        variable is autogenerated in
        :py:func:`cytominer_eval.transform.util.set_grit_column_info`.
    replicate_summary_method : {'mean', 'median'}, optional
        how replicate z-scores to control perts are summarized. Defaults to "mean".

    Returns
    -------
    dict
        A return bundle of identifiers (perturbation, group) and results (grit score).
        The dictionary has keys ("perturbation", "group", "grit_score"). "grit_score"
        will be NaN if no other profiles exist in the defined group.
    """
    # Confirm that we support the provided summary method
    check_grit_replicate_summary_method(replicate_summary_method)

    group_entry = get_grit_entry(replicate_group_df, column_id_info["group"]["id"])
    pert = get_grit_entry(replicate_group_df, column_id_info["profile"]["id"])

    # Define distributions for control perturbations
    control_distrib = replicate_group_df.loc[
        replicate_group_df.loc[:, column_id_info["profile"]["comparison"]].isin(
            control_perts
        ),
        "similarity_metric",
    ].values.reshape(-1, 1)

    assert len(control_distrib) > 1, "Error! No control perturbations found."

    # Define distributions for same group (but not same perturbation)
    same_group_distrib = replicate_group_df.loc[
        (
            replicate_group_df.loc[:, column_id_info["group"]["comparison"]]
            == group_entry
        )
        & (replicate_group_df.loc[:, column_id_info["profile"]["comparison"]] != pert),
        "similarity_metric",
    ].values.reshape(-1, 1)

    return_bundle = {"perturbation": pert, "group": group_entry}
    if len(same_group_distrib) == 0:
        return_bundle["grit"] = np.nan

    else:
        scaler = StandardScaler()
        scaler.fit(control_distrib)
        grit_z_scores = scaler.transform(same_group_distrib)

        if replicate_summary_method == "mean":
            grit_score = np.mean(grit_z_scores)
        elif replicate_summary_method == "median":
            grit_score = np.median(grit_z_scores)

        return_bundle["grit"] = grit_score

    return pd.Series(return_bundle)


def get_grit_entry(df: pd.DataFrame, col: str) -> str:
    """Helper function to define the perturbation identifier of interest

    Grit must be calculated using unique perturbations. This may or may not mean unique
    perturbations.
    """
    entries = df.loc[:, col]
    assert (
        len(entries.unique()) == 1
    ), "grit is calculated for each perturbation independently"
    return str(list(entries)[0])


class MahalanobisEstimator:
    """
    Store location and dispersion estimators of the empirical distribution of data
    provided in an array and allow computation of statistical distances.

    Parameters
    ----------
    arr : {pandas.DataFrame, np.ndarray}
        the matrix used to calculate covariance

    Attributes
    ----------
    sigma : np.array
        Fitted covariance matrix of sklearn.covariance.EmpiricalCovariance()

    Methods
    -------
    mahalanobis(X)
        Computes mahalanobis distance between the input array (self.arr) and the X
        array as provided
    """

    def __init__(self, arr: Union[pd.DataFrame, np.ndarray]):
        self.sigma = EmpiricalCovariance().fit(arr)

    def mahalanobis(self, X: Union[pd.DataFrame, np.ndarray]) -> np.ndarray:
        """Compute the mahalanobis distance between the empirical distribution described
        by this object and points in an array `X`.

        Parameters
        ----------
        X : {pandas.DataFrame, np.ndarray}
            A samples by features array-like matrix to compute mahalanobis distance
            between self.arr

        Returns
        -------
        numpy.array
            Mahalanobis distance between the input array and the original sigma
        """
        return self.sigma.mahalanobis(X)


def calculate_mahalanobis(pert_df: pd.DataFrame, control_df: pd.DataFrame) -> pd.Series:
    """Given perturbation and control dataframes, calculate mahalanobis distance per
    perturbation

    Usage: Designed to be called within a pandas.DataFrame().groupby().apply(). See
    :py:func:`cytominer_eval.operations.util.calculate_mp_value`.

    Parameters
    ----------
    pert_df : pandas.DataFrame
        A pandas dataframe of replicate perturbations (samples by features)
    control_df : pandas.DataFrame
        A pandas dataframe of control perturbations (samples by features). Must have the
        same feature measurements as pert_df

    Returns
    -------
    float
        The mahalanobis distance between perturbation and control
    """
    assert len(control_df) > 1, "Error! No control perturbations found."

    # Get dispersion and center estimators for the control perturbations
    control_estimators = MahalanobisEstimator(control_df)

    # Distance between mean of perturbation and control
    maha = control_estimators.mahalanobis(np.array(np.mean(pert_df, 0)).reshape(1, -1))[
        0
    ]
    return maha


def default_mp_value_parameters():
    """Set the different default parameters used for mp-values.

    Returns
    -------
    dict
        A default parameter set with keys: rescale_pca (whether the PCA should be
        scaled by variance explained) and nb_permutations (how many permutations to
        calculate empirical p-value). Defaults to True and 100, respectively.
    """
    params = {"rescale_pca": True, "nb_permutations": 100}
    return params


def calculate_mp_value(
    pert_df: pd.DataFrame,
    control_df: pd.DataFrame,
    params: dict = {},
) -> pd.Series:
    """Given perturbation and control dataframes, calculate mp-value per perturbation

    Usage: Designed to be called within a pandas.DataFrame().groupby().apply(). See
    :py:func:`cytominer_eval.operations.mp_value.mp_value`.

    Parameters
    ----------
    pert_df : pandas.DataFrame
        A pandas dataframe of replicate perturbations (samples by features)
    control_df : pandas.DataFrame
        A pandas dataframe of control perturbations (samples by features). Must have the
        same feature measurements as pert_df
    params : {dict}, optional
        the parameters to use when calculating mp value. See
        :py:func:`cytominer_eval.operations.util.default_mp_value_parameters`.

    Returns
    -------
    float
        The mp value for the given perturbation

    """
    assert len(control_df) > 1, "Error! No control perturbations found."

    # Assign parameters
    p = default_mp_value_parameters()

    assert all(
        [x in p.keys() for x in params.keys()]
    ), "Unknown parameters provided. Only {e} are supported.".format(e=p.keys())
    for (k, v) in params.items():
        p[k] = v

    merge_df = pd.concat([pert_df, control_df]).reset_index(drop=True)

    # We reduce the dimensionality with PCA
    # so that 90% of the variance is conserved
    pca = PCA(n_components=0.9, svd_solver="full")
    pca_array = pca.fit_transform(merge_df)
    # We scale columns by the variance explained
    if p["rescale_pca"]:
        pca_array = pca_array * pca.explained_variance_ratio_
    # This seems useless, as the point of using the Mahalanobis
    # distance instead of the Euclidean distance is to be independent
    # of axes scales

    # Distance between mean of perturbation and control
    obs = calculate_mahalanobis(
        pert_df=pca_array[: pert_df.shape[0]],
        control_df=pca_array[-control_df.shape[0] :],
    )
    # In the paper's methods section it mentions the covariance used
    # might be modified to include variation of the perturbation as well.

    # Permutation test
    sim = np.zeros(p["nb_permutations"])
    pert_mask = np.zeros(pca_array.shape[0], dtype=bool)
    pert_mask[: pert_df.shape[0]] = 1
    for i in range(p["nb_permutations"]):
        pert_mask_perm = np.random.permutation(pert_mask)
        pert_perm = pca_array[pert_mask_perm]
        control_perm = pca_array[np.logical_not(pert_mask_perm)]
        sim[i] = calculate_mahalanobis(pert_df=pert_perm, control_df=control_perm)

    return np.mean([x >= obs for x in sim])
