"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const redshift_data_1 = require("./redshift-data");
const util_1 = require("./util");
async function handler(props, event) {
    const username = props.username;
    const tablePrivileges = props.tablePrivileges;
    const clusterProps = props;
    if (event.RequestType === 'Create') {
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { PhysicalResourceId: util_1.makePhysicalId(username, clusterProps, event.RequestId) };
    }
    else if (event.RequestType === 'Delete') {
        await revokePrivileges(username, tablePrivileges, clusterProps);
        return;
    }
    else if (event.RequestType === 'Update') {
        const { replace } = await updatePrivileges(username, tablePrivileges, clusterProps, event.OldResourceProperties);
        const physicalId = replace ? util_1.makePhysicalId(username, clusterProps, event.RequestId) : event.PhysicalResourceId;
        return { PhysicalResourceId: physicalId };
    }
    else {
        /* eslint-disable-next-line dot-notation */
        throw new Error(`Unrecognized event type: ${event['RequestType']}`);
    }
}
exports.handler = handler;
async function revokePrivileges(username, tablePrivileges, clusterProps) {
    await Promise.all(tablePrivileges.map(({ tableName, actions }) => {
        return redshift_data_1.executeStatement(`REVOKE ${actions.join(', ')} ON ${tableName} FROM ${username}`, clusterProps);
    }));
}
async function grantPrivileges(username, tablePrivileges, clusterProps) {
    await Promise.all(tablePrivileges.map(({ tableName, actions }) => {
        return redshift_data_1.executeStatement(`GRANT ${actions.join(', ')} ON ${tableName} TO ${username}`, clusterProps);
    }));
}
async function updatePrivileges(username, tablePrivileges, clusterProps, oldResourceProperties) {
    const oldClusterProps = oldResourceProperties;
    if (clusterProps.clusterName !== oldClusterProps.clusterName || clusterProps.databaseName !== oldClusterProps.databaseName) {
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { replace: true };
    }
    const oldUsername = oldResourceProperties.username;
    if (oldUsername !== username) {
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { replace: true };
    }
    const oldTablePrivileges = oldResourceProperties.tablePrivileges;
    if (oldTablePrivileges !== tablePrivileges) {
        await revokePrivileges(username, oldTablePrivileges, clusterProps);
        await grantPrivileges(username, tablePrivileges, clusterProps);
        return { replace: false };
    }
    return { replace: false };
}
//# sourceMappingURL=data:application/json;base64,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