import { JsonPatch } from './json-patch';
/**
 * Options for `YamlFile`
 */
export interface YamlFileOptions {
    /**
     * The object that will be serialized. You can modify the object's contents
     * before synthesis.
     *
     * @default {} an empty object
     */
    readonly obj?: any;
}
/**
 * Represents a Yaml File.
 */
export declare class YamlFile {
    /**
     * The path to the file that the object will be written to.
     */
    private readonly filePath;
    /**
     * The output object. This object can be mutated until the project is synthesized.
     */
    private obj;
    /**
     * Patches to be applied to `obj` after the resolver is called.
     */
    private readonly patchOperations;
    constructor(filePath: string, options?: YamlFileOptions);
    /**
     * Update the output object.
     */
    update(obj: any): void;
    /**
     * Applies an RFC 6902 JSON-patch to the synthesized object file.
     * See https://datatracker.ietf.org/doc/html/rfc6902 for more information.
     *
     * For example, with the following yaml file
     * ```yaml
     * name: deploy
     * on:
     *   push:
     *     branches:
     *       - main
     *   workflow_dispatch: {}
     * ...
     * ```
     *
     * modified in the following way:
     *
     * ```ts
     * pipeline.workflowFile.patch(JsonPatch.add("/on/workflow_call", "{}"));
     * pipeline.workflowFile.patch(JsonPatch.remove("/on/workflow_dispatch"));
     * ```
     *
     * would result in the following yaml file:
     *
     * ```yaml
     * name: deploy
     * on:
     *   push:
     *     branches:
     *       - main
     *   workflow_call: {}
     * ...
     * ```
     *
     * @param patches - The patch operations to apply
     */
    patch(...patches: JsonPatch[]): void;
    /**
     * Returns the patched yaml file.
     */
    toYaml(): string;
    /**
     * Write the patched yaml file to the specified location.
     */
    writeFile(): void;
}
