import pytest

from vyper import compile_code

fail_list = [
    # VALUE is not set in the constructor
    """
VALUE: immutable(uint256)

@external
def __init__():
    pass
    """,
    # no `__init__` function, VALUE not set
    """
VALUE: immutable(uint256)

@view
@external
def get_value() -> uint256:
    return VALUE
    """,
    # VALUE given an initial value
    """
VALUE: immutable(uint256) = 3

@external
def __init__():
    pass
    """,
    # setting value outside of constructor
    """
VALUE: immutable(uint256)

@external
def __init__():
    VALUE = 0

@external
def set_value(_value: uint256):
    VALUE = _value
    """,
    # modifying immutable multiple times in constructor
    """
VALUE: immutable(uint256)

@external
def __init__(_value: uint256):
    VALUE = _value * 3
    VALUE = VALUE + 1
    """,
]


@pytest.mark.parametrize("bad_code", fail_list)
def test_compilation_fails_with_exception(bad_code):
    with pytest.raises(Exception):
        compile_code(bad_code)


types_list = (
    "uint256",
    "int256",
    "int128",
    "address",
    "bytes32",
    "decimal",
    "bool",
    "Bytes[64]",
    "String[10]",
)


@pytest.mark.parametrize("typ", types_list)
def test_compilation_simple_usage(typ):
    code = f"""
VALUE: immutable({typ})

@external
def __init__(_value: {typ}):
    VALUE = _value

@view
@external
def get_value() -> {typ}:
    return VALUE
    """

    assert compile_code(code)


pass_list = [
    # using immutable allowed in constructor
    """
VALUE: immutable(uint256)

@external
def __init__(_value: uint256):
    VALUE = _value * 3
    x: uint256 = VALUE + 1
    """
]


@pytest.mark.parametrize("good_code", pass_list)
def test_compilation_success(good_code):
    assert compile_code(good_code)
