"use strict";
// Integration test to deploy some resources, create an alarm on it and create a dashboard.
//
// Because literally every other library is going to depend on @aws-cdk/aws-cloudwatch, we drop down
// to the very lowest level to create CloudFormation resources by hand, without even generated
// library support.
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const cloudwatch = require("../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, `aws-cdk-cloudwatch`);
const queue = new cdk.CfnResource(stack, 'queue', { type: 'AWS::SQS::Queue' });
const metricA = new cloudwatch.Metric({
    namespace: 'AWS/SQS',
    metricName: 'ApproximateNumberOfMessagesVisible',
    dimensions: { QueueName: queue.getAtt('QueueName') },
    period: cdk.Duration.seconds(10),
    label: "Visible Messages"
});
const metricB = new cloudwatch.Metric({
    namespace: 'AWS/SQS',
    metricName: 'ApproximateNumberOfMessagesNotVisible',
    dimensions: { QueueName: queue.getAtt('QueueName') },
    period: cdk.Duration.seconds(30),
    label: "NotVisible Messages"
});
const sumExpression = new cloudwatch.MathExpression({
    expression: "m1+m2",
    usingMetrics: {
        m1: metricA,
        m2: metricB
    },
    label: "Total Messages",
    period: cdk.Duration.minutes(1)
});
const alarm = sumExpression.createAlarm(stack, 'Alarm', {
    threshold: 100,
    evaluationPeriods: 3,
});
const dashboard = new cloudwatch.Dashboard(stack, 'Dash', {
    dashboardName: 'MyMathExpressionDashboardName'
});
dashboard.addWidgets(new cloudwatch.AlarmWidget({
    title: 'Total messages in queue',
    alarm,
}));
dashboard.addWidgets(new cloudwatch.GraphWidget({
    title: 'More total messages in queue with alarm annotation',
    left: [sumExpression],
    right: [metricA, metricB],
    leftAnnotations: [alarm.toAnnotation()]
}));
dashboard.addWidgets(new cloudwatch.SingleValueWidget({
    title: 'Current total messages in queue',
    metrics: [sumExpression]
}));
app.synth();
//# sourceMappingURL=data:application/json;base64,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