"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '>=',
};
/**
 * Specify how missing data points are treated during alarm evaluation
 */
var TreatMissingData;
(function (TreatMissingData) {
    /**
     * Missing data points are treated as breaching the threshold
     */
    TreatMissingData["BREACHING"] = "breaching";
    /**
     * Missing data points are treated as being within the threshold
     */
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    /**
     * The current alarm state is maintained
     */
    TreatMissingData["IGNORE"] = "ignore";
    /**
     * The alarm does not consider missing data points when evaluating whether to change state
     */
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric
 */
class Alarm extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...this.renderMetric(props.metric),
            ...object_1.dropUndefined({
                // Alarm overrides
                period: props.period && props.period.toSeconds(),
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            })
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        this.annotation = {
            // tslint:disable-next-line:max-line-length
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${props.evaluationPeriods} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Trigger this action if the alarm fires
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addAlarmAction(...actions) {
        if (this.alarmActionArns === undefined) {
            this.alarmActionArns = [];
        }
        this.alarmActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Trigger this action if there is insufficient data to evaluate the alarm
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addInsufficientDataAction(...actions) {
        if (this.insufficientDataActionArns === undefined) {
            this.insufficientDataActionArns = [];
        }
        this.insufficientDataActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Trigger this action if the alarm returns from breaching state into ok state
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addOkAction(...actions) {
        if (this.okActionArns === undefined) {
            this.okActionArns = [];
        }
        this.okActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Turn this alarm into a horizontal annotation
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *   you have both a "small margin/long period" alarm as well as a
     *   "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    }))
                };
            }
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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