import { Construct, IResource, Resource } from '@aws-cdk/core';
import { IAlarmAction } from './alarm-action';
import { HorizontalAnnotation } from './graph';
import { CreateAlarmOptions } from './metric';
import { IMetric } from './metric-types';
export interface IAlarm extends IResource {
    /**
     * @attribute
     */
    readonly alarmArn: string;
    /**
     * @attribute
     */
    readonly alarmName: string;
}
/**
 * Properties for Alarms
 */
export interface AlarmProps extends CreateAlarmOptions {
    /**
     * The metric to add the alarm on
     *
     * Metric objects can be obtained from most resources, or you can construct
     * custom Metric objects by instantiating one.
     */
    readonly metric: IMetric;
}
/**
 * Comparison operator for evaluating alarms
 */
export declare enum ComparisonOperator {
    GREATER_THAN_OR_EQUAL_TO_THRESHOLD = "GreaterThanOrEqualToThreshold",
    GREATER_THAN_THRESHOLD = "GreaterThanThreshold",
    LESS_THAN_THRESHOLD = "LessThanThreshold",
    LESS_THAN_OR_EQUAL_TO_THRESHOLD = "LessThanOrEqualToThreshold"
}
/**
 * Specify how missing data points are treated during alarm evaluation
 */
export declare enum TreatMissingData {
    /**
     * Missing data points are treated as breaching the threshold
     */
    BREACHING = "breaching",
    /**
     * Missing data points are treated as being within the threshold
     */
    NOT_BREACHING = "notBreaching",
    /**
     * The current alarm state is maintained
     */
    IGNORE = "ignore",
    /**
     * The alarm does not consider missing data points when evaluating whether to change state
     */
    MISSING = "missing"
}
/**
 * An alarm on a CloudWatch metric
 */
export declare class Alarm extends Resource implements IAlarm {
    static fromAlarmArn(scope: Construct, id: string, alarmArn: string): IAlarm;
    /**
     * ARN of this alarm
     *
     * @attribute
     */
    readonly alarmArn: string;
    /**
     * Name of this alarm.
     *
     * @attribute
     */
    readonly alarmName: string;
    /**
     * The metric object this alarm was based on
     */
    readonly metric: IMetric;
    private alarmActionArns?;
    private insufficientDataActionArns?;
    private okActionArns?;
    /**
     * This metric as an annotation
     */
    private readonly annotation;
    constructor(scope: Construct, id: string, props: AlarmProps);
    /**
     * Trigger this action if the alarm fires
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addAlarmAction(...actions: IAlarmAction[]): void;
    /**
     * Trigger this action if there is insufficient data to evaluate the alarm
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addInsufficientDataAction(...actions: IAlarmAction[]): void;
    /**
     * Trigger this action if the alarm returns from breaching state into ok state
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addOkAction(...actions: IAlarmAction[]): void;
    /**
     * Turn this alarm into a horizontal annotation
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *   you have both a "small margin/long period" alarm as well as a
     *   "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation(): HorizontalAnnotation;
    private renderMetric;
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    private validateMetricStat;
}
