"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceHandler = void 0;
/* eslint-disable no-console */
const https = require("https");
const url = require("url");
// eslint-disable-next-line import/no-extraneous-dependencies
const AWS = require("aws-sdk");
// eslint-disable-next-line @typescript-eslint/ban-types
class CustomResourceHandler {
    constructor(event, context) {
        this.event = event;
        this.context = context;
        this.timedOut = false;
        this.timeout = setTimeout(async () => {
            await this.respond({
                status: 'FAILED',
                reason: 'Lambda Function Timeout',
                data: this.context.logStreamName,
            });
            this.timedOut = true;
        }, context.getRemainingTimeInMillis() - 1200);
        this.event = event;
        this.physicalResourceId = extractPhysicalResourceId(event);
    }
    /**
     * Handles executing the custom resource event. If `stateMachineArn` is present
     * in the props then trigger the waiter statemachine
     */
    async handle() {
        try {
            if ('stateMachineArn' in this.event.ResourceProperties) {
                const req = {
                    stateMachineArn: this.event.ResourceProperties.stateMachineArn,
                    name: this.event.RequestId,
                    input: JSON.stringify(this.event),
                };
                await this.startExecution(req);
                return;
            }
            else {
                const response = await this.processEvent(this.event.ResourceProperties);
                return response;
            }
        }
        catch (e) {
            console.log(e);
            throw e;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Handle async requests from the waiter state machine
     */
    async handleIsComplete() {
        try {
            const result = await this.processEvent(this.event.ResourceProperties);
            return result;
        }
        catch (e) {
            console.log(e);
            return;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Start a step function state machine which will wait for the request
     * to be successful.
     */
    async startExecution(req) {
        try {
            const sfn = new AWS.StepFunctions();
            await sfn.startExecution(req).promise();
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    respond(response) {
        if (this.timedOut) {
            return;
        }
        const cfResponse = {
            Status: response.status,
            Reason: response.reason,
            PhysicalResourceId: this.physicalResourceId,
            StackId: this.event.StackId,
            RequestId: this.event.RequestId,
            LogicalResourceId: this.event.LogicalResourceId,
            NoEcho: false,
            Data: response.data,
        };
        const responseBody = JSON.stringify(cfResponse);
        console.log('Responding to CloudFormation', responseBody);
        const parsedUrl = url.parse(this.event.ResponseURL);
        const requestOptions = {
            hostname: parsedUrl.hostname,
            path: parsedUrl.path,
            method: 'PUT',
            headers: { 'content-type': '', 'content-length': responseBody.length },
        };
        return new Promise((resolve, reject) => {
            try {
                const request = https.request(requestOptions, resolve);
                request.on('error', reject);
                request.write(responseBody);
                request.end();
            }
            catch (e) {
                reject(e);
            }
            finally {
                clearTimeout(this.timeout);
            }
        });
    }
}
exports.CustomResourceHandler = CustomResourceHandler;
function extractPhysicalResourceId(event) {
    switch (event.RequestType) {
        case 'Create':
            return event.LogicalResourceId;
        case 'Update':
        case 'Delete':
            return event.PhysicalResourceId;
    }
}
//# sourceMappingURL=data:application/json;base64,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