# -*- coding: utf-8 -*-
'''Chemical Engineering Design Library (ChEDL). Utilities for process modeling.
Copyright (C) 2017, Caleb Bell <Caleb.Andrew.Bell@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.'''

from fluids.numerics import assert_close, assert_close1d, assert_close2d, assert_close3d, assert_close4d, derivative
import pytest
import numpy as np
from math import *

from thermo.activity import GibbsExcess
from thermo.unifac import *
from fluids.numerics import *
from fluids.constants import R
from thermo.unifac import UFIP, LLEUFIP, LUFIP, DOUFIP2006, DOUFIP2016, NISTUFIP, NISTKTUFIP, PSRKIP, VTPRIP, DOUFSG
import types
import pickle, json
from thermo.test_utils import check_np_output_activity

'''
Test suite currently takes ~0.2 seconds :)
'''

def test_UNIFAC_data():
    # Test the interaction pairs
    assert len(UFIP) == 54
    assert sum([len(i) for i in UFIP.values()]) == 1270
    val_sum = sum([np.sum(np.abs(list(i.values()))) for i in UFIP.values()])
    assert_close(val_sum, 449152.27169999998)
    assert_close(UFIP[1][2], 86.02)

    for G in UFIP.keys():
        assert G in UFMG
        for G2 in UFIP[G].keys():
            assert G2 in UFMG

    for G in UFSG.values():
        assert G.main_group_id in UFMG
        assert UFMG[G.main_group_id][0] == G.main_group

    # subgroup strings:
    # [i.group for i in UFSG.values()]
    # ['CH3', 'CH2', 'CH', 'C', 'CH2=CH', 'CH=CH', 'CH2=C', 'CH=C', 'ACH', 'AC', 'ACCH3', 'ACCH2', 'ACCH', 'OH', 'CH3OH', 'H2O', 'ACOH', 'CH3CO', 'CH2CO', 'CHO', 'CH3COO', 'CH2COO', 'HCOO', 'CH3O', 'CH2O', 'CHO', 'THF', 'CH3NH2', 'CH2NH2', 'CHNH2', 'CH3NH', 'CH2NH', 'CHNH', 'CH3N', 'CH2N', 'ACNH2', 'C5H5N', 'C5H4N', 'C5H3N', 'CH3CN', 'CH2CN', 'COOH', 'HCOOH', 'CH2CL', 'CHCL', 'CCL', 'CH2CL2', 'CHCL2', 'CCL2', 'CHCL3', 'CCL3', 'CCL4', 'ACCL', 'CH3NO2', 'CH2NO2', 'CHNO2', 'ACNO2', 'CS2', 'CH3SH', 'CH2SH', 'FURFURAL', 'DOH', 'I', 'BR', 'CH=-C', 'C=-C', 'DMSO', 'ACRY', 'CL-(C=C)', 'C=C', 'ACF', 'DMF', 'HCON(..', 'CF3', 'CF2', 'CF', 'COO', 'SIH3', 'SIH2', 'SIH', 'SI', 'SIH2O', 'SIHO', 'SIO', 'NMP', 'CCL3F', 'CCL2F', 'HCCL2F', 'HCCLF', 'CCLF2', 'HCCLF2', 'CCLF3', 'CCL2F2', 'AMH2', 'AMHCH3', 'AMHCH2', 'AM(CH3)2', 'AMCH3CH2', 'AM(CH2)2', 'C2H5O2', 'C2H4O2', 'CH3S', 'CH2S', 'CHS', 'MORPH', 'C4H4S', 'C4H3S', 'C4H2S', 'NCO', '(CH2)2SU', 'CH2CHSU', 'IMIDAZOL', 'BTI']
    # Main group strings:
    # [i[0] for i in UFMG.values()]
    # ['CH2', 'C=C', 'ACH', 'ACCH2', 'OH', 'CH3OH', 'H2O', 'ACOH', 'CH2CO', 'CHO', 'CCOO', 'HCOO', 'CH2O', 'CNH2', 'CNH', '(C)3N', 'ACNH2', 'PYRIDINE', 'CCN', 'COOH', 'CCL', 'CCL2', 'CCL3', 'CCL4', 'ACCL', 'CNO2', 'ACNO2', 'CS2', 'CH3SH', 'FURFURAL', 'DOH', 'I', 'BR', 'C=-C', 'DMSO', 'ACRY', 'CLCC', 'ACF', 'DMF', 'CF2', 'COO', 'SIH2', 'SIO', 'NMP', 'CCLF', 'CON(AM)', 'OCCOH', 'CH2S', 'MORPH', 'THIOPHEN', 'NCO', 'SULFONES', 'IMIDAZOL', 'BTI']

def test_modified_UNIFAC_data():
    assert len(DOUFIP2006) == 61
    assert len(DOUFIP2016) == 65
    assert sum([len(i) for i in DOUFIP2016.values()]) == 1516
    assert sum([len(i) for i in DOUFIP2006.values()]) == 1318


    def sum_unifac_ips(dat):
        tots = [0.0, 0.0, 0.0]
        for a in dat.values():
            if a:
                for b in a.values():
                    tots[0] += abs(b[0])
                    tots[1] += abs(b[1])
                    tots[2] += abs(b[2])
        return tots

#    val_sum = np.sum(np.abs(np.vstack([np.array(list(i.values())) for i in DOUFIP2006.values() if i.values()])), axis=0)
    assert_close1d(sum_unifac_ips(DOUFIP2006), [831285.1119000008, 2645.011300000001, 3.068751211])
#    val_sum = np.sum(np.abs(np.vstack([np.array(list(i.values())) for i in DOUFIP2016.values() if i.values()])), axis=0)
    assert_close1d(sum_unifac_ips(DOUFIP2016), [1011296.5521000021, 3170.4274820000005, 3.7356898600000004])

    assert_close1d(DOUFIP2016[1][2], (189.66, -0.2723, 0.0))
    assert_close1d(DOUFIP2006[1][2], (189.66, -0.2723, 0.0))

    for G in DOUFSG.values():
        assert G.main_group_id in DOUFMG
        assert DOUFMG[G.main_group_id][0] == G.main_group

    for d in [DOUFIP2006]:
        for G in d.keys():
            assert G in DOUFMG
            for G2 in d[G].keys():
                assert G2 in DOUFMG
    # Missing some of them for DOUFIP2016 - the actual groups are known but not the numbers


    # [i.group for i in DOUFSG.values()]
    # ['CH3', 'CH2', 'CH', 'C', 'CH2=CH', 'CH=CH', 'CH2=C', 'CH=C', 'ACH', 'AC', 'ACCH3', 'ACCH2', 'ACCH', 'OH(P)', 'CH3OH', 'H2O', 'ACOH', 'CH3CO', 'CH2CO', 'CHO', 'CH3COO', 'CH2COO', 'HCOO', 'CH3O', 'CH2O', 'CHO', 'THF', 'CH3NH2', 'CH2NH2', 'CHNH2', 'CH3NH', 'CH2NH', 'CHNH', 'CH3N', 'CH2N', 'ACNH2', 'AC2H2N', 'AC2HN', 'AC2N', 'CH3CN', 'CH2CN', 'COOH', 'HCOOH', 'CH2CL', 'CHCL', 'CCL', 'CH2CL2', 'CHCL2', 'CCL2', 'CHCL3', 'CCL3', 'CCL4', 'ACCL', 'CH3NO2', 'CH2NO2', 'CHNO2', 'ACNO2', 'CS2', 'CH3SH', 'CH2SH', 'FURFURAL', 'DOH', 'I', 'BR', 'CH=-C', 'C=-C', 'DMSO', 'ACRY', 'CL-(C=C)', 'C=C', 'ACF', 'DMF', 'HCON(..', 'CF3', 'CF2', 'CF', 'COO', 'CY-CH2', 'CY-CH', 'CY-C', 'OH(S)', 'OH(T)', 'CY-CH2O', 'TRIOXAN', 'CNH2', 'NMP', 'NEP', 'NIPP', 'NTBP', 'CONH2', 'CONHCH3', 'CONHCH2', 'AM(CH3)2', 'AMCH3CH2', 'AM(CH2)2', 'AC2H2S', 'AC2HS', 'AC2S', 'H2COCH', 'COCH', 'HCOCH', '(CH2)2SU', 'CH2SUCH', '(CH3)2CB', '(CH2)2CB', 'CH2CH3CB', 'H2COCH2', 'CH3S', 'CH2S', 'CHS', 'H2COC', 'C3H2N2+', 'BTI-', 'C3H3N2+', 'C4H8N+', 'BF4-', 'C5H5N+', 'OTF-', '-S-S-']
    # [i[0] for i in UFMG.values()]
    # [i[0] for i in DOUFMG.values()]
    # ['CH2', 'C=C', 'ACH', 'ACCH2', 'OH', 'CH3OH', 'H2O', 'ACOH', 'CH2CO', 'CHO', 'CCOO', 'HCOO', 'CH2O', 'CH2NH2', 'CH2NH', '(C)3N', 'ACNH2', 'PYRIDINE', 'CH2CN', 'COOH', 'CCL', 'CCL2', 'CCL3', 'CCL4', 'ACCL', 'CNO2', 'ACNO2', 'CS2', 'CH3SH', 'FURFURAL', 'DOH', 'I', 'BR', 'C=-C', 'DMSO', 'ACRY', 'CLCC', 'ACF', 'DMF', 'CF2', 'COO', 'CY-CH2', 'CY-CH2O', 'HCOOH', 'CHCL3', 'CY-CONC', 'CONR', 'CONR2', 'HCONR', 'ACS', 'EPOXIDES', 'CARBONAT', 'SULFONE', 'SULFIDES', 'IMIDAZOL', 'BTI', 'PYRROL', 'BF4', 'PYRIDIN', 'OTF', 'DISULFIDES']
def test_modified_UNIFAC_NIST_data():
    pass

def test_UNIFAC():
    # Gmehling
    # 05.22 VLE of Hexane-Butanone-2 Via UNIFAC (p. 289)
    # Mathcad (2001) - Solution (zip) - step by step
    # http://chemthermo.ddbst.com/Problems_Solutions/Mathcad_Files/05.22a%20VLE%20of%20Hexane-Butanone-2%20Via%20UNIFAC%20-%20Step%20by%20Step.xps
    gammas_0522A = UNIFAC_gammas(chemgroups=[{1:2, 2:4}, {1:1, 2:1, 18:1}], T=60+273.15, xs=[0.5, 0.5])
    assert_close1d(gammas_0522A, [1.4276025835624173, 1.3646545010104225])
    assert_close1d(gammas_0522A, [1.428, 1.365], atol=0.001)
    assert_close1d(gammas_0522A, [1.4276, 1.36466], atol=0.0001) # Another calculator



    # Example 4.14 Activity Coefficients of Ethanol + Benzene with the UNIFAC method
    # Walas, Phase Equilibria in Chemical Engineering
    gammas_414 = UNIFAC_gammas(chemgroups=[{1:1, 2:1, 14:1}, {9:6}], T=345., xs=[0.2, 0.8])
    assert_close1d(gammas_414, [2.90999524962436, 1.1038643452317465])
    # Matches faily closely. Confirmed it uses the same coefficients.
    assert_close1d(gammas_414, [2.9119, 1.10832], atol=0.005)

    # Examples from ACTCOEFF.XLS, chethermo, from Introductory Chemical Engineering
    # Thermodynamics, 2nd Ed.: an undergraduate chemical engineering text,
    #  J.Richard Elliott and Carl T. Lira, http://chethermo.net/
    # All match exactly; not even a different gas constant used
    # isopropyl alcohol-water
    gammas_ACTCOEFF_1 = UNIFAC_gammas(chemgroups=[{1:2, 3:1, 14:1}, {16:1}], T=80.37+273.15, xs=[0.5, 0.5])
    gammas_ACTCOEFF_1_expect = [1.2667572876079400, 1.700192255741180]
    assert_close1d(gammas_ACTCOEFF_1, gammas_ACTCOEFF_1_expect)
    gammas_ACTCOEFF_2 = UNIFAC_gammas(chemgroups=[{1:2, 3:1, 14:1}, {16:1}], T=80.37+273.15, xs=[0.1, 0.9])
    gammas_ACTCOEFF_2_expect = [5.0971362612830500, 1.058637792621310]
    assert_close1d(gammas_ACTCOEFF_2, gammas_ACTCOEFF_2_expect)
    # Add in Propionic acid
    gammas_ACTCOEFF_3 = UNIFAC_gammas(chemgroups=[{1:2, 3:1, 14:1}, {16:1}, {1:1, 2:1, 42:1}], T=80.37+273.15, xs=[0.1, 0.1, 0.8])
    gammas_ACTCOEFF_3_expect = [0.9968890535625640, 2.170957708441830, 1.0011209111895400]
    assert_close1d(gammas_ACTCOEFF_3, gammas_ACTCOEFF_3_expect)
    # Add in ethanol
    gammas_ACTCOEFF_4 = UNIFAC_gammas(chemgroups=[{1:2, 3:1, 14:1}, {16:1}, {1:1, 2:1, 42:1}, {1:1, 2:1, 14:1}], T=80.37+273.15, xs=[0.01, 0.01, 0.01, .97])
    gammas_ACTCOEFF_4_expect = [1.0172562157805600, 2.721887947655120, 0.9872477280109450, 1.000190624095510]
    assert_close1d(gammas_ACTCOEFF_4, gammas_ACTCOEFF_4_expect)
    # Add in pentane
    gammas_ACTCOEFF_5 = UNIFAC_gammas(chemgroups=[{1:2, 3:1, 14:1}, {16:1}, {1:1, 2:1, 42:1}, {1:1, 2:1, 14:1}, {1:2, 2:3}], T=80.37+273.15, xs=[.1, .05, .1, .25, .5])
    gammas_ACTCOEFF_5_expect = [1.2773557137580500, 8.017146862811100, 1.1282116576861800, 1.485860948162550, 1.757426505841570]
    assert_close1d(gammas_ACTCOEFF_5, gammas_ACTCOEFF_5_expect)


    # Acetone and Pentane at 307 K and x1 = 0.047
    # Example 8-12 in Poling et al., 5E
    gammas_Poling_5e = UNIFAC_gammas(chemgroups=[{1:1, 18:1}, {1:2, 2:3}], T=307, xs=[0.047, 0.953])
    gammas_Poling_known = [4.992034311484559, 1.00526021118788]
    assert_close1d(gammas_Poling_5e, gammas_Poling_known)
    assert_close1d(gammas_Poling_5e, [4.99, 1.005], atol=0.003)

    gammas_Poling_with_cache = UNIFAC_gammas(chemgroups=[{1:1, 18:1}, {1:2, 2:3}], T=307, xs=[0.047, 0.953], cached=([2.5735, 3.8254], [2.336, 3.316], {1: 3, 18: 1, 2: 3}))
    assert_close1d(gammas_Poling_with_cache, gammas_Poling_known)
    # Test the caching

    # Another case with the same mixture
    gammas_custom = UNIFAC_gammas(chemgroups=[{1:1, 18:1}, {1:2, 2:3}], T=307, xs=[.674747, .325251])
    assert_close1d(gammas_custom, [1.1645751997624518, 2.105331695192004])


def test_UNIFAC_modified_2006():
    # 11.02 Azeotropic Points in the Quaternary System Benzene - Cyclohexane - Acetone - Ethanol Using Mod. UNIFAC-1.xps
    # Note this test does not sum up to 1?
    gammas_1102_1 = UNIFAC_gammas(chemgroups=[{9:6}, {78:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}], T=373.15, xs=[0.2, 0.3, 0.2, 0.2],
                             subgroup_data=DOUFSG, interaction_data=DOUFIP2006, modified=True)
    # Values in .xps
    gammas_1102_1_known = [1.18643111, 1.44028013, 1.20447983, 1.97207061]
    assert_close1d(gammas_1102_1, gammas_1102_1_known)
    # Recalculated values with more precision, still matching exactly
    gammas_1102_1_known2 = [1.18643111370682970, 1.44028013391119700, 1.20447983349960850, 1.97207060902998130]
    assert_close1d(gammas_1102_1, gammas_1102_1_known2, rtol=1E-14)
    # 290 K, x3=0.3 to balance
    gammas_1102_2 = UNIFAC_gammas(chemgroups=[{9:6}, {78:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}], T=290, xs=[0.2, 0.3, 0.3, 0.2],
                             subgroup_data=DOUFSG, interaction_data=DOUFIP2006, modified=True)
    gammas_1102_2_known = [1.2555831362844658, 2.002790560351622, 1.313653013490284, 2.4472442902051923]
    assert_close1d(gammas_1102_2_known, gammas_1102_2, rtol=1E-13)

    # 0.01 mole fractions except last, 250 K
    gammas_1102_3 = UNIFAC_gammas(chemgroups=[{9:6}, {78:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}], T=250, xs=[0.01, 0.01, 0.01, 0.97], subgroup_data=DOUFSG, interaction_data=DOUFIP2006, modified=True)
    gammas_1102_3_known = [6.233033961983859, 10.01994111294437, 3.376394671321658, 1.00137007335149700]
    assert_close1d(gammas_1102_3_known, gammas_1102_3, rtol=1E-13)


def test_UNIFAC_misc():
    from math import log
    T = 273.15 + 60

    def gE_T(T):
        xs = [0.5, 0.5]
        gammas = UNIFAC_gammas(chemgroups=[{1:2, 2:4}, {1:1, 2:1, 18:1}], T=T, xs=xs)
        return R*T*sum(xi*log(gamma) for xi, gamma in zip(xs, gammas))

    def hE_T(T):
        to_diff = lambda T: gE_T(T)/T
        return -derivative(to_diff, T,dx=1E-5, order=7)*T**2

    # A source gives 854.758 for hE, matching to within a gas constant
    assert_close(hE_T(T), 854.7719207160634)
    assert_close(gE_T(T), 923.6411976689174)



def test_Van_der_Waals_area():
    # DIPPR and YAWS, hexane, units are good
    assert_close(Van_der_Waals_area(3.856), 964000.0)

def test_Van_der_Waals_volume():
    # DIPPR and YAWS, hexane, units are good
    assert_close(Van_der_Waals_volume(4.4998), 6.826196599999999e-05)


def test_UNIFAC_psi():
    assert_close(UNIFAC_psi(307, 18, 1, UFSG, UFIP), 0.9165248264184787)

    assert_close(UNIFAC_psi(373.15, 9, 78, DOUFSG, DOUFIP2006, modified=True), 1.3703140538273264)



def test_UNIFAC_flash_1():
    from chemicals.rachford_rice import flash_inner_loop
    from chemicals.flash_basic import K_value
    def flash_UNIFAC_sequential_substitution(T, P, zs, Psats, chemgroups):
        gammas = UNIFAC_gammas(chemgroups=chemgroups, T=T, xs=zs)
        Ks = [K_value(P=P, Psat=Psat, gamma=gamma) for Psat, gamma in zip(Psats, gammas)]
        V_over_F, xs, ys = flash_inner_loop(zs, Ks)
        for i in range(100):
            gammas = UNIFAC_gammas(chemgroups=chemgroups, T=T, xs=xs)
            Ks = [K_value(P=P, Psat=Psat, gamma=gamma) for Psat, gamma in zip(Psats, gammas)]
            V_over_F, xs_new, ys_new = flash_inner_loop(zs, Ks)
            err = (sum([abs(x_new - x_old) for x_new, x_old in zip(xs_new, xs)]) +
                  sum([abs(y_new - y_old) for y_new, y_old in zip(ys_new, ys)]))
            xs, ys = xs_new, ys_new
            if err < 1E-11:
                break
        return V_over_F, xs, ys

    T = 307
    P = 1E5
    zs = [0.5, 0.5]
    chemgroups = [{1:1, 18:1}, {1:2, 2:3}]
    Psats = [44501.41359963363, 93853.94807811991]
    ans = flash_UNIFAC_sequential_substitution(T=T, P=P, zs=zs, Psats=Psats, chemgroups=chemgroups)
    assert_close(ans[0], 0.5101142364235425)
    assert_close1d(ans[1], [0.6594292844045343, 0.34057071559546576])
    assert_close1d(ans[2], [0.3468928503651561, 0.653107149634844])

def test_UNIFAC_class():
    T = 373.15
    xs = [0.2, 0.3, 0.1, 0.4]
    chemgroups = [{9:6}, {78:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}]
    # m = Mixture(['benzene', 'cyclohexane', 'acetone', 'ethanol'], zs=xs, T=T, P=1e5)
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=1,
                               interaction_data=DOUFIP2006, subgroups=DOUFSG)

    Vis_expect = [0.7607527334602491, 1.4426605118183198, 0.7875398015398353, 0.840743299021177]
    assert_close1d(GE.Vis(), Vis_expect, rtol=1e-12)

    Fis_expect = [0.7601728758495722, 1.5191142751587723, 0.8006943182018096, 0.780404276155682]
    assert_close1d(GE.Fis(), Fis_expect, rtol=1e-12)

    def to_jac_Vis(xs):
        return GE.to_T_xs(T, xs).Vis()

    dVis_dxs_expect = [[-0.5787447214672409, -1.0975079278209487, -0.5991230567301719, -0.6395977628687479], [-1.0975079278209487, -2.0812693523598966, -1.1361525731667568, -1.2129071580737139], [-0.5991230567301719, -1.1361525731667568, -0.6202189390094032, -0.6621188108570841], [-0.6395977628687479, -1.2129071580737139, -0.6621188108570841, -0.7068492948490122]]
    dVis_dxs_analytical = GE.dVis_dxs()
    assert_close2d(dVis_dxs_expect, dVis_dxs_analytical, rtol=1e-12)
    dVis_dxs_numerical = jacobian(to_jac_Vis, xs, scalar=False, perturbation=1e-9)
    assert_close2d(dVis_dxs_numerical, dVis_dxs_analytical, rtol=1e-6)

    def to_jac_Fis(xs):
        return GE.to_T_xs(T, xs).Fis()

    dFis_dxs_expect = [[-0.5778628011774091, -1.1547894672915822, -0.608666102543882, -0.5932421629305685], [-1.1547894672915822, -2.3077081809911624, -1.2163461688188895, -1.1855232763030454], [-0.608666102543882, -1.2163461688188895, -0.6411113912006607, -0.6248652698182505], [-0.5932421629305685, -1.1855232763030454, -0.6248652698182505, -0.6090308342420739]]
    dFis_dxs_analytical = GE.dFis_dxs()
    assert_close2d(dFis_dxs_expect, dFis_dxs_analytical, rtol=1e-12)
    dFis_dxs_numerical = jacobian(to_jac_Fis, xs, scalar=False, perturbation=4e-8)
    assert_close2d(dFis_dxs_numerical, dFis_dxs_analytical, rtol=1e-7)

    # Checked to higher precision with numdifftools
    d2Vis_dxixjs_numerical = hessian(to_jac_Vis, xs, scalar=False, perturbation=4e-5)
    d2Vis_dxixjs_expect = [[[0.880563257663788, 1.6698643121681611, 0.9115690061730761, 0.9731514928349204], [1.669864312168161, 3.166662697749667, 1.7286623513290118, 1.8454448718761605], [0.9115690061730761, 1.7286623513290118, 0.943666506390438, 1.0074173904699528], [0.9731514928349204, 1.8454448718761605, 1.0074173904699528, 1.075475066401671]], [[1.669864312168161, 3.166662697749667, 1.7286623513290118, 1.8454448718761605], [3.1666626977496675, 6.005130218214624, 3.278164905416909, 3.499626522909456], [1.7286623513290118, 3.278164905416909, 1.7895307439814416, 1.9104253251112364], [1.8454448718761605, 3.4996265229094554, 1.9104253251112364, 2.0394871309705884]], [[0.9115690061730761, 1.7286623513290118, 0.943666506390438, 1.0074173904699528], [1.7286623513290118, 3.278164905416909, 1.7895307439814416, 1.9104253251112364], [0.9436665063904379, 1.7895307439814416, 0.9768942002774252, 1.0428898337963595], [1.0074173904699528, 1.9104253251112364, 1.0428898337963595, 1.1133439067679272]], [[0.9731514928349204, 1.8454448718761605, 1.0074173904699528, 1.075475066401671], [1.8454448718761605, 3.4996265229094554, 1.9104253251112364, 2.0394871309705884], [1.0074173904699528, 1.9104253251112364, 1.0428898337963595, 1.1133439067679272], [1.075475066401671, 2.039487130970589, 1.1133439067679272, 1.188557616124302]]]
    d2Vis_dxixjs_analytical = GE.d2Vis_dxixjs()
    assert_close3d(d2Vis_dxixjs_numerical, d2Vis_dxixjs_analytical, rtol=1e-4)
    assert_close3d(d2Vis_dxixjs_expect, d2Vis_dxixjs_analytical, rtol=1e-12)

    # Checked to higher precision with numdifftools
    d2Fis_dxixjs_numerical = hessian(to_jac_Fis, xs, scalar=False, perturbation=4e-5)
    d2Fis_dxixjs_expect = [[[0.878551254835041, 1.7556792607036749, 0.9253829232058668, 0.9019332021403013], [1.7556792607036749, 3.508514329131273, 1.8492667303593284, 1.8024052766677856], [0.9253829232058666, 1.8492667303593284, 0.9747109799778525, 0.9500112583525165], [0.9019332021403013, 1.8024052766677856, 0.9500112583525168, 0.9259374414937229]], [[1.7556792607036749, 3.508514329131273, 1.8492667303593284, 1.8024052766677856], [3.5085143291312737, 7.011344881288717, 3.6955376571749134, 3.601890665129907], [1.8492667303593284, 3.6955376571749134, 1.9478429326796476, 1.8984835028636846], [1.8024052766677856, 3.601890665129907, 1.8984835028636846, 1.850374868617981]], [[0.9253829232058666, 1.8492667303593284, 0.9747109799778525, 0.9500112583525165], [1.8492667303593284, 3.6955376571749134, 1.9478429326796476, 1.8984835028636846], [0.9747109799778525, 1.9478429326796476, 1.0266684965376531, 1.0006521423702277], [0.9500112583525168, 1.8984835028636846, 1.0006521423702277, 0.9752950571746734]], [[0.9019332021403013, 1.8024052766677856, 0.9500112583525168, 0.9259374414937229], [1.8024052766677856, 3.601890665129907, 1.8984835028636846, 1.850374868617981], [0.9500112583525168, 1.8984835028636846, 1.0006521423702277, 0.9752950571746734], [0.9259374414937227, 1.850374868617981, 0.9752950571746734, 0.9505805347063537]]]
    d2Fis_dxixjs_analytical = GE.d2Fis_dxixjs()
    assert_close3d(d2Fis_dxixjs_numerical, d2Fis_dxixjs_analytical, rtol=1e-4)
    assert_close3d(d2Fis_dxixjs_expect, d2Fis_dxixjs_analytical, rtol=1e-12)

    # Not able to calculate numerical third derivatives
    d3Vis_dxixjxks_sympy = [[[[-2.009672715757163, -3.8110615199689386, -2.0804358395514293, -2.2209829747352616], [-3.8110615199689386, -7.227141909778993, -3.945253827010098, -4.211781692189962], [-2.0804358395514293, -3.945253827010098, -2.153690622634226, -2.2991866006062214], [-2.2209829747352616, -4.211781692189962, -2.2991866006062214, -2.4545117896002413]], [[-3.8110615199689386, -7.227141909778993, -3.945253827010098, -4.211781692189962], [-7.227141909778993, -13.705257684874537, -7.481618737588112, -7.987051330180062], [-3.945253827010098, -7.481618737588112, -4.084171215285101, -4.360083864450171], [-4.211781692189962, -7.987051330180062, -4.360083864450171, -4.654636229228626]], [[-2.0804358395514293, -3.945253827010098, -2.153690622634226, -2.2991866006062214], [-3.945253827010098, -7.481618737588112, -4.084171215285101, -4.360083864450171], [-2.153690622634226, -4.084171215285101, -2.229524799487544, -2.3801438752754542], [-2.2991866006062214, -4.360083864450171, -2.3801438752754542, -2.540938261065038]], [[-2.2209829747352616, -4.211781692189962, -2.2991866006062214, -2.4545117896002413], [-4.211781692189962, -7.987051330180062, -4.360083864450171, -4.654636229228626], [-2.2991866006062214, -4.360083864450171, -2.3801438752754542, -2.540938261065038], [-2.4545117896002413, -4.654636229228626, -2.540938261065038, -2.7125953660246798]]], [[[-3.8110615199689386, -7.227141909778993, -3.945253827010098, -4.211781692189962], [-7.227141909778993, -13.705257684874537, -7.481618737588112, -7.987051330180062], [-3.945253827010098, -7.481618737588112, -4.084171215285101, -4.360083864450171], [-4.211781692189962, -7.987051330180062, -4.360083864450171, -4.654636229228626]], [[-7.227141909778993, -13.705257684874537, -7.481618737588112, -7.987051330180062], [-13.705257684874537, -25.990092702435476, -14.187837180820823, -15.146318972140557], [-7.481618737588112, -14.187837180820823, -7.745056017080649, -8.268285531946963], [-7.987051330180062, -15.146318972140557, -8.268285531946963, -8.826862644638714]], [[-3.945253827010098, -7.481618737588112, -4.084171215285101, -4.360083864450171], [-7.481618737588112, -14.187837180820823, -7.745056017080649, -8.268285531946963], [-4.084171215285101, -7.745056017080649, -4.2279800608937315, -4.513607944184332], [-4.360083864450171, -8.268285531946963, -4.513607944184332, -4.818531871102873]], [[-4.211781692189962, -7.987051330180062, -4.360083864450171, -4.654636229228626], [-7.987051330180062, -15.146318972140557, -8.268285531946963, -8.826862644638714], [-4.360083864450171, -8.268285531946963, -4.513607944184332, -4.818531871102873], [-4.654636229228626, -8.826862644638714, -4.818531871102873, -5.144055416410341]]], [[[-2.0804358395514293, -3.945253827010098, -2.153690622634226, -2.2991866006062214], [-3.945253827010098, -7.481618737588112, -4.084171215285101, -4.360083864450171], [-2.153690622634226, -4.084171215285101, -2.229524799487544, -2.3801438752754542], [-2.2991866006062214, -4.360083864450171, -2.3801438752754542, -2.540938261065038]], [[-3.945253827010098, -7.481618737588112, -4.084171215285101, -4.360083864450171], [-7.481618737588112, -14.187837180820823, -7.745056017080649, -8.268285531946963], [-4.084171215285101, -7.745056017080649, -4.2279800608937315, -4.513607944184332], [-4.360083864450171, -8.268285531946963, -4.513607944184332, -4.818531871102873]], [[-2.153690622634226, -4.084171215285101, -2.229524799487544, -2.3801438752754542], [-4.084171215285101, -7.745056017080649, -4.2279800608937315, -4.513607944184332], [-2.229524799487544, -4.2279800608937315, -2.3080291938356967, -2.4639517582076884], [-2.3801438752754542, -4.513607944184332, -2.4639517582076884, -2.630407918144793]], [[-2.2991866006062214, -4.360083864450171, -2.3801438752754542, -2.540938261065038], [-4.360083864450171, -8.268285531946963, -4.513607944184332, -4.818531871102873], [-2.3801438752754542, -4.513607944184332, -2.4639517582076884, -2.630407918144793], [-2.540938261065038, -4.818531871102873, -2.630407918144793, -2.8081092873635765]]], [[[-2.2209829747352616, -4.211781692189962, -2.2991866006062214, -2.4545117896002413], [-4.211781692189962, -7.987051330180062, -4.360083864450171, -4.654636229228626], [-2.2991866006062214, -4.360083864450171, -2.3801438752754542, -2.540938261065038], [-2.4545117896002413, -4.654636229228626, -2.540938261065038, -2.7125953660246798]], [[-4.211781692189962, -7.987051330180062, -4.360083864450171, -4.654636229228626], [-7.987051330180062, -15.146318972140557, -8.268285531946963, -8.826862644638714], [-4.360083864450171, -8.268285531946963, -4.513607944184332, -4.818531871102873], [-4.654636229228626, -8.826862644638714, -4.818531871102873, -5.144055416410341]], [[-2.2991866006062214, -4.360083864450171, -2.3801438752754542, -2.540938261065038], [-4.360083864450171, -8.268285531946963, -4.513607944184332, -4.818531871102873], [-2.3801438752754542, -4.513607944184332, -2.4639517582076884, -2.630407918144793], [-2.540938261065038, -4.818531871102873, -2.630407918144793, -2.8081092873635765]], [[-2.4545117896002413, -4.654636229228626, -2.540938261065038, -2.7125953660246798], [-4.654636229228626, -8.826862644638714, -4.818531871102873, -5.144055416410341], [-2.540938261065038, -4.818531871102873, -2.630407918144793, -2.8081092873635765], [-2.7125953660246798, -5.144055416410341, -2.8081092873635765, -2.9978155537712734]]]]
    d3Vis_dxixjxks_analytical = GE.d3Vis_dxixjxks()
    assert_close4d(d3Vis_dxixjxks_analytical, d3Vis_dxixjxks_sympy, rtol=1e-12)
    '''
    cmps = range(4)
    xs = x0, x1, x2, x3 = symbols('x0, x1, x2, x3')
    rs = r0, r1, r2, r3 = symbols('r0, r1, r2, r3')
    rsxs = sum([rs[i]*xs[i] for i in cmps])
    Vis = [rs[i]/rsxs for i in cmps]

    # qsxs = sum([qs[i]*xs[i] for i in cmps])
    # Fis = [qs[i]/qsxs for i in cmps]
    to_subs = {rs[i]: v for i, v in zip(cmps, [2.2578, 4.2816, 2.3373, 2.4952])}
    for xi, xv in zip(xs, [0.2, 0.3, 0.1, 0.4]):
        to_subs[xi] = xv
    print([[[[float(diff(Vis[i], xk, xj, xi).subs(to_subs)) for i in cmps] for xk in xs] for xj in xs] for xi in xs])
    '''

    d3Fis_dxixjxks_sympy = [[[[-2.0035525019076115, -4.003859258035692, -2.1103529939864636, -2.05687546828562], [-4.003859258035692, -8.00123228260546, -4.21728722589056, -4.110418807832985], [-2.1103529939864636, -4.21728722589056, -2.2228465463157576, -2.166518371053911], [-2.05687546828562, -4.110418807832985, -2.166518371053911, -2.1116175832712356]], [[-4.003859258035692, -8.00123228260546, -4.21728722589056, -4.110418807832985], [-8.00123228260546, -15.989502605947267, -8.427742465995134, -8.21417875622259], [-4.21728722589056, -8.427742465995134, -4.442092091515058, -4.329526992374571], [-4.110418807832985, -8.21417875622259, -4.329526992374571, -4.219814355831316]], [[-2.1103529939864636, -4.21728722589056, -2.2228465463157576, -2.166518371053911], [-4.21728722589056, -8.427742465995134, -4.442092091515058, -4.329526992374571], [-2.2228465463157576, -4.442092091515058, -2.3413366306715533, -2.282005850371835], [-2.166518371053911, -4.329526992374571, -2.282005850371835, -2.224178545243034]], [[-2.05687546828562, -4.110418807832985, -2.166518371053911, -2.1116175832712356], [-4.110418807832985, -8.21417875622259, -4.329526992374571, -4.219814355831316], [-2.166518371053911, -4.329526992374571, -2.282005850371835, -2.224178545243034], [-2.1116175832712356, -4.219814355831316, -2.224178545243034, -2.1678166163830594]]], [[[-4.003859258035692, -8.00123228260546, -4.21728722589056, -4.110418807832985], [-8.00123228260546, -15.989502605947267, -8.427742465995134, -8.21417875622259], [-4.21728722589056, -8.427742465995134, -4.442092091515058, -4.329526992374571], [-4.110418807832985, -8.21417875622259, -4.329526992374571, -4.219814355831316]], [[-8.00123228260546, -15.989502605947267, -8.427742465995134, -8.21417875622259], [-15.989502605947267, -31.953102291681244, -16.841832028203655, -16.415050580879907], [-8.427742465995134, -16.841832028203655, -8.876988014402343, -8.652040171060962], [-8.21417875622259, -16.415050580879907, -8.652040171060962, -8.432792631937842]], [[-4.21728722589056, -8.427742465995134, -4.442092091515058, -4.329526992374571], [-8.427742465995134, -16.841832028203655, -8.876988014402343, -8.652040171060962], [-4.442092091515058, -8.876988014402343, -4.678880306838432, -4.560314861828465], [-4.329526992374571, -8.652040171060962, -4.560314861828465, -4.444753931537513]], [[-4.110418807832985, -8.21417875622259, -4.329526992374571, -4.219814355831316], [-8.21417875622259, -16.415050580879907, -8.652040171060962, -8.432792631937842], [-4.329526992374571, -8.652040171060962, -4.560314861828465, -4.444753931537513], [-4.219814355831316, -8.432792631937842, -4.444753931537513, -4.3321213798814435]]], [[[-2.1103529939864636, -4.21728722589056, -2.2228465463157576, -2.166518371053911], [-4.21728722589056, -8.427742465995134, -4.442092091515058, -4.329526992374571], [-2.2228465463157576, -4.442092091515058, -2.3413366306715533, -2.282005850371835], [-2.166518371053911, -4.329526992374571, -2.282005850371835, -2.224178545243034]], [[-4.21728722589056, -8.427742465995134, -4.442092091515058, -4.329526992374571], [-8.427742465995134, -16.841832028203655, -8.876988014402343, -8.652040171060962], [-4.442092091515058, -8.876988014402343, -4.678880306838432, -4.560314861828465], [-4.329526992374571, -8.652040171060962, -4.560314861828465, -4.444753931537513]], [[-2.2228465463157576, -4.442092091515058, -2.3413366306715533, -2.282005850371835], [-4.442092091515058, -8.876988014402343, -4.678880306838432, -4.560314861828465], [-2.3413366306715533, -4.678880306838432, -2.46614289556348, -2.4036494546769296], [-2.282005850371835, -4.560314861828465, -2.4036494546769296, -2.3427396325502112]], [[-2.166518371053911, -4.329526992374571, -2.282005850371835, -2.224178545243034], [-4.329526992374571, -8.652040171060962, -4.560314861828465, -4.444753931537513], [-2.282005850371835, -4.560314861828465, -2.4036494546769296, -2.3427396325502112], [-2.224178545243034, -4.444753931537513, -2.3427396325502112, -2.283373299397847]]], [[[-2.05687546828562, -4.110418807832985, -2.166518371053911, -2.1116175832712356], [-4.110418807832985, -8.21417875622259, -4.329526992374571, -4.219814355831316], [-2.166518371053911, -4.329526992374571, -2.282005850371835, -2.224178545243034], [-2.1116175832712356, -4.219814355831316, -2.224178545243034, -2.1678166163830594]], [[-4.110418807832985, -8.21417875622259, -4.329526992374571, -4.219814355831316], [-8.21417875622259, -16.415050580879907, -8.652040171060962, -8.432792631937842], [-4.329526992374571, -8.652040171060962, -4.560314861828465, -4.444753931537513], [-4.219814355831316, -8.432792631937842, -4.444753931537513, -4.3321213798814435]], [[-2.166518371053911, -4.329526992374571, -2.282005850371835, -2.224178545243034], [-4.329526992374571, -8.652040171060962, -4.560314861828465, -4.444753931537513], [-2.282005850371835, -4.560314861828465, -2.4036494546769296, -2.3427396325502112], [-2.224178545243034, -4.444753931537513, -2.3427396325502112, -2.283373299397847]], [[-2.1116175832712356, -4.219814355831316, -2.224178545243034, -2.1678166163830594], [-4.219814355831316, -8.432792631937842, -4.444753931537513, -4.3321213798814435], [-2.224178545243034, -4.444753931537513, -2.3427396325502112, -2.283373299397847], [-2.1678166163830594, -4.3321213798814435, -2.283373299397847, -2.22551134234558]]]]
    d3Fis_dxixjxks_analytical = GE.d3Fis_dxixjxks()
    assert_close4d(d3Fis_dxixjxks_analytical, d3Fis_dxixjxks_sympy, rtol=1e-12)
    '''
    cmps = range(4)
    xs = x0, x1, x2, x3 = symbols('x0, x1, x2, x3')
    qs = q0, q1, q2, q3 = symbols('q0, q1, q2, q3')

    qsxs = sum([qs[i]*xs[i] for i in cmps])
    Fis = [qs[i]/qsxs for i in cmps]
    to_subs = {qs[i]: v for i, v in zip(cmps, [2.5926, 5.181, 2.7308, 2.6616])}
    for xi, xv in zip(xs, [0.2, 0.3, 0.1, 0.4]):
        to_subs[xi] = xv
    print([[[[float(diff(Fis[i], xk, xj, xi).subs(to_subs)) for i in cmps] for xk in xs] for xj in xs] for xi in xs])
    '''

    def to_jac_Vis_modified(xs):
        return GE.to_T_xs(T, xs).Vis_modified()

    Vis_modified_expect = [0.8206306935897535, 1.3261364857739248, 0.8422082656458846, 0.8845302224632082]
    assert_close1d(GE.Vis_modified(), Vis_modified_expect, rtol=1e-12)

    dVis_modified_dxs_expect = [[-0.6734347352615999, -1.0882683041153345, -0.6911419531840057, -0.7258726499610816], [-1.0882683041153345, -1.7586379789008155, -1.1168831096933858, -1.1730078007781872], [-0.6911419531840057, -1.1168831096933858, -0.7093147627222489, -0.7449586645721071], [-0.7258726499610816, -1.1730078007781872, -0.7449586645721071, -0.7823937144508127]]
    dVis_modified_dxs_analytical = GE.dVis_modified_dxs()
    assert_close2d(dVis_modified_dxs_expect, dVis_modified_dxs_analytical, rtol=1e-12)
    dVis_modified_dxs_numerical = jacobian(to_jac_Vis_modified, xs, scalar=False, perturbation=1e-9)
    assert_close2d(dVis_modified_dxs_numerical, dVis_modified_dxs_analytical, rtol=1e-6)

    # Checked to higher precision with numdifftools
    d2Vis_modified_dxixjs_numerical = hessian(to_jac_Vis_modified, xs, scalar=False, perturbation=4e-5)
    d2Vis_modified_dxixjs_expect = [[[1.1052824277703177, 1.7861327464358234, 1.134344600820735, 1.1913467523907895], [1.7861327464358234, 2.886384608797317, 1.833097121932728, 1.9252124102775903], [1.134344600820735, 1.833097121932728, 1.1641709314124211, 1.2226718912070096], [1.1913467523907895, 1.9252124102775903, 1.2226718912070096, 1.284112593100068]], [[1.7861327464358234, 2.886384608797317, 1.833097121932728, 1.9252124102775903], [2.886384608797317, 4.664387978176171, 2.9622788842180796, 3.1111368854187704], [1.8330971219327277, 2.9622788842180796, 1.8812963734880972, 1.9758337309649805], [1.9252124102775903, 3.1111368854187704, 1.9758337309649805, 2.0751217019468173]], [[1.134344600820735, 1.833097121932728, 1.1641709314124211, 1.2226718912070096], [1.8330971219327277, 2.9622788842180796, 1.8812963734880972, 1.9758337309649805], [1.1641709314124211, 1.8812963734880972, 1.194781512218655, 1.2548206897342973], [1.2226718912070096, 1.9758337309649805, 1.2548206897342973, 1.317876906599721]], [[1.1913467523907895, 1.9252124102775903, 1.2226718912070096, 1.284112593100068], [1.9252124102775903, 3.1111368854187704, 1.9758337309649805, 2.0751217019468173], [1.2226718912070096, 1.9758337309649805, 1.2548206897342973, 1.317876906599721], [1.284112593100068, 2.0751217019468173, 1.317876906599721, 1.3841017725939864]]]
    d2Vis_modified_dxixjs_analytical = GE.d2Vis_modified_dxixjs()
    assert_close3d(d2Vis_modified_dxixjs_numerical, d2Vis_modified_dxixjs_analytical, rtol=1e-4)
    assert_close3d(d2Vis_modified_dxixjs_expect, d2Vis_modified_dxixjs_analytical, rtol=1e-12)

    # Not able to calculate numerical third derivatives
    d3Vis_modified_dxixjxks_sympy = [[[[-2.721086055941168, -4.397266063653004, -2.7926339896239365, -2.9329671351610633], [-4.397266063653004, -7.105967410452397, -4.512887287767106, -4.739665186661101], [-2.7926339896239365, -4.512887287767106, -2.8660631967060146, -3.0100862463417126], [-2.9329671351610633, -4.739665186661101, -3.0100862463417126, -3.161346623769138]], [[-4.397266063653004, -7.105967410452397, -4.512887287767106, -4.739665186661101], [-7.105967410452397, -11.48321982510726, -7.292810926086492, -7.659283260401616], [-4.512887287767106, -7.292810926086492, -4.631548643470277, -4.864289415179469], [-4.739665186661101, -7.659283260401616, -4.864289415179469, -5.108725684655301]], [[-2.7926339896239365, -4.512887287767106, -2.8660631967060146, -3.0100862463417126], [-4.512887287767106, -7.292810926086492, -4.631548643470277, -4.864289415179469], [-2.8660631967060146, -4.631548643470277, -2.9414231431806286, -3.089233118842289], [-3.0100862463417126, -4.864289415179469, -3.089233118842289, -3.2444707197865448]], [[-2.9329671351610633, -4.739665186661101, -3.0100862463417126, -3.161346623769138], [-4.739665186661101, -7.659283260401616, -4.864289415179469, -5.108725684655301], [-3.0100862463417126, -4.864289415179469, -3.089233118842289, -3.2444707197865448], [-3.161346623769138, -5.108725684655301, -3.2444707197865448, -3.4075091929278334]]], [[[-4.397266063653004, -7.105967410452397, -4.512887287767106, -4.739665186661101], [-7.105967410452397, -11.48321982510726, -7.292810926086492, -7.659283260401616], [-4.512887287767106, -7.292810926086492, -4.631548643470277, -4.864289415179469], [-4.739665186661101, -7.659283260401616, -4.864289415179469, -5.108725684655301]], [[-7.105967410452397, -11.48321982510726, -7.292810926086492, -7.659283260401616], [-11.48321982510726, -18.556845244994072, -11.785158328197802, -12.377376407972651], [-7.292810926086492, -11.785158328197802, -7.484567284310206, -7.860675601366448], [-7.659283260401616, -12.377376407972651, -7.860675601366448, -8.255683804118878]], [[-4.512887287767106, -7.292810926086492, -4.631548643470277, -4.864289415179469], [-7.292810926086492, -11.785158328197802, -7.484567284310206, -7.860675601366448], [-4.631548643470277, -7.484567284310206, -4.753330067643909, -4.992190499281962], [-4.864289415179469, -7.860675601366448, -4.992190499281962, -5.243053948802298]], [[-4.739665186661101, -7.659283260401616, -4.864289415179469, -5.108725684655301], [-7.659283260401616, -12.377376407972651, -7.860675601366448, -8.255683804118878], [-4.864289415179469, -7.860675601366448, -4.992190499281962, -5.243053948802298], [-5.108725684655301, -8.255683804118878, -5.243053948802298, -5.5065235819837515]]], [[[-2.7926339896239365, -4.512887287767106, -2.8660631967060146, -3.0100862463417126], [-4.512887287767106, -7.292810926086492, -4.631548643470277, -4.864289415179469], [-2.8660631967060146, -4.631548643470277, -2.9414231431806286, -3.089233118842289], [-3.0100862463417126, -4.864289415179469, -3.089233118842289, -3.2444707197865448]], [[-4.512887287767106, -7.292810926086492, -4.631548643470277, -4.864289415179469], [-7.292810926086492, -11.785158328197802, -7.484567284310206, -7.860675601366448], [-4.631548643470277, -7.484567284310206, -4.753330067643909, -4.992190499281962], [-4.864289415179469, -7.860675601366448, -4.992190499281962, -5.243053948802298]], [[-2.8660631967060146, -4.631548643470277, -2.9414231431806286, -3.089233118842289], [-4.631548643470277, -7.484567284310206, -4.753330067643909, -4.992190499281962], [-2.9414231431806286, -4.753330067643909, -3.018764595694322, -3.170461070393087], [-3.089233118842289, -4.992190499281962, -3.170461070393087, -3.3297804715263446]], [[-3.0100862463417126, -4.864289415179469, -3.089233118842289, -3.2444707197865448], [-4.864289415179469, -7.860675601366448, -4.992190499281962, -5.243053948802298], [-3.089233118842289, -4.992190499281962, -3.170461070393087, -3.3297804715263446], [-3.2444707197865448, -5.243053948802298, -3.3297804715263446, -3.4971058601213287]]], [[[-2.9329671351610633, -4.739665186661101, -3.0100862463417126, -3.161346623769138], [-4.739665186661101, -7.659283260401616, -4.864289415179469, -5.108725684655301], [-3.0100862463417126, -4.864289415179469, -3.089233118842289, -3.2444707197865448], [-3.161346623769138, -5.108725684655301, -3.2444707197865448, -3.4075091929278334]], [[-4.739665186661101, -7.659283260401616, -4.864289415179469, -5.108725684655301], [-7.659283260401616, -12.377376407972651, -7.860675601366448, -8.255683804118878], [-4.864289415179469, -7.860675601366448, -4.992190499281962, -5.243053948802298], [-5.108725684655301, -8.255683804118878, -5.243053948802298, -5.5065235819837515]], [[-3.0100862463417126, -4.864289415179469, -3.089233118842289, -3.2444707197865448], [-4.864289415179469, -7.860675601366448, -4.992190499281962, -5.243053948802298], [-3.089233118842289, -4.992190499281962, -3.170461070393087, -3.3297804715263446], [-3.2444707197865448, -5.243053948802298, -3.3297804715263446, -3.4971058601213287]], [[-3.161346623769138, -5.108725684655301, -3.2444707197865448, -3.4075091929278334], [-5.108725684655301, -8.255683804118878, -5.243053948802298, -5.5065235819837515], [-3.2444707197865448, -5.243053948802298, -3.3297804715263446, -3.4971058601213287], [-3.4075091929278334, -5.5065235819837515, -3.4971058601213287, -3.672839546472842]]]]
    d3Vis_modified_dxixjxks_analytical = GE.d3Vis_modified_dxixjxks()
    assert_close4d(d3Vis_modified_dxixjxks_analytical, d3Vis_modified_dxixjxks_sympy, rtol=1e-12)

    # psis
    psis_expect = [[1.0, 1.0, 0.6707604929520135, 0.035196695310572235, 0.2700092136237005, 1.140425934953746], [1.0, 1.0, 0.6707604929520135, 0.035196695310572235, 0.2700092136237005, 1.140425934953746], [1.2927014923654436, 1.2927014923654436, 1.0, 0.13639980946513558, 0.4791047146900759, 1.3703140538273264], [1.1046314487222453, 1.1046314487222453, 0.469691436026506, 1.0, 1.0619227299517133, 0.9326943648817854], [1.4015831210522953, 1.4015831210522953, 1.388780709384238, 0.4991964528404533, 1.0, 1.4462050689950563], [0.8750017310322578, 0.8750017310322578, 0.6029103149261912, 0.0345211225752076, 0.2468417790942161, 1.0]]
    psis_analytical = GE.psis()
    assert_close2d(psis_expect, psis_analytical, rtol=1e-12)


    # psis 1st T derivative
    def to_diff(T):
        return np.array(GE.to_T_xs(T, xs).psis()).ravel().tolist()
    dpsis_dT_numerical = np.array(derivative(to_diff, T, n=1, order=5, dx=T*4e-7, scalar=False)).reshape(6,6)
    dpsis_dT_expect = [[0.0, 0.0, 0.0005501317207598933, 0.000647367519092256, 0.0008408159966567053, 0.0001585324390412828], [0.0, 0.0, 0.0005501317207598933, 0.000647367519092256, 0.0008408159966567053, 0.0001585324390412828], [0.00014919258368395743, 0.00014919258368395743, 0.0, 0.002243244436897652, -0.0015269169402591476, -0.0007143860476337597], [0.01172661891603331, 0.01172661891603331, 0.0035449013241947796, 0.0, 0.004488270386007889, 0.007419012445749164], [0.002003111126252462, 0.002003111126252462, 0.0045855195246785364, 0.001865930344365094, 0.0, 0.0017469842072108397], [-5.567718366748017e-05, -5.567718366748017e-05, 0.0009273911445009344, 0.0008626432663807586, 0.0008234504937961999, 0.0]]
    dpsis_dT = GE.dpsis_dT()

    assert_close2d(dpsis_dT, dpsis_dT_expect, rtol=1e-12)
    assert_close2d(dpsis_dT, dpsis_dT_numerical, rtol=1e-9, atol=1e-12)

    # psis 2nd T derivative
    def to_diff(T):
        return np.array(GE.to_T_xs(T, xs).dpsis_dT()).ravel().tolist()
    d2psis_dT2_numerical = np.array(derivative(to_diff, T, n=1, order=5, dx=T*4e-7, scalar=False)).reshape(6,6)
    d2psis_dT2_expect = [[0.0, 0.0, -2.497385498809293e-06, 8.14459865464684e-06, -1.8882607026289393e-06, 5.162517426439307e-06], [0.0, 0.0, -2.497385498809293e-06, 8.14459865464684e-06, -1.8882607026289393e-06, 5.162517426439307e-06], [-7.824201278437826e-07, -7.824201278437826e-07, 0.0, 1.603797634138834e-05, 2.170083500037727e-06, -6.727347992871973e-06], [5.62004911546718e-05, 5.62004911546718e-05, -2.8370714993233295e-05, 0.0, 2.9189001336820514e-05, -5.303653672013689e-05], [-7.873423270360566e-06, -7.873423270360566e-06, 1.8216054771503098e-05, -5.578869568251919e-06, 0.0, -7.253123234576289e-06], [-5.752587724790159e-06, -5.752587724790159e-06, 1.487285599566105e-06, 1.8100217819471976e-05, -1.6665240719858701e-06, 0.0]]
    d2psis_dT2 = GE.d2psis_dT2()

    assert_close2d(d2psis_dT2, d2psis_dT2_expect, rtol=1e-12)
    assert_close2d(d2psis_dT2, d2psis_dT2_numerical, rtol=1e-9)

    # psis 3rd T derivative
    def to_diff(T):
        return np.array(GE.to_T_xs(T, xs).d2psis_dT2()).ravel().tolist()
    d3psis_dT3_numerical = np.array(derivative(to_diff, T, n=1, order=5, dx=T*4e-7, scalar=False)).reshape(6,6)
    d3psis_dT3_expect = [[0.0, 0.0, 1.682072114998335e-08, 4.165016652884307e-08, 2.2840705523955177e-09, -3.9180897581798757e-08], [0.0, 0.0, 1.682072114998335e-08, 4.165016652884307e-08, 2.2840705523955177e-09, -3.9180897581798757e-08], [6.153949633573174e-09, 6.153949633573174e-09, 0.0, -2.5452843203389944e-07, 3.1946609560190037e-08, 6.798965297821362e-08], [-3.0423941636258116e-07, -3.0423941636258116e-07, -6.030264891259016e-07, 0.0, 1.2759377454995282e-07, -1.3036091180326196e-06], [4.4375159106213954e-08, 4.4375159106213954e-08, 5.573000263673513e-08, -1.3773839133617482e-08, 0.0, 4.389562987455186e-08], [4.7375924278776154e-08, 4.7375924278776154e-08, 1.9860559592907367e-09, 3.0735539819699594e-07, 4.772477057976549e-10, 0.0]]
    d3psis_dT3 = GE.d3psis_dT3()

    assert_close2d(d3psis_dT3, d3psis_dT3_expect, rtol=1e-12)
    assert_close2d(d3psis_dT3, d3psis_dT3_numerical, rtol=4e-9)

    # lngammas combinatorial
    lngammas_c = GE.lngammas_c()
    lngammas_c_expect = [-0.01830902002060108, -0.009915329990301362, -0.01207311645273321, 0.030610555513890587]
    assert_close1d(lngammas_c, lngammas_c_expect, rtol=1e-12)

    def to_jac_lngammas_c(xs):
        return GE.to_T_xs(T, xs).lngammas_c()

    # lngammas combinatorial x derivative
    dlngammas_c_dxs_analytical = GE.dlngammas_c_dxs()
    dlngammas_c_dxs_expect = [[-0.14720169204015732, -0.23711219630542058, -0.1509362387874117, -0.15925421307962118], [0.2683935958787521, 0.3328254274584339, 0.25752473384564023, 0.36714416244020576], [-0.12935866673128182, -0.22640348628239915, -0.1358443244469728, -0.12603630669610544], [-0.09535468420616511, -0.07446210087051597, -0.08371434987878001, -0.16422193861631618]]
    assert_close2d(dlngammas_c_dxs_expect, dlngammas_c_dxs_analytical, rtol=1e-11)
    dlngammas_c_dxs_numerical = jacobian(to_jac_lngammas_c, xs, scalar=False, perturbation=1e-7)
    assert_close2d(dlngammas_c_dxs_analytical, dlngammas_c_dxs_numerical, rtol=1e-6)

    # lngammas combinatorial xixj derivatives
    d2lngammas_c_dxixjs = GE.d2lngammas_c_dxixjs()
    d2lngammas_c_dxixjs_numerical = np.array(hessian(to_jac_lngammas_c, xs, scalar=False, perturbation=1e-4)).T
    d2lngammas_c_dxixjs_analytical_sympy = [[[-0.431834610022220, -0.699005968004498, -0.443395964492912, -0.464561835915905], [-0.699005968004498, -1.05415684883700, -0.703959984354841, -0.811779652806081], [-0.443395964492912, -0.703959984354841, -0.452817910598213, -0.487641837829355], [-0.464561835915905, -0.811779652806081, -0.487641837829355, -0.453519975183815]], [[-1.11383953685822, -1.72452652362249, -1.12970114086422, -1.25891480362318], [-1.72452652362249, -2.46673321720012, -1.71610214156011, -2.08731980579672], [-1.12970114086422, -1.71610214156011, -1.14048618689678, -1.29897088552115], [-1.25891480362318, -2.08731980579672, -1.29897088552115, -1.33068993050655]], [[-0.461103182415314, -0.732574789932881, -0.470990720136454, -0.506727852440381], [-0.732574789932881, -1.07434727235270, -0.732917839925643, -0.870921749315069], [-0.470990720136454, -0.732917839925643, -0.478456264674908, -0.528878289507013], [-0.506727852440381, -0.870921749315069, -0.528878289507013, -0.508531084106759]], [[-0.516999750615383, -0.896519149468931, -0.541458549217459, -0.510041039673542], [-0.896519149468931, -1.50168962767411, -0.927046440399886, -0.940075844179167], [-0.541458549217459, -0.927046440399886, -0.564522191356871, -0.545966133985468], [-0.510041039673542, -0.940075844179167, -0.545966133985468, -0.448860503118729]]]
    assert_close3d(d2lngammas_c_dxixjs, d2lngammas_c_dxixjs_analytical_sympy, rtol=1e-12)
    assert_close3d(d2lngammas_c_dxixjs_numerical, d2lngammas_c_dxixjs, rtol=1e-3)

    # lngammas combinatorial xixjxk derivatives
    d3lngammas_c_dxixjxks_sympy = [[[[1.6157638281347313, 2.6137187048291146, 1.6587201153729827, 1.7395281480044291], [2.6137187048291146, 4.110995823528366, 2.6623381481448205, 2.904670949015034], [1.6587201153729827, 2.6623381481448205, 1.6990980164988494, 1.8019531515965868], [1.7395281480044291, 2.904670949015034, 1.8019531515965868, 1.80242048689384]], [[2.6137187048291146, 4.110995823528366, 2.6623381481448205, 2.904670949015034], [4.110995823528366, 6.160815002256778, 4.137938111790589, 4.7761520747739326], [2.6623381481448205, 4.137938111790589, 2.7038972231319587, 2.9919457629077058], [2.904670949015034, 4.7761520747739326, 2.9919457629077058, 3.08949331840945]], [[1.6587201153729827, 2.6623381481448205, 1.6990980164988494, 1.8019531515965868], [2.6623381481448205, 4.137938111790589, 2.7038972231319587, 2.9919457629077058], [1.6990980164988494, 2.7038972231319587, 1.7364673150844876, 1.862928127152595], [1.8019531515965868, 2.9919457629077058, 1.862928127152595, 1.8842209873970486]], [[1.7395281480044291, 2.904670949015034, 1.8019531515965868, 1.80242048689384], [2.904670949015034, 4.7761520747739326, 2.9919457629077058, 3.08949331840945], [1.8019531515965868, 2.9919457629077058, 1.862928127152595, 1.8842209873970486], [1.80242048689384, 3.08949331840945, 1.8842209873970486, 1.788495879565966]]], [[[3.294569023494623, 5.211247685889852, 3.3610906692568103, 3.6385366069018446], [5.211247685889852, 7.93881837163565, 5.26711987407594, 5.962076549915123], [3.3610906692568103, 5.26711987407594, 3.4210226652076683, 3.745107602665679], [3.6385366069018446, 5.962076549915123, 3.745107602665679, 3.880502427256209]], [[5.211247685889852, 7.93881837163565, 5.26711987407594, 5.962076549915123], [7.93881837163565, 11.313907501611073, 7.9056789654576125, 9.563376478050788], [5.26711987407594, 7.9056789654576125, 5.305998080087991, 6.096889237192718], [5.962076549915123, 9.563376478050788, 6.096889237192718, 6.5366482997368776]], [[3.3610906692568103, 5.26711987407594, 3.4210226652076683, 3.745107602665679], [5.26711987407594, 7.9056789654576125, 5.305998080087991, 6.096889237192718], [3.4210226652076683, 5.305998080087991, 3.473428161369257, 3.8473049351920636], [3.745107602665679, 6.096889237192718, 3.8473049351920636, 4.027391791992885]], [[3.6385366069018446, 5.962076549915123, 3.745107602665679, 3.880502427256209], [5.962076549915123, 9.563376478050788, 6.096889237192718, 6.5366482997368776], [3.745107602665679, 6.096889237192718, 3.8473049351920636, 4.027391791992885], [3.880502427256209, 6.5366482997368776, 4.027391791992885, 3.991668520970825]]], [[[1.687782288423389, 2.7093025236417194, 1.7289243470905367, 1.8332782904127969], [2.7093025236417194, 4.213832387211374, 2.752096474687314, 3.04256708359509], [1.7289243470905367, 2.752096474687314, 1.7670778958907294, 1.8950769256798736], [1.8332782904127969, 3.04256708359509, 1.8950769256798736, 1.9179853008967034]], [[2.7093025236417194, 4.213832387211374, 2.752096474687314, 3.04256708359509], [4.213832387211374, 6.203569871499487, 4.225015569357999, 4.9615860827389096], [2.752096474687314, 4.225015569357999, 2.7869133000998065, 3.1262918939613966], [3.04256708359509, 4.9615860827389096, 3.1262918939613966, 3.2701469966457783]], [[1.7289243470905367, 2.752096474687314, 1.7670778958907294, 1.8950769256798736], [2.752096474687314, 4.225015569357999, 2.7869133000998065, 3.1262918939613966], [1.7670778958907294, 2.7869133000998065, 1.8017756403803817, 1.9550672708058725], [1.8950769256798736, 3.1262918939613966, 1.9550672708058725, 2.000418970858349]], [[1.8332782904127969, 3.04256708359509, 1.8950769256798736, 1.9179853008967034], [3.04256708359509, 4.9615860827389096, 3.1262918939613966, 3.2701469966457783], [1.8950769256798736, 3.1262918939613966, 1.9550672708058725, 2.000418970858349], [1.9179853008967034, 3.2701469966457783, 2.000418970858349, 1.9256400463858483]]], [[[1.8255924726249153, 3.0437506128568153, 1.8902804419828647, 1.8951863276031244], [3.0437506128568153, 5.000904351395424, 3.134682371939994, 3.2394026100786686], [1.8902804419828647, 3.134682371939994, 1.953577885474481, 1.9794260027897586], [1.8951863276031244, 3.2394026100786686, 1.9794260027897586, 1.8884850590598838]], [[3.0437506128568153, 5.000904351395424, 3.134682371939994, 3.2394026100786686], [5.000904351395424, 8.010125385293254, 5.110444083939642, 5.500633116264876], [3.134682371939994, 5.110444083939642, 3.220829251438246, 3.3694349676276216], [3.2394026100786686, 5.500633116264876, 3.3694349676276216, 3.300648591617995]], [[1.8902804419828647, 3.134682371939994, 1.953577885474481, 1.9794260027897586], [3.134682371939994, 5.110444083939642, 3.220829251438246, 3.3694349676276216], [1.953577885474481, 3.220829251438246, 2.0151064483215024, 2.0634751877237676], [1.9794260027897586, 3.3694349676276216, 2.0634751877237676, 1.991864912380111]], [[1.8951863276031244, 3.2394026100786686, 1.9794260027897586, 1.8884850590598838], [3.2394026100786686, 5.500633116264876, 3.3694349676276216, 3.300648591617995], [1.9794260027897586, 3.3694349676276216, 2.0634751877237676, 1.991864912380111], [1.8884850590598838, 3.300648591617995, 1.991864912380111, 1.7868617457401452]]]]
    d3lngammas_c_dxixjxks_expct = [[[[1.6157638281347175, 2.613718704829054, 1.6587201153729723, 1.7395281480044034], [2.6137187048290684, 4.11099582352827, 2.662338148144798, 2.904670949014971], [1.658720115372958, 2.6623381481447836, 1.6990980164988514, 1.801953151596571], [1.7395281480044034, 2.904670949014985, 1.801953151596578, 1.8024204868938298]], [[2.613718704829097, 4.11099582352827, 2.6623381481447836, 2.904670949014971], [4.110995823528327, 6.160815002256527, 4.137938111790561, 4.7761520747739326], [2.6623381481448263, 4.137938111790447, 2.703897223131932, 2.991945762907676], [2.904670949014971, 4.776152074773847, 2.991945762907662, 3.089493318409396]], [[1.6587201153729723, 2.6623381481447552, 1.699098016498823, 1.801953151596571], [2.662338148144798, 4.137938111790504, 2.703897223131932, 2.991945762907676], [1.69909801649883, 2.703897223131989, 1.736467315084461, 1.8629281271525713], [1.801953151596571, 2.991945762907662, 1.8629281271525713, 1.8842209873970432]], [[1.7395281480044034, 2.904670949014985, 1.801953151596578, 1.8024204868938298], [2.9046709490150278, 4.776152074773847, 2.991945762907662, 3.089493318409424], [1.8019531515965994, 2.991945762907662, 1.8629281271525713, 1.884220987397029], [1.8024204868938298, 3.089493318409424, 1.884220987397022, 1.788495879565943]]], [[[3.294569023494617, 5.2112476858899015, 3.361090669256754, 3.638536606901809], [5.2112476858899015, 7.938818371635534, 5.267119874075888, 5.9620765499151105], [3.3610906692567255, 5.267119874075945, 3.4210226652076443, 3.7451076026656835], [3.638536606901752, 5.962076549915167, 3.745107602665655, 3.880502427256218]], [[5.21124768588993, 7.938818371635591, 5.267119874075917, 5.962076549915139], [7.9388183716356195, 11.31390750161097, 7.9056789654575255, 9.563376478050714], [5.267119874075945, 7.905678965457696, 5.30599808008796, 6.096889237192755], [5.962076549915196, 9.56337647805077, 6.09688923719267, 6.536648299736697]], [[3.3610906692567255, 5.267119874075945, 3.4210226652076443, 3.745107602665655], [5.267119874075945, 7.905678965457696, 5.30599808008796, 6.096889237192755], [3.4210226652076443, 5.305998080087932, 3.4734281613692275, 3.8473049351920565], [3.7451076026656693, 6.096889237192755, 3.8473049351919997, 4.027391791992855]], [[3.638536606901752, 5.962076549915167, 3.745107602665655, 3.8805024272561894], [5.9620765499151105, 9.563376478050657, 6.09688923719267, 6.536648299736811], [3.745107602665641, 6.096889237192755, 3.8473049351919997, 4.027391791992855], [3.880502427256218, 6.536648299736811, 4.027391791992869, 3.991668520970805]]], [[[1.6877822884233638, 2.709302523641668, 1.7289243470905333, 1.8332782904127853], [2.7093025236416537, 4.213832387211198, 2.752096474687349, 3.042567083595088], [1.7289243470905333, 2.7520964746873346, 1.7670778958906936, 1.8950769256798665], [1.8332782904127853, 3.042567083595088, 1.8950769256798665, 1.9179853008966603]], [[2.709302523641682, 4.213832387211198, 2.752096474687349, 3.042567083595088], [4.213832387211227, 6.203569871499269, 4.2250155693579075, 4.961586082738876], [2.7520964746873346, 4.225015569357851, 2.7869133000997977, 3.1262918939613797], [3.042567083595088, 4.961586082738876, 3.126291893961323, 3.2701469966457637]], [[1.7289243470905191, 2.7520964746873062, 1.7670778958906652, 1.8950769256798878], [2.752096474687363, 4.2250155693579075, 2.7869133000997977, 3.126291893961323], [1.7670778958906936, 2.786913300099755, 1.801775640380363, 1.9550672708058912], [1.8950769256798736, 3.1262918939613797, 1.955067270805884, 2.000418970858334]], [[1.8332782904127995, 3.042567083595088, 1.8950769256798665, 1.9179853008966603], [3.042567083595145, 4.961586082738876, 3.126291893961323, 3.2701469966457637], [1.8950769256798665, 3.1262918939613797, 1.955067270805877, 2.0004189708583056], [1.9179853008966674, 3.2701469966457637, 2.000418970858341, 1.9256400463858583]]], [[[1.825592472624912, 3.043750612856769, 1.8902804419828954, 1.8951863276030991], [3.043750612856769, 5.00090435139532, 3.1346823719399595, 3.2394026100787414], [1.890280441982867, 3.1346823719399595, 1.9535778854744663, 1.979426002789772], [1.8951863276030991, 3.2394026100786846, 1.9794260027897863, 1.8884850590598674]], [[3.043750612856769, 5.00090435139532, 3.1346823719399595, 3.239402610078713], [5.0009043513954055, 8.010125385292952, 5.110444083939512, 5.500633116264908], [3.1346823719399737, 5.110444083939512, 3.220829251438275, 3.3694349676275976], [3.239402610078699, 5.500633116264993, 3.369434967627683, 3.300648591617957]], [[1.8902804419828954, 3.134682371939988, 1.9535778854744663, 1.9794260027897863], [3.1346823719399737, 5.110444083939512, 3.2208292514382464, 3.369434967627626], [1.9535778854744663, 3.220829251438232, 2.0151064483215038, 2.063475187723782], [1.979426002789765, 3.369434967627626, 2.063475187723789, 1.991864912380116]], [[1.8951863276030991, 3.239402610078713, 1.979426002789772, 1.8884850590598674], [3.2394026100786846, 5.5006331162647655, 3.3694349676276403, 3.3006485916179855], [1.979426002789765, 3.369434967627569, 2.063475187723789, 1.991864912380116], [1.8884850590598603, 3.300648591617957, 1.991864912380123, 1.78686174574014]]]]
    d3lngammas_c_dxixjxks = GE.d3lngammas_c_dxixjxks()
    assert_close4d(d3lngammas_c_dxixjxks, d3lngammas_c_dxixjxks_sympy, rtol=1e-12)
    assert_close4d(d3lngammas_c_dxixjxks, d3lngammas_c_dxixjxks_expct, rtol=1e-12)

    # Residual - basic parts

    Xs_pure_expect = [[0.0, 0.0, 0.5, 0.3333333333333333],
                      [0.0, 0.0, 0.0, 0.3333333333333333],
                      [1.0, 0.0, 0.0, 0.0],
                      [0.0, 0.0, 0.0, 0.3333333333333333],
                      [0.0, 0.0, 0.5, 0.0],
                      [0.0, 1.0, 0.0, 0.0]]
    assert_close2d(GE.Xs_pure(), Xs_pure_expect, rtol=1e-12)

    Xs_expect = [0.11363636363636363, 0.09090909090909091, 0.27272727272727276, 0.09090909090909091, 0.022727272727272728, 0.40909090909090906]
    assert_close1d(Xs_expect, GE.Xs(), rtol=1e-12)

    Thetas_expect = [0.1555178945269664, 0.08304843221308064, 0.15203457516991448, 0.10469896262761912, 0.048965852914787694, 0.4557342825476318]
    assert_close1d(GE.Thetas(), Thetas_expect, rtol=1e-12)

    Thetas_pure_expect = [[0.0, 0.0, 1.0, 0.0, 0.0, 0.0],
     [0.0, 0.0, 0.0, 0.0, 0.0, 1.0],
     [0.3884575948440017, 0.0, 0.0, 0.0, 0.6115424051559982, 0.0],
     [0.3985572587917043, 0.2660429816651638, 0.0, 0.335399759543132, 0.0, 0.0]]
    assert_close2d(GE.Thetas_pure(), Thetas_pure_expect, rtol=1e-12)


    lnGammas_subgroups_pure_expect = [[0.0, 0.0, 0.2525211912149972, 0.30820317547330855],
     [0.0, 0.0, 0.0, 0.20573026824344812],
     [0.0, 0.0, 0.0, 0.0],
     [0.0, 0.0, 0.0, 0.3400841900068883],
     [0.0, 0.0, 0.07142530517046565, 0.0],
     [0.0, 0.0, 0.0, 0.0]]
    assert_close2d(GE.lnGammas_subgroups_pure(), lnGammas_subgroups_pure_expect, rtol=1e-12)

    lnGammas_subgroups_expect = [0.08369133966599032, 0.055865231539864016, 0.05507425302818821, 1.1382224417102726, 0.4125412676822281, 0.08846629261844947]
    assert_close1d(lnGammas_subgroups_expect, GE.lnGammas_subgroups(), rtol=1e-12)

    lngammas_r_expect = [0.33044551816912926, 0.5307977557106969, 0.17228611096275556, 0.423761379192482]
    assert_close1d(lngammas_r_expect, GE.lngammas_r(), rtol=1e-12)

    assert_close(GE.GE(), 1292.0910446403336)

    gammas_expect = [1.366341183343183, 1.6835125692286341, 1.1737608489858082, 1.5751837540437106]
    assert_close1d(GE.gammas(), gammas_expect, rtol=1e-12)

    dlnGammas_subgroups_dT_numerical = [i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).lnGammas_subgroups(), [GE.T], scalar=False, perturbation=1e-8)]
    assert_close1d(dlnGammas_subgroups_dT_numerical, GE.dlnGammas_subgroups_dT(), rtol=5e-7)
    dlnGammas_subgroups_dT_expect = [-0.0009677119301191769, -0.0006459623093112642, -0.00035140582140865084, -0.00743085779040337, -0.003365551965236213, -0.000493060082455736]
    assert_close1d(dlnGammas_subgroups_dT_expect, GE.dlnGammas_subgroups_dT(), rtol=1e-12)

    d2lnGammas_subgroups_dT2_numerical = [i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).dlnGammas_subgroups_dT(), [GE.T], scalar=False, perturbation=1e-8)]
    d2lnGammas_subgroups_dT2_expect = [-6.5673147545536255e-06, -4.3837816531857294e-06, 3.331409426621567e-06, -9.761393674713978e-06, 2.7776341731692812e-05, -6.963348655067715e-07]
    d2lnGammas_subgroups_dT2 = GE.d2lnGammas_subgroups_dT2()
    assert_close1d(d2lnGammas_subgroups_dT2, d2lnGammas_subgroups_dT2_expect, rtol=1e-12)
    assert_close1d(d2lnGammas_subgroups_dT2, d2lnGammas_subgroups_dT2_numerical, rtol=1e-6)

    d3lnGammas_subgroups_dT3_numerical = [i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).d2lnGammas_subgroups_dT2(), [GE.T], scalar=False, perturbation=1e-8)]
    d3lnGammas_subgroups_dT3 = GE.d3lnGammas_subgroups_dT3()
    d3lnGammas_subgroups_dT3_expect = [2.292547993512055e-08, 1.5303103640704054e-08, 1.0888701309841477e-07, 3.433594317292919e-07, -6.694401804755741e-07, 3.2918869996009466e-08]
    assert_close1d(d3lnGammas_subgroups_dT3, d3lnGammas_subgroups_dT3_expect, rtol=1e-12)
    assert_close1d(d3lnGammas_subgroups_dT3, d3lnGammas_subgroups_dT3_numerical, rtol=1e-6)

    dlnGammas_subgroups_pure_dT_expect = [[0.0, 0.0, -0.0013677908855988523, -0.002042849755954296],
     [0.0, 0.0, 0.0, -0.001363633024313006],
     [0.0, 0.0, 0.0, 0.0],
     [0.0, 0.0, 0.0, -0.004385381381959583],
     [0.0, 0.0, -0.0004368227607265967, 0.0],
     [0.0, 0.0, 0.0, 0.0]]

    dlnGammas_subgroups_pure_dT_numerical = [[i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).lnGammas_subgroups_pure()[j], [GE.T], scalar=False, perturbation=1e-8)] for j in range(6)]
    dlnGammas_subgroups_pure_dT = GE.dlnGammas_subgroups_pure_dT()
    assert_close2d(dlnGammas_subgroups_pure_dT, dlnGammas_subgroups_pure_dT_expect, rtol=1e-12)
    assert_close2d(dlnGammas_subgroups_pure_dT_numerical, dlnGammas_subgroups_pure_dT, rtol=1e-6)

    d2lnGammas_subgroups_pure_dT2_expect = [[0.0, 0.0, 5.259889238911016e-06, -1.7377017617239823e-05],
      [0.0, 0.0, 0.0, -1.1599421356304221e-05],
      [-0.0, 0.0, 0.0, 0.0],
      [0.0, 0.0, 0.0, 1.0119388945730533e-05],
      [0.0, 0.0, 1.923660010004082e-06, 0.0],
      [0.0, -0.0, 0.0, 0.0]]
    d2lnGammas_subgroups_pure_dT2_numerical = [[i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).dlnGammas_subgroups_pure_dT()[j], [GE.T], scalar=False, perturbation=3e-7)] for j in range(6)]
    d2lnGammas_subgroups_pure_dT2 = GE.d2lnGammas_subgroups_pure_dT2()
    assert_close2d(d2lnGammas_subgroups_pure_dT2, d2lnGammas_subgroups_pure_dT2_expect, rtol=1e-12)
    assert_close2d(d2lnGammas_subgroups_pure_dT2, d2lnGammas_subgroups_pure_dT2_numerical, rtol=1e-5)

    d3lnGammas_subgroups_pure_dT3_expect = [[0.0, 0.0, -2.7154094696008944e-08, 9.320074307198575e-08],
     [0.0, 0.0, 0.0, 6.221290174328159e-08],
     [0.0, 0.0, 0.0, 0.0],
     [0.0, 0.0, 0.0, 1.6857904176843558e-07],
     [0.0, 0.0, -1.4797079241774134e-08, 0.0],
     [0.0, 0.0, 0.0, 0.0]]
    d3lnGammas_subgroups_pure_dT3_numerical = [[i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).d2lnGammas_subgroups_pure_dT2()[j], [GE.T], scalar=False, perturbation=3e-8)] for j in range(6)]
    d3lnGammas_subgroups_pure_dT3 = GE.d3lnGammas_subgroups_pure_dT3()
    assert_close2d(d3lnGammas_subgroups_pure_dT3, d3lnGammas_subgroups_pure_dT3_expect, rtol=1e-12)
    assert_close2d(d3lnGammas_subgroups_pure_dT3_numerical, d3lnGammas_subgroups_pure_dT3, rtol=1e-6)

    dlngammas_dT_numerical = [i[0] for i in jacobian(lambda T: [log(i) for i in GE.to_T_xs(T=T[0], xs=GE.xs).gammas()], [GE.T], scalar=False, perturbation=3e-8)]
    dlngammas_dT_expect = [-0.0021084349284519036, -0.002958360494734416, -0.002528650249029941, -0.0012526678676069276]
    dlngammas_dT = GE.dlngammas_dT()
    assert_close1d(dlngammas_dT, dlngammas_dT_expect, rtol=1e-12)
    assert_close1d(dlngammas_dT, dlngammas_dT_numerical, rtol=1e-7)

    d2lngammas_dT2_numerical = [i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).dlngammas_dT(), [GE.T], scalar=False, perturbation=3e-8)]
    d2lngammas_dT2_expect = [1.99884565597294e-05, -4.178009193040629e-06, 1.4025477728224088e-05, -1.8554400546398206e-06]
    d2lngammas_dT2 = GE.d2lngammas_dT2()
    assert_close1d(d2lngammas_dT2, d2lngammas_dT2_expect, rtol=1e-12)
    assert_close1d(d2lngammas_dT2, d2lngammas_dT2_numerical, rtol=1e-6)

    d3lngammas_dT3_numerical = [i[0] for i in jacobian(lambda T: GE.to_T_xs(T=T[0], xs=GE.xs).d2lngammas_dT2(), [GE.T], scalar=False, perturbation=3e-8)]
    d3lngammas_dT3_expect = [6.533220785904886e-07, 1.975132199760568e-07, -6.045635266026704e-07, 5.7595328721413586e-08]
    d3lngammas_dT3 = GE.d3lngammas_dT3()
    assert_close1d(d3lngammas_dT3, d3lngammas_dT3_expect, rtol=1e-13)
    assert_close1d(d3lngammas_dT3, d3lngammas_dT3_numerical, rtol=1e-6)

    dGE_dT_expect = -2.9382799850394155
    dGE_dT_numerical = derivative(lambda T: GE.to_T_xs(T=T, xs=GE.xs).GE(), GE.T, dx=1e-5)
    dGE_dT = GE.dGE_dT()
    assert_close(dGE_dT_expect, dGE_dT, rtol=1e-12)
    assert_close(dGE_dT_numerical, dGE_dT, rtol=1e-7)

    d2GE_dT2_expect = -0.023744489066399498
    d2GE_dT2_numerical = derivative(lambda T: GE.to_T_xs(T=T, xs=GE.xs).dGE_dT(), GE.T, dx=1e-5)
    d2GE_dT2 = GE.d2GE_dT2()
    assert_close(d2GE_dT2_expect, d2GE_dT2, rtol=1e-12)
    assert_close(d2GE_dT2_numerical, d2GE_dT2, rtol=1e-7)

    d3GE_dT3_expect = 0.0005580618737894017
    d3GE_dT3_numerical = derivative(lambda T: GE.to_T_xs(T=T, xs=GE.xs).d2GE_dT2(), GE.T, dx=1e-5)
    d3GE_dT3 = GE.d3GE_dT3()
    assert_close(d3GE_dT3_expect, d3GE_dT3, rtol=1e-12)
    assert_close(d3GE_dT3_numerical, d3GE_dT3, rtol=1e-7)

    def to_jac_Thetas(xs):
        return GE.to_T_xs(T, xs).Thetas()

    dThetas_dxs_numerical = jacobian(to_jac_Thetas, xs, scalar=False, perturbation=1e-7)
    dThetas_dxs_expect = [[-0.11822048512863448, -0.23624945361855096, 0.18651349452748248, 0.1896689591463599], [-0.06313116555021574, -0.12616005890444643, -0.0664964078085818, 0.14280972890558816], [0.6446003156140904, -0.2309578934583165, -0.12173322050877644, -0.11864843258611371], [-0.07958931151910408, -0.159049688721931, -0.08383186449755824, 0.18003988842538987], [-0.037222513228661304, -0.07438472615817876, 0.45045184893310086, -0.038213161000310476], [-0.3464368401874746, 0.8268018208614238, -0.36490385064566694, -0.35565698289091363]]
    dThetas_dxs = GE.dThetas_dxs()
    assert_close2d(dThetas_dxs, dThetas_dxs_numerical, rtol=1e-7)
    assert_close2d(dThetas_dxs_expect, dThetas_dxs, rtol=1e-10)

    d2Thetas_dxixjs_expect = [[[0.1797360123291314, 0.09598119934408587, -0.98001535138781, 0.12100327164872965, 0.05659108985476041, 0.5267037782111026], [0.359180853150208, 0.19180690958948896, -0.8036536151508472, 0.2418105185574591, 0.1130905024058913, -0.10223516855220018], [-0.04712402878215064, 0.10109753111503113, -0.42358951790270866, 0.12745341904588095, -0.31261742258385145, 0.5547800191077986], [-0.05192142601009778, -0.05929225076737804, -0.4128555225025081, -0.07474960070616916, 0.058097216985817456, 0.5407215830003357]], [[0.3591808531502079, 0.19180690958948896, -0.8036536151508472, 0.2418105185574591, 0.1130905024058913, -0.10223516855220016], [0.717779834980802, 0.3833030928732323, 0.7017028658262549, 0.48322930519408913, 0.22599779872133097, -2.5120128975957092], [-0.09417171685579047, 0.2020312846975917, 0.3698533460718659, 0.25470036414283315, -0.624728406390085, -0.10768487166641533], [-0.10375873955038062, -0.11848844836295057, 0.3604810553335572, -0.1493781074051772, 0.11610031674902423, -0.10495607676407319]], [[-0.047124028782150655, 0.1010975311150311, -0.42358951790270866, 0.12745341904588092, -0.31261742258385145, 0.5547800191077986], [-0.09417171685579047, 0.2020312846975917, 0.36985334607186593, 0.25470036414283315, -0.624728406390085, -0.1076848716664153], [-0.2986805906722391, 0.10648659182632376, 0.19494219599557058, 0.13424739517491774, -0.7213484721284674, 0.5843528798038944], [-0.2974227866177282, -0.062452857515835854, 0.1900022516705034, -0.07873417017989927, -0.32093748821614554, 0.5695450508591055]], [[-0.05192142601009778, -0.059292250767378055, -0.4128555225025081, -0.07474960070616918, 0.058097216985817456, 0.5407215830003357], [-0.10375873955038063, -0.11848844836295057, 0.3604810553335572, -0.1493781074051772, 0.11610031674902423, -0.10495607676407318], [-0.2974227866177282, -0.062452857515835854, 0.19000225167050344, -0.07873417017989927, -0.32093748821614554, 0.5695450508591053], [-0.29603693354363314, -0.22289864622910932, 0.18518748829874465, -0.28100779761153216, 0.05964342850013566, 0.5551124605853943]]]
    d2Thetas_d2xs_numerical = hessian(to_jac_Thetas, xs, scalar=False, perturbation=2e-5)
    d2Thetas_dxixjs = GE.d2Thetas_dxixjs()
    assert_close3d(d2Thetas_dxixjs, d2Thetas_dxixjs_expect, rtol=1e-10)
    assert_close3d(d2Thetas_dxixjs, d2Thetas_d2xs_numerical, rtol=2e-4)

    # dlnGammas_subgroups_dxs
    def to_jac_lnGammas_subgroups(xs):
        return GE.to_T_xs(T, xs).lnGammas_subgroups()
    dlnGammas_subgroups_dxs_expect = [[-0.0654651685450795, -0.18825976488501353, 0.14260352260642384, 0.13827652728469375], [-0.04369898741211425, -0.12566623257454568, 0.09519000222248182, 0.09230166758134582], [-0.10412106268602338, -0.05850380921367319, -0.049689867111079096, 0.10836085503103632], [0.7593292861434112, 1.0993694696726688, -0.41111230956639105, -1.10141366793461], [-0.2326740341213961, 0.7593609935739266, -0.5663536643902684, -0.3115953120221794], [-0.05850380921367342, -0.16726830038762638, 0.09518353811481882, 0.13090724536885154]]

    dlnGammas_subgroups_dxs = GE.dlnGammas_subgroups_dxs()
    dlnGammas_subgroups_dxs_num = jacobian(to_jac_lnGammas_subgroups, xs, scalar=False, perturbation=4e-8)
    assert_close2d(dlnGammas_subgroups_dxs, dlnGammas_subgroups_dxs_expect, rtol=1e-10)
    assert_close2d(dlnGammas_subgroups_dxs, dlnGammas_subgroups_dxs_num, rtol=1e-6)

    # d2lnGammas_subgroups_d2xs
    d2lnGammas_subgroups_d2xs = GE.d2lnGammas_subgroups_dxixjs()
    d2lnGammas_subgroups_d2xs_expect =[[[0.2552774537326049, 0.17040155070518243, 0.30154596958467283, -1.0339901043717137, 1.1284312219463979, 0.20434710278900145], [0.26372781634545167, 0.1760422952057073, 0.20434710278900156, -1.3064472745845013, -0.059143258778847754, 0.24001041188439687], [0.024800205100049284, 0.016554510964691644, 0.23061811261316706, -0.42245293764241, 1.0998136838175623, 0.03409742626110069], [-0.16797171903770552, -0.1121236559677595, -0.10138518332232108, -0.2958794728236934, -0.21312650253996349, -0.1444461938388904]], [[0.2637278163454517, 0.17604229520570736, 0.2043471027890015, -1.3064472745845013, -0.059143258778847844, 0.24001041188439687], [0.6193361657616334, 0.4134162320661883, 0.24001041188439684, -1.5101781289496212, -1.0797456884674184, 0.5292269737687666], [-0.16563536360291592, -0.1105641034758906, 0.0340974262611006, -0.7225650771563846, -0.5782592933612959, -0.07673492045096436], [-0.08430777938068774, -0.0562767143471578, -0.14444619383889026, -0.7819251708881101, -0.9144567648545029, -0.07957095518696673]], [[0.0248002051000494, 0.016554510964691724, 0.23061811261316695, -0.4224529376424099, 1.0998136838175618, 0.034097426261100884], [-0.16563536360291592, -0.1105641034758906, 0.03409742626110049, -0.7225650771563846, -0.5782592933612961, -0.07673492045096436], [-0.1913797637173214, -0.1277488788539171, 0.18743564815293529, 0.9847949834709216, 1.7486720110144498, -0.12398244282736859], [-0.1968374454345667, -0.13139196371815295, -0.0635163702629449, 1.53473230473674, 0.8625037863342504, -0.1664607573725318]], [[-0.16797171903770558, -0.11212365596775953, -0.10138518332232112, -0.2958794728236933, -0.21312650253996349, -0.14444619383889035], [-0.08430777938068776, -0.05627671434715781, -0.14444619383889026, -0.7819251708881101, -0.9144567648545029, -0.07957095518696675], [-0.1968374454345667, -0.13139196371815295, -0.06351637026294481, 1.5347323047367403, 0.8625037863342505, -0.1664607573725317], [-0.14926526279872426, -0.0996368142795783, -0.09599580797152626, 3.1042347082302673, 1.355768158382745, -0.15375161076932561]]]
    assert_close3d(d2lnGammas_subgroups_d2xs, d2lnGammas_subgroups_d2xs_expect, rtol=1e-10)
    d2lnGammas_subgroups_d2xs_numerical = hessian(to_jac_lnGammas_subgroups, xs, scalar=False, perturbation=4e-5)
    assert_close3d(d2lnGammas_subgroups_d2xs, d2lnGammas_subgroups_d2xs_numerical, rtol=4e-4)

    # d2lnGammas_subgroups_dTdxs
    def to_jac_dlnGammas_subgroups_dT(xs):
        return GE.to_T_xs(T, xs).dlnGammas_subgroups_dT()

    d2lnGammas_subgroups_dTdxs_expect = [[0.00017336393795856162, 0.0012100791818007632, 0.00017012629938040544, -0.0010367729301749545], [0.00011572304342803308, 0.000807746105423379, 0.00011356187084395275, -0.000692061568492539], [0.0005549001004452165, -4.353312535547032e-06, 0.0006376008184172771, -0.00043358527042526747], [-0.0028905986039381992, -0.004736079063168619, -0.003737006717513751, 0.005931610278724003], [0.0036522409725451024, -0.0013380189677480517, 0.006375757426611714, -0.002416545617114441], [-4.353312535546634e-06, 0.0006140663688973192, -2.1323297657880727e-05, -0.0004530422959907456]]
    dlnGammas_subgroups_dT_dxs_num = jacobian(to_jac_dlnGammas_subgroups_dT, xs, scalar=False, perturbation=4e-8)
    d2lnGammas_subgroups_dTdxs = GE.d2lnGammas_subgroups_dTdxs()
    assert_close2d(d2lnGammas_subgroups_dTdxs, d2lnGammas_subgroups_dTdxs_expect, rtol=1e-10)
    assert_close2d(d2lnGammas_subgroups_dTdxs, dlnGammas_subgroups_dT_dxs_num, rtol=1e-5)



    def to_jac_lngammas_r(xs):
        return GE.to_T_xs(T, xs).lngammas_r()
    dlngammas_r_dxs_numerical = jacobian(to_jac_lngammas_r, xs, scalar=False, perturbation=1e-7)
    dlngammas_r_dxs_expect = [[-0.6247263761161402, -0.35102285528203914, -0.2981392026664746, 0.6501651301862179], [-0.3510228552820405, -1.0036098023257582, 0.5711012286889129, 0.7854434722131092], [-0.2981392026664756, 0.5711012286889131, -0.4237501417838445, -0.17331878473748563], [0.6501651301862175, 0.7854434722131096, -0.1733187847374854, -0.8708354730685705]]
    dlngammas_r_dxs = GE.dlngammas_r_dxs()
    assert_close2d(dlngammas_r_dxs_expect, dlngammas_r_dxs, rtol=1e-10)
    assert_close2d(dlngammas_r_dxs, dlngammas_r_dxs_numerical, rtol=5e-7)

    def to_jac_dlngammas_r_dT(xs):
        return GE.to_T_xs(T, xs).dlngammas_r_dT()

    d2lngammas_r_dTdxs = GE.d2lngammas_r_dTdxs()
    d2lngammas_r_dTdxs_numerical = jacobian(to_jac_dlngammas_r_dT, xs, scalar=False, perturbation=1e-7)
    d2lngammas_r_dTdxs_expect = [[0.003329400602671299, -2.6119875213282193e-05, 0.0038256049105036627, -0.002601511622551605], [-2.6119875213279805e-05, 0.003684398213383915, -0.00012793978594728437, -0.0027182537759444735], [0.003825604910503664, -0.00012793978594728846, 0.0065458837259921195, -0.0034533185472893956], [-0.0026015116225516044, -0.0027182537759444765, -0.0034533185472893926, 0.00420277578005651]]
    assert_close2d(d2lngammas_r_dTdxs, d2lngammas_r_dTdxs_expect, rtol=1e-10)
    assert_close2d(d2lngammas_r_dTdxs, d2lngammas_r_dTdxs_numerical, rtol=4e-6)

    d2lngammas_r_dxixjs_expect = [[[1.809275817508037, 1.2260826167340093, 1.3837086756790025, -0.6083110999339265], [1.226082616734009, 1.440062471306381, 0.2045845575666036, -0.8666771630333416], [1.3837086756790016, 0.20458455756660293, 1.1246138889176116, -0.3810982215776694], [-0.6083110999339267, -0.8666771630333416, -0.3810982215776688, -0.5759748478291575]], [[1.2260826167340086, 1.4400624713063812, 0.20458455756660415, -0.8666771630333424], [1.4400624713063812, 3.1753618426126, -0.46040952270578617, -0.4774257311218004], [0.20458455756660532, -0.46040952270578617, -0.7438946569642115, -0.9987645442351908], [-0.8666771630333421, -0.4774257311218005, -0.9987645442351901, -0.9225096646159536]], [[1.383708675679003, 0.20458455756660393, 1.1246138889176116, -0.38109822157766904], [0.20458455756660388, -0.46040952270578495, -0.7438946569642118, -0.9987645442351907], [1.1246138889176112, -0.743894656964212, 1.5572922472971285, 0.6656663408996837], [-0.38109822157766904, -0.9987645442351907, 0.6656663408996838, 1.2065028955840207]], [[-0.6083110999339263, -0.8666771630333423, -0.38109822157766904, -0.5759748478291584], [-0.8666771630333422, -0.47742573112179953, -0.9987645442351911, -0.9225096646159556], [-0.38109822157766876, -0.9987645442351911, 0.6656663408996831, 1.2065028955840205], [-0.5759748478291584, -0.9225096646159556, 1.2065028955840207, 2.8553326311519647]]]
    d2lngammas_r_dxixjs = GE.d2lngammas_r_dxixjs()
    d2lngammas_r_dxixjs_num = hessian(to_jac_lngammas_r, xs, scalar=False, perturbation=10e-5)
    assert_close3d(d2lngammas_r_dxixjs, d2lngammas_r_dxixjs_expect, rtol=1e-7)
    assert_close3d(d2lngammas_r_dxixjs_num, d2lngammas_r_dxixjs, rtol=4e-4)


    def to_jac_GE(xs):
        return GE.to_T_xs(T, xs).GE()
    dGE_dxs = GE.dGE_dxs()
    dGE_dxs_numerical = jacobian(to_jac_GE, xs, perturbation=1e-7)
    dGE_dxs_expect = [968.4165097020538, 1616.0594601188081, 497.0675005089688, 1409.7078865334572]
    assert_close1d(dGE_dxs_expect, dGE_dxs, rtol=1e-10)
    assert_close1d(dGE_dxs_numerical, dGE_dxs, rtol=1e-6)


    # Got more decimals with numdifftools, must be correct
    d2GE_dxixjs_num = hessian(to_jac_GE, xs, perturbation=2e-5)
    d2GE_dxixjs = GE.d2GE_dxixjs()
    d2GE_dxixjs_expect = [[-4940.980009366508, -4370.754506520942, -3939.3162636852335, -1022.969364414501], [-4370.754506520916, -6195.5302935872505, -1543.5471574562987, -538.4425520948281], [-3939.316263685231, -1543.5471574563073, -4349.151467217632, -3541.7479481704927], [-1022.9693644144832, -538.4425520948315, -3541.74794817049, -5955.600731588765]]
    assert_close2d(d2GE_dxixjs, d2GE_dxixjs_expect, rtol=1e-10)
    assert_close2d(d2GE_dxixjs, d2GE_dxixjs_num, rtol=5e-4)

    def to_jac_dGE_dT(xs):
        return GE.to_T_xs(T, xs).dGE_dT()

    d2GE_dTdxs_numerical = jacobian(to_jac_dGE_dT, xs, perturbation=5e-8)
    d2GE_dTdxs_expect = [-3.9462600963845125, -4.847579418251475, -6.513157954188368, -0.10859586217058816]
    d2GE_dTdxs = GE.d2GE_dTdxs()
    assert_close1d(d2GE_dTdxs, d2GE_dTdxs_expect, rtol=1e-10)
    assert_close1d(d2GE_dTdxs, d2GE_dTdxs_numerical, rtol=4e-6)

    dgammas_dT_expect = [-0.0028808414751430728, -0.004980437077194825, -0.002968030663089562, -0.00197318207426701]
    dgammas_dT_super = GibbsExcess.dgammas_dT(GE)
    del GE._dgammas_dT
    assert_close1d(dgammas_dT_super, GE.dgammas_dT(), rtol=1e-10)
    assert_close1d(dgammas_dT_expect, GE.dgammas_dT(), rtol=1e-9)

    dgammas_dns_expect = [[-0.8096374397244642, -0.5585134719755915, -0.36851059971043854, 0.9158314737715368], [-0.6881622698759399, -1.6783287994157854, 0.8459473500316345, 1.3913408969918841], [-0.31657050204549925, 0.58980247484899, -0.47162031569240825, -0.16616152619088537], [1.05581451874053, 1.3018124232255708, -0.22298830023947544, -1.448519501729566]]
    dgammas_dns_super = GibbsExcess.dgammas_dns(GE)
    del GE._dgammas_dns
    assert_close2d(dgammas_dns_super, GE.dgammas_dns(), rtol=1e-10)
    assert_close2d(GibbsExcess.dgammas_dns(GE), dgammas_dns_expect, rtol=1e-10)

    def to_jac_gammas(xs):
        return GE.to_T_xs(T, xs).gammas()

    dgammas_dxs_expect = [[-1.0547171101004926, -0.8035931423516144, -0.6135902700864654, 0.6707518033955142], [-0.13910739679148804, -1.12927392633131, 1.395002223116095, 1.940395770076354], [-0.5017802621239354, 0.40459271477056125, -0.6568300757708443, -0.35137128626931663], [0.8739284010815254, 1.1199263055665736, -0.4048744178984713, -1.630405619388567]]

    dgammas_dxs_num = jacobian(to_jac_gammas, xs, scalar=False, perturbation=1e-6)
    assert_close2d(GE.dgammas_dxs(), dgammas_dxs_expect, rtol=1e-11)
    assert_close2d(GE.dgammas_dxs(), dgammas_dxs_num, rtol=2e-6)


def test_UNIFAC_class_Lyngby():
    T = 373.15
    xs = [0.2, 0.3, 0.1, 0.4]

    chemgroups = [{1: 1, 2: 1, 12: 1}, {1: 2, 2: 3}, {1: 2, 2: 6}, {1: 2, 2: 7}]
    # m = Mixture(['ethanol', 'pentane', 'octane', 'nonane'], zs=xs, T=T, P=1e5)
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=4,
                               interaction_data=LUFIP, subgroups=LUFSG)

    def to_diff(T):
        T = float(T[0])
        return np.array(GE.to_T_xs(T, xs).psis())
    dpsis_dT_numerical = jacobian(to_diff, [T], scalar=True)[0]
    dpsis_dT = GE.dpsis_dT()

    dpsis_dT_expect = [[0.0, 0.0, 0.0009554723111888648],
     [0.0, 0.0, 0.0009554723111888648],
     [0.009512842969259994, 0.009512842969259994, 0.0]]
    assert_close2d(dpsis_dT, dpsis_dT_numerical, rtol=1e-7)
    assert_close2d(dpsis_dT, dpsis_dT_expect, rtol=1e-12)


    def to_diff_dT(T):
        T = float(T[0])
        return np.array(GE.to_T_xs(T, xs).dpsis_dT())

    d2psis_dT2_numerical = jacobian(to_diff_dT, [T], scalar=True)[0]
    d2psis_dT2 = GE.d2psis_dT2()
    d2psis_dT2_expect = [[0.0, 0.0, 1.092520326434123e-05],
     [0.0, 0.0, 1.092520326434123e-05],
     [0.0001033194479619687, 0.0001033194479619687, 0.0]]
    assert_close2d(d2psis_dT2_expect, d2psis_dT2, rtol=1e-12)
    assert_close2d(d2psis_dT2_numerical, d2psis_dT2_expect, rtol=4e-7)


    def to_diff_dT2(T):
        T = float(T[0])
        return np.array(GE.to_T_xs(T, xs).d2psis_dT2())

    d3psis_dT3_numerical = jacobian(to_diff_dT2, [T], scalar=True, perturbation=1e-8)[0]
    d3psis_dT3 = GE.d3psis_dT3()
    d3psis_dT3_expect = [[-0.0, -0.0, 1.1123247721639152e-07],
     [-0.0, -0.0, 1.1123247721639152e-07],
     [3.5174180159656834e-07, 3.5174180159656834e-07, -0.0]]

    assert_close2d(d3psis_dT3, d3psis_dT3_expect, rtol=1e-12)
    assert_close2d(d3psis_dT3, d3psis_dT3_numerical, rtol=1e-7)

    gammas_expect = [3.5161656946379742, 1.0675536248586395, 1.1080642060130936, 1.1064155445595314]
    assert_close1d(GE.gammas(), gammas_expect, rtol=1e-10)

    lngammas_c_expect = [-0.07358508262390817, -0.010165852885790083, -0.009823186951023244, -0.023612893471248952]
    lngammas_c = GE.lngammas_c()
    assert_close1d(lngammas_c_expect, lngammas_c, rtol=1e-10)

    def to_jac_lngammas_c(xs):
        return GE.to_T_xs(T, xs).lngammas_c()

    dlngammas_c_dxs_analytical = GE.dlngammas_c_dxs()
    dlngammas_c_dxs_expect = [[-0.22317648209326857, -0.2905306531432139, -0.38557482608277627, -0.41467289025986764], [-0.09020365158080279, -0.11742691507576175, -0.15584194599753676, -0.1676028251819719], [0.09743569580450218, 0.12684157432733068, 0.1683365160688155, 0.18104031936863674], [0.15488205578111058, 0.2016251192965952, 0.2675847435223364, 0.287778484174253]]
    assert_close2d(dlngammas_c_dxs_expect, dlngammas_c_dxs_analytical, rtol=1e-11)
    dlngammas_c_dxs_numerical = jacobian(to_jac_lngammas_c, xs, scalar=False, perturbation=1e-7)
    assert_close2d(dlngammas_c_dxs_analytical, dlngammas_c_dxs_numerical, rtol=1e-6)


    d2lngammas_c_dxixjs = GE.d2lngammas_c_dxixjs()
    d2lngammas_c_dxixjs_numerical = np.array(hessian(to_jac_lngammas_c, xs, scalar=False, perturbation=1e-4)).T
    d2lngammas_c_dxixjs_expect = [[[-0.14432272112331146, -0.18787900068179408, -0.24934172084340922, -0.26815872056519685], [-0.1878790006817942, -0.24458046953694845, -0.32459250335442735, -0.3490884321731367], [-0.24934172084340922, -0.32459250335442724, -0.430779667049324, -0.46328919191985163], [-0.26815872056519696, -0.3490884321731367, -0.46328919191985163, -0.49825210372608697]], [[-0.32085183119425986, -0.4176842076044004, -0.554325383439667, -0.5961584972510324], [-0.4176842076044004, -0.54374038207215, -0.7216195640062312, -0.7760778194847415], [-0.554325383439667, -0.7216195640062312, -0.9576901262548605, -1.0299638507576838], [-0.5961584972510324, -0.7760778194847415, -1.0299638507576838, -1.107691835579486]], [[-0.5699538987411796, -0.7419647308249718, -0.9846910092009157, -1.0590024015483555], [-0.7419647308249718, -0.9658880534093235, -1.2818685883212124, -1.3786069953082918], [-0.9846910092009157, -1.2818685883212124, -1.7012189683106773, -1.8296043694586293], [-1.0590024015483555, -1.3786069953082918, -1.8296043694586293, -1.9676785946409665]], [[-0.6462172584395759, -0.8412442046129456, -1.116448761524964, -1.2007034781602075], [-0.8412442046129456, -1.0951298538570984, -1.4533905402775567, -1.563073144935711], [-1.116448761524964, -1.4533905402775567, -1.9288525969121502, -2.074416759446583], [-1.2007034781602075, -1.5630731449357107, -2.074416759446583, -2.23096617064557]]]
    assert_close3d(d2lngammas_c_dxixjs, d2lngammas_c_dxixjs_expect, rtol=1e-12)
    assert_close3d(d2lngammas_c_dxixjs_numerical, d2lngammas_c_dxixjs, rtol=1e-3)


    # Not checked with sympy or anything
    d3lngammas_c_dxixjxks_expect = [[[[0.5798576717408318, 0.7548574407161887, 1.0018014390996541, 1.0774040993245881], [0.754857440716189, 0.9826717547668697, 1.3041429083006988, 1.402562284968421], [1.0018014390996541, 1.3041429083006986, 1.730780107416949, 1.8613963905226303], [1.0774040993245881, 1.4025622849684218, 1.8613963905226298, 2.0018698549878797]], [[0.754857440716189, 0.9826717547668697, 1.3041429083006988, 1.402562284968421], [0.98267175476687, 1.2792399273436608, 1.6977303674064883, 1.8258524953163264], [1.3041429083006988, 1.6977303674064883, 2.253125734121853, 2.423161723962275], [1.4025622849684218, 1.8258524953163264, 2.423161723962275, 2.606029770799412]], [[1.0018014390996544, 1.3041429083006988, 1.7307801074169493, 1.8613963905226307], [1.3041429083006988, 1.6977303674064883, 2.253125734121853, 2.423161723962275], [1.7307801074169495, 2.2531257341218525, 2.9902130934473914, 3.2158746424138487], [1.8613963905226298, 2.423161723962275, 3.215874642413849, 3.4585661264018346]], [[1.0774040993245881, 1.4025622849684218, 1.8613963905226298, 2.0018698549878797], [1.4025622849684218, 1.8258524953163264, 2.423161723962275, 2.606029770799412], [1.8613963905226298, 2.423161723962275, 3.215874642413849, 3.4585661264018346], [2.0018698549878793, 2.606029770799412, 3.4585661264018346, 3.719572738605169]]], [[[0.931386550787137, 1.2124769616894755, 1.6091265708969567, 1.7305620616542559], [1.2124769616894757, 1.5783998398788617, 2.0947574280582923, 2.252841882627931], [1.6091265708969567, 2.094757428058292, 2.780036193327388, 2.9898363828001067], [1.7305620616542559, 2.252841882627931, 2.9898363828001067, 3.215469502652809]], [[1.2124769616894755, 1.5783998398788617, 2.094757428058292, 2.252841882627931], [1.5783998398788617, 2.054757437253204, 2.726950608962622, 2.93274460395315], [2.094757428058292, 2.726950608962622, 3.6190449971858443, 3.8921624220389077], [2.252841882627931, 2.93274460395315, 3.8921624220389077, 4.185891120809913]], [[1.6091265708969567, 2.094757428058292, 2.780036193327388, 2.9898363828001067], [2.094757428058292, 2.726950608962622, 3.6190449971858443, 3.8921624220389077], [2.780036193327388, 3.6190449971858443, 4.802979067024024, 5.165444102807996], [2.9898363828001067, 3.8921624220389077, 5.165444102807996, 5.555263182890825]], [[1.7305620616542559, 2.252841882627931, 2.9898363828001067, 3.215469502652809], [2.252841882627931, 2.93274460395315, 3.8921624220389077, 4.185891120809913], [2.9898363828001067, 3.8921624220389077, 5.165444102807996, 5.555263182890825], [3.215469502652809, 4.185891120809914, 5.555263182890825, 5.974500627043071]]], [[[1.4274326167175215, 1.8582286384438755, 2.4661293957744554, 2.652240071505787], [1.8582286384438755, 2.4190379512788622, 3.210401819088636, 3.4526802870974285], [2.4661293957744546, 3.210401819088636, 4.260652394708743, 4.582189819952624], [2.652240071505787, 3.4526802870974285, 4.582189819952624, 4.927992617316712]], [[1.8582286384438755, 2.4190379512788622, 3.210401819088636, 3.4526802870974285], [2.419037951278863, 3.1490982802997944, 4.179294021500825, 4.494691597862456], [3.210401819088636, 4.179294021500825, 5.546507909079841, 5.965084621508939], [3.4526802870974285, 4.494691597862455, 5.965084621508939, 6.415249941952303]], [[2.466129395774455, 3.210401819088636, 4.260652394708744, 4.582189819952624], [3.210401819088636, 4.179294021500825, 5.546507909079841, 5.965084621508939], [4.260652394708744, 5.546507909079841, 7.360992030524254, 7.916501892740387], [4.582189819952624, 5.965084621508939, 7.916501892740387, 8.513934257486031]], [[2.652240071505787, 3.4526802870974285, 4.582189819952624, 4.927992617316712], [3.4526802870974285, 4.494691597862455, 5.965084621508939, 6.415249941952303], [4.582189819952624, 5.965084621508939, 7.916501892740387, 8.513934257486031], [4.927992617316713, 6.415249941952303, 8.513934257486031, 9.156452878166625]]], [[[1.579298636640852, 2.055927488899572, 2.728503431204184, 2.9344146125828887], [2.0559274888995716, 2.676401879636475, 3.551959760885402, 3.8200144835619834], [2.728503431204184, 3.5519597608854028, 4.713947572276673, 5.069693693928565], [2.9344146125828887, 3.820014483561983, 5.069693693928565, 5.4522868055246505]], [[2.055927488899572, 2.676401879636475, 3.551959760885402, 3.820014483561984], [2.676401879636475, 3.4841340757380985, 4.623933398310231, 4.972886446260883], [3.551959760885402, 4.623933398310231, 6.136606573465285, 6.599716091579724], [3.820014483561983, 4.972886446260883, 6.599716091579724, 7.097774962109155]], [[2.728503431204184, 3.5519597608854028, 4.713947572276673, 5.069693693928565], [3.551959760885402, 4.62393339831023, 6.136606573465285, 6.599716091579723], [4.713947572276673, 6.136606573465285, 8.144135521341859, 8.758746647473991], [5.069693693928565, 6.599716091579724, 8.758746647473988, 9.41974045417123]], [[2.9344146125828887, 3.820014483561983, 5.069693693928565, 5.4522868055246505], [3.820014483561983, 4.972886446260883, 6.599716091579724, 7.097774962109155], [5.069693693928565, 6.599716091579724, 8.758746647473988, 9.41974045417123], [5.4522868055246505, 7.097774962109154, 9.41974045417123, 10.130617289808248]]]]
    d3lngammas_c_dxixjxks = GE.d3lngammas_c_dxixjxks()
    assert_close4d(d3lngammas_c_dxixjxks_expect, d3lngammas_c_dxixjxks, rtol=1e-13)

def test_VTPR_GE():
    T = 328.15
    P = 37316.9
    xs = [0.625, 1-.625]

    chemgroups = [{1: 1, 18: 1}, {1: 1, 2: 1, 14: 1}]
    # m.UNIFAC_Dortmund_groups
    # m = Mixture(['acetone', 'ethanol'], zs=xs, T=T, P=P)
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=3,
                               interaction_data=VTPRIP, subgroups=VTPRSG)

    gammas_expect = [1.084965825553096, 1.2579548067749646]
    assert_close1d(GE.gammas(), gammas_expect)
    assert_close(GE.GE(), 373.85917799452574)
    
    # Derivatives - skip_comb
    dgammas_dT_expect = [-0.000908456121317832, -0.002432810818717207]
    dgammas_dT_super = GibbsExcess.dgammas_dT(GE)
    del GE._dgammas_dT
    assert_close1d(dgammas_dT_super, GE.dgammas_dT(), rtol=1e-10)
    assert_close1d(dgammas_dT_expect, GE.dgammas_dT(), rtol=1e-9)

    dgammas_dns_expect = [[-0.17331531861998678, 0.288858864366645],
     [0.3349150621627528, -0.5581917702712555]]

    dgammas_dns_super = GibbsExcess.dgammas_dns(GE)
    del GE._dgammas_dns
    assert_close2d(dgammas_dns_super, GE.dgammas_dns(), rtol=1e-10)
    assert_close2d(GE.dgammas_dns(), dgammas_dns_expect, rtol=1e-10)

    def to_jac_gammas(xs):
        return GE.to_T_xs(T, xs).gammas()

    dgammas_dxs_expect = [[-0.17331531861998728, 0.28885886436664376],
     [0.33491506216275213, -0.5581917702712555]]

    dgammas_dxs_num = jacobian(to_jac_gammas, xs, scalar=False, perturbation=1e-6)
    assert_close2d(GE.dgammas_dxs(), dgammas_dxs_expect, rtol=1e-11)
    assert_close2d(GE.dgammas_dxs(), dgammas_dxs_num, rtol=2e-6)

def test_NISTUF_2011():
    T = 330.0
    P = 1e5
    xs = [1-.01, .01]

    chemgroups = {1:1, 15:5, 19:1}, {15:4, 18:2} # from https://trc.nist.gov/TDE/Help/TDE103b/NIST-KT-UNIFAC-AC-Model.htm
    #m = Mixture(['ethylbenzene', 'p-xylene'], zs=xs, T=T, P=P)

    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=5,
                               interaction_data=NISTKTUFIP, subgroups=NISTKTUFSG)

    gammas_expect = [0.9999968672576434, 0.9737803219928437]
    assert_close1d(GE.gammas(), gammas_expect)
    
def test_UNIFAC_default_data():
    # TODO: PSRK
    
    # VTPR ['acetone', 'ethanol']
    T = 328.15
    P = 37316.9
    xs = [0.625, 1-.625]
    chemgroups = [{1: 1, 18: 1}, {1: 1, 2: 1, 14: 1}]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=3,
                               interaction_data=VTPRIP, subgroups=VTPRSG)
    GEd = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=3)
    assert_close1d([GEd.rs, GEd.qs, GEd.Qs, GEd.vs], [GE.rs, GE.qs, GE.Qs, GE.vs])

    # NISTUF 2011 ['ethylbenzene', 'p-xylene']
    T = 330.0
    P = 1e5
    xs = [1-.01, .01]
    chemgroups = {1:1, 15:5, 19:1}, {15:4, 18:2} # from https://trc.nist.gov/TDE/Help/TDE103b/NIST-KT-UNIFAC-AC-Model.htm
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=5,
                               interaction_data=NISTKTUFIP, subgroups=NISTKTUFSG)
    GEd = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=5)
    assert_close1d([GEd.rs, GEd.qs, GEd.Qs, GEd.vs], [GE.rs, GE.qs, GE.Qs, GE.vs])

    # Lyngby ['ethanol', 'pentane', 'octane', 'nonane']
    T = 373.15
    xs = [0.2, 0.3, 0.1, 0.4]
    chemgroups = [{1: 1, 2: 1, 12: 1}, {1: 2, 2: 3}, {1: 2, 2: 6}, {1: 2, 2: 7}]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=4,
                               interaction_data=LUFIP, subgroups=LUFSG)
    GEd = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=4)
    assert_close1d([GEd.rs, GEd.qs, GEd.Qs, GEd.vs], [GE.rs, GE.qs, GE.Qs, GE.vs])

    # Dortmund ['benzene', 'cyclohexane', 'acetone', 'ethanol']
    T = 373.15
    xs = [0.2, 0.3, 0.1, 0.4]
    chemgroups = [{9:6}, {78:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=1,
                               interaction_data=DOUFIP2016, subgroups=DOUFSG)
    GEd = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=1)
    assert_close1d([GEd.rs, GEd.qs, GEd.Qs, GEd.vs], [GE.rs, GE.qs, GE.Qs, GE.vs])

    # UNIFAC
    T = 373.15
    xs = [0.2, 0.3, 0.1, 0.4]
    chemgroups = [{9:6}, {2:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=0,
                               interaction_data=UFIP, subgroups=UFSG)
    GEd = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=0)
    assert_close1d([GEd.rs, GEd.qs, GEd.Qs, GEd.vs], [GE.rs, GE.qs, GE.Qs, GE.vs])


def call_all_methods_first_UNIFAC(kwargs):
    cls = UNIFAC
    skip_methods = ('__init__', 'to_T_xs', '__delattr__', '__format__', '__getattribute__', '__setattr__', 'from_json', 'model_id', '_regress_binary_parameters')
    special_methods = {1: ('Vis_modified', 'dVis_modified_dxs', 'd2Vis_modified_dxixjs', 'd3Vis_modified_dxixjxks')}
    special_methods[4] = special_methods[1]

    restricted_methods = {}
    for k, methods in special_methods.items():
        for name in methods:
            try:
                restricted_methods[name].add(k)
            except:
                restricted_methods[name] = set([k])

    for s in dir(cls):
        attr = getattr(cls, s)
        if isinstance(attr, types.MethodType) or type(attr) is property:
            if (s in restricted_methods and not kwargs['version'] in restricted_methods[s]) or s in skip_methods:
                continue
            base = cls(**kwargs)
            v = getattr(base, s)()


#@pytest.mark.fuzz
def test_UNIFAC_class_all_methods_first():
    # Technically a fuzz test - but takes 1/20th a second, allowed for test coverage
    # and finding real bugs
    '''T = 300.0
    xs = [0.1, 0.9]
    chemgroups = [{16: 1}, {1: 1, 2: 5, 14: 1}]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups,
                               interaction_data=UFIP, subgroups=UFSG, version=0)
    '''
    kwargs = dict(T=300.0, xs=[0.1, 0.9], rs=[0.92, 5.2730999999999995], qs=[1.4, 4.748],
                  Qs=[0.848, 0.54, 1.2, 1.4], vs=[[0, 1], [0, 5], [0, 1], [1, 0]],
                  psi_abc=([[0.0, 0.0, 986.5, 1318.0], [0.0, 0.0, 986.5, 1318.0], [156.4, 156.4, 0.0, 353.5], [300.0, 300.0, -229.1, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0], [0.0, 0.0, 0.0, 0.0]]),
                  version=0)
    call_all_methods_first_UNIFAC(kwargs)
    '''xs = [0.1, 0.9]
    chemgroups = [{16: 1}, {1: 1, 2: 5, 14: 1}]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups,
                               interaction_data=DOUFIP2006, subgroups=DOUFSG, version=1)
    '''
    kwargs= dict(T=300.0, xs=[0.1, 0.9], rs=[1.7334, 5.0252], qs=[2.4561, 5.494], Qs=[1.0608, 0.7081, 0.8927, 2.4561], vs=[[0, 1], [0, 5], [0, 1], [1, 0]], psi_abc=([[0.0, 0.0, 2777.0, 1391.3], [0.0, 0.0, 2777.0, 1391.3], [1606.0, 1606.0, 0.0, -801.9], [-17.253, -17.253, 1460.0, 0.0]], [[0.0, 0.0, -4.674, -3.6156], [0.0, 0.0, -4.674, -3.6156], [-4.746, -4.746, 0.0, 3.824], [0.8389, 0.8389, -8.673, 0.0]], [[0.0, 0.0, 0.001551, 0.001144], [0.0, 0.0, 0.001551, 0.001144], [0.0009181, 0.0009181, 0.0, -0.007514], [0.0009021, 0.0009021, 0.01641, 0.0]]), version=1)
    call_all_methods_first_UNIFAC(kwargs)


def test_UNIFAC_RQ():
    assert_close1d(UNIFAC_RQ({1:2, 2:4}), (4.4998000000000005, 3.856))
    assert_close1d(UNIFAC_RQ({1: 1, 2: 1, 14: 1}, subgroup_data=DOUFSG), (2.4951999999999996, 2.6616))


def test_UNIFAC_initialization():
    T = 321.56
    chemgroups = [{9: 6}, {2: 6}, {1: 1, 18: 1}, {1: 1, 2: 1, 14: 1}]
    xs = [0.2, 0.3, 0.1, 0.4]
    GE = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups)

    # Check the __repr__ is working
    assert eval(str(GE)).__dict__ == GE.__dict__

    # Check the json export/import is working
    assert UNIFAC.from_json(GE.as_json()).__dict__ == GE.__dict__

    gammas_expect = [1.4938332119259123, 1.960091090828185, 1.4125828059033487, 1.651847113952877]
    assert_close1d(GE.gammas(), gammas_expect)

    psi_coeffs = [[(GE.psi_a[i][j], GE.psi_b[i][j], GE.psi_c[i][j]) for j in range(GE.N_groups)] for i in range(GE.N_groups)]
    kwargs = dict(T=T, xs=xs, rs=[3.1878, 4.0464, 2.5735, 2.5755], qs=[2.4000000000000004, 3.24, 2.336, 2.588], Qs=[0.848, 0.54, 0.4, 1.2, 1.488], vs=[[0, 0, 1, 1], [0, 6, 0, 1], [6, 0, 0, 0], [0, 0, 0, 1], [0, 0, 1, 0]], version=0)

    # make a new instance without any dictionary
    GE2 = UNIFAC(psi_coeffs=psi_coeffs, **kwargs)
    assert_close2d(GE2.psi_a, GE.psi_a)
    assert_close2d(GE2.psi_b, GE.psi_b)
    assert_close2d(GE2.psi_c, GE.psi_c)
    assert_close1d(GE2.gammas(), gammas_expect)

    with pytest.raises(Exception):
        # Missing psis
        UNIFAC(**kwargs)


    # Just one more test case for d3lngammas_c_dxixjxks
    d3lngammas_c_dxixjxks_expect = [[[[1.5429848357872018, 2.657397093598547, 2.6841516344618768, 3.5893805439280797], [2.657397093598476, 4.15301440526072, 3.7506510113450986, 4.761455484200397], [2.6841516344618626, 3.7506510113450844, 2.874094083201655, 3.320324355902983], [3.589380543928094, 4.761455484200411, 3.320324355902997, 3.5882462018814962]], [[2.657397093598476, 4.15301440526072, 3.7506510113450986, 4.761455484200397], [4.153014405260734, 6.116830289204955, 5.092619245086354, 6.188948959613072], [3.750651011345113, 5.092619245086382, 3.7108238236768187, 4.142478410245076], [4.761455484200397, 6.188948959613072, 4.142478410245019, 4.333151315847843]], [[2.6841516344618768, 3.750651011345127, 2.874094083201612, 3.3203243559030255], [3.750651011345127, 5.092619245086354, 3.7108238236768045, 4.142478410245133], [2.874094083201612, 3.7108238236768756, 2.4491534600375857, 2.531989163753728], [3.320324355902997, 4.142478410245019, 2.531989163753714, 2.4407252124374637]], [[3.5893805439280655, 4.761455484200411, 3.3203243559029687, 3.5882462018814962], [4.761455484200454, 6.1889489596131, 4.142478410245076, 4.333151315847914], [3.320324355902997, 4.142478410245076, 2.5319891637536855, 2.440725212437492], [3.588246201881539, 4.333151315847914, 2.4407252124374637, 2.1562959242053807]]], [[[3.2215756996496765, 4.869148513901692, 4.206110463843032, 5.217268897803777], [4.869148513901706, 7.025847497688829, 5.670751853338885, 6.767530864627446], [4.206110463843018, 5.670751853338928, 4.07851467661726, 4.510455014771168], [5.2172688978037485, 6.767530864627389, 4.5104550147711535, 4.701413894032044]], [[4.869148513901706, 7.025847497688829, 5.670751853338885, 6.767530864627503], [7.025847497688744, 9.795768634243387, 7.4784145310018175, 8.618391408155674], [5.670751853338842, 7.47841453100186, 5.154977444255977, 5.521243352211499], [6.767530864627432, 8.618391408155617, 5.521243352211499, 5.561847446438804]], [[4.206110463843018, 5.670751853338956, 4.07851467661726, 4.510455014771168], [5.670751853338928, 7.478414531001832, 5.154977444255934, 5.521243352211485], [4.078514676617218, 5.154977444255948, 3.2471988254865565, 3.2212362470963427], [4.510455014771168, 5.521243352211457, 3.221236247096286, 2.9600250014559464]], [[5.2172688978037485, 6.767530864627389, 4.510455014771125, 4.701413894032044], [6.767530864627375, 8.618391408155617, 5.521243352211499, 5.561847446438804], [4.510455014771168, 5.521243352211457, 3.2212362470963143, 2.960025001455918], [4.701413894032044, 5.561847446438861, 2.9600250014559606, 2.437826168415725]]], [[[2.280500437516558, 3.238280660846513, 2.5482278571715824, 2.9942048823686918], [3.238280660846499, 4.44224739419937, 3.297188998089247, 3.728522127633525], [2.5482278571715753, 3.2971889980892257, 2.186082774082223, 2.268714031957316], [2.994204882368713, 3.728522127633525, 2.2687140319573302, 2.1772454759145816]], [[3.238280660846513, 4.442247394199342, 3.297188998089233, 3.7285221276335534], [4.4422473941993985, 5.91902547163113, 4.163210010415824, 4.52870516458556], [3.2971889980892044, 4.163210010415781, 2.6164372865908305, 2.5899845107738173], [3.728522127633539, 4.52870516458556, 2.5899845107739026, 2.328282686748892]], [[2.548227857171547, 3.2971889980892684, 2.186082774082223, 2.268714031957323], [3.2971889980892044, 4.163210010415781, 2.6164372865908305, 2.5899845107738457], [2.18608277408223, 2.616437286590795, 1.4386671671461855, 1.235017109421932], [2.2687140319573373, 2.5899845107738457, 1.235017109421932, 0.861468772552179]], [[2.9942048823687415, 3.728522127633525, 2.268714031957302, 2.1772454759145816], [3.728522127633539, 4.52870516458556, 2.5899845107738457, 2.328282686748892], [2.2687140319573373, 2.5899845107738457, 1.2350171094219036, 0.861468772552179], [2.1772454759145887, 2.328282686748892, 0.8614687725521506, 0.2994643786447213]]], [[[3.1870435296197783, 4.250753277444197, 2.995519065005695, 3.26318848798757], [4.250753277444211, 5.540694549230338, 3.730190271376017, 3.920542766534794], [2.9955190650057233, 3.7301902713759816, 2.269774967090136, 2.1783072355549677], [3.2631884879876196, 3.920542766534851, 2.1783072355549606, 1.8936740087357933]], [[4.250753277444183, 5.540694549230338, 3.730190271376003, 3.920542766534794], [5.540694549230324, 7.061119789289279, 4.530822605089938, 4.570656992111481], [3.7301902713760384, 4.530822605089924, 2.59133119749243, 2.3296304200472946], [3.9205427665348225, 4.570656992111452, 2.329630420047323, 1.8069416747619869]], [[2.995519065005709, 3.730190271375996, 2.269774967090136, 2.1783072355549535], [3.73019027137601, 4.530822605089867, 2.59133119749243, 2.329630420047266], [2.26977496709015, 2.591331197492444, 1.2358735987136882, 0.8623259274660668], [2.178307235554982, 2.3296304200472946, 0.862325927466074, 0.3003221996980514]], [[3.2631884879876196, 3.920542766534851, 2.1783072355549606, 1.8936740087358217], [3.920542766534851, 4.570656992111452, 2.3296304200472946, 1.806941674762058], [2.178307235554982, 2.329630420047323, 0.862325927466074, 0.3003221996980514], [1.8936740087358075, 1.8069416747620721, 0.3003221996980656, -0.47069758224442637]]]]
    assert_close4d(GE.d3lngammas_c_dxixjxks(), d3lngammas_c_dxixjxks_expect)


def test_unifac_np_output_and_hash():
    N = 4
    T = 373.15
    xs = [0.2, 0.3, 0.1, 0.4]
    chemgroups = [{9:6}, {78:6}, {1:1, 18:1}, {1:1, 2:1, 14:1}]
    model = UNIFAC.from_subgroups(T=T, xs=xs, chemgroups=chemgroups, version=1,
                               interaction_data=DOUFIP2006, subgroups=DOUFSG)


    modelnp = UNIFAC.from_subgroups(T=T, xs=np.array(xs), chemgroups=chemgroups, version=1,
                           interaction_data=DOUFIP2006, subgroups=DOUFSG)
    modelnp2 = modelnp.to_T_xs(T=T, xs=np.array(xs))

    check_np_output_activity(model, modelnp, modelnp2)

    json_string = modelnp.as_json()
    new = UNIFAC.from_json(json_string)
    assert new == modelnp

    new2 = UNIFAC.from_json(json.loads(json.dumps(modelnp.as_json())))
    assert new2 == modelnp
    new2 = UNIFAC.from_json(json.loads(json.dumps(modelnp2.as_json())))
    assert new2 == modelnp2

    assert model.model_hash() == modelnp.model_hash()
    assert new.model_hash() == modelnp.model_hash()
    assert new.model_hash() == modelnp2.model_hash()

    assert model.state_hash() == modelnp.state_hash()
    assert new.state_hash() == modelnp.state_hash()
    assert new.state_hash() == modelnp2.state_hash()

    # Pickle checks
    modelnp_pickle = pickle.loads(pickle.dumps(modelnp))
    assert modelnp_pickle == modelnp
    model_pickle = pickle.loads(pickle.dumps(model))
    assert model_pickle == model






def test_UNIFAC_one_component():
    GE = UNIFAC(T=300.0, xs=[1.0], rs=[4.8907], qs=[4.096], Qs=[0.848, 0.54, 1.088], vs=[[1], [4], [1]], psi_abc=([[0.0, 0.0, 298.9], [0.0, 0.0, 298.9], [-72.88, -72.88, 0.0]], [[0.0, 0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0]], [[0.0, 0.0, 0.0], [0.0, 0.0, 0.0], [0.0, 0.0, 0.0]]), version=0)
    
    assert GE.GE() == 0
    assert GE.HE() == 0
    assert GE.SE() == 0
    assert GE.dGE_dT() == 0
    assert GE.CpE() == 0
    assert GE.dHE_dT() == 0
    assert GE.d2GE_dT2() == 0
    
    
    assert GE.lngammas_r()[0] == 0
    assert GE.lngammas_c()[0] == 0
    assert GE.gammas()[0] == 1
    assert GE.d2GE_dxixjs()[0][0] == 0
    
    for s in GE._point_properties:
        if hasattr(GE, s):
            res = getattr(GE, s)()


def test_UNIFAC_large():
    constants, correlations = ChemicalConstantsPackage.from_IDs(IDs=list(dippr_compounds())[0:200])
    groups, CASs = [], []

    for g, c in zip(constants.UNIFAC_groups, constants.CASs):
        if g is not None:
            groups.append(g)
            CASs.append(c)


    xs = normalize([random() for i in range(len(groups))])

    GE = UNIFAC.from_subgroups(T=300.0, xs=xs, chemgroups=groups)
    GE.GE()
    GE.to_T_xs(T=310.0, xs=xs).gammas() # 16.7 ms at 200 components

del test_UNIFAC_large