import io

import pytest

from dissect.ntfs.c_ntfs import FILE_NUMBER_MFT
from dissect.ntfs.exceptions import MftNotAvailableError, VolumeNotAvailableError
from dissect.ntfs.mft import Mft, MftRecord
from dissect.ntfs.ntfs import NTFS


def test_mft(mft_bin):
    fs = NTFS(mft=mft_bin)

    assert fs.mft
    assert len(list(fs.mft.segments())) == 37

    with pytest.raises(VolumeNotAvailableError):
        assert fs.mft.get(FILE_NUMBER_MFT).open()


def test_mft_record_get_no_mft(mft_bin):
    fs = NTFS(mft=mft_bin)

    root = fs.mft.root
    fs.mft = None

    with pytest.raises(MftNotAvailableError):
        root.get("$MFT")


def test_mft_record():
    # Single MFT record of the $MFT file itself.
    data = bytes.fromhex(
        "46494c453000030051511000000000000100010038000100a001000000040000"
        "0000000000000000070000000000000002000000000000001000000060000000"
        "00001800000000004800000018000000d2145d665666d801d2145d665666d801"
        "d2145d665666d801d2145d665666d80106000000000000000000000000000000"
        "0000000000010000000000000000000000000000000000003000000068000000"
        "00001800000003004a000000180001000500000000000500d2145d665666d801"
        "d2145d665666d801d2145d665666d801d2145d665666d8010040000000000000"
        "00400000000000000600000000000000040324004d0046005400000000000000"
        "8000000048000000010040000000060000000000000000003f00000000000000"
        "4000000000000000000004000000000000000400000000000000040000000000"
        "2140550200000000b00000005000000001004000000005000000000000000000"
        "0100000000000000400000000000000000200000000000000810000000000000"
        "0810000000000000210154022101d1fd0000000000000000ffffffff00000000"
        "00000400000000002140550200000000b0000000500000000100400000000500"
        "0000000000000000010000000000000040000000000000000020000000000000"
        "08100000000000000810000000000000210154022101d1fd0000000000000200"
        "ffffffff00000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000000"
        "0000000000000000000000000000000000000000000000000000000000000200"
    )
    record = MftRecord.from_bytes(data)
    attrs = record.attributes

    assert not record.resident
    assert record.filename == "$MFT"
    assert record.filenames() == ["$MFT"]
    assert record.is_file()
    assert record.size() == 262144
    assert len(attrs) == 4

    mft = Mft(io.BytesIO(data))
    record = mft.get(0)
    assert record.filename == "$MFT"
    assert record.filenames() == ["$MFT"]
