"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const codedeploy = require("../../lib");
function mockFunction(stack, id) {
    return new lambda.Function(stack, id, {
        code: lambda.Code.fromInline('mock'),
        handler: 'index.handler',
        runtime: lambda.Runtime.NODEJS_10_X
    });
}
function mockAlias(stack) {
    return new lambda.Alias(stack, 'Alias', {
        aliasName: 'my-alias',
        version: new lambda.Version(stack, 'Version', {
            lambda: mockFunction(stack, 'Function')
        })
    });
}
module.exports = {
    'CodeDeploy Lambda DeploymentGroup': {
        'can be created with default AllAtOnce IN_PLACE configuration'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: 'MyApp3CE31C26'
                },
                ServiceRoleArn: {
                    'Fn::GetAtt': [
                        'MyDGServiceRole5E94FD88',
                        'Arn'
                    ]
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE'
                    ]
                },
                DeploymentConfigName: 'CodeDeployDefault.LambdaAllAtOnce',
                DeploymentStyle: {
                    DeploymentOption: 'WITH_TRAFFIC_CONTROL',
                    DeploymentType: 'BLUE_GREEN'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
                Type: 'AWS::Lambda::Alias',
                Properties: {
                    FunctionName: {
                        Ref: 'Function76856677'
                    },
                    FunctionVersion: {
                        'Fn::GetAtt': [
                            'Version6A868472',
                            'Version'
                        ]
                    },
                    Name: 'my-alias'
                },
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26'
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: { 'Fn::Join': ['', ['codedeploy.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }]] }
                            }
                        }],
                    Version: '2012-10-17'
                },
                ManagedPolicyArns: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':iam::aws:policy/service-role/AWSCodeDeployRoleForLambda'
                            ]
                        ]
                    }
                ]
            }));
            test.done();
        },
        'can be created with explicit name'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                deploymentGroupName: 'test',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                DeploymentGroupName: 'test',
            }));
            test.done();
        },
        'can be created with explicit role'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const serviceRole = new iam.Role(stack, 'MyRole', {
                assumedBy: new iam.ServicePrincipal('not-codedeploy.test')
            });
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                role: serviceRole
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'not-codedeploy.test'
                            }
                        }],
                    Version: '2012-10-17'
                },
                ManagedPolicyArns: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':iam::aws:policy/service-role/AWSCodeDeployRoleForLambda'
                            ]
                        ]
                    }
                ]
            }));
            test.done();
        },
        'can configure blue/green traffic shifting'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: 'MyApp3CE31C26'
                },
                ServiceRoleArn: {
                    'Fn::GetAtt': [
                        'MyDGServiceRole5E94FD88',
                        'Arn'
                    ]
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE'
                    ]
                },
                DeploymentConfigName: 'CodeDeployDefault.LambdaLinear10PercentEvery1Minute',
                DeploymentStyle: {
                    DeploymentOption: 'WITH_TRAFFIC_CONTROL',
                    DeploymentType: 'BLUE_GREEN'
                }
            }));
            test.done();
        },
        'can rollback on alarm'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: 'Failures8A3E1A2F'
                            }
                        }],
                    Enabled: true
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                        'DEPLOYMENT_STOP_ON_ALARM'
                    ]
                },
            }));
            test.done();
        },
        'onPreHook throws error if pre-hook already defined'(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            test.throws(() => group.addPreHook(mockFunction(stack, 'PreHook2')));
            test.done();
        },
        'onPostHook throws error if post-hook already defined'(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            test.throws(() => group.addPostHook(mockFunction(stack, 'PostHook2')));
            test.done();
        },
        'can run pre hook lambda function before deployment'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26'
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F'
                        },
                        BeforeAllowTrafficHook: {
                            Ref: 'PreHook8B53F672'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PreHook8B53F672',
                                    'Arn'
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: '2012-10-17'
                }
            }));
            test.done();
        },
        'can add pre hook lambda function after creating the deployment group'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            group.addPreHook(mockFunction(stack, 'PreHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26'
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F'
                        },
                        BeforeAllowTrafficHook: {
                            Ref: 'PreHook8B53F672'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PreHook8B53F672',
                                    'Arn'
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: '2012-10-17'
                }
            }));
            test.done();
        },
        'can run post hook lambda function before deployment'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26'
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F'
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PostHookF2E49B30',
                                    'Arn'
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: '2012-10-17'
                },
            }));
            test.done();
        },
        'can add post hook lambda function after creating the deployment group'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            group.addPostHook(mockFunction(stack, 'PostHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: 'MyApp3CE31C26'
                        },
                        DeploymentGroupName: {
                            Ref: 'MyDGC350BD3F'
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: 'MyDGServiceRoleDefaultPolicy65E8E1EA',
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                'Fn::GetAtt': [
                                    'PostHookF2E49B30',
                                    'Arn'
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: '2012-10-17'
                },
            }));
            test.done();
        },
        'can disable rollback when alarm polling fails'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                ignorePollAlarmsFailure: true,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: 'Failures8A3E1A2F'
                            }
                        }],
                    Enabled: true,
                    IgnorePollAlarmFailure: true
                },
            }));
            test.done();
        },
        'can disable rollback when deployment fails'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    failedDeployment: false
                }
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: 'MyApp3CE31C26'
                },
                ServiceRoleArn: {
                    'Fn::GetAtt': [
                        'MyDGServiceRole5E94FD88',
                        'Arn'
                    ]
                },
                DeploymentConfigName: 'CodeDeployDefault.LambdaAllAtOnce',
                DeploymentStyle: {
                    DeploymentOption: 'WITH_TRAFFIC_CONTROL',
                    DeploymentType: 'BLUE_GREEN'
                }
            }));
            test.done();
        },
        'can enable rollback when deployment stops'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    stoppedDeployment: true
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                        'DEPLOYMENT_STOP_ON_REQUEST'
                    ]
                },
            }));
            test.done();
        },
        'can disable rollback when alarm in failure state'(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    deploymentInAlarm: false
                },
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        'DEPLOYMENT_FAILURE',
                    ]
                },
            }));
            test.done();
        },
        'imported with fromLambdaDeploymentGroupAttributes': {
            'defaults the Deployment Config to Canary10Percent5Minutes'(test) {
                const stack = new cdk.Stack();
                const lambdaApp = codedeploy.LambdaApplication.fromLambdaApplicationName(stack, 'LA', 'LambdaApplication');
                const importedGroup = codedeploy.LambdaDeploymentGroup.fromLambdaDeploymentGroupAttributes(stack, 'LDG', {
                    application: lambdaApp,
                    deploymentGroupName: 'LambdaDeploymentGroup',
                });
                test.equal(importedGroup.deploymentConfig, codedeploy.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES);
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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