# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nornir_netconf',
 'nornir_netconf.plugins.connections',
 'nornir_netconf.plugins.helpers',
 'nornir_netconf.plugins.tasks',
 'nornir_netconf.plugins.tasks.capabilities',
 'nornir_netconf.plugins.tasks.editing',
 'nornir_netconf.plugins.tasks.locking',
 'nornir_netconf.plugins.tasks.retrieval']

package_data = \
{'': ['*'], 'nornir_netconf.plugins.tasks': ['flowmon/*']}

install_requires = \
['ncclient>=0.6.9,<0.7.0', 'nornir>=3.0.0,<4.0.0', 'xmltodict>=0.12.0,<0.13.0']

entry_points = \
{'nornir.plugins.connections': ['netconf = '
                                'nornir_netconf.plugins.connections:Netconf']}

setup_kwargs = {
    'name': 'nornir-netconf',
    'version': '1.0.1',
    'description': 'Netconf plugin for nornir using ncclient',
    'long_description': '# Nornir NETCONF\n\n[![codecov](https://codecov.io/gh/h4ndzdatm0ld/nornir_netconf/branch/develop/graph/badge.svg?token=MRI39YHOOR)](https://codecov.io/gh/h4ndzdatm0ld/nornir_netconf)\n[![Build Status](https://github.com/nornir-automation/nornir_netconf/workflows/test_nornir_netconf/badge.svg)](https://github.com/nornir-automation/nornir_netconf/actions?query=workflow%3Atest_nornir_netconf)\n\nCollection of NETCONF tasks and connection plugin for [Nornir](https://github.com/nornir-automation/nornir)\n\n## Documentation\n\nDocumentation is generated with Sphinx and hosted with Github Pages. [Documentation](https://h4ndzdatm0ld.github.io/nornir_netconf/)\n\nTo generate the latest documentation locally:\n\n```bash\nsphinx-build -vvv -b html ./docs ./docs/public\ncd docs/public\npython -m http.server\n```\n\n## Installation\n\n------------\n\n```bash\npip install nornir_netconf\n```\n\n## Plugins\n\n------------\n\n### Connections\n\n------------\n\n* **netconf** - Connect to network devices using [ncclient](https://github.com/ncclient/ncclient)\n\n### Tasks\n\n------------\n\n* **netconf_capabilities** - Return server capabilities from target\n* **netconf_get** - Returns state data based on the supplied xpath\n* **netconf_get_config** - Returns configuration from specified configuration store (default="running")\n* **netconf_edit_config** - Edits configuration on specified datastore (default="running")\n* **netconf_lock** - Locks or Unlocks a specified datastore (default="lock")\n* **netconf_commit** - Commits a change\n\n## Response Result\n\nThe goal of the task results is to put the NETCONF RPC-reply back in your hands. An \'rpc\' key will be available which can then be used to access \'data_xml\' or \'xml\' depending on the type of response or any other attributes available, such as \'error\', \'errors\'. Some of the RPC is unpacked and provided back as part of the Result by default, including the \'error\', \'errors\' and \'ok\' if available. Anything else can be accessed directly from the rpc.\n\nFurthermore, some tasks allow the \'xml_dict\' boolean argument. This will take the response RPC XML and convert it into a python dictionary. Keep in mind, this may not be perfect as XML does\'t quite translate 100% into a python dictionary.\n\nFor example, an xml response can include a collapsed response with open/close as so:  ```<ok/>```\nIf parsed into python dictionary using xml_dict argument, the key of \'ok\' will have a value of none.  However, if we were to be parsing ```<enabled>True</enabled>``` this would show a key of \'enabled\' and a value of \'True\'.\n\nThis is a simple built-in solution available, but not the only one. You have the RPC as part of the response and you are able to parse it anyway or method which works better for you.\n\n## Global Lock\n\nThe `netconf_lock` task will always return the Manager object, which is the established (and locked) agent used to send RPC\'s back and forth. The idea of retrieving the Manager is to carry this established locked session from task to task and only lock and unlock once during a run of tasks.  Please review the examples below to see how to extract the manager and store it under the `task.host` dictionary as a variable which can be used across multiple tasks. The Manager is passed into other tasks and re-used to send RPCs to the remote server.\n\n### Examples\n\nHead over to the [Examples directory](https://github.com/h4ndzdatm0ld/nornir_netconf/tree/develop/examples) if you\'d like to review the files.\n\n<details><summary>Directory Structure</summary>\n\n```bash\n├── example-project\n│   ├── config.yml\n│   ├── inventory\n│   │   ├── groups.yml\n│   │   ├── hosts-local.yml\n│   │   └── ssh_config\n│   ├── logs\n│   │   └── nornir.log\n│   └── nr-get-config.py\n└── README.md\n```\n\n</details>\n\n<details><summary>Netconf Connection Plugin</summary>\n\nBelow is the snippet of a host inside the host-local.yml file and it\'s associated group, \'sros\'.\n\n```yaml\nnokia_rtr:\n  hostname: "192.168.1.205"\n  port: 830\n  groups:\n    - "sros"\n```\n\n```yaml\nsros:\n  username: "netconf"\n  password: "NCadmin123"\n  port: 830\n  platform: "sros"\n  connection_options:\n    netconf:\n      extras:\n        hostkey_verify: false\n        timeout: 300\n        allow_agent: false\n        look_for_keys: false\n```\n\n</details>\n\n<details><summary>Task: Get Config</summary>\n\n```python\n"""Nornir NETCONF Example Task: \'get-config\'."""\nfrom nornir import InitNornir\nfrom nornir_utils.plugins.functions import print_result\nfrom nornir_netconf.plugins.tasks import netconf_get_config\n\n\n__author__ = "Hugo Tinoco"\n__email__ = "hugotinoco@icloud.com"\n\nnr = InitNornir("config.yml")\n\n# Filter the hosts by \'west-region\' assignment\nwest_region = nr.filter(region="west-region")\n\n\ndef example_netconf_get_config(task):\n    """Test get config."""\n\n    task.run(\n        netconf_get_config,\n        source="running",\n        path="""\n        <configure xmlns="urn:nokia.com:sros:ns:yang:sr:conf">\n            <router>\n                <router-name>Base</router-name>\n            </router>\n        </configure>\n        """,\n        filter_type="subtree",\n    )\n\n\ndef main():\n    """Execute Nornir Script."""\n    print_result(west_region.run(task=example_netconf_get_config))\n\n\nif __name__ == "__main__":\n    main()\n\n```\n\n</details>\n\n<details><summary>Task: Get Capabilities</summary>\n\n```python\n"""Nornir NETCONF Example Task: \'get-config\'."""\nfrom nornir import InitNornir\nfrom nornir_utils.plugins.functions import print_result\nfrom nornir_netconf.plugins.tasks import netconf_capabilities\n\n\n__author__ = "Hugo Tinoco"\n__email__ = "hugotinoco@icloud.com"\n\nnr = InitNornir("config.yml")\n\n# Filter the hosts by \'west-region\' assignment\nwest_region = nr.filter(region="west-region")\n\n\ndef example_netconf_get_capabilities(task):\n    """Test get capabilities."""\n    task.run(netconf_capabilities)\n\n\ndef main():\n    """Execute Nornir Script."""\n    print_result(west_region.run(task=example_netconf_get_capabilities))\n\n\nif __name__ == "__main__":\n    main()\n```\n\n</details>\n\n<details><summary>Task: Edit-Config with Global Lock</summary>\n\n\n```python\n"""Nornir NETCONF Example Task: \'edit-config\', \'netconf_lock\'."""\nfrom nornir import InitNornir\nfrom nornir_utils.plugins.functions import print_result\nfrom nornir_netconf.plugins.tasks import netconf_edit_config, netconf_lock, netconf_commit\n\n\n__author__ = "Hugo Tinoco"\n__email__ = "hugotinoco@icloud.com"\n\nnr = InitNornir("config.yml")\n\n# Filter the hosts by \'west-region\' assignment\nwest_region = nr.filter(region="west-region")\n\n\ndef example_global_lock(task):\n    """Test global lock operation of \'candidate\' datastore."""\n    lock = task.run(netconf_lock, datastore="candidate", operation="lock")\n    # Retrieve the Manager(agent) from lock operation and store for further\n    # operations.\n    task.host["manager"] = lock.result["manager"]\n\n\ndef example_edit_config(task):\n    """Test edit-config with global lock using manager agent."""\n\n    config_payload = """\n    <config>\n        <configure xmlns="urn:nokia.com:sros:ns:yang:sr:conf">\n            <router>\n                <router-name>Base</router-name>\n                <interface>\n                    <interface-name>L3-OAM-eNodeB069420-X1</interface-name>\n                    <admin-state>disable</admin-state>\n                    <ingress-stats>false</ingress-stats>\n                </interface>\n            </router>\n        </configure>\n    </config>\n    """\n\n    result = task.run(\n        netconf_edit_config, config=config_payload, target="candidate", manager=task.host["manager"], xmldict=True\n    )\n\n    # Access the RPC response object directly.\n    # Or you can check the \'ok\' attr from an rpc response as well, if it exists.\n    if "ok" in result.result["rpc"].data_xml:\n        task.run(netconf_commit, manager=task.host["manager"], xmldict=True)\n\n    # Check OK key exists, as we passed in \'xmldict=True\'\n    print(result.result["xml_dict"].keys())\n\ndef example_unlock(task):\n    """Unlock candidate datastore."""\n    task.run(netconf_lock, datastore="candidate", operation="unlock", manager=task.host["manager"])\n\n\ndef main():\n    """Execute Nornir Script."""\n    print_result(west_region.run(task=example_global_lock))\n    print_result(west_region.run(task=example_edit_config))\n    print_result(west_region.run(task=example_unlock))\n\n\nif __name__ == "__main__":\n    main()\n\n```\n\n</details>\n\n### Additional Documentation\n\n* [NCClient](https://ncclient.readthedocs.io/en/latest/)\n* [Sysrepo](https://www.sysrepo.org/)\n\n## Contributions\n\n------------\n\nNo line of code shall go un tested! Any contribution will need to be accounted by the coverage report and satisfy all linting.\n\nLinters:\n\n* Fake8\n* Black\n* Yamllint\n* Pylint\n* Pydocstyle\n* Bandit\n* MyPy\n\n### Testing\n\nTo test within a local docker environment\n\n```bash\ngit clone https://github.com/h4ndzdatm0ld/nornir_netconf\n```\n\n```bash\ndocker-compose build && docker-compose run test\n```\n\nTo test locally with pytest\n\n```bash\ndocker-compose up netconf1\n```\n\n```bash\npoetry install && poetry shell\n```\n\n```bash\npytest --cov=nornir_netconf --color=yes --disable-pytest-warnings -vvv\n```\n\n### Integration Tests\n\nDevices with full integration tests (ContainerLab)\n\n* Nokia SROS - TiMOS-B-21.2.R1\n* Cisco IOSxR - Cisco IOS XR Software, Version 6.1.3\n\nDevices testing against Always-ON Sandboxes (Cisco DevNet)\n\n* Cisco IOS-XE - Cisco IOS XE Software, Version 17.03.01a\n\nThese tests are ran locally, and will soon be integrated into Github CI.\n\n### Sysrepo: netopeer2\n\nMajority of integration tests are ran against a docker instance of [netopeer2](https://hub.docker.com/r/sysrepo/sysrepo-netopeer2)\n\nFrom the [Sysrepo](https://www.sysrepo.org/) website:\n\n"Netopeer2 and Sysrepo provide a fully open source and standards compliant implementation of a NETCONF server and YANG configuration data stores."\n',
    'author': 'Hugo Tinoco',
    'author_email': 'hugotinoco@icloud.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/h4ndzdatm0ld/nornir_netconf',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.0,<4.0.0',
}


setup(**setup_kwargs)
