"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DualAlbFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class DualAlbFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.hasInternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.enableLoadBalancerAlias = ((_b = props.route53Ops) === null || _b === void 0 ? void 0 : _b.enableLoadBalancerAlias) != false;
        this.vpc = (_c = props.vpc) !== null && _c !== void 0 ? _c : getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.external) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.internal) {
                this.hasInternalLoadBalancer = true;
            }
        });
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
        }
        if (this.hasInternalLoadBalancer) {
            this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
                vpc: this.vpc,
                internetFacing: false,
            });
        }
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
                circuitBreaker: props.circuitBreaker != false ? {
                    rollback: true,
                } : undefined,
            });
            this.service.push(svc);
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_f = (_e = t.scalingPolicy) === null || _e === void 0 ? void 0 : _e.maxCapacity) !== null && _f !== void 0 ? _f : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_h = (_g = t.scalingPolicy) === null || _g === void 0 ? void 0 : _g.targetCpuUtilization) !== null && _h !== void 0 ? _h : 50,
            });
            if (t.external) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                    healthCheck: t.healthCheck,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.external.port}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.external.port,
                    protocol: t.external.certificate ? elbv2.ApplicationProtocol.HTTPS : elbv2.ApplicationProtocol.HTTP,
                    certificates: t.external.certificate,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_k = (_j = t.scalingPolicy) === null || _j === void 0 ? void 0 : _j.requestPerTarget) !== null && _k !== void 0 ? _k : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            if (t.internal) {
                const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                    healthCheck: t.healthCheck,
                });
                // listener for the internal ALB
                new elbv2.ApplicationListener(this, `IntAlbListener${t.internal.port}`, {
                    loadBalancer: this.internalAlb,
                    open: true,
                    port: t.internal.port,
                    protocol: t.internal.certificate ? elbv2.ApplicationProtocol.HTTPS : elbv2.ApplicationProtocol.HTTP,
                    certificates: t.internal.certificate,
                    defaultTargetGroups: [inttg],
                });
                // extra scaling policy
                scaling.scaleOnRequestCount('RequestScaling2', {
                    requestsPerTarget: (_m = (_l = t.scalingPolicy) === null || _l === void 0 ? void 0 : _l.requestPerTarget) !== null && _m !== void 0 ? _m : 1000,
                    targetGroup: inttg,
                });
                inttg.addTarget(svc);
            }
        });
        // Route53
        const zoneName = (_e = (_d = props.route53Ops) === null || _d === void 0 ? void 0 : _d.zoneName) !== null && _e !== void 0 ? _e : 'svc.local';
        const externalAlbRecordName = (_g = (_f = props.route53Ops) === null || _f === void 0 ? void 0 : _f.externalAlbRecordName) !== null && _g !== void 0 ? _g : 'external';
        const internalAlbRecordName = (_j = (_h = props.route53Ops) === null || _h === void 0 ? void 0 : _h.internalAlbRecordName) !== null && _j !== void 0 ? _j : 'internal';
        if (this.enableLoadBalancerAlias) {
            const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
                zoneName,
                vpc: this.vpc,
            });
            if (this.hasInternalLoadBalancer) {
                new route53.ARecord(this, 'InternalAlbAlias', {
                    zone,
                    recordName: internalAlbRecordName,
                    target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
                });
            }
            if (this.hasExternalLoadBalancer) {
                new route53.ARecord(this, 'ExternalAlbAlias', {
                    zone,
                    recordName: externalAlbRecordName,
                    target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
                });
            }
            if (this.hasExternalLoadBalancer) {
                new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
                new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
            }
            if (this.hasInternalLoadBalancer) {
                new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
                new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
            }
        }
        else {
            if (this.hasExternalLoadBalancer) {
                new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            }
            if (this.hasInternalLoadBalancer) {
                new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
            }
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.DualAlbFargateService = DualAlbFargateService;
_a = JSII_RTTI_SYMBOL_1;
DualAlbFargateService[_a] = { fqn: "cdk-fargate-patterns.DualAlbFargateService", version: "0.2.16" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,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