from proto_topy.entities import (
    ProtoCollection,
    ProtoModule,
    CompilationFailed,
    DelimitedMessageFactory,
)
import pytest
import os
import sys
from pathlib import Path
from distutils.spawn import find_executable
from array import array
from io import BytesIO


protoc_path = Path(find_executable("protoc") or os.environ.get('PROTOC'))


def test_add_proto():
    proto = ProtoModule(file_path=Path("test1.proto"), source="")
    modules = ProtoCollection(compiler_path=protoc_path)
    modules.add_proto(proto)
    assert Path("test1.proto") in modules.modules


def test_add_proto2():
    modules = ProtoCollection(
        protoc_path,
        *(
            ProtoModule(file_path=Path("test2.proto"), source=""),
            ProtoModule(file_path=Path("test3.proto"), source=""),
        ),
    )
    assert Path("test2.proto") in modules.modules
    assert Path("test3.proto") in modules.modules


def test_bad_protoc():
    with pytest.raises(FileNotFoundError):
        ProtoCollection(Path('dummy')).compile()


def test_no_protoc():
    with pytest.raises(TypeError):
        ProtoCollection()


def test_compile_invalid_source():
    with pytest.raises(CompilationFailed):
        ProtoModule(file_path=Path("test4.proto"), source="").compiled(protoc_path)


def test_compile_redundant_proto():
    proto_source = 'syntax = "proto3"; message TestR { int32 foo = 1; };'
    proto1 = ProtoModule(file_path=Path("testr.proto"), source=proto_source)
    proto2 = ProtoModule(file_path=Path("testr.proto"), source=proto_source)
    with pytest.raises(KeyError, match=r"testr.proto already added"):
        ProtoCollection(protoc_path, proto1, proto2).compile()


def test_compile_minimal_proto():
    from google.protobuf.timestamp_pb2 import Timestamp

    proto = ProtoModule(
        file_path=Path("test5.proto"),
        source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message Test5 {
        google.protobuf.Timestamp created = 1;
    }
    """,
    ).compiled(protoc_path)
    sys.modules["test5"] = proto.py
    atest5 = proto.py.Test5()
    assert isinstance(atest5.created, Timestamp)
    del sys.modules["test5"]


def test_compile_minimal_proto_in_a_package():
    from google.protobuf.timestamp_pb2 import Timestamp

    proto = ProtoModule(
        file_path=Path("p1/p2/p3/thing.proto"),
        source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message Thing {
        google.protobuf.Timestamp created = 1;
    }
    """,
    ).compiled(protoc_path)
    assert (
        "\n".join(proto.py_source.split("\n")[:4])
        == '''# -*- coding: utf-8 -*-
# Generated by the protocol buffer compiler.  DO NOT EDIT!
# source: p1/p2/p3/thing.proto
"""Generated protocol buffer code."""'''
    )
    sys.modules["thing"] = proto.py
    athing = proto.py.Thing()
    assert isinstance(athing.created, Timestamp)


def test_compile_missing_dependency():
    with pytest.raises(CompilationFailed, match=r'other.proto: File not found.*'):
        ProtoModule(
            file_path=Path("test.proto"),
            source='syntax = "proto3"; import "other.proto";',
        ).compiled(protoc_path)


def test_compile_ununsed_dependency():
    proto = ProtoModule(
        file_path=Path("test.proto"),
        source="""
    syntax = "proto3";
    import "other.proto";
    """,
    )
    other_proto = ProtoModule(
        file_path=Path("other.proto"),
        source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message OtherThing {
        google.protobuf.Timestamp created = 1;
    }
    """,
    )
    modules = ProtoCollection(protoc_path, proto, other_proto)
    try:
        modules.compile()
    except CompilationFailed:
        pytest.fail("Unexpected CompilationFailed ..")


def test_compile_simple_dependency():
    from google.protobuf.timestamp_pb2 import Timestamp

    proto = ProtoModule(
        file_path=Path("p3/p4/test6.proto"),
        source="""
    syntax = "proto3";
    import "p1/p2/other2.proto";
    message Test6 {
        OtherThing2 foo = 1;
    };
    """,
    )
    other_proto = ProtoModule(
        file_path=Path("p1/p2/other2.proto"),
        source="""
    syntax = "proto3";
    import "google/protobuf/timestamp.proto";
    message OtherThing2 {
        google.protobuf.Timestamp created = 1;
    }
    """,
    )
    modules = ProtoCollection(protoc_path, proto, other_proto)
    modules.compile()
    sys.modules.update({proto.name: proto.py for proto in modules.modules.values()})
    atest6 = modules.modules[Path("p3/p4/test6.proto")].py.Test6()
    assert isinstance(atest6.foo.created, Timestamp)
    for proto in modules.modules.values():
        del sys.modules[proto.name]


def test_encode_message():
    proto_source = 'syntax = "proto3"; message Test{n} {{ int32 foo = 1; }};'
    proto1 = ProtoModule(file_path=Path("test7.proto"), source=proto_source.format(n=7))
    proto2 = ProtoModule(file_path=Path("test8.proto"), source=proto_source.format(n=8))
    ProtoCollection(protoc_path, proto1, proto2).compile()
    assert array('B', proto1.py.Test7(foo=124).SerializeToString()) == array(
        'B', [8, 124]
    )
    assert array('B', proto2.py.Test8(foo=123).SerializeToString()) == array(
        'B', [8, 123]
    )


def test_decode_message():
    proto = ProtoModule(
        file_path=Path("test9.proto"),
        source='syntax = "proto3"; message Test9 { int32 foo = 1; };',
    ).compiled(protoc_path)
    aTest9 = proto.py.Test9()
    aTest9.ParseFromString(bytes(array('B', [8, 124])))
    assert aTest9.foo == 124


def test_decode_messages_stream():
    proto = ProtoModule(
        file_path=Path("test10.proto"),
        source='syntax = "proto3"; message Test10 { int32 foo = 1; };',
    ).compiled(protoc_path)
    factory = DelimitedMessageFactory(
        BytesIO(), *(proto.py.Test10(foo=foo) for foo in [1, 12])
    )
    factory.stream.seek(0)
    assert [thing.foo for _, thing in factory.message_read(proto.py.Test10)] == [1, 12]


def test_decode_messages_stream2():
    proto = ProtoModule(
        file_path=Path("test10.proto"),
        source='syntax = "proto3"; message Test10 { int32 foo = 1; };',
    ).compiled(protoc_path)
    message = DelimitedMessageFactory(
        BytesIO(), *(proto.py.Test10(foo=foo) for foo in [1, 12])
    )

    for fn in message.read, message.bytes_read:
        message.stream.seek(0)
        foos = []
        for offset, data in fn():
            aTest10 = proto.py.Test10()
            aTest10.ParseFromString(data)
            foos.append(aTest10.foo)
        assert foos == [1, 12]
