""" 
This script gather functions related to the SZ spectrum
"""

import numpy as np
import astropy.units as u
from astropy import constants as const
from astropy.cosmology import Planck15 as cosmo


#===================================================
#========== CMB intensity
#===================================================
def get_I0_CMB():
    """
    Compute the CMB intensity

    Parameters
    ----------

    Outputs
    --------
    - I0 (quantity): the CMB intensity (homogeneous to MJy/sr)
    """

    I0 = 2*(const.k_B*cosmo.Tcmb0)**3/(const.h*const.c)**2*u.sr**-1
    return I0.to('MJy sr-1')

#===================================================
#========== Non relativistic tSZ spectrum
#===================================================
def tsz_spec(frequency):
    """
    Compute the non relativistic SZ spectrum, f(nu)
    as in delta I_nu = I0 f(nu) y
    
    Parameters
    ----------
    - frequency (quantity): frequency array homogeneous to GHz

    Outputs
    --------
    - SZ spectrum: f(nu)
    """
    
    x = const.h * frequency / (const.k_B * cosmo.Tcmb0)
    f_nu = x**4 * np.exp(x) / (np.exp(x)-1)**2 * (x*(np.exp(x)+1)/(np.exp(x)-1) - 4)
    
    return f_nu

#===================================================
#========== Relativistic tSZ spectrum
#===================================================
def tsz_spec_relativistic(frequency, kBT):
    """
    Compute the relativistic SZ spectrum, f(nu, T)
    as in delta I_nu = I0 f(nu, T) y

    Parameters
    ----------
    - frequency (quantity): frequency array homogeneous to GHz
    - temperature (quantity): frequency array homogeneous to GHz
    
    Outputs
    --------
    - SZ spectrum: f(nu, T)
    """
    
    #========== Make sure that frequency and temperature are arrays
    if type(frequency.to_value()) == float:
        frequency = np.array([frequency.to_value()]) * frequency.unit
    if type(kBT.to_value()) == float:
        kBT = np.array([kBT.to_value()]) * kBT.unit

    #========== Replicate to work with grids
    f_grid = (np.tile(frequency, [len(kBT),1])).T
    t_grid = (np.tile(kBT, [len(frequency),1]))

    #========== Function variable
    theta = t_grid.to_value('keV')/(const.m_e*const.c**2).to_value('keV')
    x = (const.h*f_grid/(const.k_B*cosmo.Tcmb0)).to_value('')
    
    #========== Region where x < 1.2
    f1 = x**4 * np.exp(x)/(np.exp(x)-1)**2
    xtil = x*(np.exp(x)+1)/(np.exp(x)-1)
    s = 2*x/(np.exp(x/2)-np.exp(-x/2))
  
    y0 = xtil-4.0

    y1a = -10.+47./2.*xtil-42./5.*xtil**(2.)
    y1b = 0.7*xtil**(3.)+s**(2.)*(-21./5.+7./5.*xtil)
    y1 = y1a+y1b

    y2a = -15/2.+1023./8.*xtil-868./5.*xtil**(2.)
    y2b = 329./5.*xtil**(3.)-44./5.*xtil**(4.)
    y2c = 11./30.*xtil**(5.)
    y2d = -434./5.+658/5.*xtil-242./5.*xtil**(2.)+143./30.*xtil**(3.)
    y2e = -44./5.+187./60.*xtil
    y2 = y2a+y2b+y2c+s**(2.)*y2d+s**(4.)*y2e
    
    y3a = 15./2.+2505./8.*xtil-7098./5.*xtil**(2.)
    y3b = 1425.3*xtil**(3.)-18594./35.*xtil**(4.)
    y3c = 12059./140.*xtil**(5.)-128./21.*xtil**(6.)+16./105.*xtil**(7.)
    y3d1 = -709.8+14253/5.*xtil-102267./35.*xtil**(2.)
    y3d2 = 156767./140.*xtil**(3.)-1216./7.*xtil**(4.)+64./7.*xtil**(5.)
    y3d = s**(2.)*(y3d1+y3d2)
    y3e1 = -18594./35.+205003./280.*xtil
    y3e2 = -1920./7.*xtil**(2.)+1024./35.*xtil**(3.)
    y3e = s**(4.)*(y3e1+y3e2)
    y3f = s**(6.)*(-544./21.+922./105.*xtil)
    y3 = y3a+y3b+y3c+y3d+y3e+y3f
    
    y4a = -135./32.+30375./128.*xtil-6239.1*xtil**(2.)
    y4b = 61472.7/4.*xtil**(3.)-12438.9*xtil**(4.)
    y4c = 35570.3/8.*xtil**(5.)-16568./21.*xtil**(6.)
    y4d = 7516./105.*xtil**(7.)-22./7.*xtil**(8.)+11./210.*xtil**(9.)
    y4e1 = -62391./20.+614727./20.*xtil
    y4e2 = -1368279./20.*xtil**(2.)+4624139./80.*xtil**(3.)
    y4e3 = -157396./7.*xtil**(4.)+30064./7.*xtil**(5.)
    y4e4 = -2717./7.*xtil**(6.)+2761./210.*xtil**(7.)
    y4e = s**(2.)*(y4e1+y4e2+y4e3+y4e4)
    y4f1 = -12438.9+6046951./160.*xtil
    y4f2 = -248520./7.*xtil**(2.)+481024./35.*xtil**(3.)
    y4f3 = -15972./7.*xtil**(4.)+18689./140.*xtil**(5.)
    y4f = s**(4.)*(y4f1+y4f2+y4f3)
    y4g1 = -70414./21.+465992./105.*xtil
    y4g2 = -11792./7.*xtil**(2.)+19778./105.*xtil**(3.)
    y4g = s**(6.)*(y4g1+y4g2)
    y4h = s**(8.)*(-682./7.+7601./210.*xtil)
    y4 = y4a+y4b+y4c+y4d+y4e+y4f+y4g+y4h

    DI_over_tau_over_theta_lt12 = f1*(y0+theta*y1+theta**(2.)*y2+theta**(3.)*y3+theta**(4.)*y4)
  
    #========== Region where x > 1.2 if T > 20.0 keV
    Tlim = 20.0
    x_0 = 3.830 * (1.0 + 1.1674*theta - 0.8533*theta**2.)

    a_ij = np.array([
        [[-1.81317E+1+x*0],[ 9.97038E+1+x*0],[-6.07438E+1+x*0],[ 1.05143E+3+x*0],[-2.86734E+3+x*0],[ 7.73353E+3+x*0],[-8.16644E+3+x*0],[-5.37712E+3+x*0],[ 1.52226E+4+x*0],[ 7.18726E+3+x*0],[-1.39548E+4+x*0],[-2.08464E+4+x*0],[ 1.79040E+4+x*0]],
        [[ 1.68733E+2+x*0],[-6.07829E+2+x*0],[ 1.14933E+3+x*0],[-2.42382E+2+x*0],[-7.73030E+2+x*0],[ 5.33993E+3+x*0],[-4.03443E+3+x*0],[ 3.00692E+3+x*0],[ 9.58809E+3+x*0],[ 8.16574E+3+x*0],[-6.13322E+3+x*0],[-1.48117E+4+x*0],[ 3.43816E+4+x*0]],
        [[-6.69883E+2+x*0],[ 1.59654E+3+x*0],[-3.33375E+3+x*0],[-2.13234E+3+x*0],[-1.80812E+2+x*0],[ 3.75605E+3+x*0],[-4.75180E+3+x*0],[-4.50495E+3+x*0],[ 5.38753E+3+x*0],[ 5.03355E+3+x*0],[-1.18396E+4+x*0],[-8.58473E+3+x*0],[ 3.96316E+4+x*0]],
        [[ 1.56222E+3+x*0],[-1.78598E+3+x*0],[ 5.13747E+3+x*0],[ 4.10404E+3+x*0],[ 5.54775E+2+x*0],[-3.89994E+3+x*0],[-1.22455E+3+x*0],[ 1.03747E+3+x*0],[ 4.32237E+3+x*0],[ 1.03805E+3+x*0],[-1.47172E+4+x*0],[-1.23591E+4+x*0],[ 1.77290E+4+x*0]],
        [[-2.34712E+3+x*0],[ 2.78197E+2+x*0],[-5.49648E+3+x*0],[-5.94988E+2+x*0],[-1.47060E+3+x*0],[-2.84032E+2+x*0],[-1.15352E+3+x*0],[-1.17893E+3+x*0],[ 7.01209E+3+x*0],[ 4.75631E+3+x*0],[-5.13807E+3+x*0],[-8.73615E+3+x*0],[ 9.41580E+3+x*0]],
        [[ 1.92894E+3+x*0],[ 1.17970E+3+x*0],[ 3.13650E+3+x*0],[-2.91121E+2+x*0],[-1.15006E+3+x*0],[ 4.17375E+3+x*0],[-3.31788E+2+x*0],[ 1.37973E+3+x*0],[-2.48966E+3+x*0],[ 4.82005E+3+x*0],[-1.06121E+4+x*0],[-1.19394E+4+x*0],[ 1.34908E+4+x*0]],
        [[ 6.40881E+2+x*0],[-6.81789E+2+x*0],[ 1.20037E+3+x*0],[-3.27298E+3+x*0],[ 1.02988E+2+x*0],[ 2.03514E+3+x*0],[-2.80502E+3+x*0],[ 8.83880E+2+x*0],[ 1.68409E+3+x*0],[ 4.26227E+3+x*0],[-6.37868E+3+x*0],[-1.11597E+4+x*0],[ 1.46861E+4+x*0]],
        [[-4.02494E+3+x*0],[-1.37983E+3+x*0],[-1.65623E+3+x*0],[ 7.36120E+1+x*0],[ 2.66656E+3+x*0],[-2.30516E+3+x*0],[ 5.22182E+3+x*0],[-8.53317E+3+x*0],[ 3.75800E+2+x*0],[ 8.49249E+2+x*0],[-6.88736E+3+x*0],[-1.01475E+4+x*0],[ 4.75820E+3+x*0]],
        [[ 4.59247E+3+x*0],[ 3.04203E+3+x*0],[-2.11039E+3+x*0],[ 1.32383E+3+x*0],[ 1.10646E+3+x*0],[-3.53827E+3+x*0],[-1.12073E+3+x*0],[-5.47633E+3+x*0],[ 9.85745E+3+x*0],[ 5.72138E+3+x*0],[ 6.86444E+3+x*0],[-5.72696E+3+x*0],[ 1.29053E+3+x*0]],
        [[-1.61848E+3+x*0],[-1.83704E+3+x*0],[ 2.06738E+3+x*0],[ 4.00292E+3+x*0],[-3.72824E+1+x*0],[ 9.10086E+2+x*0],[ 3.72526E+3+x*0],[ 3.41895E+3+x*0],[ 1.31241E+3+x*0],[ 6.68089E+3+x*0],[-4.34269E+3+x*0],[-5.42296E+3+x*0],[ 2.83445E+3+x*0]],
        [[-1.00239E+3+x*0],[-1.24281E+3+x*0],[ 2.46998E+3+x*0],[-4.25837E+3+x*0],[-1.83515E+2+x*0],[-6.47138E+2+x*0],[-7.35806E+3+x*0],[-1.50866E+3+x*0],[-2.47275E+3+x*0],[ 9.09399E+3+x*0],[-2.75851E+3+x*0],[-6.75104E+3+x*0],[ 7.00899E+2+x*0]],
        [[ 1.04911E+3+x*0],[ 2.07475E+3+x*0],[-3.83953E+3+x*0],[ 7.79924E+2+x*0],[-4.08658E+3+x*0],[ 4.43432E+3+x*0],[ 3.23015E+2+x*0],[ 6.16180E+3+x*0],[-1.00851E+4+x*0],[ 7.65063E+3+x*0],[ 1.52880E+3+x*0],[-6.08330E+3+x*0],[ 1.23369E+3+x*0]],
        [[-2.61041E+2+x*0],[-7.22803E+2+x*0],[ 1.34581E+3+x*0],[ 5.90851E+2+x*0],[ 3.32198E+2+x*0],[ 2.58340E+3+x*0],[-5.97604E+2+x*0],[-4.34018E+3+x*0],[-3.58925E+3+x*0],[ 2.59165E+3+x*0],[ 6.76140E+3+x*0],[-6.22138E+3+x*0],[ 4.40668E+3+x*0]]
    ])[:,:,0,:]
    
    theta_ei = np.array([
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]],
        [[x*0+(10*theta)**0.],[x*0+(10*theta)**1.],[x*0+(10*theta)**2.],[x*0+(10*theta)**3.],[x*0+(10*theta)**4.],[x*0+(10*theta)**5.],[x*0+(10*theta)**6.],[x*0+(10*theta)**7.],[x*0+(10*theta)**8.],[x*0+(10*theta)**9.],[x*0+(10*theta)**10.],[x*0+(10*theta)**11.],[x*0+(10*theta)**12.]]
    ])[:,:,0,:,:]
    theta_ei = np.transpose(theta_ei, (1,0,2,3))
    
    Zj = np.array([
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]], 
        [[(0.05*x)**0.],[(0.05*x)**1.],[(0.05*x)**2.],[(0.05*x)**3.],[(0.05*x)**4.],[(0.05*x)**5.],[(0.05*x)**6.],[(0.05*x)**7.],[(0.05*x)**8.],[(0.05*x)**9.],[(0.05*x)**10.],[(0.05*x)**11.],[(0.05*x)**12.]]
    ])[:,:,0,:,:]
    
    G_theta_x = np.sum(np.sum(a_ij*theta_ei*Zj, 1), 0)

    DI_over_tau_over_theta_gt12 = x**2.0 * np.exp(-x) * (x-x_0) * G_theta_x

    #========== Pick the region
    f_nu = DI_over_tau_over_theta_lt12
    w_gt12 = (x > 1.2) * (t_grid > 20*u.keV)
    f_nu[w_gt12] = DI_over_tau_over_theta_gt12[w_gt12]
     
    return f_nu
