"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const appmesh = require("../lib");
module.exports = {
    'When creating a Mesh': {
        'with no spec applied': {
            'should defaults to DROP_ALL egress filter'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                new appmesh.Mesh(stack, 'mesh', { meshName: 'test-mesh' });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::Mesh', {
                    Spec: {},
                }));
                test.done();
            },
        },
        'with spec applied': {
            'should take egress filter from props'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                    egressFilter: appmesh.MeshFilterType.ALLOW_ALL,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::Mesh', {
                    Spec: {
                        EgressFilter: {
                            Type: 'ALLOW_ALL',
                        },
                    },
                }));
                test.done();
            },
        },
    },
    'When adding a Virtual Router to existing mesh': {
        'with at least one complete port mappings': {
            'shoulld create proper router'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualRouter('router', {
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualRouter', {
                    Spec: {
                        Listeners: [
                            {
                                PortMapping: {
                                    Port: 8080,
                                    Protocol: 'http',
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
    },
    'VirtualService can use CloudMap service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const mesh = new appmesh.Mesh(stack, 'mesh', {
            meshName: 'test-mesh',
        });
        const vpc = new ec2.Vpc(stack, 'vpc');
        const namespace = new cloudmap.PrivateDnsNamespace(stack, 'test-namespace', {
            vpc,
            name: 'domain.local',
        });
        const service = namespace.createService('Svc');
        // WHEN
        new appmesh.VirtualNode(stack, 'test-node', {
            mesh,
            cloudMapService: service,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
            Spec: {
                ServiceDiscovery: {
                    AWSCloudMap: {
                        NamespaceName: 'domain.local',
                        ServiceName: { 'Fn::GetAtt': ['testnamespaceSvcB55702EC', 'Name'] },
                    },
                },
            },
        }));
        test.done();
    },
    'When adding a VirtualService to a mesh': {
        'with VirtualRouter and VirtualNode as providers': {
            'should throw error'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const testNode = new appmesh.VirtualNode(stack, 'test-node', {
                    mesh,
                    dnsHostName: 'test-node',
                });
                const testRouter = mesh.addVirtualRouter('router', {
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                // THEN
                test.throws(() => {
                    mesh.addVirtualService('service', {
                        virtualServiceName: 'test-service.domain.local',
                        virtualNode: testNode,
                        virtualRouter: testRouter,
                    });
                });
                test.done();
            },
        },
        'with single virtual router provider resource': {
            'should create service'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const testRouter = mesh.addVirtualRouter('test-router', {
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                mesh.addVirtualService('service', {
                    virtualServiceName: 'test-service.domain.local',
                    virtualRouter: testRouter,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualService', {
                    Spec: {
                        Provider: {
                            VirtualRouter: {
                                VirtualRouterName: {
                                    'Fn::GetAtt': ['meshtestrouterF78D72DD', 'VirtualRouterName'],
                                },
                            },
                        },
                    },
                }));
                test.done();
            },
        },
        'with single virtual node provider resource': {
            'should create service'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const node = mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                mesh.addVirtualService('service2', {
                    virtualServiceName: 'test-service.domain.local',
                    virtualNode: node,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualService', {
                    Spec: {
                        Provider: {
                            VirtualNode: {
                                VirtualNodeName: {
                                    'Fn::GetAtt': ['meshtestnodeF93946D4', 'VirtualNodeName'],
                                },
                            },
                        },
                    },
                }));
                test.done();
            },
        },
    },
    'When adding a VirtualNode to a mesh': {
        'with empty default listeners and backends': {
            'should create default resource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::AppMesh::VirtualNode', {
                    MeshName: {
                        'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
                    },
                    Spec: {
                        // Specifically: no Listeners and Backends
                        ServiceDiscovery: {
                            DNS: {
                                Hostname: 'test.domain.local',
                            },
                        },
                    },
                }));
                test.done();
            },
        },
        'with added listeners': {
            'should create listener resource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    MeshName: {
                        'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
                    },
                    Spec: {
                        Listeners: [
                            {
                                PortMapping: {
                                    Port: 8080,
                                    Protocol: 'http',
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
        'with added listeners with healthchecks': {
            'should create healthcheck resource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                        healthCheck: {
                            healthyThreshold: 3,
                            path: '/',
                            interval: cdk.Duration.seconds(5),
                            timeout: cdk.Duration.seconds(2),
                            unhealthyThreshold: 2,
                        },
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    MeshName: {
                        'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
                    },
                    Spec: {
                        Listeners: [
                            {
                                HealthCheck: {
                                    HealthyThreshold: 3,
                                    IntervalMillis: 5000,
                                    Path: '/',
                                    Port: 8080,
                                    Protocol: 'http',
                                    TimeoutMillis: 2000,
                                    UnhealthyThreshold: 2,
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
        'with backends': {
            'should create resource with service backends'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const service1 = new appmesh.VirtualService(stack, 'service-1', {
                    virtualServiceName: 'service1.domain.local',
                    mesh,
                });
                mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test.domain.local',
                    listener: {
                        portMapping: {
                            port: 8080,
                            protocol: appmesh.Protocol.HTTP,
                        },
                    },
                    backends: [
                        service1,
                    ],
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    Spec: {
                        Backends: [
                            {
                                VirtualService: {
                                    VirtualServiceName: {
                                        'Fn::GetAtt': ['service1A48078CF', 'VirtualServiceName'],
                                    },
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
    },
    'Can construct a mesh from a name'(test) {
        // WHEN
        const stack2 = new cdk.Stack();
        const mesh2 = appmesh.Mesh.fromMeshName(stack2, 'imported-mesh', 'abc');
        mesh2.addVirtualService('service', {
            virtualServiceName: 'test.domain.local',
        });
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualService', {
            MeshName: 'abc',
            Spec: {},
            VirtualServiceName: 'test.domain.local',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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