from pathlib import Path
from typing import Annotated, Optional

import click
import typer

from cycode.cli.cli_types import SbomFormatOption, SbomOutputFormatOption
from cycode.cli.utils.sentry import add_breadcrumb
from cycode.cyclient.report_client import ReportParameters

_OUTPUT_RICH_HELP_PANEL = 'Output options'


def sbom_command(
    ctx: typer.Context,
    sbom_format: Annotated[
        SbomFormatOption,
        typer.Option(
            '--format',
            '-f',
            help='SBOM format.',
            case_sensitive=False,
            show_default=False,
        ),
    ],
    output_format: Annotated[
        SbomOutputFormatOption,
        typer.Option(
            '--output-format',
            '-o',
            help='Specify the output file format.',
            rich_help_panel=_OUTPUT_RICH_HELP_PANEL,
        ),
    ] = SbomOutputFormatOption.JSON,
    output_file: Annotated[
        Optional[Path],
        typer.Option(
            help='Output file.',
            show_default='Autogenerated filename saved to the current directory',
            dir_okay=False,
            writable=True,
            rich_help_panel=_OUTPUT_RICH_HELP_PANEL,
        ),
    ] = None,
    include_vulnerabilities: Annotated[
        bool, typer.Option('--include-vulnerabilities', help='Include vulnerabilities.', show_default=False)
    ] = False,
    include_dev_dependencies: Annotated[
        bool, typer.Option('--include-dev-dependencies', help='Include dev dependencies.', show_default=False)
    ] = False,
) -> int:
    """Generate SBOM report."""
    add_breadcrumb('sbom')

    sbom_format_parts = sbom_format.split('-')
    if len(sbom_format_parts) != 2:
        raise click.ClickException('Invalid SBOM format.')

    sbom_format, sbom_format_version = sbom_format_parts

    report_parameters = ReportParameters(
        entity_type='SbomCli',
        sbom_report_type=sbom_format,
        sbom_version=sbom_format_version,
        output_format=output_format,
        include_vulnerabilities=include_vulnerabilities,
        include_dev_dependencies=include_dev_dependencies,
    )
    ctx.obj['report_parameters'] = report_parameters
    ctx.obj['output_file'] = output_file

    return 1
