"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BudgetNotifier = void 0;
const aws_budgets_1 = require("@aws-cdk/aws-budgets");
const core_1 = require("@aws-cdk/core");
const NotificationType_1 = require("./NotificationType");
const TimeUnit_1 = require("./TimeUnit");
/**
 *
 */
class BudgetNotifier extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProperties(props);
        const costFilters = this.createCostFilters(props);
        const subscribers = this.createSubscribers(props);
        new aws_budgets_1.CfnBudget(this, 'MonthlyBudget_' + id, {
            budget: {
                budgetType: 'COST',
                timeUnit: props.timeUnit ? props.timeUnit : TimeUnit_1.TimeUnit.MONTHLY,
                budgetLimit: {
                    amount: props.limit,
                    unit: props.unit,
                },
                costFilters: costFilters,
            },
            notificationsWithSubscribers: [
                {
                    notification: {
                        comparisonOperator: 'GREATER_THAN',
                        threshold: props.threshold,
                        thresholdType: 'PERCENTAGE',
                        notificationType: props.notificationType
                            ? props.notificationType
                            : NotificationType_1.NotificationType.ACTUAL,
                    },
                    subscribers: subscribers,
                },
            ],
        });
    }
    validateProperties(props) {
        if (props.recipients && props.recipients.length > 10) {
            throw new Error('The maximum number of 10 e-mail recipients is exceeded.');
        }
        if (props.threshold <= 0) {
            throw new Error('Thresholds less than or equal to 0 are not allowed.');
        }
    }
    createSubscribers(props) {
        const subscribers = new Array();
        if (props.recipients) {
            for (const recipient of props.recipients) {
                subscribers.push({
                    address: recipient,
                    subscriptionType: 'EMAIL',
                });
            }
        }
        if (props.topicArn) {
            subscribers.push({
                address: props.topicArn,
                subscriptionType: 'SNS',
            });
        }
        return subscribers;
    }
    createCostFilters(props) {
        const tags = [];
        if (props.application) {
            tags.push('user:Application$' + props.application);
        }
        if (props.costCenter) {
            tags.push('user:Cost Center$' + props.costCenter);
        }
        if (props.service) {
            tags.push('user:Service$' + props.service);
        }
        const costFilters = {};
        if (tags && tags.length > 0) {
            costFilters.TagKeyValue = tags;
        }
        const availabilityZones = [];
        if (props.availabilityZones) {
            for (const az of props.availabilityZones) {
                availabilityZones.push(az);
            }
            costFilters.AZ = availabilityZones;
        }
        return costFilters;
    }
}
exports.BudgetNotifier = BudgetNotifier;
//# sourceMappingURL=data:application/json;base64,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