# -*- coding: utf-8 -*-
"""Parse BEL Excel(.xls) output files."""

from itertools import product

import xlrd

from adsorption_file_parser.bel_common import _META_DICT
from adsorption_file_parser.bel_common import _check
from adsorption_file_parser.bel_common import _parse_header

from .utils import common_utils as util


def parse(path):
    """
    Parse an xls file generated by BEL software.

    Parameters
    ----------
    path: str
        The location of an xls file generated by a belsorp instrument.

    Returns
    -------
    dict
        A dictionary containing report information.
    """
    meta = {}
    data = {}

    # open the workbook
    workbook = xlrd.open_workbook(path, encoding_override='latin-1')
    sheet = workbook.sheet_by_name('AdsDes')

    # local for efficiency
    meta_dict = _META_DICT.copy()

    # iterate over all cells in the notebook
    for row, col in product(range(sheet.nrows), range(sheet.ncols)):

        # check if empty
        cell_value = sheet.cell(row, col).value
        if not isinstance(cell_value, str) or cell_value == '':
            continue

        # check if we are in the data section
        if cell_value != 'No':
            cell_value = cell_value.strip().lower()
            try:
                key = util.search_key_in_def_dict(cell_value, meta_dict)
            except StopIteration:
                continue

            ref = meta_dict[key]['xl_ref']
            tp = meta_dict[key]['type']
            unit_key = meta_dict[key].get('unit')
            del meta_dict[key]  # delete for efficiency

            val = sheet.cell(row + ref[0], col + ref[1]).value
            if val == '':
                meta[key] = None
            elif tp == 'numeric':
                meta[key] = util.handle_string_numeric(val)
            elif tp == 'string':
                meta[key] = util.handle_excel_string(val)
            elif tp == 'datetime':
                meta[key] = util.handle_xlrd_datetime(val, sheet)
            elif tp == 'date':
                meta[key] = util.handle_xlrd_date(val, sheet)
            elif tp == 'time':
                meta[key] = util.handle_xlrd_time(val, sheet)
            elif tp == 'timedelta':
                meta[key] = _handle_bel_xl_timedelta(val)

            if unit_key:
                unit = sheet.cell(row + ref[0], col + ref[1] + 1).value.strip('[]')
                meta[unit_key] = unit

        else:  # If "data" section

            header_list = _get_header(sheet, row)
            head, units = _parse_header(header_list)  # header
            meta.update(units)

            (ads_start, ads_end, des_start, des_end) = _parse_data(sheet, row, col)
            data['branch'] = [0] * (ads_end - ads_start) + [1] * (des_end - des_start)
            for i, item in enumerate(head[1:]):
                ads_points = [sheet.cell(r, i).value for r in range(ads_start, ads_end)]
                des_points = [sheet.cell(r, i).value for r in range(des_start, des_end)]
                data[item] = ads_points + des_points

    _check(meta, data, path)

    # Set extra metadata
    meta['apparatus'] = f'BEL {meta["serialnumber"]}'

    return meta, data


def _get_header(sheet, row):
    """Return list of data headers."""
    return [
        sheet.cell(row, rcol).value.strip()
        for rcol in range(sheet.ncols)
        if sheet.cell(row, rcol).value.strip() != ''
    ]


def _parse_data(sheet, row, col):
    """Return start and stop points for adsorption and desorption."""
    rowc = 1

    # Check for adsorption branch
    if sheet.cell(row + rowc, col).value == 'ADS':
        ads_start_row = row + rowc + 1
        ads_final_row = ads_start_row

    point = sheet.cell(ads_final_row, col).value

    while point != 'DES':
        ads_final_row += 1
        point = sheet.cell(ads_final_row, col).value

    if sheet.cell(ads_final_row, col).value == 'DES':
        des_start_row = ads_final_row + 1
        des_final_row = des_start_row

    if des_final_row < sheet.nrows:

        point = sheet.cell(des_final_row, col).value

        while str(point).strip():
            des_final_row += 1
            if des_final_row < sheet.nrows:
                point = sheet.cell(des_final_row, col).value
            else:
                point = None

    return (ads_start_row, ads_final_row, des_start_row, des_final_row)


def _handle_bel_xl_timedelta(val):
    seconds = int(round(val * 86400.0))
    minutes, second = divmod(seconds, 60)
    hour, minute = divmod(minutes, 60)
    return f'{hour}:{minute}:{second}'
