"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
const vpc_1 = require("./vpc");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets || [{ subnetType: vpc_1.SubnetType.PRIVATE }];
        const routeTableIds = util_1.allRouteTableIds(...subnets.map(s => props.vpc.selectSubnets(s)));
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        const subnets = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        const subnetIds = subnets.subnetIds;
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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