"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Either an IPv4 or an IPv6 CIDR
 *
 * @experimental
 */
class AclCidr {
    /**
     * An IP network range in CIDR notation (for example, 172.16.0.0/24).
     */
    static ipv4(ipv4Cidr) {
        return new AclCidrImpl({
            cidrBlock: ipv4Cidr,
        });
    }
    /**
     * The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0)
     */
    static anyIpv4() {
        return AclCidr.ipv4('0.0.0.0/0');
    }
    /**
     * An IPv6 network range in CIDR notation (for example, 2001:db8::/48)
     */
    static ipv6(ipv6Cidr) {
        return new AclCidrImpl({
            ipv6CidrBlock: ipv6Cidr,
        });
    }
    /**
     * The CIDR containing all IPv6 addresses (i.e., ::/0)
     */
    static anyIpv6() {
        return AclCidr.ipv6('::/0');
    }
}
exports.AclCidr = AclCidr;
class AclCidrImpl extends AclCidr {
    constructor(config) {
        super();
        this.config = config;
    }
    toCidrConfig() {
        return this.config;
    }
}
/**
 * The traffic that is configured using a Network ACL entry
 *
 * @experimental
 */
class AclTraffic {
    /**
     * Apply the ACL entry to all traffic
     */
    static allTraffic() {
        return new AclTrafficImpl({
            protocol: -1,
        });
    }
    /**
     * Apply the ACL entry to ICMP traffic of given type and code
     */
    static icmp(props) {
        return new AclTrafficImpl({
            protocol: 1,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to ICMPv6 traffic of given type and code
     *
     * Requires an IPv6 CIDR block.
     */
    static icmpv6(props) {
        return new AclTrafficImpl({
            protocol: 58,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port
     */
    static tcpPort(port) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port range
     */
    static tcpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port
     */
    static udpPort(port) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port range
     */
    static udpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
}
exports.AclTraffic = AclTraffic;
class AclTrafficImpl extends AclTraffic {
    constructor(config) {
        super();
        this.config = config;
    }
    toTrafficConfig() {
        return this.config;
    }
}
//# sourceMappingURL=data:application/json;base64,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