import * as iam from '@aws-cdk/aws-iam';
import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
import { Connections, IConnectable } from './connections';
import { CfnInstance } from './ec2.generated';
import { InstanceType } from './instance-types';
import { IMachineImage, OperatingSystemType } from './machine-image';
import { ISecurityGroup } from './security-group';
import { UserData } from './user-data';
import { BlockDevice } from './volume';
import { IVpc, SubnetSelection } from './vpc';
export interface IInstance extends IResource, IConnectable, iam.IGrantable {
    /**
     * The instance's ID
     *
     * @attribute
     */
    readonly instanceId: string;
    /**
     * The availability zone the instance was launched in
     *
     * @attribute
     */
    readonly instanceAvailabilityZone: string;
    /**
     * Private DNS name for this instance
     * @attribute
     */
    readonly instancePrivateDnsName: string;
    /**
     * Private IP for this instance
     *
     * @attribute
     */
    readonly instancePrivateIp: string;
    /**
     * Publicly-routable DNS name for this instance.
     *
     * (May be an empty string if the instance does not have a public name).
     *
     * @attribute
     */
    readonly instancePublicDnsName: string;
    /**
     * Publicly-routable IP  address for this instance.
     *
     * (May be an empty string if the instance does not have a public IP).
     *
     * @attribute
     */
    readonly instancePublicIp: string;
}
/**
 * Properties of an EC2 Instance
 */
export interface InstanceProps {
    /**
     * Name of SSH keypair to grant access to instance
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * Where to place the instance within the VPC
     *
     * @default - Private subnets.
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * In which AZ to place the instance within the VPC
     *
     * @default - Random zone.
     */
    readonly availabilityZone?: string;
    /**
     * Whether the instance could initiate connections to anywhere by default.
     * This property is only used when you do not provide a security group.
     *
     * @default true
     */
    readonly allowAllOutbound?: boolean;
    /**
     * The length of time to wait for the resourceSignalCount
     *
     * The maximum value is 43200 (12 hours).
     *
     * @default Duration.minutes(5)
     */
    readonly resourceSignalTimeout?: Duration;
    /**
     * VPC to launch the instance in.
     */
    readonly vpc: IVpc;
    /**
     * Security Group to assign to this instance
     *
     * @default - create new security group
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * Type of instance to launch
     */
    readonly instanceType: InstanceType;
    /**
     * AMI to launch
     */
    readonly machineImage: IMachineImage;
    /**
     * Specific UserData to use
     *
     * The UserData may still be mutated after creation.
     *
     * @default - A UserData object appropriate for the MachineImage's
     * Operating System is created.
     */
    readonly userData?: UserData;
    /**
     * An IAM role to associate with the instance profile assigned to this Auto Scaling Group.
     *
     * The role must be assumable by the service principal `ec2.amazonaws.com`:
     *
     * @example
     * const role = new iam.Role(this, 'MyRole', {
     *   assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
     * });
     *
     * @default - A role will automatically be created, it can be accessed via the `role` property
     */
    readonly role?: iam.IRole;
    /**
     * The name of the instance
     *
     * @default - CDK generated name
     */
    readonly instanceName?: string;
    /**
     * Specifies whether to enable an instance launched in a VPC to perform NAT.
     * This controls whether source/destination checking is enabled on the instance.
     * A value of true means that checking is enabled, and false means that checking is disabled.
     * The value must be false for the instance to perform NAT.
     *
     * @default true
     */
    readonly sourceDestCheck?: boolean;
    /**
     * Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.
     *
     * Each instance that is launched has an associated root device volume,
     * either an Amazon EBS volume or an instance store volume.
     * You can use block device mappings to specify additional EBS volumes or
     * instance store volumes to attach to an instance when it is launched.
     *
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
     *
     * @default - Uses the block device mapping of the AMI
     */
    readonly blockDevices?: BlockDevice[];
    /**
     * Defines a private IP address to associate with an instance.
     *
     * Private IP should be available within the VPC that the instance is build within.
     *
     * @default - no association
     */
    readonly privateIpAddress?: string;
}
/**
 * This represents a single EC2 instance
 */
export declare class Instance extends Resource implements IInstance {
    /**
     * The type of OS the instance is running.
     */
    readonly osType: OperatingSystemType;
    /**
     * Allows specify security group connections for the instance.
     */
    readonly connections: Connections;
    /**
     * The IAM role assumed by the instance.
     */
    readonly role: iam.IRole;
    /**
     * The principal to grant permissions to
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * UserData for the instance
     */
    readonly userData: UserData;
    /**
     * the underlying instance resource
     */
    readonly instance: CfnInstance;
    /**
     * @attribute
     */
    readonly instanceId: string;
    /**
     * @attribute
     */
    readonly instanceAvailabilityZone: string;
    /**
     * @attribute
     */
    readonly instancePrivateDnsName: string;
    /**
     * @attribute
     */
    readonly instancePrivateIp: string;
    /**
     * @attribute
     */
    readonly instancePublicDnsName: string;
    /**
     * @attribute
     */
    readonly instancePublicIp: string;
    private readonly securityGroup;
    private readonly securityGroups;
    constructor(scope: Construct, id: string, props: InstanceProps);
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup: ISecurityGroup): void;
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands: string[]): void;
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Apply CloudFormation update policies for the instance
     */
    private applyUpdatePolicies;
}
