# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['feedsearch_crawler',
 'feedsearch_crawler.crawler',
 'feedsearch_crawler.feed_spider']

package_data = \
{'': ['*']}

install_requires = \
['aiodns>=2.0.0,<3.0.0',
 'aiohttp>=3.7.4,<4.0.0',
 'beautifulsoup4>=4.9.3,<5.0.0',
 'brotlipy>=0.7.0,<0.8.0',
 'cchardet>=2.1.7,<3.0.0',
 'feedparser>=6.0.10,<7.0.0',
 'python-dateutil>=2.8.1,<3.0.0',
 'uvloop>=0.15.2,<0.16.0',
 'w3lib>=1.22.0,<2.0.0',
 'yarl>=1.6.3,<2.0.0']

setup_kwargs = {
    'name': 'feedsearch-crawler',
    'version': '1.0.3',
    'description': 'Search sites for RSS, Atom, and JSON feeds',
    'long_description': '# Feedsearch Crawler\n[![PyPI](https://img.shields.io/pypi/v/feedsearch-crawler.svg)](https://pypi.org/project/feedsearch-crawler/)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/feedsearch-crawler.svg)\n![PyPI - License](https://img.shields.io/pypi/l/feedsearch-crawler.svg)\n\nFeedsearch Crawler is a Python library for searching websites for [RSS](https://en.wikipedia.org/wiki/RSS), [Atom](https://en.wikipedia.org/wiki/Atom_(Web_standard)), and [JSON](https://jsonfeed.org/) feeds.\n\nIt is a continuation of my work on [Feedsearch](https://github.com/DBeath/feedsearch), which is itself a continuation of the work done by [Dan Foreman-Mackey](http://dfm.io/) on [Feedfinder2](https://github.com/dfm/feedfinder2), which in turn is based on [feedfinder](http://www.aaronsw.com/2002/feedfinder/) - originally written by [Mark Pilgrim](http://en.wikipedia.org/wiki/Mark_Pilgrim_(software_developer)) and subsequently maintained by\n[Aaron Swartz](http://en.wikipedia.org/wiki/Aaron_Swartz) until his untimely death.\n\nFeedsearch Crawler differs with all of the above in that it is now built as an asynchronous [Web crawler](https://en.wikipedia.org/wiki/Web_crawler) for [Python 3.7](https://www.python.org/downloads/release/python-370/) and above, using [asyncio](https://docs.python.org/3/library/asyncio.html) and [aiohttp](https://aiohttp.readthedocs.io/en/stable/), to allow much more rapid scanning of possible feed urls.\n\nAn implementation using this library to provide a public Feed Search API is available at https://feedsearch.dev\n\nPull requests and suggestions are welcome.\n\n## Installation\nThe library is available on [PyPI](https://pypi.org/project/feedsearch-crawler/):\n\n```\npip install feedsearch-crawler\n```\n\nThe library requires Python 3.7+.\n\n## Usage\nFeedsearch Crawler is called with the single function ``search``:\n\n``` python\n>>> from feedsearch_crawler import search\n>>> feeds = search(\'xkcd.com\')\n>>> feeds\n[FeedInfo(\'https://xkcd.com/rss.xml\'), FeedInfo(\'https://xkcd.com/atom.xml\')]\n>>> feeds[0].url\nURL(\'https://xkcd.com/rss.xml\')\n>>> str(feeds[0].url)\n\'https://xkcd.com/rss.xml\'\n>>> feeds[0].serialize()\n{\'url\': \'https://xkcd.com/rss.xml\', \'title\': \'xkcd.com\', \'version\': \'rss20\', \'score\': 24, \'hubs\': [], \'description\': \'xkcd.com: A webcomic of romance and math humor.\', \'is_push\': False, \'self_url\': \'\', \'favicon\': \'https://xkcd.com/s/919f27.ico\', \'content_type\': \'text/xml; charset=UTF-8\', \'bozo\': 0, \'site_url\': \'https://xkcd.com/\', \'site_name\': \'xkcd: Chernobyl\', \'favicon_data_uri\': \'\', \'content_length\': 2847}\n```\n\nIf you are already running in an [asyncio event loop](https://docs.python.org/3/library/asyncio-eventloop.html), then you can import and await ``search_async`` instead. The ``search`` function is only a wrapper that runs ``search_async`` in a new asyncio event loop.\n\n``` python\nfrom feedsearch_crawler import search_async\n\nfeeds = await search_async(\'xkcd.com\')\n```\n\nA search will always return a list of *FeedInfo* objects, each of which will always have a *url* property, which is a [URL](https://yarl.readthedocs.io/en/latest/api.html) object that can be decoded to a string with ``str(url)``.\nThe returned *FeedInfo* are sorted by the *score* value from highest to lowest, with a higher score theoretically indicating a more relevant feed compared to the original URL provided. A *FeedInfo* can also be serialized to a JSON compatible dictionary by calling it\'s ``.serialize()`` method.\n\nThe crawl logs can be accessed with:\n\n``` python\nimport logging\n\nlogger = logging.getLogger("feedsearch_crawler")\n```\n\nFeedsearch Crawler also provides a handy function to output the returned feeds as an [OPML](https://en.wikipedia.org/wiki/OPML) subscription list, encoded as a UTF-8 bytestring. \n\n``` python\nfrom feedsearch_crawler import output_opml\n\noutput_opml(feeds).decode()\n```\n\n## Search Arguments\n``search`` and ``search_async`` take the following arguments:\n\n``` python\nsearch(\n    url: Union[URL, str, List[Union[URL, str]]],\n    crawl_hosts: bool=True,\n    try_urls: Union[List[str], bool]=False,\n    concurrency: int=10,\n    total_timeout: Union[float, aiohttp.ClientTimeout]=10,\n    request_timeout: Union[float, aiohttp.ClientTimeout]=3,\n    user_agent: str="Feedsearch Bot",\n    max_content_length: int=1024 * 1024 * 10,\n    max_depth: int=10,\n    headers: dict={"X-Custom-Header": "Custom Header"},\n    favicon_data_uri: bool=True,\n    delay: float=0\n)\n```\n\n- **url**: *Union[str, List[str]]*: The initial URL or list of URLs at which to search for feeds. You may also provide [URL](https://yarl.readthedocs.io/en/latest/api.html) objects.\n- **crawl_hosts**: *bool*: (default True): An optional argument to add the site host origin URL to the list of initial crawl URLs. (e.g. add "example.com" if crawling "example.com/path/rss.xml"). If **False**, site metadata and favicon data may not be found.\n- **try_urls**: *Union[List[str], bool]*: (default False): An optional list of URL paths to query for feeds. Takes the origins of the *url* parameter and appends the provided paths. If no list is provided, but *try_urls* is **True**, then a list of common feed locations will be used.\n- **concurrency**: *int*: (default 10): An optional argument to specify the maximum number of concurrent HTTP requests.\n- **total_timeout**: *float*: (default 30.0): An optional argument to specify the time this function may run before timing out.\n- **request_timeout**: *float*: (default 3.0): An optional argument that controls how long before each individual HTTP request times out.\n- **user_agent**: *str*: An optional argument to override the default User-Agent header.\n- **max_content_length**: *int*: (default 10Mb): An optional argument to specify the maximum size in bytes of each HTTP Response.\n- **max_depth**: *int*: (default 10): An optional argument to limit the maximum depth of requests while following urls.\n- **headers**: *dict*: An optional dictionary of headers to pass to each HTTP request.\n- **favicon_data_uri**: *bool*: (default True): Optionally control whether to fetch found favicons and return them as a Data Uri.\n- **delay**: *float*: (default 0.0): An optional argument to delay each HTTP request by the specified time in seconds. Used in conjunction with the concurrency setting to avoid overloading sites.\n\n## FeedInfo Values\nIn addition to the *url*, FeedInfo objects may have the following values:\n\n- **bozo**: *int*: Set to 1 when feed data is not well formed or may not be a feed. Defaults 0.\n- **content_length**: *int*: Current length of the feed in bytes.\n- **content_type**: *str*: [Content-Type](https://en.wikipedia.org/wiki/Media_type) value of the returned feed.\n- **description**: *str*: Feed description.\n- **favicon**: *URL*: [URL](https://yarl.readthedocs.io/en/latest/api.html) of feed or site [Favicon](https://en.wikipedia.org/wiki/Favicon).\n- **favicon_data_uri**: *str*: [Data Uri](https://en.wikipedia.org/wiki/Data_URI_scheme) of Favicon.\n- **hubs**: *List[str]*: List of [Websub](https://en.wikipedia.org/wiki/WebSub) hubs of feed if available.\n- **is_podcast**: *bool*: True if the feed contains valid [podcast](https://en.wikipedia.org/wiki/Podcast) elements and enclosures.\n- **is_push**: *bool*: True if feed contains valid Websub data.\n- **item_count**: *int*: Number of items currently in the feed.\n- **last_updated**: *datetime*: Date of the latest published entry.\n- **score**: *int*: Computed relevance of feed url value to provided URL. May be safely ignored.\n- **self_url**: *URL*: *ref="self"* value returned from feed links. In some cases may be different from feed url.\n- **site_name**: *str*: Name of feed\'s website.\n- **site_url**: *URL*: [URL](https://yarl.readthedocs.io/en/latest/api.html) of feed\'s website.\n- **title**: *str*: Feed Title.\n- **url**: *URL*: [URL](https://yarl.readthedocs.io/en/latest/api.html) location of feed.\n- **velocity**: *float*: Mean number of items per day in the feed at the current time.\n- **version**: *str*: Feed version [XML values](https://pythonhosted.org/feedparser/version-detection.html),\n  or [JSON feed](https://jsonfeed.org/version/1).\n',
    'author': 'David Beath',
    'author_email': 'davidgbeath@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/DBeath/feedsearch-crawler',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
